/*
** mk_syslevel.c
**
** make syslevel file - creates a SYSLEVEL file for OS/2
**
** (c) 1997 by Dirk Ohme
*/

/*---| includes |-------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "mk_syslevel.h"



/*------------------------------------------------------------------------------
 * SysLevelAlloc()
 * allocates a new SYSLEVEL structure
 * input:  --none--
 * output: PSyslevel    == NULL         = error
 *                      != NULL         = pointer to initialized structure
 *------------------------------------------------------------------------------
 */
PSyslevel SysLevelAlloc( void )
{
        register PSyslevel      p;

        /*---| allocate memory |---*/
        if( NULL == (p=malloc(sizeof(Syslevel))) )
                return NULL;

        /*---| initialize memory |---*/
        memset( p, 0, sizeof(Syslevel) );
        p->usHdrSignature = 0xFFFF;
        strcpy( p->szHdr, "SYSLEVEL" );
        p->fType0         = 0x01;
        p->fType1         = 0x25;
        strcpy( p->szPrevCSD, "-------" );
        p->szPrevCSD[7] = '_';
        strcpy( p->szCurrCSD, "-------" );
        p->szCurrCSD[7] = '_';

        /*---| return pointer |---*/
        return p;

} /* SysLevelAlloc() */



/*------------------------------------------------------------------------------
 * SysLevelFree()
 * frees a SYSLEVEL structure
 * input:  PSyslevel p                  - pointer to structure
 * output: int  ==  0                   = okay
 *              == -1                   = parameter error
 *------------------------------------------------------------------------------
 */
int SysLevelFree( PSyslevel p )
{
        /*---| check parameter |---*/
        if( NULL == p )
                return -1;

        /*---| free structure |---*/
        free( p );

        /*---| return success |---*/
        return 0;

} /* SysLevelFree() */



/*------------------------------------------------------------------------------
 * SysLevelRead()
 * reads an existing SYSLEVEL file into a given structure
 * input:  PSyslevel p                  - pointer to structure
 *         char *pszFilename            - name of SYSLEVEL file
 * output: int  ==  0                   = okay
 *              == -1                   = parameter error
 *              == -2                   = file open error
 *              == -3                   = file read error
 *------------------------------------------------------------------------------
 */
int SysLevelRead( PSyslevel p, char *pszFilename )
{
        register FILE   *phFile;

        /*---| check parameter |---*/
        if( NULL == p || NULL == pszFilename )
                return -1;

        /*---| try to open file |---*/
        if( NULL == (phFile=fopen(pszFilename, "rb")) )
                return -2;

        /*---| read in structure |---*/
        if( sizeof(Syslevel) != fread(p, 1, sizeof(Syslevel), phFile) )
        {
                /*---| file read error |---*/
                fclose( phFile );
                return -3;
        }

        /*---| return success |---*/
        fclose( phFile );
        return 0;

} /* SysLevelRead() */



/*------------------------------------------------------------------------------
 * SysLevelSetVersion()
 * sets the version information in the structure
 * input:  PSyslevel p                  - pointer to SYSLEVEL structure
 *         char *pszTitle               - title string
 *         int iMajorVersion            - main version number
 *         int iMinorVersion            - minor version number
 *         int iRevision                - revision number
 *         char fPatchLevel             - patch level as character
 * output: int  ==  0                   = no error
 *              == -1                   = parameter error
 *------------------------------------------------------------------------------
 */
int SysLevelSetVersion( PSyslevel p, char *pszTitle, int iMajorVersion,
                        int iMinorVersion, int iRevision, char fPatchLevel )
{
        char     szCSD[16];
        char    *psz;

        /*---| check parameter |---*/
        if( NULL == p || NULL == pszTitle )
                return -1;
        if( '\0' == *pszTitle )
                pszTitle = "<unknown>";

        /*---| set version |---*/
        p->iVersion = ((iMajorVersion & 0x0F) << 4) | (iMinorVersion & 0x0F);

        /*---| copy program information string |---*/
        strncpy( p->szName, pszTitle, 80 );

        /*---| set component ID |---*/
        sprintf( p->szCompID, " %d.%02d.%d %c",
                 iMajorVersion, iMinorVersion,
                 iRevision,     fPatchLevel );

        /*---| set new CSD level |---*/
        memcpy( p->szPrevCSD, p->szCurrCSD, 8 );
        sprintf( szCSD, "%02X%02X%02X%c", (iMajorVersion & 0xFF),
                 (iMinorVersion & 0xFF), (iRevision & 0xFF), fPatchLevel );
        strncpy( p->szCurrCSD, szCSD, 8 );
        p->szCurrCSD[7] = '_';

        /*---| return success |---*/
        return 0;

} /* SysLevelSetVersion() */



/*------------------------------------------------------------------------------
 * SysLevelWrite()
 * writes the given structure to a SYSLEVEL file
 * input:  PSyslevel p                  - pointer to structure
 *         char *pszFilename            - name of SYSLEVEL file
 * output: int  ==  0                   = okay
 *              == -1                   = parameter error
 *              == -2                   = file create error
 *              == -3                   = file write error
 *------------------------------------------------------------------------------
 */
int SysLevelWrite( PSyslevel p, char *pszFilename )
{
        register FILE   *phFile;

        /*---| check parameter |---*/
        if( NULL == p || NULL == pszFilename )
                return -1;

        /*---| try to open file |---*/
        if( NULL == (phFile=fopen(pszFilename, "wb")) )
                return -2;

        /*---| read in structure |---*/
        if( sizeof(Syslevel) != fwrite(p, 1, sizeof(Syslevel), phFile) )
        {
                /*---| file read error |---*/
                fclose( phFile );
                return -3;
        }

        /*---| return success |---*/
        fclose( phFile );
        return 0;

} /* SysLevelWrite() */



/*===| end of 'mk_syslevel.c' |===============================================*/
