//
// Sendmail.java
//
// a simple class abling mail sending via SMTP protocol
//
// (c) 1997 by Dirk Ohme
//

//---| package |---
package iODBC.tools;

//---| imports |---
import java.io.*;
import java.net.*;
import java.lang.*;

//---| class Sendmail |---
public class Sendmail {
        /*
        ** constants
        */
        private static final int        SMTPport        = 25;
        private static final String     defaultSMTPhost = "mail";

        /*
        ** private data
        */
        private Socket                  socket          = null;
        private DataInputStream         streamIn        = null;
        private PrintStream             streamOut       = null;

        /*
        ** constructors
        */
        public Sendmail( String smtpHost, int smtpPort, String receiver,
                         String subject ) throws IOException {
                initSMTP( smtpHost, smtpPort, receiver, subject );
        }
        public Sendmail( String smtpHost, String receiver, String subject )
            throws IOException {
                initSMTP( smtpHost, SMTPport, receiver, subject );
        }
        public Sendmail( String receiver, String subject )
            throws IOException {

                String remoteHost = InetAddress.getLocalHost().toString();
                int idx = remoteHost.indexOf( '/' );
                if( 0 < idx ) {
                        remoteHost = remoteHost.substring( idx + 1 );
                }
                initSMTP( remoteHost, SMTPport, receiver, subject );
        }

        /*
        ** destructors
        */
        public void finalize() throws IOException {
                //---| close SMTP channel |---
                close();
        }

        /*
        ** private methods
        */
        //---| initialize SMTP connection |---
        private void initSMTP( String smtpHost, int smtpPort, String receiver,
                               String subject ) throws IOException {
                //---| connect to SMTP host |---
                try {
                        socket = new Socket( smtpHost, smtpPort );
                } catch( Exception eFirst ) {
                        try {
                                socket = new Socket( defaultSMTPhost, SMTPport );
                        } catch( Exception eSecond ) {
                                throw new IOException(
                                        "SMTP: neither can connect to " +
                                        smtpHost + "/" + smtpPort + ", nor to " +
                                        defaultSMTPhost + "/" + smtpPort + " !"
                                );
                        }
                }

                //---| create I/O channels |---
                streamIn  = new DataInputStream( socket.getInputStream() );
                streamOut = new PrintStream( socket.getOutputStream() );

                //---| check for origin |---
                String sender = System.getProperty( "user.name" );

                //---| initiate connection |---
                sendSMTP( "HELO" );
                waitForOkayOnSMTP();
                sendSMTP( "MAIL FROM: " + sender );
                waitForOkayOnSMTP();
                sendSMTP( "RCPT TO: " + receiver );
                waitForOkayOnSMTP();
                sendSMTP( "DATA" );
                waitForOkayOnSMTP();
                sendSMTP( "Subject: " + subject + "\n" );
        }

        //---| send data |---
        private void sendSMTP( String data ) throws IOException {
                //---| send data to stream, then flush |---
                streamOut.println( data );
                streamOut.flush();
        }

        //---| wait for okay on channel |---
        private void waitForOkayOnSMTP() throws IOException {
                //---| wait a moment |---
                try {
                        do {
                                Thread.sleep( 200 );
                        } while( 0 == streamIn.available() );
                } catch( Exception e ) {
                        /*---| do nothing |---*/
                }

                //---| get response message |---
                String s = streamIn.readLine();
                if( null == s || 0 == s.length() ||
                    ( '2' != s.charAt(0) &&
                      '3' != s.charAt(0) ) ) {
                        //---| throw Exception |---
                        throw new IOException(
                                "SMTP error: " + s
                        );
                }
        }

        /*
        ** methods
        */
        //---| close SMTP channel |---
        public void close() throws IOException {

                IOException smtpException = null;

                //---| end message |---
                try {
                        sendSMTP( "\n." );
                        waitForOkayOnSMTP();
                } catch( IOException e ) {
                        smtpException = e;
                }

                //---| send termination code |---
                sendSMTP( "QUIT" );

                //---| terminate connection |---
                socket.close();
                streamOut = null;
                streamIn  = null;
                socket    = null;

                //---| raise exception if necessary |---
                if( null != smtpException ) throw smtpException;
        }

        //---| send a line to the SMTP channel |---
        public void send( String line ) throws IOException {
                //---| send line |---
                sendSMTP( line );
        }
        public void send() throws IOException {
                //---| send line |---
                sendSMTP( "" );
        }

        //---| testing routine |---
        public static void main( String argv[] ) {

                iODBC.tools.Sendmail obj = null;

                //---| try to send a test mail to the current user |---
                try {
                        //---| display status |---
                        String receiver = System.getProperty( "user.name" );
                        System.out.println( "initialization" );
                        System.out.println(
                                "  receiver ='" + receiver + "'"
                        );

                        //---| create channel |---
                        obj = new iODBC.tools.Sendmail(
                                receiver, "Testing Mail"
                        );

                        //---| display status |---
                        System.out.println( "sending mail" );

                        //---| ouput some info |---
                        obj.send( "First Line" );
                        obj.send( "Second Line" );
                        obj.send( "Third Line" );
                        obj.send();
                        obj.send( "Last Line" );

                } catch( Exception e ) {
                        //---| display error message |---
                        System.out.println( "!!! " + e.getMessage() );
                }

                //---| close connection |---
                try {
                        //---| display status |---
                        System.out.println( "closing" );

                        //---| close channel |---
                        if( null != obj ) {
                                obj.close();
                                obj = null;
                        }

                } catch( Exception e ) {
                        //---| display error message |---
                        System.out.println( "!!! " + e.getMessage() );
                }

                //---| display status |---
                System.out.println( "done." );

        }

} /*---| class Sendmail |---*/

//===| end of file |===
