//
// JdbcOdbcStatement.java
//
// JDBC-ODBC-bridge, statement handling
//
// (c) 1997 by Dirk Ohme, all rights reserved
//

//---| package |---------------------------------------------------------------
package iODBC.sql;

//---| imports |---------------------------------------------------------------
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import iODBC.sql.dll.J2ODBC;
import iODBC.sql.dll.J2ODBCException;
import iODBC.sql.dll.J2ODBCWarning;

//---| class JdbcOdbcStatement |-----------------------------------------------
public class JdbcOdbcStatement implements java.sql.Statement {
        /*
        ** data
        */
        private J2ODBC                 db      = null;  // ODBC object
        private Connection             connect = null;  // current DB connection
        private String                 sqlStmt = null;  // SQL statement
        private java.sql.ResultSet     currRS  = null;  // current result set
        private boolean           stmtExecuted = false; // true, if a SQL
                                                        // statement has been
                                                        // executed

        /*
        ** constructor
        */
        public JdbcOdbcStatement( Connection dbConnection, J2ODBC odbcObj )
                throws SQLException {
                //---| initialize |---
                stmtExecuted = false;

                //---| store data |---
                db      = new iODBC.sql.dll.J2ODBC( odbcObj );
                connect = dbConnection;

                //---| allocate new statement handle |---
                try {
                        db.allocStmt();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        /*
        ** methods
        */
        //---| ... execute a query |---
        public java.sql.ResultSet executeQuery( String sql )
                throws SQLException {
                //---| execute, then return result set |---
                if( execute(sql) ) {
                        return getResultSet();
                } else {
                        return null;
                }
        }

        //---| ... execute a query with no result set |---
        public int executeUpdate( String sql ) throws SQLException {
                //---| execute, then return number of rows affected |---
                execute( sql );
                return getUpdateCount();
        }

        //---| ... close statement |---
        public void close() throws SQLException {
                //---| close statement handle |---
                try {
                        db.freeStmt( db.DROP );
                        db = null;
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check maximum size of a column |---
        public int getMaxFieldSize() throws SQLException {
                //---| get maximum size of a column |---
                try {
                        return db.getStmtOptionNum( db.MAX_LENGTH );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... set maximum length of a column |---
        public void setMaxFieldSize( int max ) throws SQLException {
                //---| set maximum size of a column |---
                try {
                        db.setStmtOptionNum( db.MAX_LENGTH, max );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check maximum rows in a result set |---
        public int getMaxRows() throws SQLException {
                //---| get maximum number of rows |---
                try {
                        return db.getStmtOptionNum( db.MAX_ROWS );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... set maximum rows in a result set |---
        public void setMaxRows( int max ) throws SQLException {
                //---| set maximum number of rows |---
                try {
                        db.setStmtOptionNum( db.MAX_ROWS, max );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... set automatic scan for escape sequences |---
        public void setEscapeProcessing( boolean enable )
                throws SQLException {
                //---| set escape processing |---
                try {
                        db.setStmtOptionNum(
                                db.NOSCAN, (enable) ? db.FALSE : db.TRUE
                        );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get timeout on queries |---
        public int getQueryTimeout() throws SQLException {
                //---| get timeout value |---
                try {
                        return db.getStmtOptionNum( db.QUERY_TIMEOUT );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... set timeout on queries |---
        public void setQueryTimeout( int seconds ) throws SQLException {
                //---| set timeout value |---
                try {
                        db.setStmtOptionNum( db.QUERY_TIMEOUT, seconds );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... cancel current transaction |---
        public void cancel() throws SQLException {
                //---| cancel statement |---
                try {
                        db.cancel();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get latest waring |---
        public SQLWarning getWarnings() throws SQLException {
                //---| ??? don't know how to handle this ??? |---
                return null;
        }

        //---| ... clear latest warning |---
        public void clearWarnings() throws SQLException {
                //---| ??? don't know how to handle this ??? |---
        }

        //---| ... set cursor name |---
        public void setCursorName( String name ) throws SQLException {
                //---| set cursor name |---
                try {
                        db.setCursorName( name );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get cursor name |---
        public String getCursorName() throws SQLException {
                //---| get cursor name |---
                try {
                        return db.getCursorName();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... execute query, check for result set |---
        public boolean execute( String sql ) throws SQLException {
                //---| execute query |---
                try {
                        //---| close data on previous executed statement |---
                        if( stmtExecuted ) {
                                db.freeStmt( db.CLOSE );
                        }

                        //---| execute new query |---
                        db.execDirect( sql );

                        //---| set flag, return status |---
                        stmtExecuted = true;
                        return (0 < db.numResultCols());

                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... return a result set |---
        public java.sql.ResultSet getResultSet() throws SQLException {
                //---| return result set |---
                currRS = new iODBC.sql.JdbcOdbcResultSet( db );
                return currRS;
        }

        //---| ... check number of rows affected |---
        public int getUpdateCount() throws SQLException {
                //---| get row count |---
                try {
                        return db.rowCount();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check for next result set |---
        public boolean getMoreResults() throws SQLException {
                //---| check for more results |---
                try {
                        if( db.moreResults() ) {
                                //---| close old result set |---
                                if( null != currRS ) {
                                        currRS.close();
                                        currRS = null;
                                }
                                //---| return success |---
                                return true;
                        }

                        //---| return 'no more result sets' |---
                        return false;

                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

} /* class JdbcOdbcStatement */

//===| end of file |===========================================================
