//
// JdbcOdbcResultSetMetaData.java
//
// JDBC-ODBC-bridge, result sets meta data
//
// (c) 1997 by Dirk Ohme, all rights reserved
//

//---| package |---------------------------------------------------------------
package iODBC.sql;

//---| imports |---------------------------------------------------------------
import java.sql.SQLException;
import java.sql.Types;
import iODBC.sql.dll.J2ODBC;
import iODBC.sql.dll.J2ODBCException;
import iODBC.sql.dll.J2ODBCWarning;

//---| class JdbcOdbcResultSetMetaData |---------------------------------------
public class JdbcOdbcResultSetMetaData implements java.sql.ResultSetMetaData {
        /*
        ** definitions
        */
        public static final int columnNoNulls         = 0;
        public static final int columnNullable        = 1;
        public static final int columnNullableUnknown = 2;

        /*
        ** data
        */
        private J2ODBC     db      = null;                 // ODBC object

        /*
        ** constructor(s)
        */
        public JdbcOdbcResultSetMetaData( J2ODBC newDb ) {
                //---| set ODBC connection |---
                db = newDb;
        }

        /*
        ** methods
        */
        //---| ... get number of resulting columns |---
        public int getColumnCount() throws SQLException {
                //---| return column count |---
                try {
                        return db.numResultCols();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check whether column increments automatically or not |---
        public boolean isAutoIncrement( int columnIndex )
                throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_AUTO_INCREMENT );
                        return (db.TRUE == db.colAttributesValue());
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check whether column is case sensitive or not |---
        public boolean isCaseSensitive( int columnIndex )
                throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_CASE_SENSITIVE );
                        return (db.TRUE == db.colAttributesValue());
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check whether column can be searched or not |---
        public boolean isSearchable( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_SEARCHABLE );
                        return (db.TRUE == db.colAttributesValue());
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check whether column contains a cash value or not |---
        public boolean isCurrency( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_MONEY );
                        return (db.TRUE == db.colAttributesValue());
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check whether column can store NULLs or not |---
        public int isNullable( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_NULLABLE );
                        return (db.TRUE == db.colAttributesValue())
                               ? columnNullable : columnNoNulls;
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check whether column contains signed number or not |---
        public boolean isSigned( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_UNSIGNED );
                        return (db.FALSE == db.colAttributesValue());
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get display size of a column |---
        public int getColumnDisplaySize( int columnIndex )
                throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_DISPLAY_SIZE );
                        return db.colAttributesValue();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get column label |---
        public String getColumnLabel( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_LABEL );
                        return db.colAttributesString();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get column name |---
        public String getColumnName( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_NAME );
                        return db.colAttributesString();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get schema name |---
        public String getSchemaName( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_SCHEMA_NAME );
                        return db.colAttributesString();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get column precision |---
        public int getPrecision( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_PRECISION );
                        return db.colAttributesValue();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get column scale |---
        public int getScale( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_SCALE );
                        return db.colAttributesValue();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get table name |---
        public String getTableName( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_TABLE_NAME );
                        return db.colAttributesString();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get catalog name |---
        public String getCatalogName( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_CATALOG_NAME );
                        return db.colAttributesString();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get column type |---
        public int getColumnType( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_TYPE );
                        switch( db.colAttributesValue() ) {
                                case db.SQL_CHAR:
                                // case db.SQL_TINYINT:
                                        return Types.CHAR;
                                case db.SQL_NUMERIC:
                                        return Types.NUMERIC;
                                case db.SQL_DECIMAL:
                                        return Types.DECIMAL;
                                case db.SQL_INTEGER:
                                // case SQL_C_LONG:
                                        return Types.INTEGER;
                                case db.SQL_SMALLINT:
                                // case db.SQL_C_SHORT:
                                        return Types.SMALLINT;
                                case db.SQL_FLOAT:
                                        return Types.FLOAT;
                                case db.SQL_REAL:
                                        return Types.REAL;
                                case db.SQL_DOUBLE:
                                        return Types.DOUBLE;
                                case db.SQL_VARCHAR:
                                        return Types.VARCHAR;
                                case db.SQL_BIT:
                                        return Types.BIT;
                                case db.SQL_DATE:
                                        return Types.DATE;
                                case db.SQL_TIME:
                                        return Types.TIME;
                                case db.SQL_TIMESTAMP:
                                        return Types.TIMESTAMP;
                                case db.SQL_BINARY:
                                        return Types.BINARY;
                                case db.SQL_LONGVARCHAR:
                                        return Types.LONGVARCHAR;
                                case db.SQL_LONGVARBINARY:
                                        return Types.LONGVARBINARY;
                                case db.SQL_VARBINARY:
                                        return Types.VARBINARY;
                                case db.SQL_DATETIME:
                                case db.SQL_INTERVAL:
                                case db.SQL_BIT_VARYING:
                                case db.SQL_DBCHAR:
                                case db.SQL_BLOB:
                                case db.SQL_BLOB_LOCATOR:
                                case db.SQL_CLOB:
                                case db.SQL_CLOB_LOCATOR:
                                case db.SQL_DBCLOB:
                                case db.SQL_DBCLOB_LOCATOR:
                                case db.SQL_GRAPHIC:
                                case db.SQL_LONGVARGRAPHIC:
                                case db.SQL_VARGRAPHIC:
                                default:
                                        return Types.OTHER;
                        }
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get column type name |---
        public String getColumnTypeName( int columnIndex )
                throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_TYPE_NAME );
                        return db.colAttributesString();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check column for r/o status |---
        public boolean isReadOnly( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_UPDATABLE );
                        return (db.ATTR_READONLY == db.colAttributesValue());
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check column for r/w status |---
        public boolean isWritable( int columnIndex ) throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_UPDATABLE );
                        return (db.ATTR_WRITE == db.colAttributesValue());
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... check column for (absolute) r/w status |---
        public boolean isDefinitelyWritable( int columnIndex )
                throws SQLException {
                //---| check column |---
                try {
                        db.colAttributes( columnIndex,
                                          db.COLUMN_UPDATABLE );
                        return (db.ATTR_WRITE == db.colAttributesValue());
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

} /* class JdbcOdbcResultSetMetaData */

//===| end of file |===========================================================
