//
// JdbcOdbcResultSet.java
//
// JDBC-ODBC-bridge, result sets
//
// (c) 1997 by Dirk Ohme, all rights reserved
//

//---| package |---------------------------------------------------------------
package iODBC.sql;

//---| imports |---------------------------------------------------------------
import java.sql.Date;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import iODBC.sql.JdbcOdbcResultSetMetaData;
import iODBC.sql.dll.J2ODBC;
import iODBC.sql.dll.J2ODBCException;
import iODBC.sql.dll.J2ODBCWarning;

//---| class JdbcOdbcResultSet |-----------------------------------------------
public class JdbcOdbcResultSet implements java.sql.ResultSet {
        /*
        ** data
        */
        private J2ODBC     db      = null;                 // ODBC object
        private java.sql.ResultSetMetaData meta = null;    // meta data

        /*
        ** constructor(s)
        */
        public JdbcOdbcResultSet( J2ODBC newDb ) {
                //---| set ODBC connection |---
                db = newDb;
        }

        /*
        ** methods
        */
        //---| ... get next row of result set |---
        public boolean next() throws SQLException {
                //---| get next row |---
                try {
                        return db.fetch();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... close a result set |---
        public void close() throws SQLException {
                //---| close |---
                try {
                        db.freeStmt( db.CLOSE );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... return whether last column read was NULL or not |---
        public boolean wasNull() throws SQLException {
                //---| check for NULL |---
                try {
                        return db.getDataCheckForNull();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get string result |---
        public String getString( int columnIndex ) throws SQLException {
                //---| return column value |---
                try {
                        return db.getDataString( columnIndex );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get boolean result |---
        public boolean getBoolean( int columnIndex ) throws SQLException {
                //---| return column value |---
                try {
                        return db.getDataBoolean( columnIndex );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get byte result |---
        public byte getByte( int columnIndex ) throws SQLException {
                //---| return column value |---
                try {
                        return (byte) db.getDataInteger( columnIndex );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get short result |---
        public short getShort( int columnIndex ) throws SQLException {
                //---| return column value |---
                try {
                        return (short) db.getDataInteger( columnIndex );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get integer result |---
        public int getInt( int columnIndex ) throws SQLException {
                //---| return column value |---
                try {
                        return db.getDataInteger( columnIndex );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get long result |---
        public long getLong( int columnIndex ) throws SQLException {
                //---| return column value |---
                try {
                        return (long) db.getDataInteger( columnIndex );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get float result |---
        public float getFloat( int columnIndex ) throws SQLException {
                //---| return column value |---
                try {
                        return db.getDataFloat( columnIndex );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get double result |---
        public double getDouble( int columnIndex ) throws SQLException {
                //---| return column value |---
                try {
                        return db.getDataDouble( columnIndex );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get numeric result |---
        public Bignum getBignum( int columnIndex, int scale )
                throws SQLException {
                //---| return column value |---
                try {
                        //---| get numeric from string representation |---
                        return new Bignum( db.getDataString(columnIndex) );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get byte[] result |---
        public byte[] getBytes( int columnIndex ) throws SQLException {
                //---| get result as string |---
                String s = getString( columnIndex );
                if( null == s )
                        return null;

                //---| convert to bytes[] |---
                byte[] result = new byte[10 + s.length()];
                s.getBytes( 0, s.length(), result, 0 );
                return result;
        }

        //---| ... get date result |---
        public java.sql.Date getDate( int columnIndex ) throws SQLException {
                //---| get date by string representation |---
                try {
                        //---| get date from string representation |---
                        return Date.valueOf( db.getDataString(columnIndex) );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get time result |---
        public java.sql.Time getTime( int columnIndex ) throws SQLException {
                //---| get time by string representation |---
                try {
                        //---| get time from string representation |---
                        return Time.valueOf( db.getDataString(columnIndex) );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get time stamp result |---
        public java.sql.Timestamp getTimestamp( int columnIndex )
                throws SQLException {
                //---| get timestamp by string representation |---
                try {
                        //---| get timestamp from string representation |---
                        return Timestamp.valueOf(
                                db.getDataString(columnIndex)
                        );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get ASCII via input stream |---
        public java.io.InputStream getAsciiStream( int columnIndex )
                throws SQLException {
                //---| ??? not implemented yet ??? |---
                throw new SQLException(
                        "[J2ODBC] not implemented yet!"
                );
        }

        //---| ... get unicode via input stream |---
        public java.io.InputStream getUnicodeStream( int columnIndex )
                throws SQLException {
                //---| ??? not implemented yet ??? |---
                throw new SQLException(
                        "[J2ODBC] not implemented yet!"
                );
        }

        //---| ... get binary via input stream |---
        public java.io.InputStream getBinaryStream( int columnIndex )
                throws SQLException {
                //---| ??? not implemented yet ??? |---
                throw new SQLException(
                        "[J2ODBC] not implemented yet!"
                );
        }

        //---| ... get string result |---
        public String getString( String columnName ) throws SQLException {
                return getString( findColumn(columnName) );
        }

        //---| ... get boolean result |---
        public boolean getBoolean( String columnName ) throws SQLException {
                return getBoolean( findColumn(columnName) );
        }

        //---| ... get byte result |---
        public byte getByte( String columnName ) throws SQLException {
                return getByte( findColumn(columnName) );
        }

        //---| ... get short result |---
        public short getShort( String columnName ) throws SQLException {
                return getShort( findColumn(columnName) );
        }

        //---| ... get integer result |---
        public int getInt( String columnName ) throws SQLException {
                return getInt( findColumn(columnName) );
        }

        //---| ... get long result |---
        public long getLong( String columnName ) throws SQLException {
                return getLong( findColumn(columnName) );
        }

        //---| ... get float result |---
        public float getFloat( String columnName ) throws SQLException {
                return getFloat( findColumn(columnName) );
        }

        //---| ... get double result |---
        public double getDouble( String columnName ) throws SQLException {
                return getDouble( findColumn(columnName) );
        }

        //---| ... get numeric result |---
        public Bignum getBignum( String columnName, int scale )
                throws SQLException {
                return getBignum( findColumn(columnName), scale );
        }

        //---| ... get byte[] result |---
        public byte[] getBytes( String columnName ) throws SQLException {
                return getBytes( findColumn(columnName) );
        }

        //---| ... get date result |---
        public java.sql.Date getDate( String columnName )
                throws SQLException {
                return getDate( findColumn(columnName) );
        }

        //---| ... get time result |---
        public java.sql.Time getTime( String columnName )
                throws SQLException {
                return getTime( findColumn(columnName) );
        }

        //---| ... get time stamp result |---
        public java.sql.Timestamp getTimestamp( String columnName )
                throws SQLException {
                return getTimestamp( findColumn(columnName) );
        }

        //---| ... get ASCII via input stream |---
        public java.io.InputStream getAsciiStream( String columnName )
                throws SQLException {
                return getAsciiStream( findColumn(columnName) );
        }

        //---| ... get unicode via input stream |---
        public java.io.InputStream getUnicodeStream( String columnName )
                throws SQLException {
                return getUnicodeStream( findColumn(columnName) );
        }

        //---| ... get binary via input stream |---
        public java.io.InputStream getBinaryStream( String columnName )
                throws SQLException {
                return getBinaryStream( findColumn(columnName) );
        }

        //---| ... get latest waring |---
        public SQLWarning getWarnings() throws SQLException {
                //---| ??? don't know how to handle this ??? |---
                return null;
        }

        //---| ... clear latest warning |---
        public void clearWarnings() throws SQLException {
                //---| ??? don't know how to handle this ??? |---
        }

        //---| ... get cursor name |---
        public String getCursorName() throws SQLException {
                //---| get cursor name |---
                try {
                        return db.getCursorName();
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get meta information |---
        public ResultSetMetaData getMetaData() throws SQLException {
                //---| return meta data |---
                if( null == meta ) {
                        meta = new iODBC.sql.JdbcOdbcResultSetMetaData( db );
                }
                return meta;
        }

        //---| ... get object by index |---
        public Object getObject( int columnIndex, int targetSqlType,
                                 int scale ) throws SQLException {
                //---| return object by identified type |---
                switch( targetSqlType ) {
                        case Types.BIT:
                                return new Boolean( getBoolean(columnIndex) );
                        case Types.TINYINT:
                                return new Character(
                                        (char)getByte(columnIndex)
                                );
                        case Types.SMALLINT:
                                return new Integer( getInt(columnIndex) );
                        case Types.INTEGER:
                                return new Integer( getInt(columnIndex) );
                        case Types.BIGINT:
                                return new Long( getLong(columnIndex) );
                        case Types.FLOAT:
                                return new Float( getFloat(columnIndex) );
                        case Types.REAL:
                                return new Float( getFloat(columnIndex) );
                        case Types.DOUBLE:
                                return new Double( getDouble(columnIndex) );
                        case Types.NUMERIC:
                                return getBignum( columnIndex, scale );
                        case Types.DECIMAL:
                                return getBignum( columnIndex, scale );
                        case Types.CHAR:
                                return getString( columnIndex );
                        case Types.VARCHAR:
                                return getString( columnIndex );
                        case Types.LONGVARCHAR:
                                return getString( columnIndex );
                        case Types.DATE:
                                return getDate( columnIndex );
                        case Types.TIME:
                                return getTime( columnIndex );
                        case Types.TIMESTAMP:
                                return getTimestamp( columnIndex );
                        case Types.BINARY:
                                return getBytes( columnIndex );
                        case Types.VARBINARY:
                                return getBytes( columnIndex );
                        case Types.LONGVARBINARY:
                                return getBytes( columnIndex );
                        default:
                                return null;
                }
        }
        public Object getObject( int columnIndex, int targetSqlType )
                throws SQLException {
                return getObject( columnIndex, targetSqlType, 0 );
        }
        public Object getObject( int columnIndex ) throws SQLException {
                return getObject( columnIndex,
                                  getMetaData().getColumnType(columnIndex),
                                  0 );
        }

        //---| ... get object by column name |---
        public Object getObject( String columnName, int targetSqlType,
                                 int scale ) throws SQLException {
                return getObject( findColumn(columnName),
                                  targetSqlType, scale );
        }
        public Object getObject( String columnName, int targetSqlType )
                throws SQLException {
                return getObject( findColumn(columnName), targetSqlType, 0 );
        }
        public Object getObject( String columnName ) throws SQLException {
                //---| get column index |---
                int columnIndex = findColumn( columnName );

                //---| return object |---
                return getObject( columnIndex,
                                  getMetaData().getColumnType(columnIndex),
                                  0 );
        }

        //---| ... get the index number to a column name |---
        public int findColumn( String columnName ) throws SQLException {
                int i = 0;

                //---| find column index by name |---
                try {
                        //---| scan all resulting columns |---
                        i = db.numResultCols();
                        while( i > 0 ) {
                                //---| check column name |---
                                db.describeCol( i );
                                if(db.describeColName().equals(columnName)) {
                                        //---| return column index |---
                                        return i;
                                }

                                //---| check next column |---
                                i--;
                        }

                        //---| not found -> exception |---
                        throw new SQLException(
                                "invalid column name '" + columnName + "'"
                        );

                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

} /* class JdbcOdbcResultSet */

//===| end of file |===========================================================
