//
// JdbcOdbcConnection.java
//
// JDBC-ODBC-bridge, connection class
//
// (c) 1997 by Dirk Ohme, all rights reserved
//

//---| package |---------------------------------------------------------------
package iODBC.sql;

//---| imports |---------------------------------------------------------------
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import iODBC.sql.dll.J2ODBC;
import iODBC.sql.dll.J2ODBCException;
import iODBC.sql.dll.J2ODBCWarning;

//---| class JdbcOdbcConnection |----------------------------------------------
public class JdbcOdbcConnection implements java.sql.Connection {
        /*
        ** definitions
        */
        public static final int TRANSACTION_NONE                = 0;
        public static final int TRANSACTION_READ_UNCOMMITTED    = 1;
        public static final int TRANSACTION_READ_COMMITTED      = 2;
        public static final int TRANSACTION_REPEATABLE_READ     = 3;
        public static final int TRANSACTION_SERIALIZEABLE       = 4;

        /*
        ** data
        */
        private J2ODBC connection = null;        // connection handle to ODBC
        private String currUser   = null;        // name od database user
        private String currDB     = null;        // name of ODBC object

        /*
        ** constructors
        */
        public JdbcOdbcConnection( String user, String passwd, String db )
                throws SQLException {
                //---| store info about user and currently used database |---
                currUser = user;
                currDB   = db;

                //---| connect to ODBC |---
                try {
                        connection = new iODBC.sql.dll.J2ODBC();
                        connection.allocEnv();
                        connection.allocConnect();
                        connection.connect( db, user, passwd );

                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        /*
        ** destructors
        */
        public void finalize() throws SQLException {
                close();
        }

        /*
        ** methods
        */
        //---| ... for returning the ODBC interface |---
        public J2ODBC queryOdbcBridge() throws J2ODBCException {
                return connection;
        }

        //---| ... for creating a statement |---
        public Statement createStatement() throws SQLException {
                return new iODBC.sql.JdbcOdbcStatement( this, connection );
        }

        //---| ... for creating a prepared statement |---
        public PreparedStatement prepareStatement( String sql )
                throws SQLException {
                return new iODBC.sql.JdbcOdbcPreparedStatement(
                        this, connection, sql
                );
        }

        //---| ... for creating a prepared stored-procedure call |---
        public CallableStatement prepareCall( String sql )
                throws SQLException {
                return new iODBC.sql.JdbcOdbcCallableStatement(
                        this, connection, sql
                );
        }

        //---| ... for simplifying a SQL statement |---
        public String nativeSQL( String sql ) throws SQLException {
                //---| simplify statement |---
                try {
                        return connection.nativeSql( sql );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... for setting AUTOCOMMIT |---
        public void setAutoCommit( boolean autoCommit ) throws SQLException {
                //---| set/reset AUTOCOMMIT |---
                try {
                        connection.setConnectOptionNum(
                                connection.AUTOCOMMIT,
                                (autoCommit) ? connection.AUTOCOMMIT_ON
                                             : connection.AUTOCOMMIT_OFF
                        );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... for querying AUTOCOMMIT setting |---
        public boolean getAutoCommit() throws SQLException {
                //---| check AUTOCOMMIT |---
                try {
                        int i = connection.getConnectOptionNum(
                                connection.AUTOCOMMIT
                        );
                        return (connection.AUTOCOMMIT_ON == i);

                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... for committing all transactions |---
        public void commit() throws SQLException {
                //---| commit |---
                try {
                        connection.transact( connection.COMMIT );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... for rollback all transactions |---
        public void rollback() throws SQLException {
                //---| rollback |---
                try {
                        connection.transact( connection.ROLLBACK );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... for closing statement |---
        public void close() throws SQLException {
                Exception e = null;

                //---| close connection to database |---
                try {
                        //---| drop cursor |---
                        connection.freeStmt(connection.DROP);
                } catch( Exception newE ) {
                        /*---| do nothing |---*/
                }
                try { connection.disconnect(); } catch( Exception newE ) {
                        if( null == e ) e = newE;
                }
                try { connection.freeConnect(); } catch( Exception newE ) {
                        if( null == e ) e = newE;
                }
                try { connection.freeEnv(); } catch( Exception newE ) {
                        if( null == e ) e = newE;
                }
                connection = null;

                //---| check for exception(s) |---
                if( null != e )
                {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... for checking on an open statement |---
        public boolean isClosed() throws SQLException {
                return (connection == null);
        }

        //---| ... for returning meta data |---
        public DatabaseMetaData getMetaData() throws SQLException {
                //---| ??? not implemented ??? |---
                throw new SQLException(
                        "[JdbcOdbcConnection] not implemented yet!"
                );
                // return new iODBC.sql.JdbcOdbcDatabaseMetaData( connection );
        }

        //---| ... for setting a session read-only |---
        public void setReadOnly( boolean readOnly ) throws SQLException {
                //---| set session r/o |---
                try {
                        connection.setConnectOptionNum(
                                connection.ACCESS_MODE,
                                (readOnly) ? connection.MODE_READ_ONLY
                                           : connection.MODE_READ_WRITE
                        );
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... for checking whether a session is read-only or not |---
        public boolean isReadOnly() throws SQLException {
                //---| check AUTOCOMMIT |---
                try {
                        int i = connection.getConnectOptionNum(
                                connection.ACCESS_MODE
                        );
                        return (connection.MODE_READ_ONLY == i);

                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... for setting a sub-space name |---
        public void setCatalog( String catalog ) throws SQLException {
                //---| ignore data |---
        }

        //---| ... for getting the sub-space name |---
        public String getCatalog() throws SQLException {
                //---| return database name |---
                return currDB;
        }

        //---| ... for setting the transaction isolation level |---
        public void setTransactionIsolation( int level ) throws SQLException {
                throw new SQLException(
                        "[JdbcOdbcConnection] not supported! "
                );
        }

        //---| ... for getting the transaction isolation level |---
        public int getTransactionIsolation() throws SQLException {
                //---| get isolation level |---
                try {
                        int i = connection.getInfoNum(
                                connection.TXN_ISOLATION_OPTION
                        );
                        switch( i )
                        {
                                case connection.TXN_READ_UNCOMMITTED:
                                        return TRANSACTION_READ_UNCOMMITTED;
                                case connection.TXN_READ_COMMITTED:
                                        return TRANSACTION_READ_COMMITTED;
                                case connection.TXN_REPEATABLE_READ:
                                        return TRANSACTION_REPEATABLE_READ;
                                case connection.TXN_SERIALIZABLE:
                                        return TRANSACTION_SERIALIZEABLE;
                                case connection.TXN_VERSIONING:
                                default:
                                        return TRANSACTION_NONE;
                        }
                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }

        }

        //---| ... for setting automatic close |---
        public void setAutoClose( boolean autoClose ) throws SQLException {
                throw new SQLException(
                        "[JdbcOdbcConnection] not supported!"
                );
        }

        //---| ... for getting automatic close setting |---
        public boolean getAutoClose() throws SQLException {
                //---| check AUTOCLOSE |---
                try {
                        int i = connection.getConnectOptionNum(
                                connection.CURSOR_COMMIT_BEHAVIOR
                        );
                        return (connection.CB_CLOSE == i);

                } catch( Exception e ) {
                        throw new SQLException(
                                "[J2ODBC] " + e.getMessage()
                        );
                }
        }

        //---| ... get latest waring |---
        public SQLWarning getWarnings() throws SQLException {
                //---| ??? don't know how to handle this ??? |---
                return null;
        }

        //---| ... clear latest warning |---
        public void clearWarnings() throws SQLException {
                //---| ??? don't know how to handle this ??? |---
        }

} /* class JdbcOdbcConnection */

//===| end of file |===========================================================
