/*
** RESULTS.C - This is the module containing the code for
** results gathering and processing
**
** (c) 1997 by Dirk Ohme - all rights reserved
*/

/*---| includes |------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <javaString.h>
#include <sqlcli_.h>

#include "iODBC_sql_dll_J2ODBC.h"

/*---------------------------------------------------------------------------*/
/*       Get Number of Result Columns                                        */
/*---------------------------------------------------------------------------*/
long iODBC_sql_dll_J2ODBC_numResultCols( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SWORD           cCols;

        /*
        ** get number of result columns
        */
        hstmt = (SQLHSTMT) unhand(this)->hstmt;
        cCols = 0;

        if( SQL_SUCCESS != (rc=SQLNumResultCols(hstmt, &cCols)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:numResultCols] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "numResultCols: cannot get value" );
                return 0;
        }

        /*
        ** return result
        */
        return (long) cCols;
}

/*---------------------------------------------------------------------------*/
/*       Get a Description on a specified Column in the Result Set           */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_describeCol( struct HiODBC_sql_dll_J2ODBC *this,
                                       long colIndex )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLCHAR         szColName[ SQL_MAX_ID_LENGTH + 1 ];
        SQLSMALLINT     fSqlType;
        UDWORD          cbColDef;
        SQLSMALLINT     ibScale;
        SQLSMALLINT     fNullable;

        /*
        ** get informations about a certain column in the result set
        */
        hstmt = (SQLHSTMT) unhand(this)->hstmt;

        szColName[0] = '\0';
        fSqlType     = SQL_UNKNOWN;
        cbColDef     = 0;
        ibScale      = 0;
        fNullable    = SQL_FALSE;

        if( SQL_SUCCESS != (rc=SQLDescribeCol(hstmt, colIndex, szColName,
                                              SQL_MAX_ID_LENGTH, NULL,
                                              &fSqlType, &cbColDef,
                                              &ibScale, &fNullable)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:describeCol] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "describeCol: cannot gather info about column" );
        }
        else
        {
                /*---| set informations |---*/
                unhand(this)->colName      = makeJavaString( szColName,
                                                             strlen(szColName) );
                unhand(this)->colSqlType   = (long) fSqlType;
                unhand(this)->colPrecision = (long) cbColDef;
                unhand(this)->colScale     = (long) ibScale;
                unhand(this)->colNullable  = (long) fNullable;
        }
}

/*---------------------------------------------------------------------------*/
/*       Get Attributes on a Result Column                                   */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_colAttributes( struct HiODBC_sql_dll_J2ODBC *this,
                                         long colIndex, long fDescType )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLCHAR         szDesc[ SQL_MAX_MESSAGE_LENGTH + 1 ];
        SQLINTEGER      fDesc;

        /*
        ** get column attribute
        */
        hstmt     = (SQLHSTMT) unhand(this)->hstmt;
        szDesc[0] = '\0';
        fDesc     = 0;

        if( SQL_SUCCESS != (rc=SQLColAttributes(hstmt, colIndex, fDescType,
                                                szDesc, sizeof(szDesc),
                                                NULL, &fDesc)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:colAttributes] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "colAttributes: cannot get attribute" );
                return;
        }
        else
        {
                /*---| set informations |---*/
                unhand(this)->attrDesc  = makeJavaString( szDesc,
                                                          strlen(szDesc) );
                unhand(this)->attrValue = (long) fDesc;
        }
}

/*---------------------------------------------------------------------------*/
/*       Bind Variables to Result Columns                                    */
/*---------------------------------------------------------------------------*/
/*
 * not implemented yet!
 *
 * RETCODE SQL_API SQLBindCol(
 *         LPSTMT  lpstmt,
 *         UWORD   icol,
 *         SWORD   fCType,
 *         PTR     rgbValue,
 *         SDWORD  cbValueMax,
 *         SDWORD FAR *pcbValue)
 */

/*---------------------------------------------------------------------------*/
/*       Get next Row of Result Set                                          */
/*---------------------------------------------------------------------------*/
long iODBC_sql_dll_J2ODBC_fetch( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;

        /*
        ** get next row
        */
        /*---| initialize |---*/
        hstmt = (SQLHSTMT) unhand(this)->hstmt;

        /*---| try to fetch data |---*/
        rc = SQLFetch( hstmt );

        /*---| no more data? -> return 'false' |---*/
        if( SQL_NO_DATA_FOUND == rc )
                return SQL_FALSE;

        /*---| any other error? |---*/
        if( SQL_SUCCESS != rc )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:fetch] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "fetch: cannot get next result row" );
                return SQL_FALSE;
        }

        /*
        ** return success
        */
        return SQL_TRUE;
}

/*---------------------------------------------------------------------------*/
/*       Get Column Data as Boolean                                          */
/*---------------------------------------------------------------------------*/
long iODBC_sql_dll_J2ODBC_getDataBoolean( struct HiODBC_sql_dll_J2ODBC *this,
                                          long colIndex )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLBIT          fBit;
        SQLINTEGER      cbResult;

        /*
        ** get column data
        */
        hstmt    = (SQLHSTMT) unhand(this)->hstmt;
        fBit     = SQL_FALSE;
        cbResult = 0;

        if( SQL_SUCCESS != (rc=SQLGetData(hstmt, colIndex, SQL_C_BIT,
                                          &fBit, sizeof(fBit), &cbResult)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:getDataBoolean] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "getDataBoolean: cannot get data" );
                return 0;
        }

        /*
        ** check for NULL value in column
        */
        if( SQL_NULL_DATA == cbResult )
                unhand(this)->colIsNull = (long) TRUE;
        else
                unhand(this)->colIsNull = (long) FALSE;

        /*
        ** return result
        */
        return (long) fBit;
}

/*---------------------------------------------------------------------------*/
/*       Get Column Data as Date                                             */
/*---------------------------------------------------------------------------*/
/*
 * not implemented yet!
 *
 * ??? iODBC_sql_dll_J2ODBC_getDataDate( struct HiODBC_sql_dll_J2ODBC *this,
 *                                       long colIndex )
 *
 */

/*---------------------------------------------------------------------------*/
/*       Get Column Data as Double                                           */
/*---------------------------------------------------------------------------*/
double iODBC_sql_dll_J2ODBC_getDataDouble( struct HiODBC_sql_dll_J2ODBC *this,
                                           long colIndex )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLDOUBLE       rResult;
        SQLINTEGER      cbResult;

        /*
        ** get column data
        */
        hstmt    = (SQLHSTMT) unhand(this)->hstmt;
        rResult  = 0.0;
        cbResult = 0;

        if( SQL_SUCCESS != (rc=SQLGetData(hstmt, colIndex, SQL_C_DOUBLE,
                                          &rResult, sizeof(rResult),
                                          &cbResult)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:getDataDouble] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "getDataDouble: cannot get data" );
                return 0.0;
        }

        /*
        ** check for NULL value in column
        */
        if( SQL_NULL_DATA == cbResult )
                unhand(this)->colIsNull = (long) TRUE;
        else
                unhand(this)->colIsNull = (long) FALSE;

        /*
        ** return result
        */
        return (double) rResult;
}

/*---------------------------------------------------------------------------*/
/*       Get Column Data as Float                                            */
/*---------------------------------------------------------------------------*/
float iODBC_sql_dll_J2ODBC_getDataFloat( struct HiODBC_sql_dll_J2ODBC *this,
                                         long colIndex )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLREAL         rResult;
        SQLINTEGER      cbResult;

        /*
        ** get column data
        */
        hstmt    = (SQLHSTMT) unhand(this)->hstmt;
        rResult  = 0.0;
        cbResult = 0;

        if( SQL_SUCCESS != (rc=SQLGetData(hstmt, colIndex, SQL_C_REAL,
                                          &rResult, sizeof(rResult),
                                          &cbResult)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:getDataFloat] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "getDataFloat: cannot get data" );
                return 0.0;
        }

        /*
        ** check for NULL value in column
        */
        if( SQL_NULL_DATA == cbResult )
                unhand(this)->colIsNull = (long) TRUE;
        else
                unhand(this)->colIsNull = (long) FALSE;

        /*
        ** return result
        */
        return (float) rResult;
}

/*---------------------------------------------------------------------------*/
/*       Get Column Data as Integer                                          */
/*---------------------------------------------------------------------------*/
long iODBC_sql_dll_J2ODBC_getDataInteger( struct HiODBC_sql_dll_J2ODBC *this,
                                          long colIndex )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLINTEGER      iResult;
        SQLINTEGER      cbResult;

        /*
        ** get column data
        */
        hstmt    = (SQLHSTMT) unhand(this)->hstmt;
        iResult  = 0;
        cbResult = 0;

        if( SQL_SUCCESS != (rc=SQLGetData(hstmt, colIndex, SQL_C_LONG,
                                          &iResult, sizeof(iResult),
                                          &cbResult)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:getDataInteger] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "getDataInteger: cannot get data" );
                return 0;
        }

        /*
        ** check for NULL value in column
        */
        if( SQL_NULL_DATA == cbResult )
                unhand(this)->colIsNull = (long) TRUE;
        else
                unhand(this)->colIsNull = (long) FALSE;

        /*
        ** return result
        */
        return (long) iResult;
}

/*---------------------------------------------------------------------------*/
/*       Get Column Data as String                                           */
/*---------------------------------------------------------------------------*/
Hjava_lang_String * iODBC_sql_dll_J2ODBC_getDataString(
                             struct HiODBC_sql_dll_J2ODBC *this,
                             long colIndex )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLCHAR         szResult[2048];
        SQLINTEGER      cbResult;

        /*
        ** get column data
        */
        hstmt       = (SQLHSTMT) unhand(this)->hstmt;
        szResult[0] = '\0';
        cbResult    = 0;

        if( SQL_SUCCESS != (rc=SQLGetData(hstmt, colIndex, SQL_C_CHAR,
                                          &szResult, sizeof(szResult),
                                          &cbResult)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:getDataString] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "getDataString: cannot get data" );
                return NULL;
        }

        /*
        ** check for NULL value in column
        */
        if( SQL_NULL_DATA == cbResult )
                unhand(this)->colIsNull = (long) TRUE;
        else
                unhand(this)->colIsNull = (long) FALSE;

        /*
        ** return result
        */
        return makeJavaString(szResult, cbResult);
}

/*---------------------------------------------------------------------------*/
/*       Determine whether there are more Results Sets or not                */
/*---------------------------------------------------------------------------*/
long iODBC_sql_dll_J2ODBC_moreResults( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;

        /*
        ** check for more result sets
        */
        /*---| initialize |---*/
        hstmt = (SQLHSTMT) unhand(this)->hstmt;

        /*---| try to fetch data |---*/
        rc = SQLMoreResults( hstmt );

        /*---| no more data? -> return 'false' |---*/
        if( SQL_NO_DATA_FOUND == rc )
                return SQL_FALSE;

        /*---| any other error? |---*/
        if( SQL_SUCCESS != rc )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:moreResults] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "moreResults: cannot get next result row" );
                return SQL_FALSE;
        }

        /*
        ** return success
        */
        return SQL_TRUE;
}

/*---------------------------------------------------------------------------*/
/*       Determine the Number of Rows associated with a Query                */
/*---------------------------------------------------------------------------*/
long iODBC_sql_dll_J2ODBC_rowCount( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SDWORD          cbRows;

        /*
        ** get number of rows
        */
        hstmt  = (SQLHSTMT) unhand(this)->hstmt;
        cbRows = 0;

        if( SQL_SUCCESS != (rc=SQLRowCount(hstmt, &cbRows)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:rowCount] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "rowCount: cannot determine number of rows" );
                return 0;
        }

        /*
        ** return success
        */
        return (long) cbRows;
}

/*---------------------------------------------------------------------------*/
/*       Set Cursor to a specified Row                                       */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_setPos( struct HiODBC_sql_dll_J2ODBC *this,
                                  long rowIndex, long fRefresh, long fLock )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;

        /*
        ** set position
        */
        hstmt  = (SQLHSTMT) unhand(this)->hstmt;

        if( SQL_SUCCESS != (rc=SQLSetPos(hstmt, rowIndex, fRefresh, fLock)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:setPos] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "setPos: cannot set position" );
        }
}

/*---------------------------------------------------------------------------*/
/*       Fetch a Block of Data at once                                       */
/*---------------------------------------------------------------------------*/
/*
 * not implemented yet!
 *
 * RETCODE SQL_API SQLExtendedFetch(
 *         LPSTMT  lpstmt,
 *     UWORD   fFetchType,
 *     SDWORD  irow,
 *         UDWORD FAR *pcrow,
 *     UWORD FAR *rgfRowStatus)
 */

/*---------------------------------------------------------------------------*/
/*       Get Error Message                                                   */
/*---------------------------------------------------------------------------*/
Hjava_lang_String * iODBC_sql_dll_J2ODBC_error( struct HiODBC_sql_dll_J2ODBC
                                                *this )
{
        RETCODE         rc;
        SQLHENV         henv;
        SQLHDBC         hdbc;
        SQLHSTMT        hstmt;
        SQLCHAR         szErrorMsg[ SQL_MAX_MESSAGE_LENGTH + 3 + 5 + 1 ];
        SQLCHAR         szSqlState[ 5 + 1 ];
        SQLSMALLINT     cbErrorMsg;
        SQLINTEGER      iErrorCode;

        /*
        ** get error message
        */
        henv  = (SQLHENV)  unhand(this)->henv;
        hdbc  = (SQLHDBC)  unhand(this)->hdbc;
        hstmt = (SQLHSTMT) unhand(this)->hstmt;

        szErrorMsg[0] =
        szSqlState[0] = '\0';
        cbErrorMsg    = 0;
        iErrorCode    = 0;

        if( SQL_SUCCESS != (rc=SQLError(henv, hdbc, hstmt,
                                        szSqlState,
                                        &iErrorCode,
                                        szErrorMsg,
                                        SQL_MAX_MESSAGE_LENGTH - 1,
                                        &cbErrorMsg)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:error] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "error: can't get error message" );
        }

        /*
        ** store results
        */
        strcat( szErrorMsg, " [" );
        strcat( szErrorMsg, szSqlState );
        strcat( szErrorMsg, "]" );
        return makeJavaString(szErrorMsg, cbErrorMsg);
}

/*===| end of file |=========================================================*/
