/*
** PREPARE.C - This is the module containing the code for ODBC for
** preparing SQL Commands and other functions prior to execution.
**
** (c) 1997 by Dirk Ohme - all rights reserved
*/

/*---| includes |------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <javaString.h>
#include <sqlcli_.h>

#include "iODBC_sql_dll_J2ODBC.h"

/*---------------------------------------------------------------------------*/
/*       Allocate a Statement Handle                                         */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_allocStmt( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHENV         hdbc;
        SQLHSTMT        hstmt;

        /*
        ** allocate statement handle
        */
        hdbc = (SQLHDBC) unhand(this)->hdbc;

        if( SQL_SUCCESS != (rc=SQLAllocStmt(hdbc, &hstmt)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:allocStmt] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "allocStmt: Allocation of statement handle failed" );
        }
        else
        {
                /*---| store handle |---*/
                unhand(this)->hstmt = (unsigned long) hstmt;
        }
}

/*---------------------------------------------------------------------------*/
/*       Free a Statement Handle                                             */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_freeStmt( struct HiODBC_sql_dll_J2ODBC *this,
                                    long fOption )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;

        /*
        ** free statement handle
        */
        hstmt = (SQLHSTMT) unhand(this)->hstmt;

        if( SQL_SUCCESS != (rc=SQLFreeStmt(hstmt, fOption)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:freeStmt] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "freeStmt: Freeing of statement handle failed" );
        }

        /*
        ** clear variable
        */
        if( SQL_DROP == fOption )
        {
                unhand(this)->hstmt = 0ul;
        }
}

/*---------------------------------------------------------------------------*/
/*       Prepare a SQL statement for Execution                               */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_prepare( struct HiODBC_sql_dll_J2ODBC *this,
                                   Hjava_lang_String *sqlStmt )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLCHAR        *pszSqlStmt = NULL;

        /*
        ** prepare SQL statement
        */
        /*---| initialize parameters |---*/
        hstmt      = (SQLHSTMT) unhand(this)->hstmt;
        pszSqlStmt = allocCString( sqlStmt );

        /*---| perform preparation |---*/
        rc = SQLPrepare( hstmt, pszSqlStmt, SQL_NTS );

        /*---| free temporary memory |---*/
        if( NULL != pszSqlStmt )
        {
                free( pszSqlStmt );
                pszSqlStmt = NULL;
        }

        /*---| check results |---*/
        if( SQL_SUCCESS != rc )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:prepare] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "prepare: cannot prepare SQL statement" );
        }
}

/*---------------------------------------------------------------------------*/
/*       Set Parameters on a Statement Handle                                */
/*---------------------------------------------------------------------------*/
/*
 * not implemented yet!
 *
 * RETCODE SQL_API SQLSetParam(
 *         LPSTMT  lpstmt,
 *         UWORD   ipar,
 *         SWORD   fCType,
 *         SWORD   fSqlType,
 *         UDWORD  cbColDef,
 *         SWORD   ibScale,
 *         PTR     rgbValue,
 *        SDWORD FAR *pcbValue)
 */

/*---------------------------------------------------------------------------*/
/*       Get a Description on a specified Parameter                          */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_describeParam( struct HiODBC_sql_dll_J2ODBC *this,
                                         long paramIndex )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLSMALLINT     fSqlType;
        UDWORD          cbColDef;
        SQLSMALLINT     ibScale;
        SQLSMALLINT     fNullable;

        /*
        ** get informations about a certain column in the result set
        */
        hstmt        = (SQLHSTMT) unhand(this)->hstmt;

        fSqlType     = SQL_UNKNOWN;
        cbColDef     = 0;
        ibScale      = 0;
        fNullable    = SQL_FALSE;

        if( SQL_SUCCESS != (rc=SQLDescribeParam(hstmt, paramIndex,
                                                &fSqlType, &cbColDef,
                                                &ibScale, &fNullable)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:describeParam] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "describeParam: cannot gather info about parameter" );
        }
        else
        {
                /*---| set informations |---*/
                unhand(this)->colSqlType   = (long) fSqlType;
                unhand(this)->colPrecision = (long) cbColDef;
                unhand(this)->colScale     = (long) ibScale;
                unhand(this)->colNullable  = (long) fNullable;
        }
}

/*---------------------------------------------------------------------------*/
/*       Sets multiple Values for a set of Parameters                        */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_paramOptions( struct HiODBC_sql_dll_J2ODBC *this,
                                        long numOfParams )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        UDWORD         *pirow;

        /*
        ** set multiple values
        */
        hstmt      = (SQLHSTMT) unhand(this)->hstmt;
        pirow      = (UDWORD *) &(unhand(this)->paramResultNumber);
        *pirow     = 0;

        if( SQL_SUCCESS != (rc=SQLParamOptions(hstmt, numOfParams, pirow)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:paramOptions] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "paramOptions: cannot set number of parameter options" );
        }
}

/*---------------------------------------------------------------------------*/
/*       Returns the Number of Parameter Markers set                         */
/*---------------------------------------------------------------------------*/
long iODBC_sql_dll_J2ODBC_numParams( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLSMALLINT     cParams;

        /*
        ** get number of parameter markers
        */
        hstmt   = (SQLHSTMT) unhand(this)->hstmt;
        cParams = 0;

        if( SQL_SUCCESS != (rc=SQLNumParams(hstmt, &cParams)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:numParams] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "numParams: cannot get number of parameters" );
                return 0;
        }
        return cParams;
}

/*---------------------------------------------------------------------------*/
/*       Sets Cursor Control Options                                         */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_setScrollOptions( struct HiODBC_sql_dll_J2ODBC *this,
                                            long fConcurrency,
                                            long crowKeyset,
                                            long crowRowset )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;

        /*
        ** set cursor control options
        */
        hstmt   = (SQLHSTMT) unhand(this)->hstmt;

        if( SQL_SUCCESS != (rc=SQLSetScrollOptions(hstmt, fConcurrency,
                                                   crowKeyset, crowRowset)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:setScrollOptions] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "setScrollOptions: cannot set options" );
        }
}

/*---------------------------------------------------------------------------*/
/*       Sets a new Cursor Name                                              */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_setCursorName( struct HiODBC_sql_dll_J2ODBC *this,
                                         struct Hjava_lang_String *cursorName )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLCHAR         szCursorName[ SQL_MAX_CURSOR_NAME + 1 ];

        /*
        ** set cursor control options
        */
        hstmt   = (SQLHSTMT) unhand(this)->hstmt;
        javaString2CString( cursorName, szCursorName, sizeof(szCursorName) );

        if( SQL_SUCCESS != (rc=SQLSetCursorName(hstmt, szCursorName, SQL_NTS)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:setCursorName] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "setCursorName: cannot set new cursor name" );
        }
}

/*---------------------------------------------------------------------------*/
/*       Gets the Current Cursor Name                                        */
/*---------------------------------------------------------------------------*/
Hjava_lang_String * iODBC_sql_dll_J2ODBC_getCursorName(
                             struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHSTMT        hstmt;
        SQLCHAR         szCursorName[ SQL_MAX_CURSOR_NAME + 1 ];
        SWORD           cbCursorName;

        /*
        ** set cursor control options
        */
        hstmt           = (SQLHSTMT) unhand(this)->hstmt;
        szCursorName[0] = '\0';
        cbCursorName    = 0;

        if( SQL_SUCCESS != (rc=SQLGetCursorName(hstmt, szCursorName,
                                                sizeof(szCursorName),
                                                &cbCursorName)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:getCursorName] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "getCursorName: cannot get current cursor name" );
                return NULL;
        }
        return makeJavaString(szCursorName, cbCursorName);
}

/*===| end of file |=========================================================*/
