/*
** CONNECT.C - This is the module containing the code for ODBC for
** allocation and connection.
**
** (c) 1997 by Dirk Ohme - all rights reserved
*/

/*---| includes |------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <javaString.h>
#include <sqlcli_.h>

#include "iODBC_sql_dll_J2ODBC.h"

/*---------------------------------------------------------------------------*/
/*       Allocate an Environment (ENV) Block                                 */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_allocEnv( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHENV         henv;

        /*
        ** allocate environment handle
        */
        if( SQL_SUCCESS != (rc=SQLAllocEnv(&henv)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:allocEnv] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "allocEnv: Allocation of environment failed" );
        }
        else
        {
                /*---| store handle |---*/
                unhand(this)->henv = (unsigned long) henv;
        }
}

/*---------------------------------------------------------------------------*/
/*       Allocate a Database Context (DBC) Block                             */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_allocConnect( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHENV         henv;
        SQLHDBC         hdbc;

        /*
        ** get environment handle
        */
        henv = (SQLHENV) unhand(this)->henv;

        /*
        ** allocate database context
        */
        if( SQL_SUCCESS != (rc=SQLAllocConnect(henv, &hdbc)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:allocConnect] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "allocConnect: Allocation of database context failed" );
        }
        else
        {
                /*---| store handle |---*/
                unhand(this)->hdbc = (unsigned long) hdbc;
        }
}

/*---------------------------------------------------------------------------*/
/*       Connect to Database                                                 */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_connect( struct HiODBC_sql_dll_J2ODBC *this,
                                   struct Hjava_lang_String *dsn,
                                   struct Hjava_lang_String *uid,
                                   struct Hjava_lang_String *pwd )
{
        RETCODE         rc;
        SQLHDBC         hdbc;
        SQLCHAR         szDSN[ SQL_MAX_DSN_LENGTH + 1];
        SQLCHAR         szUID[ SQL_MAX_ID_LENGTH + 1];
        SQLCHAR         szPWD[ SQL_MAX_ID_LENGTH + 1];

        /*
        ** connect to database
        */
        javaString2CString( dsn, szDSN, sizeof(szDSN) );
        javaString2CString( uid, szUID, sizeof(szUID) );
        javaString2CString( pwd, szPWD, sizeof(szPWD) );

        hdbc = (SQLHDBC) unhand(this)->hdbc;

        if( SQL_SUCCESS != (rc=SQLConnect(hdbc, szDSN, SQL_NTS,
                                                szUID, SQL_NTS,
                                                szPWD, SQL_NTS)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:connect] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "connect: cannot initiate database connection" );
        }
}

/*---------------------------------------------------------------------------*/
/*       Connect to Database, Query for Optional Data                        */
/*---------------------------------------------------------------------------*/
Hjava_lang_String * iODBC_sql_dll_J2ODBC_driverConnect(
                                       struct HiODBC_sql_dll_J2ODBC *this,
                                       struct Hjava_lang_String *connStr,
                                       long fDrvCmpl
                                       )
{
        RETCODE         rc;
        SQLHDBC         hdbc;
        SQLCHAR         szConnStrIn[ SQL_MAX_DSN_LENGTH + 1];
        SQLCHAR         szConnStrOut[ SQL_MAX_DSN_LENGTH + 1];
        SQLSMALLINT     cbConnStrOut;

        /*
        ** browse connection string
        */
        javaString2CString( connStr, szConnStrIn, sizeof(szConnStrIn) );
        szConnStrOut[0] = '\0';
        cbConnStrOut    = 0;

        hdbc = (SQLHDBC) unhand(this)->hdbc;

        if( SQL_SUCCESS != (rc=SQLDriverConnect(hdbc, (HWND) NULL,
                                                szConnStrIn,  SQL_NTS,
                                                szConnStrOut, SQL_NTS,
                                                &cbConnStrOut,
                                                (SQLUSMALLINT) fDrvCmpl)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:driverConnect] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "driverConnect: cannot initiate database connection" );
        }

        /*
        ** store results
        */
        return makeJavaString(szConnStrOut, cbConnStrOut);
}

/*---------------------------------------------------------------------------*/
/*       Check for a Connection                                              */
/*---------------------------------------------------------------------------*/
Hjava_lang_String * iODBC_sql_dll_J2ODBC_browseConnect(
                                       struct HiODBC_sql_dll_J2ODBC *this,
                                       struct Hjava_lang_String *connStr
                                       )
{
        RETCODE         rc;
        SQLHDBC         hdbc;
        SQLCHAR         szConnStrIn[ SQL_MAX_DSN_LENGTH + 1];
        SQLCHAR         szConnStrOut[ SQL_MAX_DSN_LENGTH + 1];
        SQLSMALLINT     cbConnStrOut;

        /*
        ** browse connection string
        */
        javaString2CString( connStr, szConnStrIn, sizeof(szConnStrIn) );
        szConnStrOut[0] = '\0';
        cbConnStrOut    = 0;

        hdbc = (SQLHDBC) unhand(this)->hdbc;

        if( SQL_SUCCESS != (rc=SQLBrowseConnect(hdbc,
                                                szConnStrIn,  SQL_NTS,
                                                szConnStrOut, SQL_NTS,
                                                &cbConnStrOut)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:browseConnect] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "browseConnect: error during browsing" );
        }

        /*
        ** store results
        */
        return makeJavaString(szConnStrOut, cbConnStrOut);
}

/*---------------------------------------------------------------------------*/
/*       Disconnect from Database                                            */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_disconnect( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHDBC         hdbc;

        /*
        ** disconnect
        */
        hdbc = (SQLHDBC) unhand(this)->hdbc;
        if( SQL_SUCCESS != (rc=SQLDisconnect(hdbc)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:disconnect] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "disconnect: cannot perform database disconnection" );
        }
}

/*---------------------------------------------------------------------------*/
/*       Free Database Context                                               */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_freeConnect( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHDBC         hdbc;

        /*
        ** free database context
        */
        hdbc = (SQLHDBC) unhand(this)->hdbc;
        if( SQL_SUCCESS != (rc=SQLFreeConnect(hdbc)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:freeConnect] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "freeConnect: Freeing of database context failed" );
        }

        /*
        ** clear variable
        */
        unhand(this)->hdbc = 0ul;
}

/*---------------------------------------------------------------------------*/
/*       Free Environment Block                                              */
/*---------------------------------------------------------------------------*/
void iODBC_sql_dll_J2ODBC_freeEnv( struct HiODBC_sql_dll_J2ODBC *this )
{
        RETCODE         rc;
        SQLHENV         henv;

        /*
        ** allocate environment handle
        */
        henv = (SQLHENV) unhand(this)->henv;
        if( SQL_SUCCESS != (rc=SQLFreeEnv(henv)) )
        {
                /*---| throw exception |---*/
#ifdef DEBUG
                fprintf( stderr, "! [J2ODBC:%s:freeEnv] rc = %d\n", __FILE__, rc );
#endif
                SignalError( 0, "iODBC/sql/dll/J2ODBCException",
                             "freeEnv: Freeing of environment failed" );
        }

        /*
        ** clear variable
        */
        unhand(this)->henv = 0ul;
}

/*===| end of file |=========================================================*/
