/********************************************************************
    POSTODBCL.DLL - A library to talk to Postgres95 by using the
                    WINDOWS ODBC Interface

    Copyright (C) 1996; Christian Czezatke

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details. 

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
    How to contact the author:
    
    email to: e9025461@student.tuwien.ac.at
********************************************************************/ 
#include <winsock.h>

#include <stdio.h>
#include <stdlib.h>
#ifdef __WATCOMC__
#include <mem.h>
#else
#include <memory.h>
#include <string.h>
#endif
#include "sockio.h"
#include "../globals.h"
#include <windows.h>

StreamSocket::StreamSocket(UInt2 bufsize):
  sock(-1),
  buffersize(bufsize),
  buffer_filled_in(0),
  buffer_filled_out(0),
  buffer_read_in(0),
  buffer_in(new unsigned char[bufsize]),
  buffer_out(new unsigned char[bufsize])
{
}


char StreamSocket::connect_to(UInt2 port, char *hostname)
{
 struct hostent *host;
 struct sockaddr_in sadr;

 if (sock != -1) {
     errnumber = SOCKET_ALREADY_CONNECTED;
     errorMsg = "Socket is already connected";
     return 0;
 }

 host = gethostbyname(hostname);
 if (host == NULL) {
     errnumber = SOCKET_HOST_NOT_FOUND;
     errorMsg = "Could not resolve hostname.";
     return 0;
 }

 memset((char *)&sadr, 0, sizeof(sadr));
 memcpy(&(sadr.sin_addr), host->h_addr, host->h_length);
 sadr.sin_family = AF_INET;
 sadr.sin_port = htons(port);

 sock = socket(AF_INET, SOCK_STREAM, 0);
 if (-1 == sock) {
     errnumber = SOCKET_COULD_NOT_CREATE_SOCKET;
     errorMsg = "Could not create Socket.";
     return 0;
 }

 if ( connect(sock, (struct sockaddr *)&(sadr),
              sizeof(sadr))  < 0) {
    errnumber = SOCKET_COULD_NOT_CONNECT;
    errorMsg = "Could not connect to remote socket.";
    closesocket(sock);
    sock = -1;
    return 0;
 }
 return 1;
}



StreamSocket::~StreamSocket()
{
 int lf = 0;
 int shutdown_ok;

 if (-1 != sock) {

   // CC: at fist shutdown the socket and then remove it
   // 2 -> disallow both sends and receives
   shutdown_ok = shutdown(sock, 2);
   if (0 == shutdown_ok)
     closesocket(sock);
 }
}


char StreamSocket::get_char()
{
    return (char) get_next_byte();
}


void StreamSocket::put_char(char c)
{
 put_next_byte((unsigned char)c);
}


void StreamSocket::put_n_char(char *buffer, Int4 len)
{
 Int4 lf;

 if (NULL == buffer) {
     errnumber = SOCKET_NULLPOINTER_PARAMETER;
     errorMsg = "put_n_char was called with NULL-Pointer";
     return;
 }

 for(lf=0; lf < len; lf++)
   put_next_byte((unsigned char)buffer[lf]);
}


void StreamSocket::put_string(char *string)
{
 Int4 lf, eostr;

 eostr = strlen(string)+1;

 for(lf =0; lf < eostr; lf++)
   put_next_byte((unsigned char)string[lf]);
}


void StreamSocket::put_int(Int4 value, Int2 len)
{
 Int2 lf;
 unsigned char next_byte;

 if ((len < 1) || (len > 4)) {
     errnumber = SOCKET_PUT_INT_WRONG_LENGTH;
     errorMsg = "Cannot write ints of that length";
     return;
 }

 for(lf=0; lf < len; lf++) {
   next_byte = (unsigned char) (value & 0xff);
   value >>= 8;
   put_next_byte(next_byte);
 }
}





void StreamSocket::get_n_char(char *buffer, Int4 len)
{
 Int4 lf;

 if (NULL == buffer) {
     errnumber = SOCKET_NULLPOINTER_PARAMETER;
     errorMsg = "put_n_char was called with NULL-Pointer";
     return;
 }
 for(lf=0; lf < len; lf++)
   buffer[lf] = get_next_byte();
}



void StreamSocket::get_string(char *buffer, Int4 bufsize)
{
 Int4 lf;

 lf = 0;
 do {
     buffer[lf] = get_next_byte();
 } while ( (buffer[lf++]!= '\0') && (lf < bufsize-1));
 buffer[lf-1] = '\0';
}



Int4 StreamSocket::get_int(Int2 len)
{
 Int2 lf;
 unsigned char next_byte;
 Int4 rv;


 if ((len < 1) || (len > 4)) {
     errnumber = SOCKET_GET_INT_WRONG_LENGTH;
     errorMsg = "Cannot read ints of that length";
     return 0;
 }

 rv = 0;
 for(lf=0; lf < len; lf++) {
   next_byte = get_next_byte();
   rv += (Int4)next_byte << 8*lf;
 }
 return rv;
}




void StreamSocket::flush_output()
{
 Int4 written;

 written = send(sock, (char *)buffer_out, buffer_filled_out, 0);
 if (written != buffer_filled_out) {
     errnumber = SOCKET_WRITE_ERROR;
     errorMsg = "Could not flush socket buffer.";
 }
 buffer_filled_out = 0;
}

////////////   PRIVATE FUNCTIONS ////////

unsigned char StreamSocket::get_next_byte()
{
 if (buffer_read_in >= buffer_filled_in) {
     // there are no more bytes left in the buffer ->
     // reload the buffer

     buffer_read_in = 0;
     buffer_filled_in = recv(sock, (char *)buffer_in, buffersize, 0);
     if (buffer_filled_in == -1) {
         errnumber = SOCKET_READ_ERROR;
         errorMsg = "Error while reading from the socket.";
         buffer_filled_in = 0;
     }
 }
 return buffer_in[buffer_read_in++];
}


void StreamSocket::put_next_byte(unsigned char next_byte)
{
 Int4 bytes_sent;

 buffer_out[buffer_filled_out++] = next_byte;

 if (buffer_filled_out == buffersize) {
     // buffer is full, so write it out
     bytes_sent = send(sock, (char *)buffer_out, buffersize, 0);
     if (bytes_sent != buffersize) {
         errnumber = SOCKET_WRITE_ERROR;
         errorMsg = "Error while writing to the socket.";
     }
     buffer_filled_out = 0;
 }
}



