/* -*- mode: c++; c-basic-offset: 4 -*- */
/********************************************************************
    POSTODBCL.DLL - A library to talk to Postgres95 by using the
                    WINDOWS ODBC Interface

    Copyright (C) 1996; Christian Czezatke

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details. 

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
    How to contact the author:
    
    email to: e9025461@student.tuwien.ac.at
********************************************************************/   


#include "connect.h"
#include "errclass.h"
#include <stdlib.h>
#ifdef __WATCOMC__
#include <mem.h>
#else
#include <memory.h>
#include <string.h>
#endif
#include <stdio.h>
#include "../globals.h"
#include <windows.h>




Connection::Connection():
  sock(new StreamSocket()),
  dbasename(NULL)
{
  char address[20];

  debug_decls;

  debug_out("in Connection constructor.\n");
  if(errorMsg) {
    debug_out("errorMsg is not null.\n");
  } else {
    debug_out("errorMsg is null.\n");
  }
  sprintf(address, "%x", this);
  debug_out("address of this Connection object is ");
  debug_out(address);
  debug_out("\n");

  dbasename = 0;
  DSN = 0;
}

#define PATH_SIZE 64
#define NAMEDATALEN 16
#define ARGV_SIZE 64

#define NO_AUTHENTIFICATION 7


char Connection::open_db(char *in_DSN, char *machine, UInt2 port, 
                         char *database, char *user)
{
 // Authentication Block:
 /**************************
  len (4 bytes)
  type (4 bytes -> 7)
  database (PATH_SIZE 64 bytes)
  user (NAMEDATALEN 16 bytes)
  options (ARGV_SIZE 64 bytes)
  execfile (ARGV_SIZE 64 bytes);
  tty (PATH_SIZE 64 bytes)
 ****************************/
 char path[PATH_SIZE];
 char name[NAMEDATALEN];
 char argv[ARGV_SIZE];
 Int2 cplen;
 QResult *res;
 debug_decls;

 debug_out("in Connect::open_db...\n");

 // if there is already a database name, discard it
 if (NULL != dbasename) {
     free(dbasename);
 }
 dbasename = strdup(database);

 if(!DSN) { free(DSN); }
 DSN = strdup(in_DSN);

 // At first connect to the server's socket
 debug_out("connecting to the server socket...\n");
 sock->connect_to(port, machine);
 if (sock->get_errcode() != 0) {
   debug_out("connection to the server socket failed.\n");
     errnumber = CONNECTION_SERVER_NOT_REACHED;
     errorMsg = "Could not connect to the server";
     return 0;
 }
 debug_out("connection to the server socket succeeded.\n");

 // Send length auf Authentication Block
 sock->put_int(htonl(4+4+PATH_SIZE+NAMEDATALEN+2*ARGV_SIZE+PATH_SIZE+5),4);

 // now send the fields in the right order and length
 sock->put_int(htonl(NO_AUTHENTIFICATION), 4);

 // The database name
 memset(path, 0, PATH_SIZE);
 cplen = strlen(database) < PATH_SIZE ? strlen(database) : PATH_SIZE;
 strncpy(path, database, cplen);
 sock->put_n_char(path, PATH_SIZE);

 // The user name
 memset(name, 0, NAMEDATALEN);
 cplen = strlen(user) < NAMEDATALEN ? strlen(user) : NAMEDATALEN;
 strncpy(name, user, cplen);
 sock->put_n_char(name, NAMEDATALEN);

 // The options, the execfile and the tty are all 0
 memset(argv, 0, ARGV_SIZE);
 sock->put_n_char(argv, ARGV_SIZE); // Options
 sock->put_n_char(argv, ARGV_SIZE); // ExecFile
 memset(path, 0, PATH_SIZE);
 sock->put_n_char(path, PATH_SIZE); // tty

 // I'm not sure about the following two put_*chars.
 // it does not matter what they contain, but
 // if you do not send them, you cannot connect...
 sock->put_n_char("...",4);
 sock->put_char('W');

 sock->flush_output();
 debug_out("sent the authentication block.\n");

 if(sock->get_errcode() != 0) {
   debug_out("couldn't send the authentication block properly.\n");
     errnumber = CONNECTION_COULD_NOT_SEND_AUTHENTICATION;
     errorMsg = "Sending the authentication packet failed";
     return 0;
 }
 debug_out("sent the authentication block successfully.\n");

 /* send an empty query in order to find out whether the specified */
 /* database really exists on the server machine */
 debug_out("sending an empty query...\n");
 res = send_query(" ");
 if ((NULL == res)||(res->get_status() != PGRES_EMPTY_QUERY)) {
   debug_out("got no result from the empty query.  (probably database does not exist)\n");
     errnumber = CONNECTION_NO_SUCH_DATABASE;
     errorMsg = "A database with the specified name does not exist on the host";
     if (NULL != res)
       delete res;
     return 0;
 }
 if (NULL != res)
    delete res;
 debug_out("empty query seems to be OK.\n");

 return 1;
}


QResult *Connection::send_query(char *query)
{
 QResult *res = NULL;
 char id, swallow, clear, *msg;
 static char cmdstatus[MAX_MESSAGE_LEN+1];

 // Indicate that we are sending a query to the backend
 if(strlen(query) > MAX_MESSAGE_LEN-2) {
     errnumber = CONNECTION_MSG_TOO_LONG;
     errorMsg = "Query string is too long";
     return NULL;
 }
 if ((NULL == query) || (query[0] == '\0'))
   return NULL;
 sock->put_char('Q');
 sock->put_string(query);
 sock->flush_output();
 if (sock->get_errcode() != 0) {
     errnumber = CONNECTION_COULD_NOT_SEND;
     errorMsg = "Could not send Query to backend";
     return NULL;
 }

 while(1) {
   /* what type of message is comming now ? */
   id = sock->get_char();
   if ((sock->get_errcode() != 0) || (id == EOF)) {
       errnumber = CONNECTION_NO_RESPONSE;
       errorMsg = "No response from the backend";
       if (NULL != res)
         delete res;
       return NULL;
   }

   switch (id) {
       case 'A' : /* Asynchronous Messages are currently ignored because we
                      do not support them currently (not needed for out purposes) */
         (void)sock->get_int(4); /* id of notification */
         sock->get_string(cmdstatus, MAX_MESSAGE_LEN+1);
                   /* name of the relation the message comes from */
         break;
       case 'C' : /* portal query command, no tuples returned */
         /* read in the return message from the backend */
         sock->get_string(cmdstatus, MAX_MESSAGE_LEN+1);
         if (sock->get_errcode() != 0) {
             errnumber = CONNECTION_NO_RESPONSE;
             errorMsg = "No response from backend while receiving a portal query answer";
             return NULL;
         } else {
             /* (Quotation from the original comments)
                since backend may produze more than one result for some commands
                we need to poll until clear
                 so we send an empty query, and keep reading out of the pipe
                 until an 'I' is received
              */
              clear = 0;
              sock->put_string("Q ");
              sock->flush_output();
              while(!clear) {
                  sock->get_string(msgbuffer, ERROR_MSG_LENGTH+1);
                  clear = msgbuffer[0] == 'I';
              }
              res = new QResult(PGRES_COMMAND_OK);
              msg = new char[strlen(cmdstatus)+1];
              memcpy(msg, cmdstatus, strlen(cmdstatus)+1);
              res->set_message(msg);
              return res;
         }
       case 'E' : /* The backend sent us an error message */
         msg = new char[ERROR_MSG_LENGTH+1];
         sock->get_string(msg, ERROR_MSG_LENGTH+1);
         res = new QResult(PGRES_NONFATAL_ERROR);
         res->set_message(msg);
         return res;
       case 'I' : /* The server sends an empty query */
         /* There is a closing '\0' following the 'I', so we eat it */
         swallow = sock->get_char();
         if ((swallow != '\0') || sock->get_errcode() != 0) {
                errnumber = CONNECTION_BACKEND_CRAZY;
                errorMsg = "Unexpected protocol character from backend";
                return new QResult(PGRES_FATAL_ERROR);
         } else {
             /* We return the empty query */
             return new QResult(PGRES_EMPTY_QUERY);
         }
         break;
       case 'N' : /* notices from the backend -> store them in the error var */
         errnumber = CONNECTION_SERVER_REPORTED_ERROR;
         errorMsg = msgbuffer;
         sock->get_string(errorMsg, ERROR_MSG_LENGTH+1);
         return NULL;
       case 'P' : /* get the Portal name */
         sock->get_string(msgbuffer, MAX_MESSAGE_LEN+1);
         break;
       case 'T': /* Tuple results start here */
         res = new QResult();
         res->make_result(msgbuffer, sock);
         return res;
       case 'D': /* Copy in command began successfully */
         res = new QResult(PGRES_COPY_IN);
         return res;
       case 'B': /* Copy out command began successfully */
         res = new QResult(PGRES_COPY_OUT);
         return res;
       default:
         errnumber = CONNECTION_BACKEND_CRAZY;
         errorMsg = "Unexpected protocol character from backend";
         return NULL;
   }
 }
}


Connection::~Connection()
{
    delete sock;

    if (NULL != dbasename) {
       free(dbasename);
    }
}


/************************************/
/* CLASS pgresAttDescList           */
/************************************/
/* This class describes the structure of tuples  */
/* as they are sent by the backend in response   */
/* to a select query                             */
/* this class is supposed to be used only by     */
/* the QResult class                             */
/*************************************************/



pgresAttDescList::pgresAttDescList():
        num_fields(0),
        name(NULL),
        adtid(NULL),
        adtsize(NULL)
{
}


char buf[50];
char pgresAttDescList::make_result(StreamSocket *sock)
{
    Int2 lf;
    int new_num_fields;
    Oid new_adtid;
    Int2 new_adtsize;
    char *new_field_name;

    new_field_name = new char[MAX_MESSAGE_LEN+1];
    if(!new_field_name)
        return 0;
    
    /* at first read in the number of fields that are in the query */
    new_num_fields = (Int2)sock->get_int(sizeof(Int2));

    /* according to that allocate memory */
    set_num_fields(new_num_fields);
    
    /* now read in the descriptions */
    for(lf = 0; lf < new_num_fields; lf++) {
        sock->get_string(new_field_name, MAX_MESSAGE_LEN+1);
        new_adtid = (Oid)sock->get_int(4);
        new_adtsize = (Int2)sock->get_int(2);
        set_field_info(lf, new_field_name, new_adtid, new_adtsize);
    }

    delete [] new_field_name;

    return (sock->get_errcode() == 0);
}


pgresAttDescList::~pgresAttDescList()
{
  free_memory();
}

void pgresAttDescList::free_memory(void)
{
    Int2 lf;

    for(lf = 0; lf < num_fields; lf++) {
        if(name[lf])
            delete [] name[lf];
    }

    delete [] name;
    delete [] adtid;
    delete [] adtsize;

}

void pgresAttDescList::set_num_fields(int new_num_fields)
{
    free_memory();

    num_fields = new_num_fields;

    name = new char*[num_fields];
    adtid = new Oid[num_fields];
    adtsize = new Int2[num_fields];    
}

void pgresAttDescList::set_field_info(int field_num, char *new_name, 
                                      Oid new_adtid, Int2 new_adtsize)
{
    
    // check bounds
    if((field_num < 0) || (field_num >= num_fields)) {
        return;
    }

    // store the info
    name[field_num] = new char[strlen(new_name)+1];
    strcpy(name[field_num], new_name);

    adtid[field_num] = new_adtid;
    adtsize[field_num] = new_adtsize;
}

char *pgresAttDescList::get_fieldname(Int2 which)
{
    char *rv = NULL;
    
    if (NULL == name)
        return NULL;
    
    if ((which >= 0) && (which < num_fields))
        rv = name[which];
    return rv;
}


Int2 pgresAttDescList::get_fieldsize(Int2 which)
{
    Int2 rv = 0;

    if (NULL == adtsize)
        return 0;
    
    if ((which >= 0) && (which < num_fields))
        rv = adtsize[which];
    return rv;
}



Oid pgresAttDescList::get_oid(Int2 which)
{
 Oid rv = 0;

 if (NULL == adtid)
   return 0;

 if ((which >= 0) && (which < num_fields))
   rv = adtid[which];
 return rv;
}



/* CC: do not inline because Visual C++ can't link in that case */
Int2 pgresAttDescList::get_num_fields()
{
 return num_fields;
}



/************************************/
/* CLASS TupleList                  */
/************************************/


TupleList::TupleList(UInt4 fieldcnt):
        num_fields(fieldcnt),
        num_tuples(0),
        list_start(NULL),
        list_end(NULL),
        lastref(NULL),
        last_indexed(-1)
{}


void *TupleList::get_fieldval(Int4 tupleno, Int2 fieldno)
{
 Int4 lf;
 Int4 delta, from_end;
 char end_is_closer, start_is_closer;
 TupleNode *rv;

 if (-1==last_indexed)
   /* we have an empty tuple list */
   return NULL;

 /* some more sanity checks */
 if ((tupleno >= num_tuples) || (tupleno < 0))
   /* illegal tuple number range */
   return NULL;

 if ((fieldno >= num_fields) || (fieldno < 0))
   /* illegel field number range */
   return NULL;

 /* check if we are accessing the same tuple that was used in
    the last fetch (e.g: for fetching all the fields one after
    another. Do this to speed things up
 */
 if (tupleno == last_indexed)
   return lastref->tuple[fieldno].value;

 /* now for the tricky part... */

 /*
 Since random access is quite inefficient for linked lists we use
 the lastref pointer that points to the last element referenced
 by a get_fieldval() call in conjunction with the its index number
 that is stored in last_indexed. (So we use some locality of
 reference principle to speed things up)
 */

 delta = tupleno - last_indexed;
 /* if delta is positive, we have to go forward */

 /* now check if we are closer to the start or the end of the list
    than to our last_indexed pointer
 */
 from_end = (num_tuples-1) - tupleno;

 start_is_closer = labs(delta) > tupleno;
 /* true if we are closer to the start of the list than to the
    last_indexed pointer
 */

 end_is_closer = labs(delta) > from_end;
 /* true if we are closer at the end of the list */

 if (end_is_closer) {
   /* scanning from the end is the shortest way. so we do that... */
   rv = list_end;
   for (lf=0; lf < from_end; lf++)
      rv = rv->prev;
 } else if (start_is_closer) {
    /* the shortest way is to start the search from the head of the list */
    rv = list_start;
    for (lf=0; lf < tupleno; lf++)
      rv = rv->next;
 } else {
    /* the closest way is starting from our lastref - pointer */
    rv = lastref;
    /* at first determine whether we have to search forward or backwards */
    if (delta < 0) {
      /* we have to search backwards */
      for(lf=0; lf < (-1)*delta; lf++)
        rv = rv->prev;
    } else {
      /* ok, we have to search forward... */
      for (lf=0; lf < delta; lf++)
        rv = rv->next;
    }
 }

 /* now we have got our return pointer, so update the lastref
    and the last_indexed values
 */
 lastref = rv;
 last_indexed = tupleno;
 return rv->tuple[fieldno].value;
}


TupleList::~TupleList()
{
    UInt4 lf;
    TupleNode *node, *hlp;

    node = list_start;
    while(NULL != node) {
        for (lf=0; lf < num_fields; lf++)
            if (node->tuple[lf].value != NULL) {
                // delete [] node->tuple[lf].value;
                free(node->tuple[lf].value);
            }
        hlp = node->next;
        // delete [] ((char *)node);
        free(node);
        node = hlp;
    }
}

TupleNode *TupleList::read_next_tuple(StreamSocket *sock, char binary)
/* CC: 23.05.96: Return value changed from char to TupleNode* to
       allow update of the field length. */
{
    TupleNode *new_node;
    char rv;
    
    new_node = read_one_tuple(sock, binary);
    /* if we could not read in a tuple, we return an error */
    if (NULL == new_node) {
        return 0;
    }
    
    rv = add_tuple(new_node);
    
    if (!rv) { 
        /* inserting the tuple failed */
        // delete [] ((char *)new_node);
        free(new_node);
        new_node = NULL;
    }

 return new_node;
}


char TupleList::add_tuple(TupleNode *new_field)
{
 /* we append the tuple at the end of the doubly linked list
    of the tuples we have already read in
 */

 new_field->prev = NULL;
 new_field->next = NULL;

 if (NULL == list_start) {
     /* the list is empty, we have to add the first tuple */
     list_start = new_field;
     list_end = new_field;
     lastref = new_field;
     last_indexed = 0;
 } else {
     /* there is already an element in the list, so add the new
        one at the end of the list
     */
     list_end->next = new_field;
     new_field->prev = list_end;
     list_end = new_field;
 }
 num_tuples++;
 /* this method of building a list cannot fail, so we return 1 */
 return 1;
}



TupleNode *TupleList::read_one_tuple(StreamSocket *sock, char binary)
{
 Int2 field_lf;
 TupleNode *this_tuplenode;
 char bmp, bitmap[MAX_FIELDS];        /* Max. len of the bitmap */
 Int2 bitmaplen;                       /* len of the bitmab in bytes */
 Int2 bitmap_pos;
 Int2 bitcnt;
 Int4 len;
 char *buffer;

 bitmaplen = (Int2)num_fields / BYTELEN;
 if ((num_fields % BYTELEN) > 0)
   bitmaplen++;

 /*
    at first the server sends a bitmat that indicates which
    database fields are null
 */
 sock->get_n_char(bitmap, bitmaplen);

 this_tuplenode = (TupleNode *)malloc(sizeof(TupleNode)+(num_fields-1)*sizeof(TupleField));

 bitmap_pos = 0;
 bitcnt = 0;
 bmp = bitmap[bitmap_pos];

 for(field_lf = 0; field_lf < num_fields; field_lf++) {
     /* Check if the current field ist NULL */
     if(!(bmp & 0200)) {
         /* YES, it is NULL ! */
         this_tuplenode->tuple[field_lf].len = 0;
         this_tuplenode->tuple[field_lf].value = 0;
     } else {
         /*
           NO, the field ist not null. so get at first the
           length of the field (four bytes)
         */
         len = sock->get_int(VARHDRSZ);
         if (!binary)
             len-= VARHDRSZ;
         buffer = (char *)malloc(len+1);
         sock->get_n_char(buffer, len);
         buffer[len] = '\0';

         this_tuplenode->tuple[field_lf].len = len;
         this_tuplenode->tuple[field_lf].value = buffer;
     }
     /*
       Now adjust for the next bit to be scanned in the
       next loop.
     */
     bitcnt++;
     if (BYTELEN == bitcnt) {
         bitmap_pos++;
         bmp = bitmap[bitmap_pos];
         bitcnt = 0;
     } else
         bmp <<= 1;
 }
 return this_tuplenode;
}









/************************************/
/* CLASS QResult                    */
/************************************/




QResult::QResult(QueryResultCode condition):
        status(condition),
        result_completed(0),
        fields(),
        tuples(NULL),
        message(NULL)
{

}





char QResult::make_result(char *portalname, StreamSocket *sock)
{
 char *messagebuf;
 char *backend_coredump = "Unexpected result from backend. It probably crashed";
 TupleNode *node;

 char done, id;
 /* before doing anything else check if there is not     */
 /* already a result tied to that instance of the object */
 /* At first read in the attribute description           */


 if (result_completed)
   return 0;

 /* now read in the field description */
 if (!fields.make_result(sock))
   return 0;

 /* now allocate the Tuplelist and tell it how many */
 /* fields are in one tuple */
 tuples = new TupleList(fields.get_num_fields());

 id = sock->get_char();
 done = 0;
 while(!done) {
    switch (id) {
        case 'T': /* Tuples within tuples cannot be handled */
          return 0;
        case 'B': /* Tuples in binary format */
        case 'D': /* Tuples in ASCII format  */
          node = tuples->read_next_tuple(sock, id == 0);
          if (NULL == node) {
             status = PGRES_BAD_RESPONSE;
             return 0;
          }
         /* CC: 23.05.96: Now let us see wether we have to correct the field length information.
                The new tuple we have just read is appended at the end of the list. So
                let us see whether we have to update it */
          update_fieldsizes(node);
        
          break;
        case 'C': /* End of tuple list */
          /* now allocate temporary space to hold the command tap */
          messagebuf = new char[MAX_MESSAGE_LEN+1];
          sock->get_string(messagebuf, MAX_MESSAGE_LEN+1);
          message = new char[strlen(messagebuf)+1];
          memcpy(message, messagebuf, strlen(messagebuf)+1);
          delete [] messagebuf;
          done = 1;
          status = PGRES_TUPLES_OK;
          break;
        case 'E': /* The server sent an error message */
          /* temporarily allocate space to hold the error message */
          messagebuf = new char[ERROR_MSG_LENGTH+1];
          sock->get_string(messagebuf, ERROR_MSG_LENGTH+1);
          message = new char[strlen(messagebuf)+1];
          memcpy(message, messagebuf, strlen(messagebuf)+1);
          delete [] messagebuf;
          done = 1;
          status = PGRES_FATAL_ERROR;
          break;
        case 'N': /* Asynchronous notificaions are not properly supported */
          /* temporarily allocate space to hold the notification */
          messagebuf = new char[ERROR_MSG_LENGTH+1];
          sock->get_string(messagebuf, ERROR_MSG_LENGTH+1);
          /* Anyway, discard notifications since we do not have stderr when
             running under Windows 16bit */
          delete [] messagebuf;
          break;
        default: /* this should only happen if the backend dumped core */
          messagebuf = new char[strlen(backend_coredump)+1];
          memcpy(messagebuf, backend_coredump, strlen(backend_coredump)+1);
          status = PGRES_FATAL_ERROR;
          done = 1;
          break;
    }
    if (!done)
      id = sock->get_char();
 }

 return (status != PGRES_FATAL_ERROR);
}

void QResult::set_num_fields(int new_num_fields)
{
    fields.set_num_fields(new_num_fields);
    if(tuples) {
        delete tuples;
    }
    tuples = new TupleList(new_num_fields);
}

void QResult::set_field_info(int field_num, char *name, 
                             Oid adtid, Int2 adtsize)
{
    fields.set_field_info(field_num, name, adtid, adtsize);
}

void QResult::add_tuple(TupleNode *new_tuple)
{
    tuples->add_tuple(new_tuple);
}


void QResult::update_fieldsizes(TupleNode *node)
  /*
   CC: 23.05.96: ADDED
     checks the tuple "node" whether there are larger entries than the current
     length of the respective column as stored in "fields". If that is the
     case, the length stored in fields is updated accordingly
  */   
{
    Int2 lf;
    Int2 this_size;

    for (lf=0; lf < fields.get_num_fields(); lf++) {
       this_size = (Int2)node->tuple[lf].len+1;
       // CC: there is no danger of loosing significant digits, since
       // Postgres limits tuple sizes to <8KBytes
       // The "+1" takes the ending '\0' into account for ascii rep.
       // Maybe we should change that and differentiate between
       // binary and ascii representation? 
        
       if (fields.get_fieldsize(lf) < this_size)
         set_field_info(lf, get_fieldname(lf), get_field_type(lf), this_size);
         /* CC: This only updates the field size */
         /* Maybe we should introduce an own function (set_fieldsize) for that ? */
    }
}



QResult::~QResult()
{
    delete tuples;
}
