/* -*- mode: c++; c-basic-offset: 4 -*- */
/********************************************************************
    POSTODBCL.DLL - A library to talk to Postgres95 by using the
                    WINDOWS ODBC Interface

    Copyright (C) 1996; Christian Czezatke

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details. 

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
    How to contact the author:
    
    email to: e9025461@student.tuwien.ac.at
********************************************************************/   


#ifndef __CONNECT_H__
#define __CONNECT_H__

// extern "C" {
#include "compat.h"
#include "sockio.h"
#include "errclass.h"
#include <winsock.h>

#include <stdio.h>
#include "../tuple.h"

typedef UInt4 Oid;

#define MAX_MESSAGE_LEN 8192

/*************************************************/
/* CLASS pgresAttDescList                        */
/*************************************************/
/* This class describes the structure of tuples  */
/* as they are sent by the backend in response   */
/* to a select query                             */
/* this class is supposed to be used only by     */
/* the QResult class                             */
/*************************************************/

class pgresAttDescList
{
public:
  pgresAttDescList();
  char make_result(StreamSocket *sock);
  /* This function reads in the information represented by this
     class from the socket pointed to by sock
  */   
  virtual ~pgresAttDescList();

  /* functions for setting up the fields from within the program, */
  /* without reading from a socket */
  void set_num_fields(int new_num_fields);
  void set_field_info(int field_num, char *new_name, 
                      Oid new_adtid, Int2 new_adtsize);

  /* various functions for obtaining the information */
  
  Int2 get_num_fields();
  /* How many fields are in this query result ? */
  char *get_fieldname(Int2 which);
  /* Returns the name of the nth field in the query result */
  Oid get_oid(Int2 which);
  /* Returns an oid identifying the type of the field */
  Int2 get_fieldsize(Int2 which);
  /* Returns the size of nth field in the query result
     (as it has been defined in the table)
  */   

private:
  void free_memory(void);

  Int2 num_fields;
  char** name; /* list of type names */
  Oid *adtid;  /* list of type idi */
  Int2 *adtsize; /* list type sizes */
};

/* TupleField, TupleNode type definitions moved into tuple.h */

/*************************************************/
/* CLASS TupleList                               */
/*************************************************/
/* This class describes the tuples that are re-  */
/* turned by the backend in response to a select */
/* query                                         */
/* Internally the content of the query is        */ 
/* represented by a doubly linked list of        */
/* TupleNodes                                    */
/* A pgresAttDescList--Object is used for        */
/* representing the format information           */
/* (name and type of the fields contained int    */
/* the query)                                    */
/*************************************************/

#define MAX_FIELDS 512
/* a query result cannot consist of more than MAX_FIELD columns */

#define BYTELEN 8
#define VARHDRSZ sizeof(Int4)


#define ERROR_MSG_LENGTH 4096

class TupleList {
public:
  TupleList(UInt4 fieldcnt);
  /* Create a TupleList. Each tuple consits of fieldcnt columns */
  
  void *get_fieldval(Int4 tupleno, Int2 fieldno);
  Int4 get_num_tuples()
    { return num_tuples; }

  TupleNode *read_next_tuple(StreamSocket *sock, char binary);
  /* read in next tuple from sock, binary indicates whether
     we have to process a binary representation or an ASCII

     CC: 23.05.96: Return value changed from char to TupleNode* to
         allow update of the field length. */
  
  char add_tuple(TupleNode *new_field);
  /* Add the Tuple to the list of tuples we already have got. */
       
  virtual ~TupleList();
  
private:
  Int4 num_fields;
  Int4 num_tuples;  
  TupleNode *list_start, *list_end, *lastref;
  Int4 last_indexed;
  
  TupleNode *read_one_tuple(StreamSocket *sock, char binary);
  /* Read in one tuple. from sock. binary indicates whether the
     tuple is sent in binary or ASCII representation
  */   
};    


/*************************************************/
/* CLASS QResult                                 */
/*************************************************/
/* This class provides end-user access to the    */
/* results of a query. It consits of structure   */
/* information (pgresAttDescr) and a bunch of    */
/* tuples (TupleList)                            */
/*************************************************/

enum QueryResultCode {
  PGRES_EMPTY_QUERY = 0,
  PGRES_COMMAND_OK,  /* a query command that doesn't return */
                    /* anything was executed properly by the backend */
  PGRES_TUPLES_OK,  /* a query command that returns tuples */
                   /* was executed properly by the backend, PGresult */
                   /* contains the resulttuples */
  PGRES_COPY_OUT,
  PGRES_COPY_IN,
  PGRES_BAD_RESPONSE, /* an unexpected response was recv'd from the backend */
  PGRES_NONFATAL_ERROR,
  PGRES_FATAL_ERROR
};



class QResult {
public:
    QResult(QueryResultCode condition=PGRES_EMPTY_QUERY);
    /* Create a new QResult, condition is the return value
       of the Query (By default the query is ok, but empty.) */   
    char make_result(char *portalname, StreamSocket *sock);
    /* Build a QResult-structure by reading the information
       from the StreamSocket *sock. portalname is currently
       ignored. */

    /* routines to add information to a result structure */
    /* by hand (for example, for SQLTables) */
    void set_num_fields(int new_num_fields);
    void set_field_info(int field_num, char *name, 
                        Oid adtid, Int2 adtsize);
    void add_tuple(TupleNode *new_tuple);
  
    virtual ~QResult();

  
    Int4 get_num_tuples()
        /* How many tuples are in the query result ? */
    { return (tuples == NULL) ? 0 : tuples->get_num_tuples(); }
    
    Int2 attribs_per_tuple()
        /* How many fields are in one tuple ? */
    { return fields.get_num_fields(); }

    void *get_value(Int4 tupleno, Int2 fieldno)
        /* Get the raw value of:
           Tuple number tupleno  (valid from: 0 to get_num_tuples()-1)
           field number fieldno (valid from: 0 to attribs_per_tuple()-1) */   
    { return tuples->get_fieldval(tupleno, fieldno); }

    char *get_fieldname(Int2 fieldno)
        /* get name of the fieldno'th field */
    { return fields.get_fieldname(fieldno); }
    
    Int2 get_fieldsize(Int2 fieldno)
        /* get the max. length of the fieldno'th field */
    { return fields.get_fieldsize(fieldno); }

    Oid get_field_type(Int2 fieldno)
    { return fields.get_oid(fieldno); }

    QueryResultCode get_status()
        /* What is the result status of the query ? */
    { return status; }
    
    void set_message(char *buffer)
        /* Assign a new value to the message buffer. Should only
           be called by objects that intend to build a QResult
           structure */   
    { 
        if (NULL != message)
            delete message;
        message = buffer;
    }

    char *get_message()
        /* returns the status message string */
    { return message; }
  
private:

   void update_fieldsizes(TupleNode *node);
   /*
    CC: 23.05.96: ADDED
      checks the tuple "node" whether there are larger entries than the current
      length of the respective column as stored in "fields". If that is the
      case, the length stored in fields is updated accordingly
   */
   
    QueryResultCode status;
    char result_completed;
    pgresAttDescList fields;
    TupleList *tuples;
    char *message;
};    



/*************************************************************************/

#define CONNECTION_SERVER_NOT_REACHED 1
#define CONNECTION_COULD_NOT_SEND_AUTHENTICATION 2
#define CONNECTION_MSG_TOO_LONG 3
#define CONNECTION_COULD_NOT_SEND 4
#define CONNECTION_NO_SUCH_DATABASE 5
#define CONNECTION_BACKEND_CRAZY 6
#define CONNECTION_NO_RESPONSE 7
#define CONNECTION_SERVER_REPORTED_ERROR 8


class Connection : public PgErrorClass
{
public:
  Connection();
  char open_db(char *in_DSN, char *machine, UInt2 port, char *database, char *user);
  QResult *send_query(char *query);

  char *get_dbname()  { return dbasename; }
  char *get_DSN() { return DSN; }

  /* returns the name of the database we are connected to */
  virtual ~Connection();
  
private:
  StreamSocket *sock;
  char *dbasename;
  char *DSN;
  char msgbuffer[MAX_MESSAGE_LEN];
};

// }
#endif
