/* -*- mode: c++; c-basic-offset: 4 -*- */
/********************************************************************
    POSTODBCL.DLL - A library to talk to Postgres95 by using the
                    WINDOWS ODBC Interface

    Copyright (C) 1996; Christian Czezatke, Dan McGuirk

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    How to contact the author:

    email to: e9025461@student.tuwien.ac.at     (Christian Czezatke)
              mcguirk@indirect.com              (Dan McGuirk

********************************************************************/


/*
**
** RESULTS.C - This is the PostODBC driver code for
** returning results and information about results.
**
*/

//      -       -       -       -       -       -       -       -       -

#include <string.h>
#include "globals.h"
#include "custom.h"
#include "socket/wrapper.h"
#include "pgtypes.h" /* CC: for the various PG_* -- types */


#define UNALIGNED
/* CC: We do not compile this code on MIPS machines... */

#define DEFAULT_COLSIZE 66
// CC: Hack to return a field size if it is not supplied by Postgres

/* CC: DEBUG */
char buffer[50];
//      -       -       -       -       -       -       -       -       -

//      This returns the number of columns associated with the database
//      attached to "hstmt".

RETCODE SQL_API SQLNumResultCols(
        HSTMT     hstmt,
        SWORD FAR *pccol)
{
        StatementClass *stmt = HI_stmtfromhstmt(the_handles, hstmt);
        ResultC *result;
        if (NULL == stmt)
            return SQL_INVALID_HANDLE;
        SC_clear_error(stmt);

        /* CC: Now check for the "prepared, but not executed" situation, that enables us to
               deal with "SQLPrepare -- SQLDescribeCol -- ... -- SQLExecute" situations.
               (AutoCAD 13 ASE/ASI just _loves_ that ;-) )
        */
        SC_resultinfo_preprocess(stmt);


        result = SC_get_ResultC(stmt);
        if ((NULL == result) || ((stmt->status != STMT_FINISHED) && (stmt->status != STMT_PREMATURE)) ) {
            /* no query has been executed on this statement */
            stmt->errornumber = STMT_SEQUENCE_ERROR;
#ifdef _ENGLISH_
            stmt->errormsg = "No query has been executed with that handle";
#else
            stmt->errormsg = "Mit dem Handle wurde noch keine Abfrage durchgefhrt";
#endif
            return SQL_ERROR;
        }
        *pccol = ResultC_NumResultCols(result);
        return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

//      Return information about the database column the user wants
//      information about.
/* CC: preliminary implementation */
RETCODE SQL_API SQLDescribeCol(
        HSTMT      hstmt,
        UWORD      icol,
        UCHAR  FAR *szColName,
        SWORD      cbColNameMax,
        SWORD  FAR *pcbColName,
        UNALIGNED SWORD  FAR *pfSqlType,
        UNALIGNED UDWORD FAR *pcbColDef,
        UNALIGNED SWORD  FAR *pibScale,
        SWORD  FAR *pfNullable)
{
    /* gets all the information about a specific column */
    StatementClass *stmt = HI_stmtfromhstmt(the_handles, hstmt);
    ResultC *result;
    char *name;
    Int2 fieldsize;
    Int4 fieldtype;

    if (NULL == stmt)
        return SQL_INVALID_HANDLE;
    SC_clear_error(stmt);

    /* CC: Now check for the "prepared, but not executed" situation, that enables us to
           deal with "SQLPrepare -- SQLDescribeCol -- ... -- SQLExecute" situations.
           (AutoCAD 13 ASE/ASI just _loves_ that ;-) )
    */
    SC_resultinfo_preprocess(stmt);


    result = SC_get_ResultC(stmt);
    if ( (NULL == result) || ((stmt->status != STMT_FINISHED) && (stmt->status != STMT_PREMATURE))) {
        /* no query has been executed on this statement */
        stmt->errornumber = STMT_SEQUENCE_ERROR;
#ifdef _ENGLISH_
        stmt->errormsg = "No query has been assigned to this statement.";
#else
        stmt->errormsg = "Diesem Handle wurde noch keine Abfrage zugeordnet.";
#endif
        return SQL_ERROR;
    }

    if (cbColNameMax >= 1) {
        name = ResultC_get_fieldname(result, icol-1);
        /* our indices start from 0 whereas ODBC defines indices starting from 1 */
        if (NULL != pcbColName)  {
            // we want to get the total number of bytes in the column name
            if (NULL == name)
                *pcbColName = 0;
            else
                *pcbColName = strlen(name);
        }
        if (NULL != szColName) {
            // get the column name into the buffer if there is one
            if (NULL == name)
                szColName[0] = '\0';
            else
                strncpy_null(szColName, name, cbColNameMax);
        }
    }

    /* fieldtype = ResultC_get_field_type(stmt, icol-1); */
    /* CC: stmt is a StatementHandle, whereas result is a ResultC ;-) */
    fieldtype = ResultC_get_field_type(result, icol-1);

    if (NULL != pfSqlType)
        *pfSqlType = pgtype_to_sqltype(fieldtype);
    if (NULL != pcbColDef) {
        // we can't just use pgtype_to_precision here because that gives
        // the maximum precision for the data type, and we really just
        // want the precision of the column.  if it's a string type,
        // we try to figure out the real length--otherwise, just
        // use the default precision.
        /* CC: This code is now also in SQLColAttributes/SQL_COLUMN_LENGTH */

        *pcbColDef = pgtype_precision(fieldtype);
        if(fieldtype == PG_TYPE_CHAR ||
           fieldtype == PG_TYPE_VARCHAR ||
           fieldtype == PG_TYPE_BPCHAR || /* CC: Added BPCHAR */
           fieldtype == PG_TYPE_TEXT ||
           fieldtype == PG_TYPE_NAME) {

           fieldsize = ResultC_get_fieldsize(result, icol-1);

            if (fieldsize >= 0) {
                *pcbColDef = fieldsize;
            } else {
                *pcbColDef = pgtype_precision(fieldtype);
            }
        } else {
            *pcbColDef = pgtype_precision(fieldtype);
        }

    }
    if (NULL != pibScale) {
        Int2 scale;
        scale = pgtype_scale(fieldtype);
        if(scale == -1) { scale = 0; }

        *pibScale = scale;
    }
    if (NULL != pfNullable) {
        *pfNullable = pgtype_nullable(fieldtype);
    }

    return SQL_SUCCESS;
}



//      -       -       -       -       -       -       -       -       -

//      Returns result column descriptor information for a result set.

RETCODE SQL_API SQLColAttributes(
        HSTMT      hstmt,
        UWORD      icol,
        UWORD      fDescType,
        PTR        rgbDesc,
        SWORD      cbDescMax,
        SWORD  FAR *pcbDesc,
        SDWORD FAR *pfDesc)
{
    StatementClass *stmt;
    char *value;
    Int4 field_type;


    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    /* CC: Now check for the "prepared, but not executed" situation, that enables us to
           deal with "SQLPrepare -- SQLDescribeCol -- ... -- SQLExecute" situations.
           (AutoCAD 13 ASE/ASI just _loves_ that ;-) )
    */
    SC_resultinfo_preprocess(stmt);

    if ( (NULL == stmt->result) || ((stmt->status != STMT_FINISHED) && (stmt->status != STMT_PREMATURE)) ) {
#ifdef _ENGLISH_
        stmt->errormsg = "Can't get column attributes: no result found.";
#else
        stmt->errormsg = "Kann keine ResultC Struktur zu diesem Statement finden.";
#endif
        stmt->errornumber = STMT_SEQUENCE_ERROR;
        return SQL_ERROR;
    }

    if(icol < 1) {
        // we do not support bookmarks
#ifdef _ENGLISH_
        stmt->errormsg = "Bookmarks are not currently supported.";
#else
        stmt->errormsg = "Bookmarks werden zur Zeit nicht untersttzt.";
#endif
        stmt->errornumber = STMT_NOT_IMPLEMENTED_ERROR;
        return SQL_ERROR;
    }

    icol -= 1;
    field_type = ResultC_get_field_type(stmt->result, icol);

    switch(fDescType) {
    case SQL_COLUMN_AUTO_INCREMENT:
        if (NULL != pfDesc) {
            *pfDesc = pgtype_auto_increment(field_type);

            if(*pfDesc == -1) { /* "not applicable" becomes false */
                *pfDesc = FALSE;
            }
        }
        break;
    case SQL_COLUMN_CASE_SENSITIVE:
        if (NULL != pfDesc)
          *pfDesc = pgtype_case_sensitive(field_type);
        break;
    case SQL_COLUMN_COUNT:
        if (NULL != pfDesc)
          *pfDesc = ResultC_attribs_per_tuple(stmt->result);
        break;
    case SQL_COLUMN_DISPLAY_SIZE:
        // super-fake
        /* CC: Now determines length from result set. Longest entry determines return value */
        if (NULL != pfDesc)
          *pfDesc = ResultC_get_fieldsize(stmt->result, icol);
        break;
    case SQL_COLUMN_LABEL:
    case SQL_COLUMN_NAME:
        value = ResultC_get_fieldname(stmt->result, icol);
        strncpy_null((char *)rgbDesc, value, cbDescMax);
        /* CC: Check for Nullpointesr */
        if (NULL != pcbDesc)
          *pcbDesc = strlen(value);
        break;
    case SQL_COLUMN_LENGTH:
        /* CC: Returning SQL_Error is rather rude -> Try to determine it */

        // we can't just use pgtype_to_precision here because that gives
        // the maximum precision for the data type, and we really just
        // want the precision of the column.  if it's a string type,
        // we try to figure out the real length--otherwise, just
        // use the default precision.

        if (NULL != pfDesc)
          *pfDesc = pgtype_precision(field_type);
        return SQL_SUCCESS;
        break;
    case SQL_COLUMN_MONEY:
        if (NULL != pfDesc)
          *pfDesc = pgtype_money(field_type);
        break;
    case SQL_COLUMN_NULLABLE:
        if (NULL != pfDesc)
          *pfDesc = pgtype_nullable(field_type);
        break;
    case SQL_COLUMN_OWNER_NAME:
        return SQL_ERROR;
        break;
    case SQL_COLUMN_PRECISION:
        if (NULL != pfDesc)
          *pfDesc = pgtype_precision(field_type);
        break;
    case SQL_COLUMN_QUALIFIER_NAME:
        strncpy_null((char *)rgbDesc, "", cbDescMax);
        if (NULL != pfDesc)
          *pcbDesc = 1;
        break;
    case SQL_COLUMN_SCALE:
        if (NULL != pfDesc)
          *pfDesc = pgtype_scale(field_type);
        break;
    case SQL_COLUMN_SEARCHABLE:
        if (NULL != pfDesc)
          *pfDesc = pgtype_searchable(field_type);
        break;
    case SQL_COLUMN_TABLE_NAME:
        return SQL_ERROR;
        break;
    case SQL_COLUMN_TYPE:
        if (NULL != pfDesc)
          *pfDesc = pgtype_to_sqltype(field_type);
        break;
    case SQL_COLUMN_TYPE_NAME:
        value = pgtype_to_name(field_type);
        strncpy_null((char *)rgbDesc, value, cbDescMax);
        if (NULL != pcbDesc)
          *pcbDesc = strlen(value);
        break;
    case SQL_COLUMN_UNSIGNED:
        if (NULL != pfDesc) {
            *pfDesc = pgtype_unsigned(field_type);
            if(*pfDesc == -1) {
                *pfDesc = FALSE;
            }
        }
        break;
    case SQL_COLUMN_UPDATABLE:
        // everything should be updatable, I guess, unless access permissions
        // prevent it--are we supposed to check for that here?  seems kind
        // of complicated.  hmm...
        if (NULL != pfDesc)
          *pfDesc = SQL_ATTR_WRITE;
        break;
    }

    return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

//      Associate a user-supplied buffer with a database column.
/* CC: preliminary implementation - only SQL_C_CHAR and SQL_C_DEFAULT are supported as types */
RETCODE SQL_API SQLBindCol(
        HSTMT      hstmt,
        UWORD      icol,
        SWORD      fCType,
        PTR        rgbValue,
        SDWORD     cbValueMax,
        SDWORD FAR *pcbValue)
{
        StatementClass *stmt;

        stmt = HI_stmtfromhstmt(the_handles, hstmt);
        if (NULL == stmt)
            return SQL_INVALID_HANDLE;

        if (icol < 1) {
            /* currently we do not support bookmarks */
            stmt->errormsg = "Bookmarks are not currently supported.";
            stmt->errornumber = STMT_NOT_IMPLEMENTED_ERROR;
            return SQL_ERROR;
        }

        SC_clear_error(stmt);

        /* now we have got the StatementClass instance, now let us bind */
        if (!SC_bind_cols(stmt, icol-1, cbValueMax, (char *)rgbValue, pcbValue, (Int2)fCType)) {
            // error message passed from above
            return SQL_ERROR;
        }

        return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

//      Returns data for bound columns in the current row ("hstmt->iCursor"),
//      advances the cursor.

/* CC: works */
RETCODE SQL_API SQLFetch(
        HSTMT   hstmt)
{
 StatementClass *stmt;
 int rows_in_result;

 stmt = HI_stmtfromhstmt(the_handles, hstmt);

 if (NULL == stmt)
     return SQL_INVALID_HANDLE;
 SC_clear_error(stmt);

 if (NULL == stmt->result) {
     stmt->errormsg = "Null statement result in SQLFetch.";
     stmt->errornumber = STMT_SEQUENCE_ERROR;
     return SQL_ERROR;
 }

 rows_in_result = ResultC_get_num_tuples(stmt->result);

 // increment the pointer as long as we're not _already_
 // off the end of the results
 if(stmt->currTuple < rows_in_result) {
     stmt->currTuple++;
 }

 // but we might be off the end of the results _now_,
 // so check for that
 if(stmt->currTuple >= rows_in_result) {
     return SQL_NO_DATA_FOUND;
 }

 if (SC_perform_fetch(stmt, 0) == -1) { /* 0 == don't do extended fetch */
     // error passed from above
     return SQL_ERROR;
 }

 if (STMT_TRUNCATED == stmt->errornumber)
   return SQL_SUCCESS_WITH_INFO;

 return SQL_SUCCESS;
}

//      Returns result data for a single column in the current row.

RETCODE SQL_API SQLGetData(
        HSTMT      hstmt,
        UWORD      icol,
        SWORD      fCType,
        PTR        rgbValue,
        SDWORD     cbValueMax,
        SDWORD FAR *pcbValue)
{
    StatementClass *stmt;
    int num_cols, num_rows;
    Int4 field_type;
    void *value;
    int result;

    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    if (STMT_EXECUTING == stmt->status) {
#ifdef _ENGLISH_
        stmt->errormsg = "Can't get data while statement is still executing.";
#else
        stmt->errormsg = "Kann keine Daten ermitteln whrend ein Statement ausgefhrt wird.";
#endif
        stmt->errornumber = STMT_SEQUENCE_ERROR;
        return 0;
    }

    if (stmt->status != STMT_FINISHED) {
        stmt->errornumber = STMT_STATUS_ERROR;
#ifdef _ENGLISH_
        stmt->errormsg = "GetData can only be called after the successful execution on a SQL statement";
#else
        stmt->errormsg = "GetData kann nur nach erfolgter Ausfhrung eines SQL Statements verwendet werden";
#endif
        return 0;
    }

    if(icol == 0) {
#ifdef _ENGLISH_
        stmt->errormsg = "Bookmarks are not currently supported.";
#else
        stmt->errormsg = "Bookmarks werden momentan nicht untersttzt.";
#endif
        stmt->errornumber = STMT_NOT_IMPLEMENTED_ERROR;
        return SQL_ERROR;
    }

    // use zero-based column numbers
    icol--;

    // make sure the column number is valid
    num_cols = ResultC_NumResultCols(stmt->result);

    /* CC: "icol < 0" removed in successive if statements (always FALSE since
           icol is an UWORD (=unsigned)). Anyway, Watcom C was issuing the
           warning "meaninless use of an expression
    */
    if (icol >= num_cols) {
#ifdef _ENGLISH_
        stmt->errormsg = "Invalid column number.";
#else
        stmt->errormsg = "Ungltige Spaltennummer.";
#endif
        stmt->errornumber = STMT_INVALID_COLUMN_NUMBER_ERROR;
        return SQL_ERROR;
    }

    // make sure we're positioned on a valid row
    num_rows = ResultC_get_num_tuples(stmt->result);
    if((stmt->currTuple < 0) ||
       (stmt->currTuple >= num_rows)) {
#ifdef _ENGLISH_
        stmt->errormsg = "Not positioned on a valid row for GetData.";
#else
        stmt->errormsg = "Zeilenposition ist nicht gltig fr GetData.";
#endif
        stmt->errornumber = STMT_INVALID_CURSOR_STATE_ERROR;
        return SQL_ERROR;
    }

    field_type = ResultC_get_field_type(stmt->result, icol);

    value = ResultC_get_value(stmt->result, stmt->currTuple, icol);
    result = copy_and_convert_field(field_type, value,
                                    fCType, rgbValue, cbValueMax, pcbValue);

    if(result == COPY_UNSUPPORTED_TYPE) {
#ifdef _ENGLISH_
        stmt->errormsg = "Received an unsupported type from Postgres.";
#else
        stmt->errormsg = "Von Postgres wurde ein nicht untersttzter Datentyp geliefert.";
#endif
        stmt->errornumber = STMT_RESTRICTED_DATA_TYPE_ERROR;
        return SQL_ERROR;
    } else if(result == COPY_UNSUPPORTED_CONVERSION) {
#ifdef _ENGLISH_
        stmt->errormsg = "Couldn't handle the necessary data type conversion.";
#else
        stmt->errormsg = "Datenkonversion konnte nicht durchgefhrt werden.";
#endif
        stmt->errornumber = STMT_RESTRICTED_DATA_TYPE_ERROR;
        return SQL_ERROR;
    } else if(result == COPY_RESULT_TRUNCATED) {
        stmt->errornumber = STMT_TRUNCATED;
#ifdef _ENGLISH_
        stmt->errormsg = "The buffer was too small for the result.";
#else
        stmt->errormsg = "Der Rckgabepuffer ist zu klein fr das Ergebnis.";
#endif
        return SQL_SUCCESS_WITH_INFO;
    } else if(result != COPY_OK) {
#ifdef _ENGLISH_
        stmt->errormsg = "Unrecognized return value from copy_and_convert_field.";
#else
        stmt->errormsg = "Interner Fehler: Unerwarteter Rckgabewert von copy_and_convert_field.";
#endif
        stmt->errornumber = STMT_INTERNAL_ERROR;
        return SQL_ERROR;
    }

    return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

//      This determines whether there are more results sets available for
//      the "hstmt".

/* CC: return SQL_NO_DATA_FOUND since we do not support multiple result sets */
RETCODE SQL_API SQLMoreResults(
        HSTMT   hstmt)
{
          return SQL_NO_DATA_FOUND;
}

//      -       -       -       -       -       -       -       -       -

//      This returns the number of rows associated with the database
//      attached to "hstmt".

// (it is supposed to return the number of rows affected by an update,
// delete, or insert as well... but how the hell are we supposed to know that?)

RETCODE SQL_API SQLRowCount(
        HSTMT      hstmt,
        SDWORD FAR *pcrow)
{
    StatementClass *stmt = HI_stmtfromhstmt(the_handles, hstmt);
    ResultC *res;

    if (NULL == stmt)
       return SQL_ERROR;

    if(stmt->statement_type == STMT_TYPE_SELECT) {
        if (stmt->status == STMT_FINISHED) {
            res = SC_get_ResultC(stmt);
            if(res && pcrow) {
                *pcrow = ResultC_get_num_tuples(res);
                return SQL_SUCCESS;
            }
        }
    } else {
        // it says we can return -1 if the 'number of affected
        // rows is not available'.  since I can't figure out how
        // we'll ever know the number of affected rows, just
        // return -1 always...
        if(pcrow) {
#ifdef FAKE_ROWCOUNT
            // Access relies on this--it doesn't interpret the result
            // 'not available'.  so we have to fake it out...
            *pcrow = 1;
#else
            *pcrow = -1;
#endif
            return SQL_SUCCESS;
        }
    }

    return SQL_ERROR;
}

//      -       -       -       -       -       -       -       -       -

//      This positions the cursor within a block of data.

RETCODE SQL_API SQLSetPos(
        HSTMT   hstmt,
        UWORD   irow,
        UWORD   fOption,
        UWORD   fLock)
{
        return SQL_ERROR;
}

//      -       -       -       -       -       -       -       -       -

//      This fetchs a block of data (rowset).

RETCODE SQL_API SQLExtendedFetch(
        HSTMT      hstmt,
        UWORD      fFetchType,
        SDWORD     irow,
        UDWORD FAR *pcrow,
        UWORD  FAR *rgfRowStatus)
{
    StatementClass *stmt;
    int rows_in_result;
    Int4 rows_fetched, i; /* CC: changed that to int4 for 16bit compile */

    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    if(!stmt->result) {
        stmt->errormsg = "Null statement result in SQLExtendedFetch.";
        stmt->errornumber = STMT_SEQUENCE_ERROR;
        return SQL_ERROR;
    }

    rows_in_result = ResultC_get_num_tuples(stmt->result);
    // just so I can assume below that there is at least one row,
    // and not confuse myself.
    if(rows_in_result == 0) {
        return SQL_NO_DATA_FOUND;
    }

    // first try to figure out which row we should be starting with
    // here's another example of the simplicity and elegance
    // of the ODBC spec
    switch(fFetchType) {
    case SQL_FETCH_NEXT:
        // if we haven't started yet, it's a special case.
        // start at the first row.  otherwise, skip the size
        // of one rowset.
        if(stmt->currTuple == -1) {
            stmt->currTuple = 0;
        } else {
            stmt->currTuple += stmt->rowset_size;
        }
        break;
    case SQL_FETCH_PRIOR:
        // special case for past the end of the result set--
        // move to the last rowset in the result set.
        // otherwise, go backwards by one complete rowset.
        if(stmt->currTuple >= rows_in_result) {
            stmt->currTuple = rows_in_result - stmt->rowset_size;
        } else {
            stmt->currTuple -= stmt->rowset_size;
        }
        break;
    case SQL_FETCH_RELATIVE:
        // special case again--if we're before the start
        // and irow > 0, or after the end and irow < 0,
        // then this is exactly like SQL_FETCH_ABSOLUTE.
        if((stmt->currTuple < 0 && irow > 0) ||
           (stmt->currTuple >= rows_in_result && irow < 0)) {
            if(irow > 0) {
                stmt->currTuple = irow;
            } else {
                stmt->currTuple = rows_in_result + irow;
            }
        } else {
            stmt->currTuple += irow;
        }
        break;
    case SQL_FETCH_FIRST:
        stmt->currTuple = 0;
        break;
    case SQL_FETCH_LAST:
        stmt->currTuple = rows_in_result - stmt->rowset_size;
        break;
    case SQL_FETCH_ABSOLUTE:
        // absolute position, unless it's 0, in which case
        // we return NO_DATA_FOUND and position before the
        // start of the result set.
        if(irow > 0) {
            stmt->currTuple = irow-1; /* CC: Added "-1" since our indices start with 0, whereas odbc indices start with 1 */
        } else if(irow == 0) {
            stmt->currTuple = -1;
            return SQL_NO_DATA_FOUND;
        } else {
            stmt->currTuple = rows_in_result + irow;
            /* CC: Do we have to correct the index here too ? */
        }
        break;
    default:
        stmt->errormsg = "Bad fFetchType argument to SQLExtendedFetch.";
        stmt->errornumber = STMT_SEQUENCE_ERROR;
    }

    // now see where our pointer ended up.  the table on pg. 310
    // of the ODBC Programmer's Reference says what to do.
    if(stmt->currTuple <= -(Int4)stmt->rowset_size) {  // watch sign here
        // we're all the way off the beginning of the set.  return nothing.
        return SQL_NO_DATA_FOUND;
    } else if(stmt->currTuple < 0) {
        // we're off the beginning of the set, but overlapping with
        // the beginning of the set.  so start at the beginning of the set.
        stmt->currTuple = 0;
    } else if(stmt->currTuple >= rows_in_result) {
        // we're off the end of the set (not overlapping).  return nothing.
        return SQL_NO_DATA_FOUND;
    }

    rows_fetched = SC_perform_fetch(stmt, 1);   /* 1 == do extended fetch */
    if(rows_fetched == -1) {
        // error message passed from above
        return SQL_ERROR;
    }

    // set *pcrow
    // CC: check for pcrow being NULL
    if (NULL != pcrow)
        *pcrow = rows_fetched;

    // fill in rgfRowStatus.  the only results we return are success
    // or no row, since there will never be an error fetching an individual
    // row unless there is an error with the whole fetch.
    // CC: check for rgfRowStatus being NULL
    if (NULL != rgfRowStatus) {
        for(i = 0; i < stmt->rowset_size; i++) {
            if(i < rows_fetched) {
                rgfRowStatus[i] = SQL_ROW_SUCCESS;
            } else {
                rgfRowStatus[i] = SQL_ROW_NOROW;
            }
        }
    }

    return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

//      Returns the next SQL error information.

RETCODE SQL_API SQLError(
        LPENV      henv,
        LPDBC      hdbc,
        HSTMT      hstmt,
        UCHAR  FAR *szSqlState,
        SDWORD FAR *pfNativeError,
        UCHAR  FAR *szErrorMsg,
        SWORD      cbErrorMsgMax,
        SWORD  FAR *pcbErrorMsg)
{
    char *msg;
    int status;

    if (SQL_NULL_HSTMT != hstmt) {
        // CC: return an error of a hstmt
        StatementClass *stmt = HI_stmtfromhstmt(the_handles, hstmt);

        if (NULL == stmt)
            return SQL_INVALID_HANDLE;

        if (SC_get_error(stmt, &status, &msg)) {
            if (NULL == msg) {
                if (NULL != szSqlState)
                    strcpy(szSqlState, "00000");
                if (NULL != pcbErrorMsg)
                    *pcbErrorMsg = 0;
                if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
                    szErrorMsg[0] = '\0';

                return SQL_NO_DATA_FOUND;
            }
            if (NULL != pcbErrorMsg)
                *pcbErrorMsg = (SWORD)strlen(msg);

            if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
                strncpy_null(szErrorMsg, msg, cbErrorMsgMax);

            if (NULL != pfNativeError)
                *pfNativeError = status;

            if (NULL != szSqlState)

                switch (status) {
                    // now determine the SQLSTATE to be returned
                case STMT_TRUNCATED:
                    strcpy(szSqlState, "01004");
                    // data truncated
                    break;
                case STMT_INFO_ONLY:
                    strcpy(szSqlState, "00000");
                    // just information that is returned, no error
                    break;
                case STMT_EXEC_ERROR:
                    strcpy(szSqlState, "08S01");
                    // communication link failure
                    break;
                case STMT_STATUS_ERROR:
                case STMT_SEQUENCE_ERROR:
                    strcpy(szSqlState, "S1010");
                    // Function sequence error
                    break;
                case STMT_NO_MEMORY_ERROR:
                    strcpy(szSqlState, "S1001");
                    // memory allocation failure
                    break;
                case STMT_COLNUM_ERROR:
                    strcpy(szSqlState, "S1002");
                    // invalid column number
                    break;
                case STMT_NO_STMTSTRING:
                    strcpy(szSqlState, "S1001");
                    // having no stmtstring is also a malloc problem
                    break;
                case STMT_ERROR_TAKEN_FROM_BACKEND:
                    strcpy(szSqlState, "S1000");
                    // general error
                    break;
                case STMT_INTERNAL_ERROR:
                    strcpy(szSqlState, "S1000");
                    // general error
                    break;
                case STMT_NOT_IMPLEMENTED_ERROR:
                    strcpy(szSqlState, "S1C00"); // == 'driver not capable'
                    break;
                case STMT_OPTION_OUT_OF_RANGE_ERROR:
                    strcpy(szSqlState, "S1092");
                    break;
                case STMT_BAD_PARAMETER_NUMBER_ERROR:
                    strcpy(szSqlState, "S1093");
                    break;
                case STMT_INVALID_COLUMN_NUMBER_ERROR:
                    strcpy(szSqlState, "S1002");
                    break;
                case STMT_RESTRICTED_DATA_TYPE_ERROR:
                    strcpy(szSqlState, "07006");
                    break;
                case STMT_INVALID_CURSOR_STATE_ERROR:
                    strcpy(szSqlState, "24000");
                    break;
                case STMT_OPTION_VALUE_CHANGED:
                    strcpy(szSqlState, "01S02");
                    break;
                default:
                    strcpy(szSqlState, "S1000");
                    // also a general error
                    break;
                }

        } else {
            if (NULL != szSqlState)
                strcpy(szSqlState, "00000");
            if (NULL != pcbErrorMsg)
                *pcbErrorMsg = 0;
            if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
                szErrorMsg[0] = '\0';

            return SQL_NO_DATA_FOUND;
        }
        return SQL_SUCCESS;

    } else if (SQL_NULL_HDBC != hdbc) {
        ConnectionClass *conn = HI_connfromhdbc(the_handles, hdbc);
        if (NULL == conn)
            return SQL_INVALID_HANDLE;

        if (CC_get_error(conn, &status, &msg)) {
            if (NULL == msg) {
                if (NULL != szSqlState)
                    strcpy(szSqlState, "00000");
                if (NULL != pcbErrorMsg)
                    *pcbErrorMsg = 0;
                if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
                    szErrorMsg[0] = '\0';

                return SQL_NO_DATA_FOUND;
            }

            if (NULL != pcbErrorMsg)
                *pcbErrorMsg = (SWORD)strlen(msg);
            if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
                strncpy_null(szErrorMsg, msg, cbErrorMsgMax);
            if (NULL != pfNativeError)
                *pfNativeError = status;

            if (NULL != szSqlState)
                switch(status) {
                case CONN_INIREAD_ERROR:
                    strcpy(szSqlState, "IM002");
                    // data source not found
                    break;
                case CONN_OPENDB_ERROR:
                    strcpy(szSqlState, "08001");
                    // unable to connect to data source
                    break;
                case CONN_STMT_ALLOC_ERROR:
                    strcpy(szSqlState, "S1001");
                    // memory allocation failure
                    break;
                case CONN_IN_USE:
                    strcpy(szSqlState, "S1000");
                    // general error
                    break;
                case CONN_UNSUPPORTED_OPTION:
                    strcpy(szSqlState, "IM001");
                    // driver does not support this function
                case CONN_INVALID_ARGUMENT_NO:
                    strcpy(szSqlState, "S1009");
                    // invalid argument value
                    break;
                case CONN_TRANSACT_IN_PROGRES:
                    strcpy(szSqlState, "S1010");
                    // when the user tries to switch commit mode in a transaction
                    // -> function sequence error
                    break;
                case CONN_NO_MEMORY_ERROR:
                    strcpy(szSqlState, "S1001");
                    break;
                case CONN_NOT_IMPLEMENTED_ERROR:
                    strcpy(szSqlState, "S1C00");
                    break;
                default:
                    strcpy(szSqlState, "S1000");
                    // general error
                    break;
                }

        } else {
            if (NULL != szSqlState)
                strcpy(szSqlState, "00000");
            if (NULL != pcbErrorMsg)
                *pcbErrorMsg = 0;
            if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
                szErrorMsg[0] = '\0';

            return SQL_NO_DATA_FOUND;
        }
        return SQL_SUCCESS;

    } else if (SQL_NULL_HENV != henv) {
        EnvironmentClass *env = (EnvironmentClass *)henv;
        if(EN_get_error(env, &status, &msg)) {
            if (NULL == msg) {
                if (NULL != szSqlState)
                    strcpy(szSqlState, "00000");
                if (NULL != pcbErrorMsg)
                    *pcbErrorMsg = 0;
                if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
                    szErrorMsg[0] = '\0';

                return SQL_NO_DATA_FOUND;
            }

            if (NULL != pcbErrorMsg)
                *pcbErrorMsg = (SWORD)strlen(msg);
            if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
                strncpy_null(szErrorMsg, msg, cbErrorMsgMax);
            if (NULL != pfNativeError)
                *pfNativeError = status;

            if(szSqlState) {
                switch(status) {
                case ENV_ALLOC_ERROR:
                    // memory allocation failure
                    strcpy(szSqlState, "S1001");
                    break;
                default:
                    strcpy(szSqlState, "S1000");
                    // general error
                    break;
                }
            }
        } else {
            if (NULL != szSqlState)
                strcpy(szSqlState, "00000");
            if (NULL != pcbErrorMsg)
                *pcbErrorMsg = 0;
            if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
                szErrorMsg[0] = '\0';

            return SQL_NO_DATA_FOUND;
        }

        return SQL_SUCCESS;
    }

    if (NULL != szSqlState)
        strcpy(szSqlState, "00000");
    if (NULL != pcbErrorMsg)
        *pcbErrorMsg = 0;
    if ((NULL != szErrorMsg) && (cbErrorMsgMax > 0))
        szErrorMsg[0] = '\0';

    return SQL_NO_DATA_FOUND;
}

//      -       -       -       -       -       -       -       -       -
