/* -*- mode: c++; c-basic-offset: 4 -*- */
/********************************************************************
    POSTODBCL.DLL - A library to talk to Postgres95 by using the
                    WINDOWS ODBC Interface

    Copyright (C) 1996; Christian Czezatke, Dan McGuirk

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details. 

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
    How to contact the authors:
    
    email to: e9025461@student.tuwien.ac.at     (Christian Czezatke)
              mcguirk@indirect.com              (Dan McGuirk)  
********************************************************************/ 

/*
** PREPARE.C - This is the PostODBC driver code for
** preparing SQL Commands and other functions prior to execution.
*/

//      -       -       -       -       -       -       -       -       -

#ifdef __WATCOMC__
#include <stdlib.h> /* CC: for malloc/free prototypes */
#include <string.h> /* CC: for strlen prototype */
#endif

#include "globals.h"
#include "pgtypes.h"

//      -       -       -       -       -       -       -       -       -

//      Allocate a SQL statement
/* CC: works */
RETCODE SQL_API SQLAllocStmt(
                             HDBC      hdbc,
                             HSTMT FAR *phstmt)
{
    HGLOBAL hstmt;
    ConnectionClass *conn;

    conn = HI_connfromhdbc(the_handles, hdbc);
    if(!conn) {
        return SQL_INVALID_HANDLE;
    }

    hstmt = GlobalAlloc (GMEM_MOVEABLE | GMEM_ZEROINIT, sizeof (STMT));
    if (!hstmt || (*phstmt = (HSTMT)GlobalLock (hstmt)) == SQL_NULL_HSTMT) {
        GlobalFree (hstmt);     //      Free it if lock fails
#ifdef _ENGLISH_        
        conn->errormsg = "Couldn't allocate memory for statement handle.";
#else
        conn->errormsg = "Konnte Speicher fr Statement Handle nicht anfordern.";
#endif        
        conn->errornumber = CONN_NO_MEMORY_ERROR;
        return SQL_ERROR;
    }

    CC_create_new_statement(conn, *phstmt);

    return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

RETCODE SQL_API SQLFreeStmt(
                            HSTMT     hstmt,
                            UWORD     fOption)
{
    StatementClass *my_stmt;

    my_stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if (my_stmt == NULL)
        return SQL_INVALID_HANDLE;
    
    if (fOption == SQL_DROP) {
        /* the user discards the hstmt */
        if (CC_destroy_statement(SC_get_conn(my_stmt), hstmt)) {
            GlobalUnlock (GlobalPtrHandle(hstmt));
            GlobalFree (GlobalPtrHandle(hstmt));
        } else
            return SQL_ERROR;
    } else if (fOption == SQL_UNBIND) {
	SC_unbind_cols(my_stmt);
    } else if (fOption == SQL_CLOSE) {
	/* this should discard all the results, but leave the statement */
	/* itself in place (it can be executed again) */
        if (!SC_recycle_statement(my_stmt))
            return SQL_ERROR;

    } else if(fOption == SQL_RESET_PARAMS) {
        if(my_stmt->parameters) {
            free(my_stmt->parameters);
        }
	my_stmt->parameters = 0;
        my_stmt->parameters_allocated = 0;
    } else {
#ifdef _ENGLISH_        
        my_stmt->errormsg = "Invalid option passed to SQLFreeStmt.";
#else
        my_stmt->errormsg = "Ungltige Option fr SQLFreeStmt.";
#endif        
        my_stmt->errornumber = STMT_OPTION_OUT_OF_RANGE_ERROR;
        return SQL_ERROR;
    }
    
    return SQL_SUCCESS;
}


//      -       -       -       -       -       -       -       -       -

//      Perform a Prepare on the SQL statement
/* CC: works */
RETCODE SQL_API SQLPrepare(
        HSTMT     hstmt,
        UCHAR FAR *szSqlStr,
        SDWORD    cbSqlStr)
{
    StatementClass *stmt;
    /* see if shortcut helps to find the statementclass quickly */
    if (last_hstmt == hstmt) {
        /* yes, it did help! */
        stmt = last_stmt;
    } else {
        /* no, it did not help! */
        stmt = HI_stmtfromhstmt(the_handles, hstmt);
        if (NULL == stmt) {
            return SQL_INVALID_HANDLE;
        }
        last_stmt = stmt;
        last_hstmt = stmt->hstmt;
    }
    if (!SC_do_prepare(stmt, szSqlStr, cbSqlStr))
        return SQL_ERROR;

    return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

//      Bind parameters on a statement handle

RETCODE SQL_API SQLBindParameter(
        HSTMT      hstmt,
        UWORD      ipar,
        SWORD      fParamType,
        SWORD      fCType,
        SWORD      fSqlType,
        UDWORD     cbColDef,
        SWORD      ibScale,
        PTR        rgbValue,
        SDWORD     cbValueMax,
        SDWORD FAR *pcbValue)
{
    StatementClass *stmt;

    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    if(!SC_bind_parameter(stmt, ipar-1, cbValueMax, rgbValue, pcbValue,
                          fParamType, fCType, fSqlType, cbColDef, ibScale)) {
        return SQL_ERROR;
    }

    return SQL_SUCCESS;

}

//      -       -       -       -       -       -       -       -       -

//      Returns the description of a parameter marker.

RETCODE SQL_API SQLDescribeParam(
        HSTMT      hstmt,
        UWORD      ipar,
        SWORD  FAR *pfSqlType,
        UDWORD FAR *pcbColDef,
        SWORD  FAR *pibScale,
        SWORD  FAR *pfNullable)
{
    StatementClass *stmt;

    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    /* CC: The following section got modified since ipar is an UWORD and thus
           cannot get negative values. (it would simply underflow)
           Watcom also issues a "meaningless use of an expression" warning                     
    */           

    if( (ipar < 1) || (ipar > stmt->parameters_allocated) ) {
#ifdef _ENGLISH_        
        stmt->errormsg = "Invalid parameter number for SQLDescribeParam.";
#else
        stmt->errormsg = "Unzulssige Parametergre fr SQLDescribeParam.";
#endif        
        stmt->errornumber = STMT_BAD_PARAMETER_NUMBER_ERROR;
        return SQL_ERROR;
    }
    ipar--;

    if(pfSqlType) {
        *pfSqlType = stmt->parameters[ipar].SQLType;
    }
    if(pcbColDef) {
        *pcbColDef = stmt->parameters[ipar].precision;
    }
    if(pibScale) {
        *pibScale = stmt->parameters[ipar].scale;
    }
    if(pfNullable) {
        *pfNullable = pgtype_nullable(stmt->parameters[ipar].paramType);
    }
    
    return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

//      Sets multiple values (arrays) for the set of parameter markers.

RETCODE SQL_API SQLParamOptions(
        HSTMT      hstmt,
        UDWORD     crow,
        UDWORD FAR *pirow)
{
        return SQL_ERROR;
}

//      -       -       -       -       -       -       -       -       -

//      Returns the number of parameter markers.

RETCODE SQL_API SQLNumParams(
        HSTMT      hstmt,
        SWORD  FAR *pcpar)
{
    StatementClass *stmt;
    unsigned int i;

    // I guess this is the number of actual parameter markers
    // in the statement, not the number of parameters that are bound.
    // why does this have to be driver-specific?

    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }
    
    if(!stmt->statement) {
        // no statement has been allocated
        *pcpar = 0;
#ifdef _ENGLISH_        
        stmt->errormsg = "SQLNumParams called with no statement ready.";
#else
        stmt->errormsg = "SQLNumParams wurde aufgerufen, obwohl kein Statement angefordert ist.";
#endif        
        stmt->errornumber = STMT_SEQUENCE_ERROR;
        return SQL_ERROR;
    } else {
        *pcpar = 0;
        for(i=0; i < strlen(stmt->statement); i++) {
            if(stmt->statement[i] == '?') {
                (*pcpar)++;
            }
        }

        return SQL_SUCCESS;
    }
}

//      -       -       -       -       -       -       -       -       -

//      Sets options that control the behavior of cursors.

RETCODE SQL_API SQLSetScrollOptions(
        HSTMT      hstmt,
        UWORD      fConcurrency,
        SDWORD  crowKeyset,
        UWORD      crowRowset)
{
        return SQL_ERROR;
}

//      -       -       -       -       -       -       -       -       -

//      Set the cursor name on a statement handle

RETCODE SQL_API SQLSetCursorName(
        HSTMT     hstmt,
        UCHAR FAR *szCursor,
        SWORD     cbCursor)
{
        return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

//      Return the cursor name for a statement handle

RETCODE SQL_API SQLGetCursorName(
        HSTMT     hstmt,
        UCHAR FAR *szCursor,
        SWORD     cbCursorMax,
        SWORD FAR *pcbCursor)
{
        return SQL_ERROR;
}
