/* -*- mode: c++; c-basic-offset: 4 -*- */
/********************************************************************
    POSTODBCL.DLL - A library to talk to Postgres95 by using the
                    WINDOWS ODBC Interface

    Copyright (C) 1996; Dan McGuirk

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details. 

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
    How to contact the author:
    
    email to: mcguirk@indirect.com              (Dan McGuirk)
    
********************************************************************/  


#include "pgtypes.h"
#include "custom.h"
#include "socket/compat.h"
#include <windows.h>
#include <sql.h>
#include <sqlext.h>

/* these are the types we support.  all of the pgtype_ functions should */
/* return values for each one of these.                                 */

/* to add a type:  add it to this list.  add it to all of the pgtype_*  */
/* functions.  then check copy_and_convert_field and convert_parameter  */
/* and see if anything needs to be added to either of those.            */
Int4 pgtypes_defined[]  = { PG_TYPE_CHAR,
			    PG_TYPE_CHAR8,
			    PG_TYPE_BPCHAR,
			    PG_TYPE_VARCHAR,
			    PG_TYPE_TEXT,
			    PG_TYPE_NAME,
			    PG_TYPE_INT2,
			    PG_TYPE_INT4,
			    PG_TYPE_FLOAT4,
			    PG_TYPE_FLOAT8,
			    PG_TYPE_OID,
			    PG_TYPE_DATE,
			    PG_TYPE_TIME,
			    0 };
			  
/*
 * the functions that return integers return the value, or -1 for NULL, 
 * or -666 for error.
 * the functions that return strings return the value, or 0 for NULL,
 * or "-666" for error (which is kind of silly, but it's my fault).
 *
 * CC: Added PG_TYPE_BPCHAR -- treated like PG_TYPE_TEXT -- hope this is acceptable
 */

Int2 pgtype_to_sqltype(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:
    case PG_TYPE_CHAR8:         return SQL_CHAR;
    case PG_TYPE_VARCHAR:       return SQL_VARCHAR;
    case PG_TYPE_BPCHAR:
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          
#ifdef SMALL_TEXT_FIELDS
	return SQL_VARCHAR;
#else
	return SQL_LONGVARCHAR;
#endif
    case PG_TYPE_INT2:          return SQL_SMALLINT;
    case PG_TYPE_OID:
    case PG_TYPE_INT4:          return SQL_INTEGER;
    case PG_TYPE_FLOAT4:        return SQL_REAL;
    case PG_TYPE_FLOAT8:        return SQL_FLOAT;
    case PG_TYPE_DATE:          return SQL_DATE;
    case PG_TYPE_TIME:          return SQL_TIME;
    default:                    return -666;
    }
}

Int2 pgtype_to_ctype(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:
    case PG_TYPE_CHAR8:   
    case PG_TYPE_VARCHAR:
    case PG_TYPE_BPCHAR:
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return SQL_C_CHAR;
    case PG_TYPE_INT2:          return SQL_C_SSHORT;
    case PG_TYPE_OID:
    case PG_TYPE_INT4:          return SQL_C_SLONG;
    case PG_TYPE_FLOAT4:        return SQL_C_FLOAT;
    case PG_TYPE_FLOAT8:        return SQL_C_DOUBLE;
    case PG_TYPE_DATE:          return SQL_C_DATE;
    case PG_TYPE_TIME:          return SQL_C_TIME;
    default:                    return -666;
    }
}

char *pgtype_to_name(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:          return "char";
    case PG_TYPE_CHAR8:         return "char8";
    case PG_TYPE_VARCHAR:       return "varchar";
    case PG_TYPE_BPCHAR:        return "bpchar";
    case PG_TYPE_TEXT:          return "text";
    case PG_TYPE_NAME:          return "name";
    case PG_TYPE_INT2:          return "int2";
    case PG_TYPE_OID:           return "oid";
    case PG_TYPE_INT4:          return "int4";
    case PG_TYPE_FLOAT4:        return "float4";
    case PG_TYPE_FLOAT8:        return "float8";
    case PG_TYPE_DATE:          return "date";
    case PG_TYPE_TIME:          return "time";
    default:                    return "-666";
    }    
}

Int4 pgtype_precision(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:          
#ifdef SMALL_TEXT_FIELDS
	return 255;
#else
	return 4096;
#endif
    case PG_TYPE_CHAR8:       	return 8;
    case PG_TYPE_BPCHAR:        
    case PG_TYPE_VARCHAR:       
#ifdef SMALL_TEXT_FIELDS
	return 255;
#else
	return 4096;
#endif
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return 0;
    case PG_TYPE_INT2:          return 5;
    case PG_TYPE_OID:
    case PG_TYPE_INT4:          return 10;
    case PG_TYPE_FLOAT4:        return 7;
    case PG_TYPE_FLOAT8:        return 15;
    case PG_TYPE_DATE:          return 10;
    case PG_TYPE_TIME:          return 8;
    default:                    return -666;
    }
}

Int4 pgtype_length(Int4 type)
{
    switch(type) {
    // the lengths on these two depend on the column definition
    case PG_TYPE_CHAR:         return -666;
    case PG_TYPE_CHAR8:        return 8;
    case PG_TYPE_VARCHAR:      return -666;
    case PG_TYPE_BPCHAR:
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return 8192;
    case PG_TYPE_INT2:          return 2;
    case PG_TYPE_OID:
    case PG_TYPE_INT4:          return 4;
    case PG_TYPE_FLOAT4:        return 4;
    case PG_TYPE_FLOAT8:        return 8;
    case PG_TYPE_DATE:          return 6;
    case PG_TYPE_TIME:          return 6;
    default:                    return -666;
    }
}

Int2 pgtype_scale(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:          return -1;
    case PG_TYPE_CHAR8:         return -1;
    case PG_TYPE_VARCHAR:       return -1;
    case PG_TYPE_BPCHAR:    
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return -1;
    case PG_TYPE_INT2:          return 0;
    case PG_TYPE_OID:
    case PG_TYPE_INT4:          return 0;
    case PG_TYPE_FLOAT4:        return 0;
    case PG_TYPE_FLOAT8:        return 0;
    case PG_TYPE_DATE:          return 0;
    case PG_TYPE_TIME:          return 0;
    default:                    return -666;
    }
}

Int2 pgtype_radix(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:          return -1;
    case PG_TYPE_CHAR8:         return -1;
    case PG_TYPE_VARCHAR:       return -1;
    case PG_TYPE_BPCHAR:    
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return -1;
    case PG_TYPE_INT2:          return 10;
    case PG_TYPE_OID:
    case PG_TYPE_INT4:          return 10;
    case PG_TYPE_FLOAT4:        return 10;
    case PG_TYPE_FLOAT8:        return 10;
    case PG_TYPE_DATE:          return -1;
    case PG_TYPE_TIME:          return -1;
    default:                    return -666;
    }
}

Int2 pgtype_nullable(Int4 type)
{
    switch(type) {
	/* actually everything is nullable */
    case PG_TYPE_CHAR:   
    case PG_TYPE_CHAR8:  
    case PG_TYPE_VARCHAR:
    case PG_TYPE_BPCHAR:
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:   
    case PG_TYPE_INT2:
    case PG_TYPE_OID:
    case PG_TYPE_INT4:
    case PG_TYPE_FLOAT4:
    case PG_TYPE_FLOAT8:
    case PG_TYPE_DATE:
    case PG_TYPE_TIME:          return SQL_NULLABLE;
    default:                    return -666;
    }
}

Int4 pgtype_auto_increment(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:         
    case PG_TYPE_CHAR8:
    case PG_TYPE_VARCHAR:      
    case PG_TYPE_BPCHAR:
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return -1;
    case PG_TYPE_INT2:         
    case PG_TYPE_OID:
    case PG_TYPE_INT4:         
    case PG_TYPE_FLOAT4:       
    case PG_TYPE_FLOAT8:        return FALSE;
    default:                    return -666;
    }    
}

Int4 pgtype_case_sensitive(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:          return TRUE;
    case PG_TYPE_CHAR8:         return TRUE;
    case PG_TYPE_VARCHAR:       return TRUE;
    case PG_TYPE_BPCHAR:
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return TRUE;
    case PG_TYPE_INT2:          return FALSE;
    case PG_TYPE_OID:
    case PG_TYPE_INT4:          return FALSE;
    case PG_TYPE_FLOAT4:        return FALSE;
    case PG_TYPE_FLOAT8:        return FALSE;
    case PG_TYPE_DATE:          return FALSE;
    case PG_TYPE_TIME:          return FALSE;
    default:                    return -666;
    }
}

Int4 pgtype_money(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:          return FALSE;
    case PG_TYPE_CHAR8:         return FALSE;
    case PG_TYPE_VARCHAR:       return FALSE;
    case PG_TYPE_BPCHAR:    
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return FALSE;
    case PG_TYPE_INT2:          return FALSE;
    case PG_TYPE_OID:
    case PG_TYPE_INT4:          return FALSE;
    case PG_TYPE_FLOAT4:        return FALSE;
    case PG_TYPE_FLOAT8:        return FALSE;
    case PG_TYPE_DATE:          return FALSE;
    case PG_TYPE_TIME:          return FALSE;
    default:                    return -666;
    }    
}

Int4 pgtype_searchable(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:          return SQL_SEARCHABLE;
    case PG_TYPE_CHAR8:         return SQL_SEARCHABLE;
    case PG_TYPE_VARCHAR:       return SQL_SEARCHABLE;
    case PG_TYPE_BPCHAR:
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return SQL_SEARCHABLE;
    case PG_TYPE_INT2:          return SQL_ALL_EXCEPT_LIKE;
    case PG_TYPE_OID:
    case PG_TYPE_INT4:          return SQL_ALL_EXCEPT_LIKE;
    case PG_TYPE_FLOAT4:        return SQL_ALL_EXCEPT_LIKE;
    case PG_TYPE_FLOAT8:        return SQL_ALL_EXCEPT_LIKE;
    case PG_TYPE_DATE:          return SQL_ALL_EXCEPT_LIKE;
    case PG_TYPE_TIME:          return SQL_ALL_EXCEPT_LIKE;
    default:                    return -666;
    }    
}

Int4 pgtype_unsigned(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:        
    case PG_TYPE_CHAR8:
    case PG_TYPE_VARCHAR:     
    case PG_TYPE_BPCHAR:    
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return -1;
    case PG_TYPE_INT2:          return FALSE;
    case PG_TYPE_OID:           return TRUE;
    case PG_TYPE_INT4:          return FALSE;
    case PG_TYPE_FLOAT4:        return FALSE;
    case PG_TYPE_FLOAT8:        return FALSE;
    case PG_TYPE_DATE:          return -1;
    case PG_TYPE_TIME:          return -1;
    default:                    return -666;
    }
}

char *pgtype_literal_prefix(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:
    case PG_TYPE_CHAR8:
    case PG_TYPE_VARCHAR:
    case PG_TYPE_BPCHAR:    
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return "'";
    case PG_TYPE_INT2:
    case PG_TYPE_OID:
    case PG_TYPE_INT4:
    case PG_TYPE_FLOAT4:
    case PG_TYPE_FLOAT8:        return 0;
    case PG_TYPE_DATE:          return "'";
    case PG_TYPE_TIME:          return "'";
    default:                    return "-666";
    }
}

char *pgtype_literal_suffix(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:
    case PG_TYPE_CHAR8:
    case PG_TYPE_VARCHAR:
    case PG_TYPE_BPCHAR:    
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:          return "'";
    case PG_TYPE_INT2:  
    case PG_TYPE_OID:
    case PG_TYPE_INT4:       
    case PG_TYPE_FLOAT4:      
    case PG_TYPE_FLOAT8:        return 0;
    case PG_TYPE_DATE:          return "'";
    case PG_TYPE_TIME:          return "'";
    default:                    return "-666";
    }
}

char *pgtype_create_params(Int4 type)
{
    switch(type) {
    case PG_TYPE_CHAR:         
    case PG_TYPE_VARCHAR:       return "max. length";
    case PG_TYPE_CHAR8:
    case PG_TYPE_BPCHAR:
    case PG_TYPE_TEXT:
    case PG_TYPE_NAME:         
    case PG_TYPE_INT2:  
    case PG_TYPE_OID:
    case PG_TYPE_INT4:       
    case PG_TYPE_FLOAT4:      
    case PG_TYPE_FLOAT8:        return 0;
    case PG_TYPE_DATE:          return 0;
    case PG_TYPE_TIME:          return 0;
    default:                    return "-666";
    }
}


Int2 sqltype_to_default_ctype(Int2 sqltype)
{
    // from the table on page 623 of ODBC 2.0 Programmer's Reference
    // (Appendix D)
    switch(sqltype) {
    case SQL_CHAR: 
    case SQL_VARCHAR:
    case SQL_LONGVARCHAR:
    case SQL_DECIMAL:
    case SQL_NUMERIC:
    case SQL_BIGINT:
	return SQL_C_CHAR;
    case SQL_BIT:
	return SQL_C_BIT;
    case SQL_TINYINT:
	return SQL_C_STINYINT;
    case SQL_SMALLINT:
	return SQL_C_SSHORT;
    case SQL_INTEGER:
	return SQL_C_SLONG;
    case SQL_REAL:
	return SQL_C_FLOAT;
    case SQL_FLOAT:
    case SQL_DOUBLE:
	return SQL_C_DOUBLE;
    case SQL_BINARY:
    case SQL_VARBINARY:
    case SQL_LONGVARBINARY:
	return SQL_C_BINARY;
    case SQL_DATE:
	return SQL_C_DATE;
    case SQL_TIME:
	return SQL_C_TIME;
    case SQL_TIMESTAMP:
	return SQL_C_TIMESTAMP;
    default:
	return -1;
    }
}
