/* -*- mode: c++; c-basic-offset: 4 -*- */
/********************************************************************
    POSTODBCL.DLL - A library to talk to Postgres95 by using the
                    WINDOWS ODBC Interface

    Copyright (C) 1996; Christian Czezatke, Dan McGuirk

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    How to contact the authors:

    email to: e9025461@student.tuwien.ac.at     (Christian Czezatke)
              mcguirk@indirect.com              (Dan McGuirk)
********************************************************************/


/*
** INFO.C - This is the PostODBC driver code for
** executing information functions.
**
*/

//      -       -       -       -       -       -       -       -       -


/* CC: Set this to something else when producing a MIPS version of that file... */
#define UNALIGNED

#include <string.h>
#include "postodbc.h"

#include <sql.h> /* for some defines */
#include <sqlext.h> /* still more defines */
#ifdef __WATCOMC__
#include <stdlib.h> /* CC: Watcom likes this for malloc/free definitions */
#endif
#include "socket\wrapper.h"
#include "globals.h"
#include "tuple.h"
#include "pgtypes.h"

//      -       -       -       -       -       -       -       -       -

RETCODE SQL_API SQLGetInfo(
        HDBC      hdbc,
        UWORD     fInfoType,
        PTR       rgbInfoValue,
        SWORD     cbInfoValueMax,
        SWORD FAR *pcbInfoValue)
{
    ConnectionClass *conn;
    char *hlp;

    conn = HI_connfromhdbc(the_handles, hdbc);
    if(!conn) { return SQL_INVALID_HANDLE; }

    /* CC: Some sanity checks */
    if ((NULL == (char *)rgbInfoValue) ||
        (cbInfoValueMax == 0))

        /* removed: */
        /* || (NULL == pcbInfoValue) */

        /* pcbInfoValue is ignored for non-character output. */
        /* some programs (at least Microsoft Query) seem to just send a NULL, */
        /* so let them get away with it... */

        return SQL_INVALID_HANDLE;


    switch (fInfoType) {
    case SQL_ACCESSIBLE_PROCEDURES: /* ODBC 1.0 */
        // can the user call all functions returned by SQLProcedures?
        // I assume access permissions could prevent this in some cases(?)
        // anyway, SQLProcedures doesn't exist yet.
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "N", (size_t)cbInfoValueMax);
        break;

    case SQL_ACCESSIBLE_TABLES: /* ODBC 1.0 */
        // is the user guaranteed "SELECT" on every table?
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "N", (size_t)cbInfoValueMax);
        break;

    case SQL_ACTIVE_CONNECTIONS: /* ODBC 1.0 */
        // how many simultaneous connections do we support?
        *((WORD *)rgbInfoValue) = MAX_CONNECTIONS;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_ACTIVE_STATEMENTS: /* ODBC 1.0 */
        // no limit on the number of active statements.
        *((WORD *)rgbInfoValue) = (WORD)0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_ALTER_TABLE: /* ODBC 2.0 */
        // what does 'alter table' support? (bitmask)
        // postgres doesn't seem to let you drop columns.
        *((DWORD *)rgbInfoValue) = SQL_AT_ADD_COLUMN;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_BOOKMARK_PERSISTENCE: /* ODBC 2.0 */
        // through what operations do bookmarks persist? (bitmask)
        // bookmarks don't exist yet, so they're not very persistent.
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_COLUMN_ALIAS: /* ODBC 2.0 */
        // do we support column aliases?  guess not.
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "N", (size_t)cbInfoValueMax);
        break;

    case SQL_CONCAT_NULL_BEHAVIOR: /* ODBC 1.0 */
        // how does concatenation work with NULL columns?
        // not sure how you do concatentation, but this way seems
        // more reasonable
        *((WORD *)rgbInfoValue) = SQL_CB_NON_NULL;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

        // which types of data-conversion do we support?
        // currently we don't support any, except converting a type
        // to itself.
    case SQL_CONVERT_BIGINT:
    case SQL_CONVERT_BINARY:
    case SQL_CONVERT_BIT:
    case SQL_CONVERT_CHAR:
    case SQL_CONVERT_DATE:
    case SQL_CONVERT_DECIMAL:
    case SQL_CONVERT_DOUBLE:
    case SQL_CONVERT_FLOAT:
    case SQL_CONVERT_INTEGER:
    case SQL_CONVERT_LONGVARBINARY:
    case SQL_CONVERT_LONGVARCHAR:
    case SQL_CONVERT_NUMERIC:
    case SQL_CONVERT_REAL:
    case SQL_CONVERT_SMALLINT:
    case SQL_CONVERT_TIME:
    case SQL_CONVERT_TIMESTAMP:
    case SQL_CONVERT_TINYINT:
    case SQL_CONVERT_VARBINARY:
    case SQL_CONVERT_VARCHAR: /* ODBC 1.0 */
        // only return the type we were called with (bitmask)
        *((DWORD *)rgbInfoValue) = fInfoType;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_CONVERT_FUNCTIONS: /* ODBC 1.0 */
        // which conversion functions do we support? (bitmask)
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_CORRELATION_NAME: /* ODBC 1.0 */
        // I don't know what a correlation name is, so I guess we don't
        // support them.

        // *((WORD *)rgbInfoValue) = (WORD)SQL_CN_NONE;

        // well, let's just say we do--otherwise Query won't work.
        *((WORD *)rgbInfoValue) = (WORD)SQL_CN_ANY;
        if(pcbInfoValue) { *pcbInfoValue = 2; }

        break;

    case SQL_CURSOR_COMMIT_BEHAVIOR: /* ODBC 1.0 */
        // postgres definitely closes cursors when a transaction ends,
        // but you shouldn't have to re-prepare a statement after
        // commiting a transaction (I don't think)
        *((WORD *)rgbInfoValue) = (WORD)SQL_CB_CLOSE;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_CURSOR_ROLLBACK_BEHAVIOR: /* ODBC 1.0 */
        // see above
        *((WORD *)rgbInfoValue) = (WORD)SQL_CB_CLOSE;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_DATA_SOURCE_NAME: /* ODBC 1.0 */
        hlp = ConnectionC_get_DSN(conn->connection);
        if(!hlp) {
            /* CC: Check for Nullpointers */
            if (NULL != rgbInfoValue)
              ((char *)rgbInfoValue)[0] = '\0';
            if (NULL != pcbInfoValue)
              *pcbInfoValue = 0;
        } else {
            if (NULL != pcbInfoValue)
              *pcbInfoValue = strlen(hlp);
            if (NULL != rgbInfoValue)
              strncpy_null((char *)rgbInfoValue, hlp, (size_t)cbInfoValueMax);
        }
        break;

    case SQL_DATA_SOURCE_READ_ONLY: /* ODBC 1.0 */
        // postgres data sources should never be read-only.
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "N", (size_t)cbInfoValueMax);
        break;

    case SQL_DATABASE_NAME: /* Support for old ODBC 1.0 Apps */
        // case SQL_CURRENT_QUALIFIER:
        // this tag doesn't seem to be in ODBC 2.0, and it conflicts
        // with a valid tag (SQL_TIMEDATE_ADD_INTERVALS).

        /* CC: get the database name we are connected to */
        hlp = ConnectionC_get_dbname(conn->connection);
        if (NULL == hlp) {
            /* There is no database name specified */
            ((char *)rgbInfoValue)[0] = '\0';
            *pcbInfoValue = 0;
        } else {
            *pcbInfoValue = strlen(hlp);
            strncpy_null((char *)rgbInfoValue, hlp, (size_t)cbInfoValueMax);
        }
        break;

    case SQL_DBMS_NAME: /* ODBC 1.0 */
        *pcbInfoValue = 10;
        strncpy_null((char *)rgbInfoValue, "Postgres95", (size_t)cbInfoValueMax);
        break;

    case SQL_DBMS_VER: /* ODBC 1.0 */
        *pcbInfoValue = 26;
        strncpy_null((char *)rgbInfoValue, "01.01.0000 Postgres95 1.01",
                (size_t)cbInfoValueMax);
        break;

    case SQL_DEFAULT_TXN_ISOLATION: /* ODBC 1.0 */
        // are dirty reads, non-repeatable reads, and phantoms possible? (bitmask)
        // by direct experimentation they are not.  postgres forces
        // the newer transaction to wait before doing something that
        // would cause one of these problems.
        *((DWORD *)rgbInfoValue) = SQL_TXN_SERIALIZABLE;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_DRIVER_NAME: /* ODBC 1.0 */
        // this should be the actual filename of the driver
#ifdef COMPILE_FOR_16_bit
        hlp = "POSTODBC.DLL";
#else
        hlp = "PODBC32.DLL";
#endif
        *pcbInfoValue = strlen(hlp);
        strncpy_null((char *)rgbInfoValue, hlp, (size_t)cbInfoValueMax);
        break;

    case SQL_DRIVER_ODBC_VER:
        /* I think we should return 02.00--at least, that is the version of the */
        /* spec I'm currently referring to. */
        *pcbInfoValue = 5;
        strncpy_null((char *)rgbInfoValue, "02.00", (size_t)cbInfoValueMax);
        break;

    case SQL_DRIVER_VER: /* ODBC 1.0 */
        hlp = POSTGRESDRIVERVERSION;
        *pcbInfoValue = strlen(hlp);
        strncpy_null((char *)rgbInfoValue, hlp, (size_t)cbInfoValueMax);
        break;

    case SQL_EXPRESSIONS_IN_ORDERBY: /* ODBC 1.0 */
        // can you have expressions in an 'order by' clause?
        // not sure about this.  say no for now.
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "N", (size_t)cbInfoValueMax);
        break;

    case SQL_FETCH_DIRECTION: /* ODBC 1.0 */
        // which fetch directions are supported? (bitmask)
        // I guess these apply to SQLExtendedFetch?
        *((DWORD *)rgbInfoValue) = SQL_FETCH_NEXT ||
                                   SQL_FETCH_FIRST ||
                                   SQL_FETCH_LAST ||
                                   SQL_FETCH_PRIOR ||
                                   SQL_FETCH_ABSOLUTE ||
	                           SQL_FETCH_RELATIVE;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_FILE_USAGE: /* ODBC 2.0 */
        // we are a two-tier driver, not a file-based one.
        *((WORD *)rgbInfoValue) = (WORD)SQL_FILE_NOT_SUPPORTED;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_GETDATA_EXTENSIONS: /* ODBC 2.0 */
        // (bitmask)
        // we don't have GetData
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_GROUP_BY: /* ODBC 2.0 */
        // how do the columns selected affect the columns you can group by?
        // I think this is right
        *((WORD *)rgbInfoValue) = SQL_GB_NO_RELATION;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_IDENTIFIER_CASE: /* ODBC 1.0 */
        // are identifiers case-sensitive (yes)
        *((WORD *)rgbInfoValue) = SQL_IC_SENSITIVE;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_IDENTIFIER_QUOTE_CHAR: /* ODBC 1.0 */
        // the character used to quote "identifiers" (what are they?)
        // the manual index lists 'owner names' and 'qualifiers' as
        // examples of identifiers.  it says return a blank for no
        // quote character, we'll try that...
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, " ", (size_t)cbInfoValueMax);
        break;

    case SQL_KEYWORDS: /* ODBC 2.0 */
        // do this later
        conn->errormsg = "SQL_KEYWORDS parameter to SQLGetInfo not implemented.";
        conn->errornumber = CONN_NOT_IMPLEMENTED_ERROR;
        return SQL_ERROR;
        break;

    case SQL_LIKE_ESCAPE_CLAUSE: /* ODBC 2.0 */
        // is there a character that escapes '%' and '_' in a LIKE clause?
        // not as far as I can tell
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "N", (size_t)cbInfoValueMax);
        break;

    case SQL_LOCK_TYPES: /* ODBC 2.0 */
        // which lock types does SQLSetPos support? (bitmask)
        // SQLSetPos doesn't exist yet
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_MAX_BINARY_LITERAL_LEN: /* ODBC 2.0 */
        // the maximum length of a query is 2k, so maybe we should
        // set the maximum length of all these literals to that value?
        // for now just use zero for 'unknown or no limit'

        // maximum length of a binary literal
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_MAX_CHAR_LITERAL_LEN: /* ODBC 2.0 */
        // maximum length of a character literal
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_MAX_COLUMN_NAME_LEN: /* ODBC 1.0 */
        // maximum length of a column name
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_COLUMNS_IN_GROUP_BY: /* ODBC 2.0 */
        // maximum number of columns in a 'group by' clause
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_COLUMNS_IN_INDEX: /* ODBC 2.0 */
        // maximum number of columns in an index
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_COLUMNS_IN_ORDER_BY: /* ODBC 2.0 */
        // maximum number of columns in an ORDER BY statement
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_COLUMNS_IN_SELECT: /* ODBC 2.0 */
        // I think you get the idea by now
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_COLUMNS_IN_TABLE: /* ODBC 2.0 */
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_CURSOR_NAME_LEN: /* ODBC 1.0 */
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_INDEX_SIZE: /* ODBC 2.0 */
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_MAX_OWNER_NAME_LEN: /* ODBC 1.0 */
        // the maximum length of a table owner's name.  (0 == none)
        // (maybe this should be 8)
        *((WORD *)rgbInfoValue) = (WORD)0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_PROCEDURE_NAME_LEN: /* ODBC 1.0 */
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_QUALIFIER_NAME_LEN: /* ODBC 1.0 */
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_ROW_SIZE: /* ODBC 2.0 */
        // the maximum size of one row
        // here I do know a definite value
        *((DWORD *)rgbInfoValue) = 8192;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_MAX_ROW_SIZE_INCLUDES_LONG: /* ODBC 2.0 */
        // does the preceding value include LONGVARCHAR and LONGVARBINARY
        // fields?
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "Y", (size_t)cbInfoValueMax);
        break;

    case SQL_MAX_STATEMENT_LEN: /* ODBC 2.0 */
        // there should be a definite value here (2k?)
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_MAX_TABLE_NAME_LEN: /* ODBC 1.0 */
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_TABLES_IN_SELECT: /* ODBC 2.0 */
        *((WORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MAX_USER_NAME_LEN:
        /* CC: Check for nullpointer */
        if (NULL != rgbInfoValue)
          *(UNALIGNED SWORD FAR *)rgbInfoValue = 0;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_MULT_RESULT_SETS: /* ODBC 1.0 */
        // do we support multiple result sets?
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "Y", (size_t)cbInfoValueMax);
        break;

    case SQL_MULTIPLE_ACTIVE_TXN: /* ODBC 1.0 */
        // do we support multiple simultaneous transactions?
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "Y", (size_t)cbInfoValueMax);
        break;

    case SQL_NEED_LONG_DATA_LEN: /* ODBC 2.0 */
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "N", (size_t)cbInfoValueMax);
        break;

    case SQL_NON_NULLABLE_COLUMNS: /* ODBC 1.0 */
        // I think you can have NOT NULL columns with one of dal Zotto's
        // patches, but for now we'll say no.
        *((WORD *)rgbInfoValue) = (WORD)SQL_NNC_NULL;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_NULL_COLLATION: /* ODBC 2.0 */
        // where are nulls sorted?
        *((WORD *)rgbInfoValue) = (WORD)SQL_NC_LOW;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_NUMERIC_FUNCTIONS: /* ODBC 1.0 */
        // what numeric functions are supported? (bitmask)
        // I'm not sure if any of these are actually supported
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_ODBC_API_CONFORMANCE: /* ODBC 1.0 */
        *((WORD *)rgbInfoValue) = SQL_OAC_LEVEL1; /* well, almost */
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_ODBC_SAG_CLI_CONFORMANCE: /* ODBC 1.0 */
        // can't find any reference to SAG in the ODBC reference manual
        // (although it's in the index, it doesn't actually appear on
        // the pages referenced)
        *((WORD *)rgbInfoValue) = SQL_OSCC_NOT_COMPLIANT;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_ODBC_SQL_CONFORMANCE: /* ODBC 1.0 */
        *((WORD *)rgbInfoValue) = SQL_OSC_CORE;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_ODBC_SQL_OPT_IEF: /* ODBC 1.0 */
        // do we support the "Integrity Enhancement Facility" (?)
        // (something to do with referential integrity?)
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "N", (size_t)cbInfoValueMax);
        break;

    case SQL_ORDER_BY_COLUMNS_IN_SELECT: /* ODBC 2.0 */
        // do the columns sorted by have to be in the list of
        // columns selected?
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "Y", (size_t)cbInfoValueMax);
        break;

    case SQL_OUTER_JOINS: /* ODBC 1.0 */
        // do we support outer joins?
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "N", (size_t)cbInfoValueMax);
        break;

    case SQL_OWNER_TERM: /* ODBC 1.0 */
        // what we call an owner
        if (NULL != pcbInfoValue)
          *pcbInfoValue = 5;
        if (NULL != rgbInfoValue)
          strncpy_null((char *)rgbInfoValue, "owner", (size_t)cbInfoValueMax);
        break;

    case SQL_OWNER_USAGE: /* ODBC 2.0 */
        // in which statements can "owners be used"?  (what does that mean?
        // specifying 'owner.table' instead of just 'table' or something?)
        // (bitmask)
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_POS_OPERATIONS: /* ODBC 2.0 */
        // what functions does SQLSetPos support? (bitmask)
        // SQLSetPos does not exist yet
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_POSITIONED_STATEMENTS: /* ODBC 2.0 */
        // what 'positioned' functions are supported? (bitmask)
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_PROCEDURE_TERM: /* ODBC 1.0 */
        // what do we call a procedure?
        *pcbInfoValue = 9;
        strncpy_null((char *)rgbInfoValue, "procedure", (size_t)cbInfoValueMax);
        break;

    case SQL_PROCEDURES: /* ODBC 1.0 */
        // do we support procedures?
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "Y", (size_t)cbInfoValueMax);
        break;

    case SQL_QUALIFIER_LOCATION: /* ODBC 2.0 */
        // where does the qualifier go (before or after the table name?)
        // we don't really use qualifiers, so...
        *((WORD *)rgbInfoValue) = SQL_QL_START;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_QUALIFIER_NAME_SEPARATOR: /* ODBC 1.0 */
        // not really too sure what a qualifier is supposed to do either
        // (specify the name of a database in certain cases?), so nix
        // on that, too.
        *pcbInfoValue = 0;
        strncpy_null((char *)rgbInfoValue, "", (size_t)cbInfoValueMax);
        break;

    case SQL_QUALIFIER_TERM: /* ODBC 1.0 */
        // what we call a qualifier
        if (NULL != pcbInfoValue)
          *pcbInfoValue = 0;
        if (NULL != rgbInfoValue)
          strncpy_null((char *)rgbInfoValue, "", (size_t)cbInfoValueMax);
        break;

    case SQL_QUALIFIER_USAGE: /* ODBC 2.0 */
        // where can qualifiers be used? (bitmask)
        // nowhere
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_QUOTED_IDENTIFIER_CASE: /* ODBC 2.0 */
        // are "quoted" identifiers case-sensitive?
        // well, we don't really let you quote identifiers, so...
        *((WORD *)rgbInfoValue) = SQL_IC_SENSITIVE;
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_ROW_UPDATES: /* ODBC 1.0 */
        // not quite sure what this means
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "Y", (size_t)cbInfoValueMax);
        break;

    case SQL_SCROLL_CONCURRENCY: /* ODBC 1.0 */
        // what concurrency options are supported? (bitmask)
        // taking a guess here
        *((DWORD *)rgbInfoValue) = SQL_SCCO_OPT_ROWVER;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_SCROLL_OPTIONS: /* ODBC 1.0 */
        // what options are supported for scrollable cursors? (bitmask)
        // not too sure about this one, either...
        *((DWORD *)rgbInfoValue) = SQL_SO_KEYSET_DRIVEN;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_SEARCH_PATTERN_ESCAPE: /* ODBC 1.0 */
        // this is supposed to be the character that escapes '_' or '%'
        // in LIKE clauses.  as far as I can tell postgres doesn't have one
        // (backslash generates an error).  returning an empty string means
        // no escape character is supported.
        *pcbInfoValue = 0;
        strncpy_null((char *)rgbInfoValue, "", (size_t)cbInfoValueMax);
        break;

    case SQL_SERVER_NAME: /* ODBC 1.0 */
        // where can we get this from?
        *pcbInfoValue = 0;
        strncpy_null((char *)rgbInfoValue, "", (size_t)cbInfoValueMax);
        break;

    case SQL_SPECIAL_CHARACTERS: /* ODBC 2.0 */
        // what special characters can be used in table and column names, etc.?
        // probably more than just this...
        *pcbInfoValue = 1;
        strncpy_null((char *)rgbInfoValue, "_", (size_t)cbInfoValueMax);
        break;

    case SQL_STATIC_SENSITIVITY: /* ODBC 2.0 */
        // can changes made inside a cursor be detected? (or something like that)
        // (bitmask)
        // only applies to SQLSetPos, which doesn't exist yet.
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_STRING_FUNCTIONS: /* ODBC 1.0 */
        // what string functions exist? (bitmask)
        // not sure if any of these exist, either
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_SUBQUERIES: /* ODBC 2.0 */
        // where are subqueries supported? (bitmask)
        // we do not support subqueries
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_SYSTEM_FUNCTIONS: /* ODBC 1.0 */
        // what system functions are supported? (bitmask)
        // none of these seem to be supported, either
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_TABLE_TERM: /* ODBC 1.0 */
        // what we call a table
        *pcbInfoValue = 5;
        strncpy_null((char *)rgbInfoValue, "table", (size_t)cbInfoValueMax);
        break;

    case SQL_TIMEDATE_ADD_INTERVALS: /* ODBC 2.0 */
        // what resolutions are supported by the "TIMESTAMPADD scalar
        // function" (whatever that is)? (bitmask)
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_TIMEDATE_DIFF_INTERVALS: /* ODBC 2.0 */
        // what resolutions are supported by the "TIMESTAMPDIFF scalar
        // function" (whatever that is)? (bitmask)
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_TIMEDATE_FUNCTIONS: /* ODBC 1.0 */
        // what time and date functions are supported? (bitmask)
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_TXN_CAPABLE: /* ODBC 1.0 */
        *((WORD *)rgbInfoValue) = (WORD)SQL_TC_ALL;
        // Postgres can deal with create or drop table statements in a transaction
        if(pcbInfoValue) { *pcbInfoValue = 2; }
        break;

    case SQL_TXN_ISOLATION_OPTION: /* ODBC 1.0 */
        // what transaction isolation options are available? (bitmask)
        // only the default--serializable transactions.
        *((DWORD *)rgbInfoValue) = SQL_TXN_SERIALIZABLE;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_UNION: /* ODBC 2.0 */
        // do we support the union clause? (bitmask)
        // no.
        *((DWORD *)rgbInfoValue) = 0;
        if(pcbInfoValue) { *pcbInfoValue = 4; }
        break;

    case SQL_USER_NAME: /* ODBC 1.0 */
        // where should we get this from?
        /* CC: Don't forget to check for NULL pointers */
        if (NULL != pcbInfoValue)
          *pcbInfoValue = 0;
        if (NULL != rgbInfoValue)
          strncpy_null((char *)rgbInfoValue, "", (size_t)cbInfoValueMax);
        break;

    default:
        /* unrecognized key */
        conn->errormsg = "Unrecognized key passed to SQLGetInfo.";
        conn->errornumber = CONN_NOT_IMPLEMENTED_ERROR;
        return SQL_ERROR;
    }

    return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

void typeinfo_add_type(ResultC *res,
                       Int2 null_fields,
                       char *type_name, Int2 data_type, Int4 precision,
                       char *literal_prefix, char *literal_suffix,
                       char *create_params, Int2 nullable, Int2 case_sensitive,
                       Int2 searchable, Int2 unsigned_attribute, Int2 money,
                       Int2 auto_increment, char *local_type_name,
                       Int2 minimum_scale, Int2 maximum_scale)
{
    TupleNode *row;

    row = (TupleNode *)malloc(sizeof(TupleNode) + (15 - 1)*sizeof(TupleField));

    /* this has become a mess.  maybe a macro is in order here. */
    if(((null_fields >> 0) & 1) == 1) {
        set_tuplefield_null(&row->tuple[0]);
    } else {
        set_tuplefield_string(&row->tuple[0], type_name);
    }
    if(((null_fields >> 1) & 1) == 1) {
        set_tuplefield_null(&row->tuple[1]);
    } else {
        set_tuplefield_int2(&row->tuple[1], data_type);
    }
    if(((null_fields >> 2) & 1) == 1) {
        set_tuplefield_null(&row->tuple[2]);
    } else {
        set_tuplefield_int4(&row->tuple[2], precision);
    }
    if(((null_fields >> 3) & 1) == 1) {
        set_tuplefield_null(&row->tuple[3]);
    } else {
        set_tuplefield_string(&row->tuple[3], literal_prefix);
    }
    if(((null_fields >> 4) & 1) == 1) {
        set_tuplefield_null(&row->tuple[4]);
    } else {
        set_tuplefield_string(&row->tuple[4], literal_suffix);
    }
    if(((null_fields >> 5) & 1) == 1) {
        set_tuplefield_null(&row->tuple[5]);
    } else {
        set_tuplefield_string(&row->tuple[5], create_params);
    }
    if(((null_fields >> 6) & 1) == 1) {
        set_tuplefield_null(&row->tuple[6]);
    } else {
        set_tuplefield_int2(&row->tuple[6], nullable);
    }
    if(((null_fields >> 7) & 1) == 1) {
        set_tuplefield_null(&row->tuple[7]);
    } else {
        set_tuplefield_int2(&row->tuple[7], case_sensitive);
    }
    if(((null_fields >> 8) & 1) == 1) {
        set_tuplefield_null(&row->tuple[8]);
    } else {
        set_tuplefield_int2(&row->tuple[8], searchable);
    }
    if(((null_fields >> 9) & 1) == 1) {
        set_tuplefield_null(&row->tuple[9]);
    } else {
        set_tuplefield_int2(&row->tuple[9], unsigned_attribute);
    }
    if(((null_fields >> 10) & 1) == 1) {
        set_tuplefield_null(&row->tuple[10]);
    } else {
        set_tuplefield_int2(&row->tuple[10], money);
    }
    if(((null_fields >> 11) & 1) == 1) {
        set_tuplefield_null(&row->tuple[11]);
    } else {
        set_tuplefield_int2(&row->tuple[11], auto_increment);
    }
    if(((null_fields >> 12) & 1) == 1) {
        set_tuplefield_null(&row->tuple[12]);
    } else {
        set_tuplefield_string(&row->tuple[12], local_type_name);
    }
    if(((null_fields >> 13) & 1) == 1) {
        set_tuplefield_null(&row->tuple[13]);
    } else {
        set_tuplefield_int2(&row->tuple[13], minimum_scale);
    }
    if(((null_fields >> 14) & 1) == 1) {
        set_tuplefield_null(&row->tuple[14]);
    } else {
        set_tuplefield_int2(&row->tuple[14], maximum_scale);
    }

    ResultC_add_tuple(res, row);
}

RETCODE SQL_API SQLGetTypeInfo(
        HSTMT   hstmt,
        SWORD   fSqlType)
{
    StatementClass *stmt;
    int i;
    Int4 type;

    // see SQLTables for explanation of some of this
    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    stmt->result = ResultC_Constructor();
    if(!stmt->result) {
        return SQL_ERROR;
    }

    extend_bindings(stmt, 15);

    ResultC_set_num_fields(stmt->result, 15);
    ResultC_set_field_info(stmt->result, 0, "TYPE_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 1, "DATA_TYPE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 2, "PRECISION", PG_TYPE_INT4, 4);
    ResultC_set_field_info(stmt->result, 3, "LITERAL_PREFIX", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 4, "LITERAL_SUFFIX", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 5, "CREATE_PARAMS", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 6, "NULLABLE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 7, "CASE_SENSITIVE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 8, "SEARCHABLE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 9, "UNSIGNED_ATTRIBUTE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 10, "MONEY", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 11, "AUTO_INCREMENT", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 12, "LOCAL_TYPE_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 13, "MINIMUM_SCALE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 14, "MAXIMUM_SCALE", PG_TYPE_INT2, 2);

    // cycle through the types
    for(i=0, type = pgtypes_defined[0]; type; type = pgtypes_defined[++i]) {
	if(fSqlType == SQL_ALL_TYPES ||
	   fSqlType == pgtype_to_sqltype(type)) {
	    // add to table
	    typeinfo_add_type(stmt->result,
			      ((pgtype_literal_prefix(type) == 0) << 3) &
			      ((pgtype_literal_suffix(type) == 0) << 4) &
			      ((pgtype_create_params(type) == 0) << 5) &
			      ((pgtype_unsigned(type) == -1) << 9) &
			      ((pgtype_auto_increment(type) == -1) << 11) &
			      (1 << 12) &
			      ((pgtype_scale(type) == -1) << 12) &
			      ((pgtype_scale(type) == -1) << 13),
			      pgtype_to_name(type),
			      pgtype_to_sqltype(type),
			      pgtype_precision(type),
			      (pgtype_literal_prefix(type) ?
			       pgtype_literal_prefix(type) : ""),
			      (pgtype_literal_suffix(type) ?
			       pgtype_literal_suffix(type) : ""),
			      (pgtype_create_params(type) ?
			       pgtype_create_params(type) : ""),
			      pgtype_nullable(type),
			      pgtype_case_sensitive(type),
			      pgtype_searchable(type),
			      ((pgtype_unsigned(type) != -1) ?
			       pgtype_unsigned(type) : 0),
			      pgtype_money(type),
			      ((pgtype_auto_increment(type) != -1) ?
			       pgtype_auto_increment(type) : 0),
			      "", /* never a DS-dependent type name */
			      ((pgtype_scale(type) != -1) ?
			       pgtype_scale(type) : 0),
			      ((pgtype_scale(type) != -1) ?
			       pgtype_scale(type) : 0));
	}
    }

    if(stmt->status == STMT_ALLOCATED) {
	CC_remove_from_list(stmt->conn, ALLOCATED_LIST, stmt->hstmt);
    } else if(stmt->status == STMT_READY) {
	CC_remove_from_list(stmt->conn, TO_DO_LIST, stmt->hstmt);
    }
    CC_append_to_list(stmt->conn, DONE_LIST, stmt);
    stmt->status = STMT_FINISHED;

    stmt->currTuple = -1;

    return SQL_SUCCESS;
}

//      -       -       -       -       -       -       -       -       -

/* CC: we just claim to support everything. But unsupported
       functions simply return SQL_ERROR when they are called
*/
RETCODE SQL_API SQLGetFunctions(
        LPDBC     lpdbc,
        UWORD     fFunction,
        UWORD FAR *pfExists)
{
    if (fFunction == SQL_API_ALL_FUNCTIONS) {

        // stop the lies!
        // well, or maybe don't.  it's more useful to see what
        // function a program is trying to call.


#ifdef GETINFO_LIE
        int i;
        memset(pfExists, 0, sizeof(UWORD)*100);

        pfExists[SQL_API_SQLALLOCENV] = TRUE;
        pfExists[SQL_API_SQLFREEENV] = TRUE;
        for (i = SQL_API_SQLALLOCCONNECT; i <= SQL_NUM_FUNCTIONS; i++)
            pfExists[i] = TRUE;
        for (i = SQL_EXT_API_START; i <= SQL_EXT_API_LAST; i++)
            pfExists[i] = TRUE;
#else
        memset(pfExists, 0, sizeof(UWORD)*100);

        // ODBC core functions
        pfExists[SQL_API_SQLALLOCCONNECT]     = TRUE;
        pfExists[SQL_API_SQLALLOCENV]         = TRUE;
        pfExists[SQL_API_SQLALLOCSTMT]        = TRUE;
        pfExists[SQL_API_SQLBINDCOL]          = TRUE;  // partial--
                                                       // doesn't do type conversions
        pfExists[SQL_API_SQLCANCEL]           = TRUE;
        pfExists[SQL_API_SQLCOLATTRIBUTES]    = TRUE;
        pfExists[SQL_API_SQLCONNECT]          = TRUE;
        pfExists[SQL_API_SQLDESCRIBECOL]      = TRUE;  // partial
        pfExists[SQL_API_SQLDISCONNECT]       = TRUE;
        pfExists[SQL_API_SQLERROR]            = TRUE;
        pfExists[SQL_API_SQLEXECDIRECT]       = TRUE;
        pfExists[SQL_API_SQLEXECUTE]          = TRUE;
        pfExists[SQL_API_SQLFETCH]            = TRUE;
        pfExists[SQL_API_SQLFREECONNECT]      = TRUE;
        pfExists[SQL_API_SQLFREEENV]          = TRUE;
        pfExists[SQL_API_SQLFREESTMT]         = TRUE;
        pfExists[SQL_API_SQLGETCURSORNAME]    = FALSE;
        pfExists[SQL_API_SQLNUMRESULTCOLS]    = TRUE;
        pfExists[SQL_API_SQLPREPARE]          = TRUE;  // complete?
        pfExists[SQL_API_SQLROWCOUNT]         = TRUE;
        pfExists[SQL_API_SQLSETCURSORNAME]    = FALSE;
        pfExists[SQL_API_SQLSETPARAM]         = FALSE;
        pfExists[SQL_API_SQLTRANSACT]         = TRUE;

        // ODBC level 1 functions
        pfExists[SQL_API_SQLBINDPARAMETER]    = TRUE;
        pfExists[SQL_API_SQLCOLUMNS]          = TRUE;
        pfExists[SQL_API_SQLDRIVERCONNECT]    = TRUE;
        pfExists[SQL_API_SQLGETCONNECTOPTION] = TRUE;  // partial
        pfExists[SQL_API_SQLGETDATA]          = TRUE;
        pfExists[SQL_API_SQLGETFUNCTIONS]     = TRUE;  // sadly,  I still
                                                       // had to think about
                                                       // this one
        pfExists[SQL_API_SQLGETINFO]          = TRUE;
        pfExists[SQL_API_SQLGETSTMTOPTION]    = TRUE;  // very partial
        pfExists[SQL_API_SQLGETTYPEINFO]      = TRUE;
        pfExists[SQL_API_SQLPARAMDATA]        = FALSE;
        pfExists[SQL_API_SQLPUTDATA]          = FALSE;
        pfExists[SQL_API_SQLSETCONNECTOPTION] = TRUE;  // partial
        pfExists[SQL_API_SQLSETSTMTOPTION]    = TRUE;
        pfExists[SQL_API_SQLSPECIALCOLUMNS]   = TRUE;
        pfExists[SQL_API_SQLSTATISTICS]       = TRUE;
        pfExists[SQL_API_SQLTABLES]           = TRUE;

        // ODBC level 2 functions
        pfExists[SQL_API_SQLBROWSECONNECT]    = FALSE;
        pfExists[SQL_API_SQLCOLUMNPRIVILEGES] = FALSE;
        pfExists[SQL_API_SQLDATASOURCES]      = FALSE;  // only implemented by DM
        pfExists[SQL_API_SQLDESCRIBEPARAM]    = FALSE;
        pfExists[SQL_API_SQLDRIVERS]          = FALSE;
        pfExists[SQL_API_SQLEXTENDEDFETCH]    = TRUE;
        pfExists[SQL_API_SQLFOREIGNKEYS]      = FALSE;
        pfExists[SQL_API_SQLMORERESULTS]      = TRUE;
        pfExists[SQL_API_SQLNATIVESQL]        = TRUE;
        pfExists[SQL_API_SQLNUMPARAMS]        = TRUE;
        pfExists[SQL_API_SQLPARAMOPTIONS]     = FALSE;
        pfExists[SQL_API_SQLPRIMARYKEYS]      = FALSE;
        pfExists[SQL_API_SQLPROCEDURECOLUMNS] = FALSE;
        pfExists[SQL_API_SQLPROCEDURES]       = FALSE;
        pfExists[SQL_API_SQLSETPOS]           = FALSE;
        pfExists[SQL_API_SQLSETSCROLLOPTIONS] = FALSE;
        pfExists[SQL_API_SQLTABLEPRIVILEGES]  = FALSE;
#endif
    } else {
#ifdef GETINFO_LIE
        *pfExists = TRUE;
#else
        switch(fFunction) {
        case SQL_API_SQLALLOCCONNECT:     *pfExists = TRUE; break;
        case SQL_API_SQLALLOCENV:         *pfExists = TRUE; break;
        case SQL_API_SQLALLOCSTMT:        *pfExists = TRUE; break;
        case SQL_API_SQLBINDCOL:          *pfExists = TRUE; break;  // partial
        case SQL_API_SQLCANCEL:           *pfExists = TRUE; break;
        case SQL_API_SQLCOLATTRIBUTES:    *pfExists = TRUE; break;
        case SQL_API_SQLCONNECT:          *pfExists = TRUE; break;
        case SQL_API_SQLDESCRIBECOL:      *pfExists = TRUE; break;  // partial
        case SQL_API_SQLDISCONNECT:       *pfExists = TRUE; break;
        case SQL_API_SQLERROR:            *pfExists = TRUE; break;
        case SQL_API_SQLEXECDIRECT:       *pfExists = TRUE; break;
        case SQL_API_SQLEXECUTE:          *pfExists = TRUE; break;
        case SQL_API_SQLFETCH:            *pfExists = TRUE; break;
        case SQL_API_SQLFREECONNECT:      *pfExists = TRUE; break;
        case SQL_API_SQLFREEENV:          *pfExists = TRUE; break;
        case SQL_API_SQLFREESTMT:         *pfExists = TRUE; break;
        case SQL_API_SQLGETCURSORNAME:    *pfExists = FALSE; break;
        case SQL_API_SQLNUMRESULTCOLS:    *pfExists = TRUE; break;
        case SQL_API_SQLPREPARE:          *pfExists = TRUE; break;
        case SQL_API_SQLROWCOUNT:         *pfExists = TRUE; break;
        case SQL_API_SQLSETCURSORNAME:    *pfExists = FALSE; break;
        case SQL_API_SQLSETPARAM:         *pfExists = FALSE; break;
        case SQL_API_SQLTRANSACT:         *pfExists = TRUE; break;

            // ODBC level 1 functions
        case SQL_API_SQLBINDPARAMETER:    *pfExists = TRUE; break;
        case SQL_API_SQLCOLUMNS:          *pfExists = TRUE; break;
        case SQL_API_SQLDRIVERCONNECT:    *pfExists = TRUE; break;
        case SQL_API_SQLGETCONNECTOPTION: *pfExists = TRUE; break;  // partial
        case SQL_API_SQLGETDATA:          *pfExists = TRUE; break;
        case SQL_API_SQLGETFUNCTIONS:     *pfExists = TRUE; break;
        case SQL_API_SQLGETINFO:          *pfExists = TRUE; break;
        case SQL_API_SQLGETSTMTOPTION:    *pfExists = TRUE; break;  // very partial
        case SQL_API_SQLGETTYPEINFO:      *pfExists = TRUE; break;
        case SQL_API_SQLPARAMDATA:        *pfExists = FALSE; break;
        case SQL_API_SQLPUTDATA:          *pfExists = FALSE; break;
        case SQL_API_SQLSETCONNECTOPTION: *pfExists = TRUE; break;  // partial
        case SQL_API_SQLSETSTMTOPTION:    *pfExists = TRUE; break;
        case SQL_API_SQLSPECIALCOLUMNS:   *pfExists = TRUE; break;
        case SQL_API_SQLSTATISTICS:       *pfExists = TRUE; break;
        case SQL_API_SQLTABLES:           *pfExists = TRUE; break;

            // ODBC level 2 functions
        case SQL_API_SQLBROWSECONNECT:    *pfExists = FALSE; break;
        case SQL_API_SQLCOLUMNPRIVILEGES: *pfExists = FALSE; break;
        case SQL_API_SQLDATASOURCES:      *pfExists = FALSE; break;  // only implemented by DM
        case SQL_API_SQLDESCRIBEPARAM:    *pfExists = FALSE; break;
        case SQL_API_SQLDRIVERS:          *pfExists = FALSE; break;
        case SQL_API_SQLEXTENDEDFETCH:    *pfExists = TRUE; break;
        case SQL_API_SQLFOREIGNKEYS:      *pfExists = FALSE; break;
        case SQL_API_SQLMORERESULTS:      *pfExists = TRUE; break;
        case SQL_API_SQLNATIVESQL:        *pfExists = TRUE; break;
        case SQL_API_SQLNUMPARAMS:        *pfExists = TRUE; break;
        case SQL_API_SQLPARAMOPTIONS:     *pfExists = FALSE; break;
        case SQL_API_SQLPRIMARYKEYS:      *pfExists = FALSE; break;
        case SQL_API_SQLPROCEDURECOLUMNS: *pfExists = FALSE; break;
        case SQL_API_SQLPROCEDURES:       *pfExists = FALSE; break;
        case SQL_API_SQLSETPOS:           *pfExists = FALSE; break;
        case SQL_API_SQLSETSCROLLOPTIONS: *pfExists = FALSE; break;
        case SQL_API_SQLTABLEPRIVILEGES:  *pfExists = FALSE; break;
        }
#endif
    }

    return SQL_SUCCESS;
}

// this function still ignores the szTableType parameter.  but it does
// everything else right.
RETCODE SQL_API SQLTables(
                          HSTMT       hstmt,
                          UCHAR FAR * szTableQualifier,
                          SWORD       cbTableQualifier,
                          UCHAR FAR * szTableOwner,
                          SWORD       cbTableOwner,
                          UCHAR FAR * szTableName,
                          SWORD       cbTableName,
                          UCHAR FAR * szTableType,
                          SWORD       cbTableType)
{
    StatementClass *stmt;
    TupleNode *row;
    HSTMT tables_query_stmt;
    RETCODE result;
    char *match_table_name, *match_table_owner;
    char tables_query[512];
    char table_name[128], table_owner[128];
    SDWORD table_name_len, table_owner_len;
    int length;

    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    result = SQLAllocStmt(stmt->conn->hdbc, &tables_query_stmt);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        // error is passed from above
        return SQL_ERROR;
    }

    // taken from the psql source
    // strcpy(tables_query, "select relname, usename from pg_class, pg_user where relkind = 'r' and relname !~ '^pg_' and relname !~ '^Inv' and usesysid = relowner order by relname");
    // CC: Did not work for me Postgres complained about comparing Int4 to Oid (we are using 1.0.pl14)
    //     so I casted the operands "usesysid" and "relowner" to char16 using "int4out"

//    strcpy(tables_query, "select relname, usename from pg_class, pg_user where relkind = 'r' and relname !~ '^pg_' and relname !~ '^Inv' and int4out(usesysid) = int4out(relowner) order by relname");

    strcpy(tables_query, "select relname, usename from pg_class, pg_user where relkind = 'r' ");
    strcat(tables_query, "and relname !~ '^Inv[0-9]+' and int4out(usesysid) = int4out(relowner) ");

    if(szTableOwner && (cbTableOwner > 0 || cbTableOwner == SQL_NTS)) {
        if(cbTableOwner > 0) {
            length = cbTableOwner;
        } else {
            length = strlen(szTableOwner);
        }
        match_table_owner = malloc(length+1);
        strncpy_null(match_table_owner, szTableOwner, length+1);
        strcat(tables_query, " and usename like '");
        strcat(tables_query, match_table_owner);
        strcat(tables_query, "'");
        free(match_table_owner);
    }
    if(szTableName && (cbTableName > 0 || cbTableName == SQL_NTS)) {
        if(cbTableName > 0) {
            length = cbTableName;
        } else {
            length = strlen(szTableName);
        }
        match_table_name = malloc(length+1);
        strncpy_null(match_table_name, szTableName, length+1);
        strcat(tables_query, " and relname like '");
        strcat(tables_query, match_table_name);
        strcat(tables_query, "'");
        free(match_table_name);
    }

    strcat(tables_query, " order by relname");

    result = SQLExecDirect(tables_query_stmt, tables_query, strlen(tables_query));
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }

    result = SQLBindCol(tables_query_stmt, 1, SQL_C_CHAR,
                        table_name, 128, &table_name_len);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }
    result = SQLBindCol(tables_query_stmt, 2, SQL_C_CHAR,
                        table_owner, 128, &table_owner_len);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }

    stmt->result = ResultC_Constructor();
    if(!stmt->result) {
#ifdef _ENGLISH_
        stmt->errormsg = "Couldn't allocate memory for result.";
#else
        stmt->errormsg = "Konnte Speicher fr Rckgabewerte nicht anfordern.";
#endif
        stmt->errornumber = STMT_NO_MEMORY_ERROR;
        return SQL_ERROR;
    }

    // the binding structure for a statement is not set up until
    // a statement is actually executed, so we'll have to do this ourselves.
    extend_bindings(stmt, 5);
	
    // set the field names
    ResultC_set_num_fields(stmt->result, 5);
    ResultC_set_field_info(stmt->result, 0, "TABLE_QUALIFIER", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 1, "TABLE_OWNER", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 2, "TABLE_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 3, "TABLE_TYPE", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 4, "REMARKS", PG_TYPE_TEXT, 254);

    // add the tuples
    result = SQLFetch(tables_query_stmt);
    while((result == SQL_SUCCESS) || (result == SQL_SUCCESS_WITH_INFO)) {
        row = (TupleNode *)malloc(sizeof(TupleNode) + (5 - 1) * sizeof(TupleField));

        set_tuplefield_string(&row->tuple[0], "");

        // I have to hide the table owner from Access, otherwise it
        // insists on referring to the table as 'owner.table'.
        // (this is valid according to the ODBC SQL grammar, but
        // Postgres won't support it.)

        // set_tuplefield_string(&row->tuple[1], table_owner);
        set_tuplefield_string(&row->tuple[1], "");
        set_tuplefield_string(&row->tuple[2], table_name);
        // careful: this is case-sensitive
	if(strncmp(table_name, "pg_", 3) == 0) {
	    set_tuplefield_string(&row->tuple[3], "SYSTEM TABLE");
	} else {
	    set_tuplefield_string(&row->tuple[3], "TABLE");
	}
        set_tuplefield_string(&row->tuple[4], "");

        ResultC_add_tuple(stmt->result, row);

        result = SQLFetch(tables_query_stmt);
    }
    if(result != SQL_NO_DATA_FOUND) {
        return SQL_ERROR;
    }

    // we need to move it to the DONE list
    if(stmt->status == STMT_ALLOCATED) {
	CC_remove_from_list(stmt->conn, ALLOCATED_LIST, stmt->hstmt);
    } else if(stmt->status == STMT_READY) {
	CC_remove_from_list(stmt->conn, TO_DO_LIST, stmt->hstmt);
    }
    CC_append_to_list(stmt->conn, DONE_LIST, stmt);

    // also, things need to think that this statement is finished so
    // the results can be retrieved.
    stmt->status = STMT_FINISHED;

    // set up the current tuple pointer for SQLFetch
    stmt->currTuple = -1;

    return SQL_SUCCESS;
}

RETCODE SQL_API SQLColumns(
                           HSTMT        hstmt,
                           UCHAR FAR *  szTableQualifier,
                           SWORD        cbTableQualifier,
                           UCHAR FAR *  szTableOwner,
                           SWORD        cbTableOwner,
                           UCHAR FAR *  szTableName,
                           SWORD        cbTableName,
                           UCHAR FAR *  szColumnName,
                           SWORD        cbColumnName)
{
    StatementClass *stmt;
    TupleNode *row;
    HSTMT columns_query_stmt;
    // taken (basically) from psql source.
    char columns_query[512];
    RETCODE result;
    char *match_table_name, *match_table_owner, *match_column_name;
    char table_owner[128], table_name[128], field_name[128], field_type_name[128];
    Int4 field_type;
    SDWORD table_owner_len, table_name_len, field_name_len,
        field_type_len, field_type_name_len;
    int length;

    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    // set up the query
    // strcpy(columns_query, "select u.usename, c.relname, a.attname, a.atttypid, t.typname from pg_user u, pg_class c, pg_attribute a, pg_type t where not c.relname like 'pg_%' and u.usesysid = c.relowner and c.oid = a.attrelid and a.atttypid = t.oid and a.attnum > 0");
    // CC: Did not work for me Postgres complained about comparing Int4 to Oid (we are using 1.0.pl14)
    //     so I casted the operands "usesysid" and "relowner" to char16 using "int4out"

    strcpy(columns_query, "select u.usename, c.relname, a.attname, a.atttypid,t.typname, a.attnum from pg_user u, pg_class c, pg_attribute a, pg_type t where ");

    strcat(columns_query, "int4out(u.usesysid) = int4out(c.relowner) and c.oid= a.attrelid and a.atttypid = t.oid and (a.attnum > 0");

#ifdef SHOW_OID
    strcat(columns_query, " or a.attnum = -2)");
#else
    strcat(columns_query, ")");
#endif

    // we can't assume that the strings passed into us are null-terminated
    // properly.  (e.g., Access passes in "postgres.car" with a length of 8,
    // and expects us only to use "postgres").
    if(szTableName && (cbTableName > 0 || cbTableName == SQL_NTS)) {
        if(cbTableName > 0) {
            length = cbTableName;
        } else {
            length = strlen(szTableName);
        }
        match_table_name = malloc(length+1);
        strncpy_null(match_table_name, szTableName, length+1);
        strcat(columns_query, " and c.relname like '");
        strcat(columns_query, match_table_name);
        strcat(columns_query, "'");
        free(match_table_name);
    }

    if(szTableOwner && (cbTableOwner > 0 || cbTableOwner == SQL_NTS)) {
        if(cbTableOwner > 0) {
            length = cbTableOwner;
        } else {
            length = strlen(szTableOwner);
        }
        match_table_owner = malloc(length+1);
        strncpy_null(match_table_owner, szTableOwner, length+1);
        strcat(columns_query, " and u.usename like '");
        strcat(columns_query, match_table_owner);
        strcat(columns_query, "'");
        free(match_table_owner);
    }

    if(szColumnName && (cbColumnName > 0 || cbColumnName == SQL_NTS)) {
        if(cbColumnName > 0) {
            length = cbColumnName;
        } else {
            length = strlen(szColumnName);
        }
        match_column_name = malloc(length+1);
        strncpy_null(match_column_name, szColumnName, length+1);
        strcat(columns_query, " and a.attname like '");
        strcat(columns_query, match_column_name);
        strcat(columns_query, "'");
        free(match_column_name);
    }

    // give the output in the order the columns were defined
    // when the table was created
    strcat(columns_query, " order by attnum");

    result = SQLAllocStmt(stmt->conn->hdbc, &columns_query_stmt);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }

    result = SQLExecDirect(columns_query_stmt, columns_query,
                           strlen(columns_query));
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }

    result = SQLBindCol(columns_query_stmt, 1, SQL_C_CHAR,
                        table_owner, 128, &table_owner_len);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }
    result = SQLBindCol(columns_query_stmt, 2, SQL_C_CHAR,
                        table_name, 128, &table_name_len);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }
    result = SQLBindCol(columns_query_stmt, 3, SQL_C_CHAR,
                        field_name, 128, &field_name_len);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }
    result = SQLBindCol(columns_query_stmt, 4, SQL_C_DEFAULT,
                        &field_type, 4, &field_type_len);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }
    result = SQLBindCol(columns_query_stmt, 5, SQL_C_CHAR,
                        field_type_name, 128, &field_type_name_len);
    if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
        return SQL_ERROR;
    }

    stmt->result = ResultC_Constructor();
    if(!stmt->result) {
        stmt->errormsg = "Couldn't allocate space for result.";
        stmt->errornumber = STMT_NO_MEMORY_ERROR;
        return SQL_ERROR;
    }

    // the binding structure for a statement is not set up until
    // a statement is actually executed, so we'll have to do this ourselves.
    extend_bindings(stmt, 12);

    // set the field names
    ResultC_set_num_fields(stmt->result, 12);
    ResultC_set_field_info(stmt->result, 0, "TABLE_QUALIFIER", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 1, "TABLE_OWNER", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 2, "TABLE_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 3, "COLUMN_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 4, "DATA_TYPE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 5, "TYPE_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 6, "PRECISION", PG_TYPE_INT4, 4);
    ResultC_set_field_info(stmt->result, 7, "LENGTH", PG_TYPE_INT4, 4);
    ResultC_set_field_info(stmt->result, 8, "SCALE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 9, "RADIX", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 10, "NULLABLE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 11, "REMARKS", PG_TYPE_TEXT, 254);

    result = SQLFetch(columns_query_stmt);
    while((result == SQL_SUCCESS) || (result == SQL_SUCCESS_WITH_INFO)) {
        row = (TupleNode *)malloc(sizeof(TupleNode) +
                                  (12 - 1) * sizeof(TupleField));

        set_tuplefield_string(&row->tuple[0], "");
        // see note in SQLTables()
        //      set_tuplefield_string(&row->tuple[1], table_owner);
        set_tuplefield_string(&row->tuple[1], "");
        set_tuplefield_string(&row->tuple[2], table_name);
        set_tuplefield_string(&row->tuple[3], field_name);
        set_tuplefield_int2(&row->tuple[4], pgtype_to_sqltype(field_type));
        set_tuplefield_string(&row->tuple[5], field_type_name);
        if(pgtype_precision(field_type) != -1) {
            set_tuplefield_int4(&row->tuple[6], pgtype_precision(field_type));
        } else {
            set_tuplefield_null(&row->tuple[6]);
        }
        if((field_type == PG_TYPE_CHAR) ||
           (field_type == PG_TYPE_VARCHAR)) {
            // maybe we can get away with this for now--we're
            // pretending we don't know.
            set_tuplefield_int4(&row->tuple[7], SQL_NO_TOTAL);
        } else {
            set_tuplefield_int4(&row->tuple[7], pgtype_length(field_type));
        }
        if(pgtype_scale(field_type) != -1) {
            set_tuplefield_int2(&row->tuple[8], pgtype_scale(field_type));
        } else {
            set_tuplefield_null(&row->tuple[8]);
        }
        if(pgtype_radix(field_type) != -1) {
            set_tuplefield_int2(&row->tuple[9], pgtype_radix(field_type));
        } else {
            set_tuplefield_null(&row->tuple[9]);
        }
        set_tuplefield_int2(&row->tuple[10], pgtype_nullable(field_type));
        set_tuplefield_string(&row->tuple[11], "");

        ResultC_add_tuple(stmt->result, row);

        result = SQLFetch(columns_query_stmt);
    }
    if(result != SQL_NO_DATA_FOUND) {
        return SQL_ERROR;
    }

    // we need to move it to the DONE list
    if(stmt->status == STMT_ALLOCATED) {
	CC_remove_from_list(stmt->conn, ALLOCATED_LIST, stmt->hstmt);
    } else if(stmt->status == STMT_READY) {
	CC_remove_from_list(stmt->conn, TO_DO_LIST, stmt->hstmt);
    }
    CC_append_to_list(stmt->conn, DONE_LIST, stmt);

    // also, things need to think that this statement is finished so
    // the results can be retrieved.
    stmt->status = STMT_FINISHED;

    // set up the current tuple pointer for SQLFetch
    stmt->currTuple = -1;

    return SQL_SUCCESS;
}

RETCODE SQL_API SQLSpecialColumns(
                                  HSTMT        hstmt,
                                  UWORD        fColType,
                                  UCHAR FAR *  szTableQualifier,
                                  SWORD        cbTableQualifier,
                                  UCHAR FAR *  szTableOwner,
                                  SWORD        cbTableOwner,
                                  UCHAR FAR *  szTableName,
                                  SWORD        cbTableName,
                                  UWORD        fScope,
                                  UWORD        fNullable)
{
    TupleNode *row;
    StatementClass *stmt;

    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    stmt->result = ResultC_Constructor();
    extend_bindings(stmt, 8);

    ResultC_set_num_fields(stmt->result, 8);
    ResultC_set_field_info(stmt->result, 0, "SCOPE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 1, "COLUMN_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 2, "DATA_TYPE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 3, "TYPE_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 4, "PRECISION", PG_TYPE_INT4, 4);
    ResultC_set_field_info(stmt->result, 5, "LENGTH", PG_TYPE_INT4, 4);
    ResultC_set_field_info(stmt->result, 6, "SCALE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 7, "PSEUDO_COLUMN", PG_TYPE_INT2, 2);

    /* use the oid value for the rowid */
    if(fColType == SQL_BEST_ROWID) {

        row = (TupleNode *)malloc(sizeof(TupleNode) + (8 - 1) * sizeof(TupleField));

        set_tuplefield_int2(&row->tuple[0], SQL_SCOPE_SESSION);
        set_tuplefield_string(&row->tuple[1], "oid");
        set_tuplefield_int2(&row->tuple[2], pgtype_to_sqltype(PG_TYPE_OID));
        set_tuplefield_string(&row->tuple[3], "OID");
        set_tuplefield_int4(&row->tuple[4], pgtype_precision(PG_TYPE_OID));
        set_tuplefield_int4(&row->tuple[5], pgtype_length(PG_TYPE_OID));
        set_tuplefield_int2(&row->tuple[6], pgtype_scale(PG_TYPE_OID));
        set_tuplefield_int2(&row->tuple[7], SQL_PC_PSEUDO);

        ResultC_add_tuple(stmt->result, row);

    } else if(fColType == SQL_ROWVER) {
        /* can columns automatically update? */
        /* for now assume no. */
        /* return an empty result. */
    }

    if(stmt->status == STMT_ALLOCATED) {
	CC_remove_from_list(stmt->conn, ALLOCATED_LIST, stmt->hstmt);
    } else if(stmt->status == STMT_READY) {
	CC_remove_from_list(stmt->conn, TO_DO_LIST, stmt->hstmt);
    }
    CC_append_to_list(stmt->conn, DONE_LIST, stmt);
    stmt->status = STMT_FINISHED;
    stmt->currTuple = -1;

    return SQL_SUCCESS;
}

RETCODE SQL_API SQLStatistics(
                              HSTMT         hstmt,
                              UCHAR FAR *   szTableQualifier,
                              SWORD         cbTableQualifier,
                              UCHAR FAR *   szTableOwner,
                              SWORD         cbTableOwner,
                              UCHAR FAR *   szTableName,
                              SWORD         cbTableName,
                              UWORD         fUnique,
                              UWORD         fAccuracy)
{
    StatementClass *stmt;
    char index_query[512];
    HSTMT index_query_stmt;
    RETCODE result;
    char *table_name;
    int length;
    char index_name[128];
    short fields_vector[8];
    SDWORD index_name_len, fields_vector_len;
    TupleNode *row;
    int i;
    HSTMT columns_stmt;
    char column_name[128];
    char **column_names = 0;
    Int4 column_name_len;
    int total_columns;

    stmt = HI_stmtfromhstmt(the_handles, hstmt);
    if(!stmt) {
        return SQL_INVALID_HANDLE;
    }

    stmt->result = ResultC_Constructor();
    if(!stmt->result) {
#ifdef _ENGLISH_
        stmt->errormsg = "Couldn't allocate memory for result.";
#else
        stmt->errormsg = "Konnte Speicher fr Rckgabewert nicht anfordern.";
#endif
        stmt->errornumber = STMT_NO_MEMORY_ERROR;
        return SQL_ERROR;
    }

    // the binding structure for a statement is not set up until
    // a statement is actually executed, so we'll have to do this ourselves.
    extend_bindings(stmt, 13);

    // set the field names
    ResultC_set_num_fields(stmt->result, 13);
    ResultC_set_field_info(stmt->result, 0, "TABLE_QUALIFIER", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 1, "TABLE_OWNER", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 2, "TABLE_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 3, "NON_UNIQUE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 4, "INDEX_QUALIFIER", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 5, "INDEX_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 6, "TYPE", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 7, "SEQ_IN_INDEX", PG_TYPE_INT2, 2);
    ResultC_set_field_info(stmt->result, 8, "COLUMN_NAME", PG_TYPE_TEXT, 128);
    ResultC_set_field_info(stmt->result, 9, "COLLATION", PG_TYPE_CHAR, 1);
    ResultC_set_field_info(stmt->result, 10, "CARDINALITY", PG_TYPE_INT4, 4);
    ResultC_set_field_info(stmt->result, 11, "PAGES", PG_TYPE_INT4, 4);
    ResultC_set_field_info(stmt->result, 12, "FILTER_CONDITION", PG_TYPE_TEXT, 128);

    // there are no unique indexes in postgres, so return nothing
    // if those are requested
    if(fUnique != SQL_INDEX_UNIQUE) {
        // only use the table name... the owner should be redundant, and
        // we never use qualifiers.
        if(szTableName && (cbTableName > 0 || cbTableName == SQL_NTS)) {
            if(cbTableName > 0) {
                length = cbTableName;
            } else {
                length = strlen(szTableName);
            }
            table_name = malloc(length+1);
            strncpy_null(table_name, szTableName, length+1);
        } else {
#ifdef _ENGLISH_
            stmt->errormsg = "No table name passed to SQLStatistics.";
#else
            stmt->errormsg = "SQLStatistics wurde ohne Table--Name aufgerufen.";
#endif
            stmt->errornumber = STMT_INTERNAL_ERROR;
            return SQL_ERROR;
        }

	// we need to get a list of the field names first,
	// so we can return them later.
	result = SQLAllocStmt(stmt->conn->hdbc, &columns_stmt);
	if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
	    stmt->errormsg = "SQLAllocStmt failed in SQLStatistics.";
	    stmt->errornumber = STMT_NO_MEMORY_ERROR;
            return SQL_ERROR;
	}
	result = SQLColumns(columns_stmt, "", 0, "", 0, 
			    table_name, strlen(table_name), "", 0);
	if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
	    stmt->errormsg = "SQLColumns failed in SQLStatistics.";
	    stmt->errornumber = STMT_NO_MEMORY_ERROR;
            return SQL_ERROR;
	}
	result = SQLBindCol(columns_stmt, 4, SQL_C_CHAR,
			    column_name, 128, &column_name_len);
        if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
	    stmt->errormsg = "Couldn't bind column in SQLStatistics.";
	    stmt->errornumber = STMT_INTERNAL_ERROR;
            return SQL_ERROR;
        }

	total_columns = 0;
	result = SQLFetch(columns_stmt);
        while((result == SQL_SUCCESS) || (result == SQL_SUCCESS_WITH_INFO)) {
	    total_columns++;

	    column_names = 
		(char **)realloc(column_names, 
				 total_columns * sizeof(char *));
	    column_names[total_columns-1] = 
		(char *)malloc(strlen(column_name)+1);
	    strcpy(column_names[total_columns-1], column_name);

	    result = SQLFetch(columns_stmt);
	}
	if(result != SQL_NO_DATA_FOUND ||
	   total_columns == 0) {
	    stmt->errormsg = "Couldn't get column names in SQLStatistics.";
	    stmt->errornumber = STMT_INTERNAL_ERROR;
            return SQL_ERROR;	    
	}
	
	SQLFreeStmt(columns_stmt, SQL_DROP);

        // get a list of indexes on this table
        result = SQLAllocStmt(stmt->conn->hdbc, &index_query_stmt);
        if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
            // error is NOT passed from above
	    // since that error is on a different handle
	    stmt->errormsg = "SQLAllocStmt failed in SQLStatistics.";
	    stmt->errornumber = STMT_NO_MEMORY_ERROR;
            return SQL_ERROR;
        }

        strcpy(index_query, "select c.relname, i.indkey from pg_index i, pg_class c, pg_class d where c.oid = i.indexrelid and d.relname = '");
        strcat(index_query, table_name);
        strcat(index_query, "' and d.oid = i.indrelid");

        result = SQLExecDirect(index_query_stmt, index_query, strlen(index_query));
        if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
	    stmt->errormsg = "Couldn't execute index query (w/SQLExecDirect) in SQLStatistics.";
	    stmt->errornumber = STMT_INTERNAL_ERROR;
            return SQL_ERROR;
        }

        result = SQLBindCol(index_query_stmt, 1, SQL_C_CHAR,
                            index_name, 128, &index_name_len);
        if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
	    stmt->errormsg = "Couldn't bind column in SQLStatistics.";
	    stmt->errornumber = STMT_INTERNAL_ERROR;
            return SQL_ERROR;
        }
        // bind the vector column
        result = SQLBindCol(index_query_stmt, 2, SQL_C_DEFAULT,
                            fields_vector, 16, &fields_vector_len);
        if((result != SQL_SUCCESS) && (result != SQL_SUCCESS_WITH_INFO)) {
	    stmt->errormsg = "Couldn't bind column in SQLStatistics.";
	    stmt->errornumber = STMT_INTERNAL_ERROR;
            return SQL_ERROR;
        }

        result = SQLFetch(index_query_stmt);
        while((result == SQL_SUCCESS) || (result == SQL_SUCCESS_WITH_INFO)) {
	    i = 0;
	    // add a row in this table for each field in the index
	    while(i < 8 && fields_vector[i] != 0) {
		row = (TupleNode *)malloc(sizeof(TupleNode) + 
					  (13 - 1) * sizeof(TupleField));

		// no table qualifier
		set_tuplefield_string(&row->tuple[0], "");
		// don't set the table owner, else Access tries to use it
		set_tuplefield_string(&row->tuple[1], "");
		set_tuplefield_string(&row->tuple[2], table_name);

		// Postgres95 indices always allow non-unique values.
		set_tuplefield_int2(&row->tuple[3], TRUE);
		
		// no index qualifier
		set_tuplefield_string(&row->tuple[4], "");
		set_tuplefield_string(&row->tuple[5], index_name);

		// check this--what does it mean for an index
		// to be clustered?  (none of mine seem to be--
		// we can and probably should find this out from
		// the pg_index table)
		set_tuplefield_int2(&row->tuple[6], SQL_INDEX_HASHED);
		set_tuplefield_int2(&row->tuple[7], i+1);

		if(fields_vector[i] < 0 ||
		   fields_vector[i] > total_columns) {
		    set_tuplefield_string(&row->tuple[8], "UNKNOWN");
		} else {
		    set_tuplefield_string(&row->tuple[8], 
					  column_names[fields_vector[i]-1]);
		}
		set_tuplefield_string(&row->tuple[9], "A");
		set_tuplefield_null(&row->tuple[10]);
		set_tuplefield_null(&row->tuple[11]);
		set_tuplefield_null(&row->tuple[12]);

		ResultC_add_tuple(stmt->result, row);
		i++;
	    }

            result = SQLFetch(index_query_stmt);
        }
        if(result != SQL_NO_DATA_FOUND) {
	    stmt->errormsg = "SQLFetch failed in SQLStatistics.";
	    stmt->errornumber = STMT_INTERNAL_ERROR;
            return SQL_ERROR;
        }

	SQLFreeStmt(index_query_stmt, SQL_DROP);
    }

    // we need to move it to the DONE list
    if(stmt->status == STMT_ALLOCATED) {
	CC_remove_from_list(stmt->conn, ALLOCATED_LIST, stmt->hstmt);
    } else if(stmt->status == STMT_READY) {
	CC_remove_from_list(stmt->conn, TO_DO_LIST, stmt->hstmt);
    }
    CC_append_to_list(stmt->conn, DONE_LIST, stmt);

    // also, things need to think that this statement is finished so
    // the results can be retrieved.
    stmt->status = STMT_FINISHED;

    // set up the current tuple pointer for SQLFetch
    stmt->currTuple = -1;

    free(table_name);
    for(i = 0; i < total_columns; i++) {
	free(column_names[i]);
    }
    free(column_names);

    return SQL_SUCCESS;
}

RETCODE SQL_API SQLColumnPrivileges(
                                    HSTMT        hstmt,
                                    UCHAR FAR *  szTableQualifier,
                                    SWORD        cbTableQualifier,
                                    UCHAR FAR *  szTableOwner,
                                    SWORD        cbTableOwner,
                                    UCHAR FAR *  szTableName,
                                    SWORD        cbTableName,
                                    UCHAR FAR *  szColumnName,
                                    SWORD        cbColumnName)
{
    return SQL_ERROR;
}

RETCODE SQL_API SQLForeignKeys(
                               HSTMT         hstmt,
                               UCHAR FAR *   szPkTableQualifier,
                               SWORD         cbPkTableQualifier,
                               UCHAR FAR *   szPkTableOwner,
                               SWORD         cbPkTableOwner,
                               UCHAR FAR *   szPkTableName,
                               SWORD         cbPkTableName,
                               UCHAR FAR *   szFkTableQualifier,
                               SWORD         cbFkTableQualifier,
                               UCHAR FAR *   szFkTableOwner,
                               SWORD         cbFkTableOwner,
                               UCHAR FAR *   szFkTableName,
                               SWORD         cbFkTableName)
{
    return SQL_ERROR;
}

RETCODE SQL_API SQLPrimaryKeys(
                               HSTMT         hstmt,
                               UCHAR FAR *   szTableQualifier,
                               SWORD         cbTableQualifier,
                               UCHAR FAR *   szTableOwner,
                               SWORD         cbTableOwner,
                               UCHAR FAR *   szTableName,
                               SWORD         cbTableName)
{
    return SQL_ERROR;
}

RETCODE SQL_API SQLProcedureColumns(
                                    HSTMT         hstmt,
                                    UCHAR FAR *   szProcQualifier,
                                    SWORD         cbProcQualifier,
                                    UCHAR FAR *   szProcOwner,
                                    SWORD         cbProcOwner,
                                    UCHAR FAR *   szProcName,
                                    SWORD         cbProcName,
                                    UCHAR FAR *   szColumnName,
                                    SWORD         cbColumnName)
{
    return SQL_ERROR;
}

RETCODE SQL_API SQLProcedures(
                              HSTMT          hstmt,
                              UCHAR FAR *    szProcQualifier,
                              SWORD          cbProcQualifier,
                              UCHAR FAR *    szProcOwner,
                              SWORD          cbProcOwner,
                              UCHAR FAR *    szProcName,
                              SWORD          cbProcName)
{
    return SQL_ERROR;
}

RETCODE SQL_API SQLTablePrivileges(
                                   HSTMT           hstmt,
                                   UCHAR FAR *     szTableQualifier,
                                   SWORD           cbTableQualifier,
                                   UCHAR FAR *     szTableOwner,
                                   SWORD           cbTableOwner,
                                   UCHAR FAR *     szTableName,
                                   SWORD           cbTableName)
{
    return SQL_ERROR;
}
