/* -*- mode: c++; c-basic-offset: 4 -*- */
/********************************************************************
    POSTODBCL.DLL - A library to talk to Postgres95 by using the
                    WINDOWS ODBC Interface

    Copyright (C) 1996; Christian Czezatke, Dan McGuirk

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    How to contact the authors:

    email to: e9025461@student.tuwien.ac.at     (Christian Czezatke)
              mcgurik@indirect.com              (Dan McGuirk)
********************************************************************/



#ifndef __GLOBALS_H__
#define __GLOBALS_H__

/* globals.h/globals.c: Defines handles for connections and statements */

#include "socket/wrapper.h"
#include "postodbc.h"
#include "custom.h"
#if defined(iODBC)
#  include <isql.h>
#else
#  include <sql.h>
#endif

/*
*
* Type BindInfoClass -- needed by StatemenaClass
*/

typedef struct {
    Int4 buflen; /* size of buffer */
    char *buffer; /* pointer to the buffer */
    Int4 *used; /* used space in the buffer (for strings not counting the '\0') */
    Int2 returntype; /* kind of conversion to be appied when returning (SQL_C_DEFAULT, SQL_C_CHAR...) */
} BindInfoClass;


/*
 * ParameterInfoClass -- stores information about a bound parameter
 */
typedef struct {
    Int4 buflen;
    char *buffer;
    Int4 *used;

    // probably not all of these will ever be used, but they're giving us
    // the information--might as well hold on to it
    Int2 paramType;
    Int2 CType;
    Int2 SQLType;
    UInt4 precision;
    Int2 scale;
} ParameterInfoClass;

/*
 * EnvironmentClass -- stores information (really only error messages)
 *    that applies to the whole environment
 */
#define ENV_ALLOC_ERROR 1

typedef struct {
    char *errormsg;
    int errornumber;
} EnvironmentClass;

EnvironmentClass *EN_Constructor(void);
char EN_get_error(EnvironmentClass *self, int *number, char **message);
char EN_Destructor(EnvironmentClass *self);




typedef enum {
    STMT_ALLOCATED,     /* The statement handle is allocated, but not used so far */
    STMT_READY,         /* the statement is waiting to be executed */
    STMT_PREMATURE,     /* CC: ODBC states that it is legal to call e.g. SQLDescribeCol before
                           a call to SQLExecute, but after SQLPrepare. To get all the necessary
                           information in such a case, we simply execute the query _before_ the
                           actual call to SQLExecute, so that statement is considered to be "premature".
                           Premature statements are in the same statement list as "STMT_FINISHED" statements.
                        */
    STMT_FINISHED,      /* statement execution has finished */
    STMT_EXECUTING      /* statement execution is still going on */
} STMT_Status;

#define STMT_TRUNCATED -2
#define STMT_INFO_ONLY -1 /* not an error message, just a notification to be returned by SQLError */
#define STMT_OK 0 /* will be interpreted as "no error pending" */
#define STMT_EXEC_ERROR 1
#define STMT_STATUS_ERROR 2
#define STMT_SEQUENCE_ERROR 3
#define STMT_NO_MEMORY_ERROR 4
#define STMT_COLNUM_ERROR 5
#define STMT_NO_STMTSTRING 6
#define STMT_ERROR_TAKEN_FROM_BACKEND 7
#define STMT_INTERNAL_ERROR 8
#define STMT_STILL_EXECUTING 9
#define STMT_NOT_IMPLEMENTED_ERROR 10
#define STMT_BAD_PARAMETER_NUMBER_ERROR 11
#define STMT_OPTION_OUT_OF_RANGE_ERROR 12
#define STMT_INVALID_COLUMN_NUMBER_ERROR 13
#define STMT_RESTRICTED_DATA_TYPE_ERROR 14
#define STMT_INVALID_CURSOR_STATE_ERROR 15
#define STMT_OPTION_VALUE_CHANGED 16


/* statement types */
#define STMT_TYPE_SELECT     0
#define STMT_TYPE_INSERT     1
#define STMT_TYPE_UPDATE     2
#define STMT_TYPE_DELETE     3
#define STMT_TYPE_OTHER      4
#define STMT_TYPE_UNKNOWN  666  // 'unknown' means we don't have the statement yet,
                                // or haven't looked at it to see what type it is.
                                // 'other' means we looked, but couldn't tell.
#ifdef __WATCOMC__
struct ConnectionClass_;
/* CC: Watcom C needs this forward declaration to be able to cope with the
   "ConnectionClass_ *conn" member of the following struct
*/
#endif

/* Class for handling individual statements */
struct StatementClass_ {
    HSTMT hstmt; /* Handle of the current statement */
    ResultC *result; /* result of the current statement (or NULL
                        if still executint) */

    struct ConnectionClass_ *conn;
                     /* pointer to ConnectionClass this statement belongs to */
    STMT_Status status;
    char *errormsg;
    int errornumber;

    /* information on bindings */
    BindInfoClass *bindings; /* array to store the binding information */
    int bindings_allocated;
    /* use ResultC_getNumRows(self) to determine the length of the bindings array */

    /* for extended fetching */
    UInt4 rowset_size;              /* how many rows to return at once */
                                    /* on an extended fetch */
    UInt4 binding_type_or_offset;   /* if column-wise binding, this is      */
                                    /* SQL_BIND_BY_COLUMN.  otherwise,      */
                                    /* (row-wise binding), it is the offset */
                                    /* between rows.                        */


    /* information on statement parameters */
    int parameters_allocated;
    ParameterInfoClass *parameters;

    Int4 currTuple; /* only valid if status is STMT_FINISHED or STMT_PREMATURE. In that case it points to
                        the tuple that will be returned by the next Fetch command
                        (This index is 0 -based.) */
    char *statement; /* if non--null pointer to the SQL statement that has been executed */
    int statement_type; /* According to the defines above */

    char resulterror_taken; /* if non--null the error message from the *result structure
                                has already been used by the error handling functions */
    struct StatementClass_ *next; /* chaining statements for this connection */
};

typedef struct StatementClass_  StatementClass;


StatementClass *SC_Constructor(HSTMT hstmt, struct ConnectionClass_ *conn);
/* builds a new statement (copys the message string) */
/* second parameter must be a ConnectionClass */

char SC_recycle_statement(StatementClass *self);
/* turns an already used handle back into a newly allocated one.
   CC: Parameter bindings (if they are there) are _not_ removed. This conforms to the
       ODBC specs since Parameter bindings should only be removed with the SQL_DROP
       option of SQLFreeStmt
*/


char SC_do_prepare(StatementClass *self, char *statement, int statement_len);
/* prepares a statement to be executed at lateron */

char SC_do_execute(StatementClass *self);
/* execute a statement that has been prepared by a previous call to do_prepare */


char SC_start_execution(StatementClass *self, char *statement,
                        int statement_len);
/* causes the statement to be executed */

/* CC: SC_bind_cols got return type (SQL_C_*) parameter added */
char SC_bind_cols(StatementClass *self, UInt2 colnum, Int4 buflen, char *buffer, Int4 *used, Int2 returntype);
/* used for binding and unbinding columns, again indices are 0 -based */
/* *buffer is buflen bytes long and shall contain the result value. */
/* *used tells how big the value returned actually is (not counting */
/* the terminating '\0' -- byte if using ASCII representation.      */
/* "returntype" stores information on how to return the value (SQL_C_* */
char SC_unbind_cols(StatementClass *self);
/* unbind all columns. */

char SC_bind_parameter(StatementClass *self, UInt2 which_param,
                       Int4 param_max_length, char *buffer, Int4 *param_length,
                       Int2 param_type, Int2 param_c_type, Int2 param_sql_type,
                       UInt4 precision, Int2 scale);
/* CC: Dan, could you elaborate on that ? ;-) */

Int4 SC_perform_fetch(StatementClass *self, int do_extended_fetch);
// CC: changed int -> Int4
/* copies the values of the current tuple (pointed to by currTuple) to the
   respective buffers if the row has been bound -- implments SQLFetch */

void SC_clear_error(StatementClass *self);
/* resets the error condition flags */

char SC_get_error(StatementClass *self, int *number, char **message);
/* does not copy the information, do not modify *message */
/* returns true if an error is pending */

#define SC_get_ResultC(a) (a->result);
/* returns the ResultC structure containing the result information */

char SC_Destructor(StatementClass *self);

/**********************/
/* internal use only: */
/**********************/

#define SC_get_conn(a) (a->conn)
/* return the connection class it belongs to */

void SC_resultinfo_preprocess(StatementClass *self);
 /* CC: All this function does is to transform a STMT_READY to STMT_PREMATURE or
        nothing if self's status != STMT_PREMATURE. -- Used by info functions (DescribeCol,...)
        To deal with "SQLPrepare -- <Info function calls>" -- sequences
 */




typedef enum {
    CONN_NOT_CONNECTED,      /* Connection has not been established */
    CONN_CONNECTED,      /* Connection is up and has been established */
    CONN_DOWN,            /* Connection is broken */
    CONN_EXECUTING     /* the connection is currently executing a statement */
} CONN_Status;

#define CONN_INIREAD_ERROR 1
#define CONN_OPENDB_ERROR 2
#define CONN_STMT_ALLOC_ERROR 3
#define CONN_IN_USE 4 /* freeing that conn is impossible since a query is active on it */
#define CONN_UNSUPPORTED_OPTION 5
/* Used by SetConnectoption to indicate unsupported options */
#define CONN_INVALID_ARGUMENT_NO 6
/* SetConnectOption: corresponds to ODBC--"S1009" */
#define CONN_TRANSACT_IN_PROGRES 7
#define CONN_NO_MEMORY_ERROR 8
#define CONN_NOT_IMPLEMENTED_ERROR 9


#define CONN_IN_AUTOCOMMIT 0x01
#define CONN_IN_TRANSACTION 0x02

/* Class for handling a connection */
struct ConnectionClass_ {
     HDBC hdbc; /* Connection handle */
     ConnectionC *connection;
     StatementClass *allocated; /*list of unused statement handles */
     StatementClass *to_do; /* list of statements to send to the server */
     StatementClass *done; /* list of statements already processed */
     /*SQLFreeStmt will clear all that information */

     char *username;
     char *errormsg;
     int errornumber;
     CONN_Status status;

     char transact_status; /* Indicates whether we are in AutoCommit or wether a transaction is currently in progress */
};


typedef struct ConnectionClass_ ConnectionClass;


ConnectionClass *CC_Constructor();

void CC_get_defaults(char *DSN,
                     char *server, int server_len,
                     char *port, int port_len,
                     char *database, int database_len,
                     char *username, int username_len);
/* Return Information as required by a call to SQLConnect or
   SQLDriverConnect */

char CC_connect(ConnectionClass *self, HDBC hdbc, char *DSN,
                char *username, char *machine, char *port, char *dbasename);


StatementClass *CC_stmtfromhstmt(ConnectionClass *self, HSTMT hstmt);

char CC_destroy_statement(ConnectionClass *self, HSTMT hstmt);

char CC_get_error(ConnectionClass *self, int *number, char **message);
/* does not copy the information, do not modify *message */
/* returns true if an error is pending */

StatementClass *CC_create_new_statement(ConnectionClass *self, HSTMT hstmt);
/* Creates a new statement and inserts it into the allocated list */

char CC_set_commit_behaviour(ConnectionClass *self, char do_autocommit);
/* Turn on or off Autocommit behaviour. If we are
   in manual commit and a transaction is currently in progress, 0 is returned
*/

char CC_Destructor(ConnectionClass *self);

/* for internal use only */

#define ALLOCATED_LIST 0
#define TO_DO_LIST 1
#define DONE_LIST 2
StatementClass *CC_remove_from_list(ConnectionClass *self, int which, HSTMT hstmt);

void CC_append_to_list(ConnectionClass *self, int which, StatementClass *stmt);




#define MAX_CONNECTIONS 16

/* Class for keeping all the connections and statement handles */
typedef struct {
   ConnectionClass *conns[MAX_CONNECTIONS];
   /* the current implementation limits the number of connections to 16 */
} HandleInfo;

HandleInfo *HI_Constructor();

char HI_add_connection(HandleInfo *self, ConnectionClass *conn);
char HI_remove_connection(HandleInfo *self, ConnectionClass *conn);

ConnectionClass *HI_connfromhdbc(HandleInfo *self, HDBC hdbc);
StatementClass *HI_stmtfromhstmt(HandleInfo *self, HSTMT hstmt);

char HI_Destructor(HandleInfo *self);

BindInfoClass *create_empty_bindings(int num_columns);
void extend_bindings(StatementClass *stmt, int num_columns);

void remove_newlines(char *string);
char *copy_statement_with_parameters(StatementClass *stmt, char *old_statement,
                                     int old_statement_len);
char *convert_parameter(ParameterInfoClass *param);
int count_escapes_needed(char *string, int string_len);
void escape_stuff(char *input_string, int input_string_len,
                   char *output_string);
int needs_escape(char ch);
int statement_type(char *statement);

#define COPY_OK 0
#define COPY_UNSUPPORTED_TYPE        1
#define COPY_UNSUPPORTED_CONVERSION  2
#define COPY_RESULT_TRUNCATED        3



int copy_and_convert_field_bindinfo(Int4 field_type, void *value, BindInfoClass *bic);
/* CC: Changed that. Now all other return types besides SQL_C_DEFAULT get handled
       correctly too. Simply calls copy_and_convert_field with the appropriate parameters
       taken from the BindInfoClass instance "bic"
*/


int copy_and_convert_field(Int4 field_type, void *value,
                           Int2 fCType, PTR rgbValue, SDWORD cbValueMax, SDWORD *pcbValue);



char *strncpy_null(char *dst, const char *src, size_t len);


extern HandleInfo *the_handles;

extern StatementClass *last_stmt;
extern HSTMT last_hstmt;

#endif
