/* -*- mode: c++; c-basic-offset: 4 -*- */
/********************************************************************
    POSTODBCL.DLL - A library to talk to Postgres95 by using the
                    WINDOWS ODBC Interface

    Copyright (C) 1996; Christian Czezatke, Dan McGuirk

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    How to contact the authors:

    email to: e9025461@student.tuwien.ac.at     (Christian Czezatke)
              mcguirk@indirect.com              (Dan McGuirk)
********************************************************************/

/*
**
** This file contains the main "glue logic" between
** the ODBC interface of PostODBC and its low level
** functions as implemented in the subdirectory "socket"
**
*/

#include "custom.h"
#include <windows.h>
#include "globals.h"
#include "socket\wrapper.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "odbcinst.h" /* CC: for SQLGetPrivateProfileString */
#include "pgtypes.h"


HandleInfo *the_handles = NULL;

StatementClass *last_stmt;
HSTMT last_hstmt;

/*
// That function was quite handy when debugging the driver
void write_contents(ResultC *res)
{
    FILE *fp = fopen("c:\\temp\\dbdump.txt", "w");
    Int4 lf, tuples;
    char *value;

    if (NULL == res) {
        fprintf(fp, "kein resultcode\n");
        return;
    }

    tuples = ResultC_get_noof_tuples(res);
    fprintf(fp, "Tuples: %ld\n", tuples);

    for (lf=0; lf < tuples; lf++) {
        value = ResultC_get_value(res, lf, 0);
        fprintf(fp, "%ld. Wert: %s\n", lf+1, value);
    }
    fprintf(fp,"Over and out!\n");
    fclose(fp);
}
*/



/*
 * EnvironmentClass implementation
 */

EnvironmentClass *EN_Constructor(void)
{
    EnvironmentClass *new_env;

    new_env = (EnvironmentClass *)malloc(sizeof(EnvironmentClass));
    if(!new_env) {
        return 0;
    }

    new_env->errormsg = 0;
    new_env->errornumber = 0;

    return new_env;
}

char EN_get_error(EnvironmentClass *self, int *number, char **message)
{
    if(self && self->errormsg && self->errornumber) {
        *message = self->errormsg;
        *number = self->errornumber;
        self->errormsg = 0;
        self->errornumber = 0;
        return 1;
    } else {
        return 0;
    }
}

char EN_Destructor(EnvironmentClass *self)
{
    // the error messages are static strings distributed throughout
    // the source--they should not be freed

    return 1;
}



/*
 * StatementClass implementation
 */


StatementClass *SC_Constructor(HSTMT hstmt, struct ConnectionClass_ *conn)
{
    StatementClass *rv = (StatementClass *)malloc(sizeof(StatementClass));

    if (NULL != rv) {
        rv->hstmt  = hstmt;
        rv->result = NULL;
        rv->conn = conn;
        rv->status = STMT_ALLOCATED;
        rv->errormsg = NULL;
        rv->errornumber = 0;
        rv->statement = NULL;
        rv->statement_type = STMT_TYPE_UNKNOWN;
        rv->bindings = NULL;
        rv->bindings_allocated = 0;
        rv->rowset_size = 1;
        rv->binding_type_or_offset = SQL_BIND_BY_COLUMN;
        rv->parameters_allocated = 0;
        rv->parameters = 0;
        rv->resulterror_taken = 0;
        rv->next = NULL;
        rv->currTuple = -1;
        rv->result = 0;
    }
    return rv;
}


char SC_recycle_statement(StatementClass *self)
{
 ConnectionClass *conn;
    
 if (self->status == STMT_EXECUTING)
    // a statement that is currently executing cannot be recycled since
    //   there is no way to preempt a database transaction in Postgres
    return 0;

 self->errormsg = NULL;
 self->errornumber = 0;
 self->resulterror_taken = 0;

 switch (self->status) {
     case STMT_ALLOCATED:
        /* this statement does not need to be recycled */
        return 1;
     case STMT_READY:
        CC_remove_from_list(self->conn, TO_DO_LIST, self->hstmt);
        break;
     case STMT_PREMATURE:
        /* CC: Premature execution of the statement might have caused the start of a transaction.
               If so, we have to rollback that transaction. */
        conn = SC_get_conn(self);
        if ( ! ((conn->transact_status) & CONN_IN_AUTOCOMMIT)  && /* we are in manual commit mode */
               ((conn->transact_status) & CONN_IN_TRANSACTION) /* and there is no transaction going on */
       )
       {             

          ConnectionC_send_query(conn->connection, "rollback transaction");
          conn->transact_status &= ~CONN_IN_TRANSACTION;
       }
       CC_remove_from_list(self->conn, DONE_LIST, self->hstmt);
       break;

     case STMT_FINISHED:
        CC_remove_from_list(self->conn, DONE_LIST, self->hstmt);
        break;
     default:
#ifdef _ENGLISH_
       self->errormsg = "An internal error occured while recycling statements";
#else
       self->errormsg = "Es ist ein interner Fehler beim wiederverwerten von Statements aufgetreten";
#endif
       self->errornumber = STMT_INTERNAL_ERROR;
       return 0;
 }


 /*
  * we definitely want to free all results--that is the definition of
  * SQL_CLOSE.
  */
 if (NULL != self->result) {
     ResultC_Destructor(self->result);
     self->result = NULL;
 }

 /* 
  * at this point the transaction is not on any lists.
  * for us to get here, it must have been in some state other
  * than STMT_ALLOCATED, so it has been prepared or directly
  * executed before.  so now make its status STMT_READY again.
  */
 self->status = STMT_READY;
 self->errormsg = NULL;
 self->errornumber = 0;

 /*
  * our equivalent of "closing the cursor"
  */
 self->currTuple = -1;

 self->resulterror_taken = 0;

 /*
  * and put it back on the ready list
  */
 CC_append_to_list(self->conn, TO_DO_LIST, self);
 return 1;
}





char SC_do_prepare(StatementClass *self, char *statement, int statement_len)
{
 char ok = 0;
 struct ConnectionClass_ *hlp;


  /* CC: According to the ODBC specs it is valid to call SQLPrepare mulitple times. In that case,
         the bound SQL statement is replaced by the new one */

  switch (self->status) {
     
    case STMT_PREMATURE:
      SC_recycle_statement(self); /* recycle the statement, but do not remove parameter bindings */
      /* NO Break! -- Contiue the same way as with a newly allocated statement ! */
    case STMT_ALLOCATED:
        // it is not really necessary to do any conversion of the statement
        // here--just copy it, and deal with it when it's ready to be
        // executed.
        if(statement_len == SQL_NTS) {
            self->statement = strdup(statement);
        } else {
            self->statement = (char *)malloc(statement_len+1);
            strncpy_null(self->statement, statement, statement_len+1);
        }
        self->statement_type = statement_type(self->statement);
        hlp = self->conn;
        self->status = STMT_READY;

     /* CC: move that statement to the list of pre-allocated statements */
     CC_remove_from_list(hlp, ALLOCATED_LIST, self->hstmt);
     CC_append_to_list(hlp, TO_DO_LIST, self);
     ok = 1;
      break;
    
    case STMT_READY:  /* SQLPrepare has already been called -- Just changed the SQL statement that is assigned to the handle */
       if (NULL != self->statement)
         free(self->statement);
       self->statement = (char *)malloc(statement_len+1);
       strncpy_null(self->statement, statement, statement_len+1);
       self->statement_type = statement_type(self->statement);
       ok =1;
       break;
                            
    case STMT_FINISHED:
    case STMT_EXECUTING:
#ifdef _ENGLISH_
       self->errormsg = "The handle does not point to a statement that is ready to be executed";
#else
       self->errormsg = "Dieser Handle verweist nicht auf ein auszufhrendes Statement";
#endif
       ok = 0;
       break;
     default:
       self->errormsg = "An Internal Error has occured -- Unknown statement status.";
     break;
  } /* end switch (self->status) */    
 return ok;
}



char SC_do_execute(StatementClass *self)
{
 char ok = 0, was_ok, *res;
 Int2 numcols;
 ConnectionClass *hlp;
 Int2 oldstatus;
    char *statement_with_parameters;
    
 // check if this or any other statement currently uses that connection
 // to transmit a transaction
 hlp = self->conn;
 if (CONN_EXECUTING == hlp->status)
    return 0;

 /* CC: Now check if the statement is a premature one. If so, all we have to do is to
        turn it into a "normal" finished statement */
 if (STMT_PREMATURE == self->status) {
   self->status = STMT_FINISHED;
   return 1;
 }  
        

 if (self->status != STMT_READY) {
     self->errornumber = STMT_STATUS_ERROR;
#ifdef _ENGLISH_
     self->errormsg = "The handle does not point to a statement that is ready to be executed";
#else
     self->errormsg = "Dieser Handle verweist nicht auf ein auszufhrendes Statement";
#endif
 } else {
     if (self->statement == NULL) {
         self->errornumber = STMT_NO_STMTSTRING;
#ifdef _ENGLISH_
         self->errormsg = "This handle does not have a SQL statement stored in it";
#else
         self->errormsg = "Kein SQL Statement gespeichert";
#endif
         return 0;
     }

     // copy the statement again, this time inserting the parameters
     statement_with_parameters = 
         copy_statement_with_parameters(self, self->statement,
                                        strlen(self->statement));
     if(!statement_with_parameters) {
         return 0;
     }
     oldstatus = hlp->status;
     hlp->status = CONN_EXECUTING;
     self->status = STMT_EXECUTING;
     self->result = ConnectionC_send_query( ((ConnectionClass *)hlp)->connection, statement_with_parameters);
     free(statement_with_parameters);
     hlp->status = oldstatus;
     self->status = STMT_FINISHED;
     ok = (NULL != self->result);

     if (ok) {
         // At least the postgres server has processsed the query.
         //   Now let us find out how the server got along with our statement

         res = ResultC_get_message(self->result);
         was_ok = ResultC_command_successful(self->result);

         self->errormsg = res;
         self->errornumber = was_ok ? STMT_INFO_ONLY : STMT_ERROR_TAKEN_FROM_BACKEND;
         self->resulterror_taken = 1;

         self->currTuple = -1; /* set cursor before the first tuple in the list */

         /* now move that statement from the to do to the done list */
         CC_remove_from_list(hlp, TO_DO_LIST, self->hstmt);
         CC_append_to_list(hlp, DONE_LIST, self);


         /* see if the query did return any result columns */
         numcols = ResultC_NumResultCols(self->result);
         if (0 == numcols)
             return was_ok;

         /* now allocate the array to hold the binding info */
         extend_bindings(self, numcols);
         if (NULL == self->bindings) {
             self->errornumber = STMT_NO_MEMORY_ERROR;
#ifdef _ENGLISH_
             self->errormsg = "Could not get enough free memory to store the binding information";
#else
             self->errormsg = "Kein Speicher um Bindinginformation abzulegen";
#endif
             return 0;
         }
     } else {
         self->errornumber = STMT_EXEC_ERROR;
#ifdef _ENGLISH_
         self->errormsg = "Error while executing the query";
#else
         self->errormsg = "Fehler beim Ausfhren der Query";
#endif
     }
 }
 return ok;
}

char SC_start_execution(StatementClass *self, char *statement, 
                        int statement_len)
{
 char ok = 0, was_ok, *res;
 Int2 numcols;
 struct ConnectionClass_ *hlp;
 Int2 oldstatus;
 char *statement_with_parameters;
 STMT_Status orig_stmt_status;

 // check if this or any other statement currently uses that connection
 // to transmit a transaction
 if (CONN_EXECUTING == self->conn->status) {
#ifdef _ENGLISH_
     self->errormsg = "A statement is already being executed using this handle.";
#else
     self->errormsg = "Auf dieser Verbindung wird gerade eine Query ausgefhrt.";
#endif
     self->errornumber = STMT_SEQUENCE_ERROR;
     return 0;
 }

 if (self->status != STMT_ALLOCATED &&
     self->status != STMT_READY) {
     self->errornumber = STMT_STATUS_ERROR;
#ifdef _ENGLISH_
     self->errormsg = "The handle does not point to a statement that is ready to be executed";
#else
     self->errormsg = "Dieser Handle verweist nicht auf ein auszufhrendes Statement";
#endif
 } else {
     // keep a copy of the un-parametrized statement, in case
     // they try to execute this statement again
     if(statement_len == SQL_NTS) {
         self->statement = strdup(statement);
     } else {
         self->statement = (char *)malloc(statement_len+1);
         strncpy_null(self->statement, statement, statement_len+1);
     }
     statement_with_parameters =
         copy_statement_with_parameters(self, self->statement, statement_len);
     if(!statement_with_parameters) {
         return 0;
     }
     self->statement_type = statement_type(self->statement);
     hlp = self->conn;
     oldstatus = self->conn->status;
     self->conn->status = CONN_EXECUTING;
     orig_stmt_status = self->status;
     self->status = STMT_EXECUTING;
     self->result = ConnectionC_send_query( ((ConnectionClass *)hlp)->connection, statement_with_parameters);
     free(statement_with_parameters);
     self->conn->status = oldstatus;
     self->status = STMT_FINISHED;

     ok = (NULL != self->result);
     if (ok) {

         numcols = ResultC_NumResultCols(self->result);

         res = ResultC_get_message(self->result);
         was_ok = ResultC_command_successful(self->result);

         self->errormsg = res;
         self->errornumber = was_ok ? STMT_INFO_ONLY : STMT_ERROR_TAKEN_FROM_BACKEND;
         self->resulterror_taken = 1;

         self->currTuple = -1; /* set cursor to the first tuple in the list */

         /* now move that statement to the done list */
         if(orig_stmt_status == STMT_ALLOCATED) {
             CC_remove_from_list(hlp, ALLOCATED_LIST, self->hstmt);
         } else if(orig_stmt_status == STMT_READY) {
             CC_remove_from_list(hlp, TO_DO_LIST, self->hstmt);
         }
         CC_append_to_list(hlp, DONE_LIST, self);

         /* see if the query did return any result columns */
         numcols = ResultC_NumResultCols(self->result);
         if (0 == numcols)
             return was_ok;

         /* now allocate the array to hold the binding info */
         extend_bindings(self, numcols);
         if (NULL == self->bindings) {
             self->errornumber = STMT_NO_MEMORY_ERROR;
#ifdef _ENGLISH_
             self->errormsg = "Could not get enough free memory to store the binding information";
#else
             self->errormsg = "Kein Speicher um Bindinginformation abzulegen";
#endif
             return 0;
         }
     } else {
         self->errornumber = STMT_EXEC_ERROR;
#ifdef _ENGLISH_
         self->errormsg = "Error while executing the query";
#else
         self->errormsg = "Fehler beim Ausfhren der Query";
#endif
     }
 }
 return ok;
}


Int4 SC_perform_fetch(StatementClass *self, int do_extended_fetch)
/* copies the values of the current tuple (pointed to by currTuple) to the
   respective buffers if the row has been bound -- implments SQLFetch */
{
    Int2 num_cols, rows_in_result;
    Int2 lf;
    Oid type;
    char *value;
    int result;
    Int4 tuple_num, first_tuple, last_tuple, row_offset; // CC: int ->int4 change applied

    if (STMT_EXECUTING == self->status) {
#ifdef _ENGLISH_
        self->errormsg = "Can't fetch while statement is still executing.";
#else
        self->errormsg = "SQLFetch kann nicht durchgefhrt werden whrend ein anderes Statement ausgefhrt wird.";
#endif
        self->errornumber = STMT_SEQUENCE_ERROR;
        return -1;
    }

    if (self->status != STMT_FINISHED) {
        self->errornumber = STMT_STATUS_ERROR;
#ifdef _ENGLISH_
        self->errormsg = "Fetch can only be called after the successful execution of an SQL statement";
#else
        self->errormsg = "Fetch kann nur nach erfolgter Ausfhrung eines SQL Statemens verwendet werden";
#endif
        return -1;
    }
    
    num_cols = ResultC_NumResultCols(self->result);
    rows_in_result = ResultC_get_num_tuples(self->result);
    
    if (self->bindings == NULL) {
        // just to avoid a crash if the user insists on calling this
        // function even if SQL_ExecDirect has reported an Error
#ifdef _ENGLISH_
        self->errormsg = "Bindings were not allocated properly.";
#else
        self->errormsg = "Information fr Rckgabepuffer wurde nicht korrekt angelegt.";
#endif
        self->errornumber = STMT_SEQUENCE_ERROR;
        return -1;
    }
    
    tuple_num = first_tuple = self->currTuple;
    if(do_extended_fetch) {
        last_tuple = first_tuple + self->rowset_size - 1;
    } else {
        last_tuple = first_tuple;
    }
    if(first_tuple < 0) {
        self->errormsg = "SC_perform_fetch called before beginning of result set.";
        self->errornumber = STMT_INTERNAL_ERROR;
        return -1;
    }
    if(last_tuple >= rows_in_result) {
        // stop at the last row
        last_tuple = rows_in_result - 1;
    }
    row_offset = 0;

    while(tuple_num <= last_tuple) {
        for (lf=0; lf < num_cols; lf++) {
            
            if (self->bindings[lf].buffer != NULL) {
                Int4 buflen;
                char *buffer;
                Int4 *used;

                // this column has a binding
                
                type = ResultC_get_field_type(self->result, lf);
                value = ResultC_get_value(self->result, tuple_num, lf);

                buflen = self->bindings[lf].buflen;
                buffer = self->bindings[lf].buffer;
                used = self->bindings[lf].used;

                // for extended fetch, compute some offsets here
                if(do_extended_fetch) {
                    if(self->binding_type_or_offset == SQL_BIND_BY_COLUMN) {
                        // column-wise binding.  offset the buffer by row_offset*
                        // buflen, and used by row_offset*sizeof(Int4)

                        buffer += buflen*row_offset;   /* adding a number of bytes here  */
                        used += row_offset;            /* adding a number of Int4's here */
                    } else {
                        // row-wise binding.  add the offset to both buffer
                        // and used.
                        buffer += self->binding_type_or_offset;
                        used = (Int4 *)((char *)used + self->binding_type_or_offset);
                            // we don't want the pointer magic here
                    }
                }

                result = copy_and_convert_field(type,
                                                value,
                                                self->bindings[lf].returntype,
                                                buffer,
                                                buflen,
                                                used);

                // check whether the complete result was copied
                if(result == COPY_UNSUPPORTED_TYPE) {
                    self->errormsg = "Received an unsupported type from Postgres.";
                    self->errornumber = STMT_RESTRICTED_DATA_TYPE_ERROR;
                    return -1;
                } else if(result == COPY_UNSUPPORTED_CONVERSION) {
                    self->errormsg = "Couldn't handle the necessary data type conversion.";
                    self->errornumber = STMT_RESTRICTED_DATA_TYPE_ERROR;
                    return -1;
                } else if(result == COPY_RESULT_TRUNCATED) {
                    /* The result has been truncated during the copy */
                    /* this will generate a SQL_SUCCESS_WITH_INFO result */
                    self->errornumber = STMT_TRUNCATED;
                    
#ifdef _ENGLISH_
                    self->errormsg = "A buffer was too small for the return value to fit in";
#else
                    self->errormsg = "Fetch can only be called after the successful execution on a SQL statement";
#endif
                } else if(result != COPY_OK) {
#ifdef _ENGLISH_
                    self->errormsg = "Unrecognized return value from copy_and_convert_field.";
#else
                    self->errormsg = "Interner Fehler: Unbekannter Rckgabewert von copy_and_convert_field";
#endif
                    self->errornumber = STMT_INTERNAL_ERROR;
                    return -1;
                }
            }
        }

        tuple_num++;
        row_offset++;
    }
    return row_offset;
}





char SC_bind_cols(StatementClass *self, UInt2 colnum, Int4 buflen, char *buffer, Int4 *used, Int2 returntype)
/* used for binding and unbinding columns */
{
    Int2 numcols;

    if(!self->result) {
#ifdef _ENGLISH_
        self->errormsg = "Can't bind columns with a NULL query result structure.";
#else
        self->errormsg = "Binding fr Spalten mit NULL Ergebnis unzulssig.";
#endif
        self->errornumber = STMT_SEQUENCE_ERROR;
        return 0;
    }
    if(self->status == STMT_EXECUTING) {
#ifdef _ENGLISH_
        self->errormsg = "Can't bind columns while statement is still executing.";
#else
        self->errormsg = "Binding nicht mglich whrend ein Statement ausgefhrt wird.";
#endif
        self->errornumber = STMT_SEQUENCE_ERROR;
        return 0;
    }

    numcols = ResultC_NumResultCols(self->result);

    if (colnum >= numcols) {
        self->errornumber = STMT_COLNUM_ERROR;
#ifdef _ENGLISH_
        self->errormsg = "Column number too big";
#else
        self->errormsg = "Die Spaltennummer ist zu gro";
#endif
        return 0;
    }
    if (NULL == self->bindings) {
#ifdef _ENGLISH_
        self->errormsg = "Bindings were not allocated properly.";
#else
        self->errormsg = "Bindings wurden nicht korrekt angefordert.";
#endif
        self->errornumber = STMT_SEQUENCE_ERROR;
        return 0;
    }

    if ((buflen == 0) || (buffer == NULL)) {
        /* we have to unbind the column */
        self->bindings[colnum].buflen = 0;
        self->bindings[colnum].buffer = NULL;
        self->bindings[colnum].used =   NULL;
        self->bindings[colnum].returntype = SQL_C_CHAR;
        /* CC: Just a silly default value */
    } else {
        /* ok, bind that column */
        self->bindings[colnum].buflen     = buflen;
        self->bindings[colnum].buffer     = buffer;
        self->bindings[colnum].used       = used;
        self->bindings[colnum].returntype = returntype;
    }

    return 1;
}

char SC_unbind_cols(StatementClass *self)
{
    Int2 lf;

    for(lf = 0; lf < self->bindings_allocated; lf++) {
        self->bindings[lf].buflen = 0;
        self->bindings[lf].buffer = NULL;
        self->bindings[lf].used = NULL;
        self->bindings[lf].returntype = SQL_C_CHAR;
    }

    return 1;
}

char SC_bind_parameter(StatementClass *self, UInt2 which_param,
                       Int4 param_max_length, char *buffer, Int4 *param_length,
                       Int2 param_type, Int2 param_c_type, Int2 param_sql_type,
                       UInt4 precision, Int2 scale)
{
    if(self->parameters_allocated < which_param+1) {
        ParameterInfoClass *old_parameters;
        int old_parameters_allocated;
        int i;

        old_parameters = self->parameters;
        old_parameters_allocated = self->parameters_allocated;

        self->parameters = (ParameterInfoClass *)malloc(sizeof(ParameterInfoClass)*(which_param+1));
        self->parameters_allocated = which_param+1;

        // copy the old parameters over
        for(i = 0; i < old_parameters_allocated; i++) {
            // a structure copy should work
            self->parameters[i] = old_parameters[i];
        }

        // get rid of the old parameters, if there were any
        if(old_parameters) {
            free(old_parameters);
        }

        // zero out the newly allocated parameters (in case they skipped some,
        // so we don't accidentally try to use them later)
        for(; i < self->parameters_allocated; i++) {
            self->parameters[i].buflen = 0;
            self->parameters[i].buffer = 0;
            self->parameters[i].used = 0;
            self->parameters[i].paramType = 0;
            self->parameters[i].CType = 0;
            self->parameters[i].SQLType = 0;
            self->parameters[i].precision = 0;
            self->parameters[i].scale = 0;
        }
    }

    // store the given info
    self->parameters[which_param].buflen = param_max_length;
    self->parameters[which_param].buffer = buffer;
    self->parameters[which_param].used = param_length;
    self->parameters[which_param].paramType = param_type;
    self->parameters[which_param].CType = param_c_type;
    self->parameters[which_param].SQLType = param_sql_type;
    self->parameters[which_param].precision = precision;
    self->parameters[which_param].scale = scale;

    return 1;
}



void SC_clear_error(StatementClass *self)
{
    self->errornumber = 0;
    self->errormsg = NULL;
    self->resulterror_taken = 0;
}


char SC_get_error(StatementClass *self, int *number, char **message)
{
 char rv;
 char *hlp;


 if ((self->errornumber) && (self->errormsg != NULL)) {
     *number = self->errornumber;
     *message = self->errormsg;
     self->errormsg = NULL;
 }
 rv = (self->errornumber != 0);
 self->errornumber = 0;


 if ( (!rv) && (!self->resulterror_taken)) {
     /* probably there is a message that has been returned by the QResult object ? */
     /* and that message has not been returned to the application so far */
     ResultC *res = self->result;
     *message = NULL;
     if (NULL != res) {
         hlp = NULL;
         hlp = ResultC_get_message(res);
         *message = hlp;
         self->resulterror_taken = 1;
     }
     rv = (NULL != *message);
     if (rv)
        rv = ((*message)[0] != '\0');
     return rv;
 }
 return rv;
}


void SC_resultinfo_preprocess(StatementClass *self)
{
 /* CC: All this function does is to transform a STMT_READY to STMT_PREMATURE or
        nothing if self's status != STMT_PREMATURE.
 */

 if (STMT_READY == self->status) {
   /* CC: a quick and dirty call to SQLExecute takes care of whether we have to
          start a transaction or not */
   SQLExecute(self->hstmt);
   if (STMT_FINISHED == self->status)
      self->status = STMT_PREMATURE;
 }     
}    




char SC_Destructor(StatementClass *self)
{

 if (STMT_EXECUTING == self->status)
     return 0;

 if (NULL != self->result) {
     ResultC_Destructor(self->result);
 }

 if (NULL != self->statement)
     free(self->statement);

 if (NULL != self->bindings)
     /* the memory pointed to by the bindings is not deallocated by the driver */
     /* by by the application that uses that driver, so we don't have to care  */
     /* about that here. */
     free(self->bindings);

 free(self);

 return 1;
}



/*
*
*       IMPLEMENTATION CONNECTION CLASS
*
*/

ConnectionClass *CC_Constructor()
{
    ConnectionClass *rv;
    ConnectionC *conn = ConnectionC_Constructor();

    rv = NULL;
    if (conn != NULL) {
        rv = (ConnectionClass *)malloc(sizeof(ConnectionClass));

        if (NULL != rv) {
            rv->hdbc = SQL_NULL_HDBC;
            rv->connection = conn;

            rv->allocated = NULL;
            rv->to_do = NULL;
            rv->done = NULL;

            rv->username = NULL;
            rv->errormsg = NULL;
            rv->errornumber = 0;
            rv->status = CONN_NOT_CONNECTED;
            rv->transact_status = CONN_IN_AUTOCOMMIT; // autocommit by default
        } else
            ConnectionC_Destructor(conn);
    }
    return rv;
}



void CC_get_defaults(char *DSN,
                     char *server, int server_len,
                     char *port, int port_len,
                     char *database, int database_len,
                     char *username, int username_len)
{
    // this should use the Registry in the 32-bit version.
    /* Now read in Hostname, Port and the Database name to connect to */
    /* CC: Changed GetPrivateProfileString to SQLGetPrivateProfileString
           as defined in ODBCINST. This is said to use the registry
           in WIN32 and the INI files in WIN16
       CC: We are now using the keyword strings as declared in "postodbc.h"    
    */

    SQLGetPrivateProfileString(DSN, INI_SERVER, "",
                            server, server_len, ODBC_INI);
    SQLGetPrivateProfileString(DSN, INI_PORT, "",
                            port, port_len, ODBC_INI);
    SQLGetPrivateProfileString(DSN, INI_DATABASE, "",
                            database, database_len, ODBC_INI);
    SQLGetPrivateProfileString(DSN, INI_USER, "",
                            username, username_len, ODBC_INI);
}


char CC_connect(ConnectionClass *self, HDBC hdbc, char *DSN,
                char *username, char *machine, char *port, char *dbasename)
{
    char rv;

    if (self->status != CONN_NOT_CONNECTED) {
        self->errormsg = "Already connected.";
        self->errornumber = CONN_OPENDB_ERROR;
        return 0;
    }

    self->hdbc = hdbc;
    self->username = strdup(username);

    if ( !machine   || (machine[0]   == '\0') ||
         !port      || (port[0]      == '\0') ||
         !dbasename || (dbasename[0] == '\0')) {
        self->errornumber = CONN_INIREAD_ERROR;
#ifdef _ENGLISH_
        self->errormsg = "Missing server name, port, or database name in call to CC_connect.";
#else
        self->errormsg = "Servername, Port oder Datenbankname fehlt.";
#endif
        return 0;
    }

    rv = ConnectionC_open_db(self->connection, DSN, machine, (short)atoi(port), dbasename, username);
    if (rv)
        self->status = CONN_CONNECTED;
    else {
        self->errornumber = CONN_OPENDB_ERROR;
#ifdef _ENGLISH_
        self->errormsg = "Could not establish a communications link to the server.";
#else
        self->errormsg = "Kommunikationsverbindung zum Server konnte nicht aufgebaut werden";
#endif
    }

    return rv;
}






StatementClass *CC_stmtfromhstmt(ConnectionClass *self, HSTMT hstmt)
{
  StatementClass *rv;

  rv = self->allocated;
  while ((rv != NULL) && (rv->hstmt != hstmt))
      rv = rv->next;

  if (rv != NULL)
    return rv;


  rv = self->to_do;
  while ((rv != NULL) && (rv->hstmt != hstmt))
      rv = rv->next;

  if (rv != NULL)
    return rv;

  rv = self->done;
  while ((rv != NULL) && (rv->hstmt != hstmt))
      rv = rv->next;

  return rv;
}


char CC_destroy_statement(ConnectionClass *self, HSTMT hstmt)
{

 StatementClass *hlp;
 int i;

 hlp = CC_stmtfromhstmt(self, hstmt);
 if (NULL == hlp)
     return 0;

 if (STMT_EXECUTING == hlp->status)
     return 0;

 for (i=0; i < 3; i++) { /* check the done the to_do and the allocated list */
     hlp = CC_remove_from_list(self, i, hstmt);
     if (NULL != hlp) {
         SC_Destructor(hlp);
         return 1;
     }
 }

 return 0;
}




StatementClass *CC_create_new_statement(ConnectionClass *self, HSTMT hstmt)
/* Creates a new statement and inserts it into the allocated list */
{
 StatementClass *rv = SC_Constructor(hstmt, self);

 if (NULL != rv) {
     CC_append_to_list(self, ALLOCATED_LIST, rv);
     /* set the shortcuts to this new statement */
     last_stmt = rv;
     last_hstmt= hstmt;
 } else {
     self->errornumber = CONN_STMT_ALLOC_ERROR;
#ifdef _ENGLISH_
     self->errormsg = "No more memory to allocate a further SQL-statement";
#else
     self->errormsg = "Kein freier Speicher fr ein weiteres SQL-Statement";
#endif
 }
 return rv;
}


char CC_set_commit_behaviour(ConnectionClass *self, char do_autocommit)
/* Turn on or off Autocommit behaviour. If we are
   in manual commit and a transaction is currently in progress, 0 is returned
*/
{
 if (self->transact_status & CONN_IN_TRANSACTION)
    return 0;

 self->transact_status = do_autocommit ? CONN_IN_AUTOCOMMIT : 0;
 return 1;
}



char CC_get_error(ConnectionClass *self, int *number, char **message)
{
    int rv;

    if (self->errornumber) {
        *number = self->errornumber;
        *message = self->errormsg;
    }
    rv = (self->errornumber != 0);

    self->errornumber = 0;

    // somehow an invalid assumption is being made here:
    // (conn is not really an ErrClassC * when the ErrClassC functions are
    // being called)
    // although it should be...  Connection is a descendent of PgErrorClass,
    // which is what ErrClassC is a wrapper for.

    // ok...  I made a separate Connection-specific wrapper for the functions
    // it inherits from PgErrorClass.  now it doesn't GPF anymore.

    if (!rv) {
        /* Error message from ConnectionC ? */
        ConnectionC *conn;

        conn = self->connection;
        *message = NULL;
        if (NULL != conn) {
            //     *message = ErrClassC_get_errmsg((ErrClassC *)conn);
            //     ErrClassC_clear_errmsg((ErrClassC *) conn);

            *message = ConnectionC_get_errmsg(conn);
            ConnectionC_clear_errmsg(conn);
        }
        rv = (NULL != *message) && (*message[0] != '\0');
    }
    return rv;
}



char CC_Destructor(ConnectionClass *self)
{
    StatementClass *rv;
    StatementClass *hlp;
    ResultC *res;

    if (CONN_EXECUTING == self->status)
        return 0;

    // Cancel an ongoing transaction
    if ( ! ((self->transact_status) & CONN_IN_AUTOCOMMIT)  && // we are in manual commit mode
          ((self->transact_status) & CONN_IN_TRANSACTION) // and there is a transaction going on
       )
    {
     // quit the transaction
     res = NULL;
     if (NULL != self->connection)
       // just to make sure...
       res = ConnectionC_send_query(self->connection, "abort transaction");
     if (NULL != res)
        ResultC_Destructor(res);
    }

    // now free the statement handles that might still be
    // associated with that Connection:
    rv = self->allocated;
    while (rv != NULL) {
        hlp = rv->next;
        SC_Destructor(rv);
        rv = hlp;
    }


    rv = self->to_do;
    while (rv != NULL) {
        hlp = rv->next;
        SC_Destructor(rv);
        rv = hlp;
    }

    rv = self->done;
    while (rv != NULL) {
        hlp = rv->next;
        SC_Destructor(rv);
        rv = hlp;
    }


    if (NULL != self->username)
        free(self->username);

    ConnectionC_Destructor(self->connection);
    free(self);
    return 1;
}



StatementClass *CC_remove_from_list(ConnectionClass *self, int which, HSTMT hstmt)
{
 StatementClass *hlp, *rv, *tail;
 StatementClass **anchor;

     if (which == 0)
        anchor = &(self->allocated);
     else if (which == 1)
        anchor = &(self->to_do);
     else
        anchor = &(self->done);

     if (*anchor != NULL) {
         /* The done list is NOT empty ! */

         if ((*anchor)->hstmt == hstmt) {
             /* The first item in the list ist the one we are looking for */
             tail = (*anchor)->next;
             (*anchor)->next = NULL;
             rv = *anchor;
             *anchor = tail;
             return rv;
         }
         hlp = *anchor;
         /* CC: just to make sure... */
         if (NULL == hlp)
            return NULL;
         while (hlp->next != NULL) {
             if (hlp->next->hstmt == hstmt) {
                 /* the next item is the one we are looking for */
                 tail = hlp->next->next;
                 rv = hlp->next;
                 rv->next = NULL;
                 hlp->next = tail;
                 return rv;
             }
             hlp = hlp->next;
         }
     }
 return NULL;
}


void CC_append_to_list(ConnectionClass *self, int which, StatementClass *stmt)
{
    StatementClass **anchor, *hlp;

     if (which == 0)
        anchor = &(self->allocated);
     else if (which == 1)
        anchor = &(self->to_do);
     else
        anchor = &(self->done);

     stmt->next = NULL;
     if (NULL == *anchor) {
         *anchor = stmt;
     } else {
         hlp = *anchor;
         while (hlp->next != NULL)
             hlp = hlp->next;
         hlp->next = stmt;
     }
}






/*
*
*       IMPLEMENTATION HANDLE INFO
*
*/

HandleInfo *HI_Constructor()
{
    HandleInfo *rv = (HandleInfo *)malloc(sizeof(HandleInfo));
    int lf;

    if (NULL != rv) {
        for (lf=0; lf< MAX_CONNECTIONS; lf++)
            rv->conns[lf] = NULL;
    }
    return rv;
}



char HI_add_connection(HandleInfo *self, ConnectionClass *conn)
{
    int lf;


    lf = 0;
    while ((lf < MAX_CONNECTIONS) && (self->conns[lf] != NULL))
      lf++;

    if (lf < MAX_CONNECTIONS)
      self->conns[lf] = conn;

    return lf < MAX_CONNECTIONS;
}


char HI_remove_connection(HandleInfo *self, ConnectionClass *conn)
{
    int lf;
    char removed = 0;

    lf = 0;
    while ((lf < MAX_CONNECTIONS) && (self->conns[lf] != conn))
      lf++;

    if ((lf < MAX_CONNECTIONS) && (self->conns[lf]->status != CONN_EXECUTING)) {
      self->conns[lf] = NULL;
      removed = 1;
   }

    return removed;
}


ConnectionClass *HI_connfromhdbc(HandleInfo *self, HDBC hdbc)
{
    int lf;

    lf = 0;
    while (lf < MAX_CONNECTIONS) {
       if ( (self->conns[lf] != NULL) && (self->conns[lf]->hdbc == hdbc) ) {
           /* we have found what we are looking for */
           return self->conns[lf];
       }
       lf++;
    }

    return NULL; /* we did not find what we wanted to find */
}



StatementClass *HI_stmtfromhstmt(HandleInfo *self, HSTMT hstmt)
{
    int lf;
    StatementClass *rv;

    rv = NULL;
    lf = 0;
    while (lf < MAX_CONNECTIONS) {
      if (self->conns[lf] != NULL) {
          rv = CC_stmtfromhstmt(self->conns[lf], hstmt);
          if (rv != NULL)
              /* break;  we have found the handle, so return the result */
              return rv;
      }
      lf++;
    }
    return rv;
}


char HI_Destructor(HandleInfo *self)
{
    int lf;
    char rv = 1;

    for (lf = 0; lf < MAX_CONNECTIONS; lf++)
      if (NULL != self->conns[lf])
        rv = rv && CC_Destructor(self->conns[lf]);
    if (rv)
      free(self);
    return rv;
}



/*
 *
 *
 * Implementation of various auxillary routines
 *
 */


BindInfoClass *create_empty_bindings(int num_columns)
{
    BindInfoClass *new_bindings;
    int i;

    new_bindings = (BindInfoClass *)malloc(num_columns * sizeof(BindInfoClass));
    if(!new_bindings) {
        return 0;
    }

    for(i=0; i < num_columns; i++) {
        new_bindings[i].buflen = 0;
        new_bindings[i].buffer = NULL;
        new_bindings[i].used = NULL;
    }

    return new_bindings;
}

void extend_bindings(StatementClass *stmt, int num_columns)
{
    BindInfoClass *new_bindings;
    int i;

    /* if we have too few, allocate room for more, and copy the old */
    /* entries into the new structure */
    if(stmt->bindings_allocated < num_columns) {
        new_bindings = create_empty_bindings(num_columns);
        if(stmt->bindings) {
            for(i=0; i<stmt->bindings_allocated; i++) {
                new_bindings[i] = stmt->bindings[i];
            }
            free(stmt->bindings);
        }
        stmt->bindings = new_bindings;
    } else {
        /* if we have too many, make sure the extra ones are emptied out */
        /* so we don't accidentally try to use them for anything */
        for(i = num_columns; i < stmt->bindings_allocated; i++) {
            stmt->bindings[i].buflen = 0;
            stmt->bindings[i].buffer = NULL;
            stmt->bindings[i].used = NULL;
        }
    }
}

void remove_newlines(char *string)
{
    unsigned int i;

    for(i=0; i < strlen(string); i++) {
        if((string[i] == '\n') ||
           (string[i] == '\r')) {
            string[i] = ' ';
        }
    }
}

char *copy_statement_with_parameters(StatementClass *stmt, char *old_statement,
                                     int old_statement_len)
{
    int new_statement_len;
    int old_statement_pos, new_statement_pos, param_number;
    char *new_statement;
    char *param_string;
    int param_len;
    Int2 param_ctype;
    Int2 param_sqltype;

    if(old_statement_len == SQL_NTS) {
        old_statement_len = strlen(old_statement);
    }
    new_statement_len = old_statement_len + 1;
    new_statement = (char *)malloc(new_statement_len);
    
    // copy the statement over, inserting parameters as we go
    new_statement_pos = 0;
    new_statement[0] = '0';
    param_number = 0;
    for(old_statement_pos=0;
        old_statement_pos < old_statement_len; 
        old_statement_pos++) {
        if(old_statement[old_statement_pos] == '?') {
            if(stmt->parameters_allocated > param_number &&
               stmt->parameters[param_number].buffer) {

                // the other thing is too long to type
                param_ctype = stmt->parameters[param_number].CType;
                param_sqltype = stmt->parameters[param_number].SQLType;

                // replace DEFAULT with something we can use
                if(param_ctype == SQL_C_DEFAULT) {
                    param_ctype = 
                        sqltype_to_default_ctype(stmt->parameters[param_number].SQLType);
                }

                // set 'param_string' to the value we want to insert.
                // we are responsible for freeing it.
                param_string = 
                    convert_parameter(&(stmt->parameters[param_number]));
                if(!param_string) {
                    // error.
                    free(new_statement);
                    return 0;
                }

                param_len = strlen(param_string);

                // use the SQL type instead of the C type for this,
                // because a program can, for example, pass in an integer
                // value (SQLType SQL_INTEGER) inside a string 
                // (ctype SQL_C_CHAR).
                if(param_sqltype == SQL_CHAR ||
                   param_sqltype == SQL_VARCHAR) {
                    param_len += 2; // for single quotes
                }

                // reallocate new_statement with the new length
                new_statement_len += param_len - 1; // we're losing the '?'
                new_statement = realloc(new_statement,
                                        new_statement_len);

                // putting quotes around non-string values can
                // cause problems for Postgres (it can't figure out
                // which operators to use.)  so only do it for strings.
                if(param_sqltype == SQL_CHAR ||
                   param_sqltype == SQL_VARCHAR) {
                    strcat(new_statement, "'");
                }
                strcat(new_statement, param_string);
                if(param_sqltype == SQL_CHAR ||
                   param_sqltype == SQL_VARCHAR) {
                    strcat(new_statement, "'");
                }

                new_statement_pos += param_len;

                free(param_string);
                param_number++;
            } else {
                // shouldn't happen...
                new_statement[new_statement_pos] = '?';
                new_statement_pos++;
            }
        } else if(old_statement[old_statement_pos] == '\n' ||
                  old_statement[old_statement_pos] == '\r') {
            // postgres seems to choke on these, so just convert
            // them to spaces.
            new_statement[new_statement_pos] = ' ';
            new_statement_pos++;
        } else {
            new_statement[new_statement_pos] = old_statement[old_statement_pos];
            new_statement_pos++;
        }

        // make sure new_statement is always null-terminated
        new_statement[new_statement_pos] = '\0';
    }
        
    return new_statement;
}

char *convert_parameter(ParameterInfoClass *param)
{
    Int2 param_ctype;
    char *result;

    param_ctype = param->CType;
    if(param_ctype == SQL_C_DEFAULT) {
        param_ctype =
            sqltype_to_default_ctype(param->SQLType);
    }

    if(param_ctype == SQL_C_CHAR) {
        int source_length, num_escapes;

        // figure out the length of the source parameter
        if(param->used && *(param->used) != SQL_NTS) {
            source_length = *(param->used);
        } else {
            source_length = strlen(param->buffer);
        }

        // figure out the number of escapes needed
        num_escapes = count_escapes_needed(param->buffer, source_length);

        // make space for the result
        result = (char *)malloc(source_length + num_escapes + 1);

        // and escapify it
        escape_stuff(param->buffer, source_length, result);
    } else if(param_ctype == SQL_C_DOUBLE) {
        // not sure of the real maximum length
        result = (char *)malloc(40);
        sprintf(result, "%f", 
                *((SDOUBLE *)param->buffer));
    } else if(param_ctype == SQL_C_FLOAT) {
        result = (char *)malloc(40);
        sprintf(result, "%f", 
                *((SFLOAT *)param->buffer));
    } else if(param_ctype == SQL_C_SLONG ||
              param_ctype == SQL_C_LONG) {
        result = (char *)malloc(40);
        sprintf(result, "%ld",
                *((SDWORD *)param->buffer));
    } else if(param_ctype == SQL_C_SSHORT ||
              param_ctype == SQL_C_SHORT) {
        result = (char *)malloc(40);
        sprintf(result, "%d",
                *((SWORD *)param->buffer));
    } else if(param_ctype == SQL_C_STINYINT ||
              param_ctype == SQL_C_TINYINT) {
        result = (char *)malloc(40);
        sprintf(result, "%d",
                *((SCHAR *)param->buffer));
    } else if(param_ctype == SQL_C_ULONG) {
        result = (char *)malloc(40);
        sprintf(result, "%lu",
                *((UDWORD *)param->buffer));
    } else if(param_ctype == SQL_C_USHORT) {
        result = (char *)malloc(40);
        sprintf(result, "%u",
                *((UWORD *)param->buffer));
    } else if(param_ctype == SQL_C_UTINYINT) {
        result = (char *)malloc(40);
        sprintf(result, "%u",
                *((UCHAR *)param->buffer));
    } else if(param_ctype == SQL_C_DATE) {
        DATE_STRUCT *date_struct = (DATE_STRUCT *)param->buffer;

        result = (char *)malloc(40);
        sprintf(result, "%02u-%02u-%04u",
                date_struct->month,
                date_struct->day,
                date_struct->year);
    } else if(param_ctype == SQL_C_TIME) {
        TIME_STRUCT *time_struct = (TIME_STRUCT *)param->buffer;

        result = (char *)malloc(40);
        sprintf(result, "%02u:%02u:%02u",
                time_struct->hour,
                time_struct->minute,
                time_struct->second);
    } else {
        // error--unrecognized type
        return 0;
    }
    
    return result;
}

int count_escapes_needed(char *string, int string_len)
{
    int i;
    int escapes_needed = 0;

    for(i = 0; i < string_len; i++) {
        if(needs_escape(string[i])) {
            escapes_needed++;
        }
    }

    return escapes_needed;
}

void escape_stuff(char *input_string, int input_string_len,
                   char *output_string)
{
    int input_string_pos = 0, output_string_pos = 0;

    while(input_string_pos < input_string_len) {
        if(needs_escape(input_string[input_string_pos])) {
            output_string[output_string_pos] = '\\';
            output_string[output_string_pos+1] = 
                input_string[input_string_pos];
            output_string_pos += 2;
        } else {
            output_string[output_string_pos] =
                input_string[input_string_pos];
            output_string_pos++;
        }

        input_string_pos++;
    }

    // null termination is your friend.
    output_string[output_string_pos] = '\0';
}

int needs_escape(char ch)
{
    return ch == '\'' || ch == '\n' || ch == '\r';
}

int statement_type(char *statement)
{
    if(strnicmp(statement, "SELECT", 6) == 0) {
        return STMT_TYPE_SELECT;
    } else if(strnicmp(statement, "INSERT", 6) == 0) {
        return STMT_TYPE_INSERT;
    } else if(strnicmp(statement, "UPDATE", 6) == 0) {
        return STMT_TYPE_UPDATE;
    } else if(strnicmp(statement, "DELETE", 6) == 0) {
        return STMT_TYPE_DELETE;
    } else {
        return STMT_TYPE_OTHER;
    }
}


int copy_and_convert_field_bindinfo(Int4 field_type, void *value, BindInfoClass *bic)
{
 return copy_and_convert_field(field_type, value, (Int2)bic->returntype, (PTR)bic->buffer,
                                (SDWORD)bic->buflen, (SDWORD *)bic->used);
}

int copy_and_convert_field(Int4 field_type, void *value, Int2 fCType, PTR rgbValue, SDWORD cbValueMax, SDWORD *pcbValue)
{
    Int4 len;

    if(value) {
        if(fCType == SQL_C_CHAR) {
            // character is easy, since we receive everything as character
            // in the first place (right?)
            len = strlen((char *)value);
            strncpy_null((char *)rgbValue, (char *)value, cbValueMax);
        } else if(fCType == SQL_C_DEFAULT ||

                  /* integer types can be output if it's an integer */
                  ((fCType == SQL_C_SSHORT ||
                    fCType == SQL_C_USHORT ||
                    fCType == SQL_C_SLONG ||
                    fCType == SQL_C_ULONG ||
                    fCType == SQL_C_LONG ||
                    fCType == SQL_C_SHORT) &&
                   (field_type == PG_TYPE_INT2 ||
                    field_type == PG_TYPE_INT4)) ||

                  /* float/double types can be output if it's a float */
                  ((fCType == SQL_C_FLOAT ||
                    fCType == SQL_C_DOUBLE) &&
                   (field_type == PG_TYPE_FLOAT4 ||
                    field_type == PG_TYPE_FLOAT8)) ||

                  /* date/time types OK. */
                  ((fCType == SQL_C_DATE && field_type == PG_TYPE_DATE) ||
                   (fCType == SQL_C_TIME && field_type == PG_TYPE_TIME))
                  
                  ) {
            // check the type of the field and act appropriately
            // we are still only supporting the default conversions here.
            if(field_type == PG_TYPE_CHAR ||
               field_type == PG_TYPE_VARCHAR ||
               field_type == PG_TYPE_TEXT ||
               field_type == PG_TYPE_NAME) {
                len = strlen((char *)value);
                strncpy_null((char *)rgbValue, (char *)value, cbValueMax);
            } else if(field_type == PG_TYPE_INT2) {
                len = 2;
                if(cbValueMax >= len) {
                    *((SWORD *)rgbValue) = atoi(value);
                }
            } else if(field_type == PG_TYPE_INT4 || field_type == PG_TYPE_OID) {
                len = 4;
                if(cbValueMax >= len) {
                    *((SDWORD *)rgbValue) = atoi(value);
                }
            } else if(field_type == PG_TYPE_FLOAT4) {
                len = 4;
                if(cbValueMax >= len) {
                    *((SFLOAT *)rgbValue) = (float)atof(value);
                }
            } else if(field_type == PG_TYPE_FLOAT8) {
                len = 8;
                if(cbValueMax >= len) {
                    *((SDOUBLE *)rgbValue) = atof(value);
                }
            } else if(field_type == PG_TYPE_INT28) {
                // this is an array of eight integers
                int int_array[8];
                short *short_array = (short *)rgbValue;
                int i;

                len = 16;
                // scanf is pretty flaky, but since we know
                // exactly the format we have, it'll probably be OK
                // scanf doesn't have a 'short' code, so we have
                // to read these in as ints and then convert
                sscanf(value,
                       "%d %d %d %d %d %d %d %d",
                       &int_array[0],
                       &int_array[1],
                       &int_array[2],
                       &int_array[3],
                       &int_array[4],
                       &int_array[5],
                       &int_array[6],
                       &int_array[7]);

                for(i=0; i<8; i++) {
                    short_array[i] = (short)int_array[i];
                }
            } else if(field_type == PG_TYPE_DATE) {
                DATE_STRUCT *date_struct = (DATE_STRUCT *)rgbValue;
                int year, month, day;

                /* Postgres outputs dates in the form MM-DD-YYYY */
                sscanf(value, "%u-%u-%u", &month, &day, &year);

                date_struct->year = year;
                date_struct->month = month;
                date_struct->day = day;
                len = 6;
            } else if(field_type == PG_TYPE_TIME) {
                TIME_STRUCT *time_struct = (TIME_STRUCT *)rgbValue;
                int hour, minute, second;

                /* times are in the format 'HH:MM:SS' */
                sscanf(value, "%u:%u:%u", &hour, &minute, &second);
                
                time_struct->hour = hour;
                time_struct->minute = minute;
                time_struct->second = second;
                len = 6;
            } else {
                return COPY_UNSUPPORTED_TYPE;
            }
        } else {
            return COPY_UNSUPPORTED_CONVERSION;
        }
    } else {
        /* handle a null just by returning SQL_NULL_DATA in pcbValue, */
        /* and doing nothing to the buffer.                           */
        if(pcbValue) {
            *pcbValue = SQL_NULL_DATA;
        }
    }

    // store the length of what was copied, if there's a place for it
    // unless it was a NULL (in which case it was already set above)
    if(pcbValue && value)
        *pcbValue = len;

    if(len > cbValueMax) {
        return COPY_RESULT_TRUNCATED;
    } else {
        return COPY_OK;
    }
}

// strncpy copies up to len characters, and doesn't terminate
// the destination string if src has len characters or more.
// instead, I want it to copy up to len-1 characters and always
// terminate the destination string.
/* CC: Added check for dst == NULL. In that case nothing is copied */
char *strncpy_null(char *dst, const char *src, size_t len)
{
    unsigned int i;

    if (NULL != dst) {
      for(i = 0; src[i] && i < len - 1; i++) {
          dst[i] = src[i];
      }

      if(len > 0) {
          dst[i] = '\0';
      }
    }
    return dst;
}
