/*
** log.c
**
** logging facility used within all modules
**
** (c) 1997 by Dirk Ohme - all rights reserved
*/

/*---| includes |------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <sqlcli_.h>
#include <log.h>

#ifdef USE_STDARG_H
#  include <stdarg.h>
#else
#  include <varargs.h>
#endif

/*---------------------------------------------------------------------------*/
/*      local variables                                                      */
/*---------------------------------------------------------------------------*/
static int       iIntLogLevel   = LOG_LEVEL_DEFAULT;
static char      szPrgName[256] =
#ifdef mSQL
                                        "mSQL 1.x";
#else
#  ifdef mSQL2
                                        "mSQL 2.x";
#  else
#    ifdef ORACLE
                                        "ORACLE";
#    else
#      ifdef UNIDRV
                                        "UNIDRV";
#      else
                                        "???";
#      endif
#    endif
#  endif
#endif

/*---------------------------------------------------------------------------*/
/*  LogInit                                                                  */
/*      initialize logging tool                                              */
/*    input:    char *pszPrgName                                             */
/*                name of program using this logging library                 */
/*    output:   == SQL_SUCCESS : initialization successfully completed       */
/*              == SQL_ERROR   : illegal program name                        */
/*---------------------------------------------------------------------------*/
RETCODE SQL_LOCAL_API LogInit(         char     *pszPrgName          )
{
        /*
        ** check parameter
        */
        if( NULL == pszPrgName || '\0' == *pszPrgName )
                return SQL_ERROR;

        /*
        ** set program name
        */
        strncpy( szPrgName, pszPrgName, sizeof(szPrgName) );
        return SQL_SUCCESS;
}

/*---------------------------------------------------------------------------*/
/*  LogSetLevel                                                              */
/*      set level of logging                                                 */
/*    input:    int iLogLevel                                                */
/*                level of logging, see LOG_LEVEL_...                        */
/*    output:   == SQL_SUCCESS : level has been changes successfully         */
/*              == SQL_ERROR   : illegal level, no chances                   */
/*---------------------------------------------------------------------------*/
RETCODE SQL_LOCAL_API LogSetLevel( int iLogLevel )
{
        if( LOG_LEVEL_DISABLED > iLogLevel ||
            LOG_LEVEL_EXTENDED < iLogLevel
          )
        {
                return SQL_ERROR;
        }
        iIntLogLevel = iLogLevel;
        return SQL_SUCCESS;
}

/*---------------------------------------------------------------------------*/
/*  LogEntry                                                                 */
/*      save an entry within the logging file/list/...                       */
/*    input:    char cLogType                                                */
/*                type of information to be logged, see LOG_...              */
/*              char *pszLogEntry, ...                                       */
/*                string and parameters to be logged                         */
/*    output:   == SQL_SUCCESS        : logging completed successfully       */
/*              == SQL_NO_DATA        : no string found (second parameter)   */
/*              == SQL_INVALID_HANDLE : invalid log type (first parameter)   */
/*              == SQL_ERROR          : error during call                    */
/*---------------------------------------------------------------------------*/
#ifdef USE_STDARG_H
  RETCODE SQL_LOCAL_API LogEntry( char cLogType, ... )
#else
  RETCODE SQL_LOCAL_API LogEntry( va_alist )
    va_dcl
#endif
{
        va_list         args;
        char           *pszLogEntry;
        char            szBuffer[2048];
#ifdef USE_STDARG_H
        if( LOG_LEVEL_DISABLED == iIntLogLevel )
        {
                return SQL_SUCCESS;
        }
        va_start( args, cLogType );
#else
        char            cLogType;

        if( LOG_LEVEL_DISABLED == iIntLogLevel )
        {
                return SQL_SUCCESS;
        }
        va_start( args );
        cLogType = (char) ((int) va_arg( args, char * ) & 0xFF);
#endif
        /*
        ** check type of logging
        */
        if( 0 == cLogType )
        {
                va_end( args );
                return SQL_INVALID_HANDLE;
        }

        /*
        ** get logging entry
        */
        pszLogEntry = (char *) va_arg( args, char * );
        if( NULL == pszLogEntry )
        {
                va_end( args );
                return SQL_NO_DATA;
        }

        /*
        ** get logging entry
        */
        (void)vsprintf( szBuffer, pszLogEntry, args );
        va_end( args );

        /*
        ** check level and save accordingly
        */
        switch( iIntLogLevel )
        {
                case LOG_LEVEL_EXTENDED:
                        break;
                case LOG_LEVEL_STANDARD:
                        if( LOG_ERROR   == cLogType ||
                            LOG_WARNING == cLogType
                          )
                        {
                                break;
                        }
                case LOG_LEVEL_DISABLED:
                        return SQL_SUCCESS;
                default:
                        return SQL_ERROR;

        } /* switch */
        fprintf( stderr, "%c [%s]%s\n", cLogType, szPrgName, szBuffer );
        fflush( stderr );

        return SQL_SUCCESS;
}

/*===| end of file |=========================================================*/
