/*
** RESULTS.C - This is the module containing the code for ODBC for
** returning results and information about results.
**
** (c) 1997 by Dirk Ohme - all rights reserved
*/

#include "cli.h"

/*---------------------------------------------------------------------------*/
/*      This returns the number of columns associated with the database      */
/*      attached to "lpstmt".                                                */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLNumResultCols(
        LPSTMT  lpstmt,
        SWORD FAR *pccol)
{                                                /*--------------------------*/
#ifdef ORACLE
        sb4      cbSize;                         /* database entry size      */
#endif
                                                 /*--------------------------*/
        /*
        ** check parameter
        */
        LogEntry( LOG_STATUS, "[SQLNumResultCols]  lpstmt = $%08lX", lpstmt );
        LogEntry( LOG_STATUS, "                   *pccol  = $%08lX", pccol  );
        if( NULL == lpstmt )
        {
                LogEntry( LOG_ERROR,      "[SQLNumResultCols] NULL == lpstmt" );
                LogEntry( LOG_RETURNCODE, "[SQLNumResultCols] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }
        if( NULL == pccol )
        {
                strcpy( lpstmt->szSqlState, "S1009" );
                lpstmt->pszSqlMsg = "[SQLNumResultCols] pccol must not be NULL";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLNumResultCols] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }
        *pccol = 0;

        /*
        ** get number of result columns
        */
        if( NULL == lpstmt->pszQuery )
        {
                strcpy( lpstmt->szSqlState, "S1010" );
                lpstmt->pszSqlMsg = "[SQLNumResultCols] no prepared statement";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLNumResultCols] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }
        *pccol         = lpstmt->ciCol;
        LogEntry( LOG_OUTPUT, "[SQLNumResultCols] columns = %d", *pccol );

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLNumResultCols] SQL_SUCCESS" );
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Return information about the database column the user wants          */
/*      information about.                                                   */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLDescribeCol(
        LPSTMT  lpstmt,
        UWORD   icol,
        UCHAR FAR *szColName,
        SWORD   cbColNameMax,
        SWORD FAR *pcbColName,
        SWORD FAR *pfSqlType,
        UDWORD FAR *pcbColDef,
        SWORD FAR *pibScale,
        SWORD FAR *pfNullable)
{                                                /*--------------------------*/
        sb4              o_cbColSize;            /* column size              */
        sb2              o_fColType;             /* column type              */
        sb1              o_szColName[40];        /* column name              */
        sb4              o_cbColName;            /* column name len          */
        sb4              o_cbColDisp;            /* column display len       */
        sb2              o_cbColPrec;            /* column precision         */
        sb2              o_cbColScale;           /* column scale             */
        sb2              o_fColNullOk;           /* column nullable          */
                                                 /*--------------------------*/
        /*
        ** check parameter
        */
        LogEntry( LOG_STATUS, "[SQLDescribeCol]  lpstmt       = $%08lX", lpstmt       );
        LogEntry( LOG_STATUS, "                  icol         =  %3d",   icol         );
        LogEntry( LOG_STATUS, "                  *szColName   = $%08lX", szColName    );
        LogEntry( LOG_STATUS, "                  cbColNameMax = $%08lX", cbColNameMax );
        LogEntry( LOG_STATUS, "                  *pcbColName  = $%08X",  pcbColName   );
        LogEntry( LOG_STATUS, "                  *pfSqlType   = $%08X",  pfSqlType    );
        LogEntry( LOG_STATUS, "                  *pcbColDef   = $%08X",  pcbColDef    );
        LogEntry( LOG_STATUS, "                  *pibScale    = $%08X",  pibScale     );
        LogEntry( LOG_STATUS, "                  *pfNullable  = $%08X",  pfNullable   );
        if( NULL == lpstmt )
        {
                LogEntry( LOG_ERROR,      "[SQLDescribeCol] NULL == lpstmt" );
                LogEntry( LOG_RETURNCODE, "[SQLDescribeCol] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }

        /*
        ** set defaults
        */
        if( NULL != szColName )
                *szColName  = '\0';
        if( NULL != pcbColName )
                *pcbColName = 0;
        if( NULL != pfSqlType )
                *pfSqlType  = 0;
        if( NULL != pcbColDef )
                *pcbColDef  = 0;
        if( NULL != pibScale )
                *pibScale   = 0;
        if( NULL != pfNullable )
                *pfNullable = SQL_FALSE;

        /*
        ** check query status
        */
        if( NULL == lpstmt->pszQuery )
        {
                strcpy( lpstmt->szSqlState, "S1010" );
                lpstmt->pszSqlMsg = "[SQLDescribeCol] no prepared statement";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLDescribeCol] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** get infos about column (Oracle)
        */
        if( NULL != pcbColDef )
                *pcbColDef = 0;
        o_cbColName = sizeof( o_szColName );
        if( odescr( &lpstmt->cda, icol, &o_cbColSize, &o_fColType,
                    &o_szColName[0], &o_cbColName, &o_cbColDisp,
                    &o_cbColPrec, &o_cbColScale, &o_fColNullOk ) )
        {
                strcpy( lpstmt->szSqlState, "07005" );
                lpstmt->pszSqlMsg = "[SQLDescribeCol] column not available";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLDescribeCol] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** set column name (Oracle)
        */
        o_szColName[o_cbColName] = '\0';
        if( NULL != szColName )
                strncpy( szColName, o_szColName, cbColNameMax );
        if( NULL != pcbColName )
                *pcbColName = strlen( szColName );

        /*
        ** set data type (Oracle)
        */
        if( NULL != pfSqlType )
        {
                switch( o_fColType )
                {
                        case VARCHAR2_TYPE:     *pfSqlType = SQL_CHAR;
                                                break;
                        case INT_TYPE:          *pfSqlType = SQL_INTEGER;
                                                break;
                        case NUMBER_TYPE:       *pfSqlType = SQL_REAL;
                                                break;
                        default:
                                strcpy( lpstmt->szSqlState, "S1097" );
                                lpstmt->pszSqlMsg = "[SQLDescribeCol] invalid column type detected";
                                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                                LogEntry( LOG_RETURNCODE,
                                          "[SQLDescribeCol] SQL_ERROR (%s)",
                                          lpstmt->szSqlState );
                                return SQL_ERROR;
                }
        }

        /*
        ** set precision (Oracle)
        */
        if( NULL != pcbColDef )
        {
                if( VARCHAR2_TYPE == o_fColType )
                        *pcbColDef = o_cbColSize;
		else
			*pcbColDef = (0 != o_cbColPrec) ? o_cbColPrec
							: o_cbColDisp + 1;
        }

        /*
        ** set scale (Oracle)
        */
        if( NULL != pibScale )
        {
                *pibScale = o_cbColScale;
        }

        /*
        ** set nullable (Oracle)
        */
        if( NULL != pfNullable )
        {
                *pfNullable = (0 != o_fColNullOk) ? SQL_NULLABLE
                                                  : SQL_NO_NULLS;
        }

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLDescribeCol] SQL_SUCCESS" );
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Returns result column descriptor information for a result set.       */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLColAttributes(
        LPSTMT  lpstmt,
        UWORD   icol,
        UWORD   fDescType,
        UCHAR FAR *rgbDesc,
        SWORD   cbDescMax,
        SWORD FAR *pcbDesc,
        SDWORD FAR *pfDesc)
{                                                /*--------------------------*/
        sb4              cbLen;                  /* length of a column       */
        sb4              o_cbColSize;            /* column size              */
        sb2              o_fColType;             /* column type              */
        sb1              o_szColName[40];        /* column name              */
        sb4              o_cbColName;            /* column name len          */
        sb4              o_cbColDisp;            /* column display len       */
        sb2              o_cbColPrec;            /* column precision         */
        sb2              o_cbColScale;           /* column scale             */
        sb2              o_fColNullOk;           /* column nullable          */
                                                 /*--------------------------*/
        /*
        ** check parameters
        */
        LogEntry( LOG_STATUS, "[SQLColAttributes]  lpstmt     = $%08lX", lpstmt    );
        LogEntry( LOG_STATUS, "                    icol       =  %5d",   icol      );
        LogEntry( LOG_STATUS, "                    fDescType  = $%08lX", fDescType );
        LogEntry( LOG_STATUS, "                    rgbDesc    = $%08lX", rgbDesc   );
        LogEntry( LOG_STATUS, "                    cbDescMax  =  %5d",   cbDescMax );
        LogEntry( LOG_STATUS, "                    pcbDesc    = $%08lX", pcbDesc   );
        LogEntry( LOG_STATUS, "                    pfDesc     = $%08lX", pfDesc    );
        if( NULL == lpstmt )
        {
                LogEntry( LOG_ERROR,      "[SQLColAttributes] NULL == lpstmt" );
                LogEntry( LOG_RETURNCODE, "[SQLColAttributes] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }

        /*
        ** check query status
        */
        if( NULL == lpstmt->pszQuery )
        {
                strcpy( lpstmt->szSqlState, "S1010" );
                lpstmt->pszSqlMsg = "[SQLColAttributes] no prepared statement";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLColAttributes] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** get infos about the desired column
        */
        o_cbColName = sizeof( o_szColName );
        if( odescr( &lpstmt->cda, (sword) icol, (sb4 *) &o_cbColSize,
	            (sb2 *) &o_fColType,  (sb1 *) &o_szColName[0],
		    (sb4 *) &o_cbColName, (sb4 *) &o_cbColDisp,
                    (sb2 *) &o_cbColPrec, (sb2 *) &o_cbColScale,
		    (sb2 *) &o_fColNullOk ) )
        {
                strcpy( lpstmt->szSqlState, "07005" );
                lpstmt->pszSqlMsg = "[SQLColAttributes] column not available";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLColAttributes] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }
        o_szColName[o_cbColName] = '\0';

        /*
        ** check pointers
        */
        switch( fDescType )
        {
                case SQL_COLUMN_AUTO_INCREMENT:
                case SQL_COLUMN_CASE_SENSITIVE:
                case SQL_COLUMN_COUNT:
                case SQL_COLUMN_DISPLAY_SIZE:
                case SQL_COLUMN_LENGTH:
                case SQL_COLUMN_MONEY:
                case SQL_COLUMN_NULLABLE:
                case SQL_COLUMN_PRECISION:
                case SQL_COLUMN_SCALE:
                case SQL_COLUMN_SEARCHABLE:
                case SQL_COLUMN_TYPE:
                case SQL_COLUMN_UNSIGNED:
                case SQL_COLUMN_UPDATABLE:
                        if( NULL == pfDesc )
                        {
                                strcpy( lpstmt->szSqlState, "S1090" );
                                lpstmt->pszSqlMsg =
                                    "[SQLColAttributes] NULL == pfDesc";
                                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                                LogEntry( LOG_RETURNCODE,
                                          "[SQLColAttributes] SQL_ERROR (%s)",
                                          lpstmt->szSqlState );
                                return SQL_ERROR;
                        }
                        break;
                case SQL_COLUMN_CATALOG_NAME:
                case SQL_COLUMN_QUALIFIER_NAME:
                case SQL_COLUMN_DISTINCT_TYPE:
                case SQL_COLUMN_LABEL:
                case SQL_COLUMN_NAME:
                case SQL_COLUMN_SCHEMA_NAME:
                case SQL_COLUMN_OWNER_NAME:
                case SQL_COLUMN_TABLE_NAME:
                case SQL_COLUMN_TYPE_NAME:
                        if( NULL == rgbDesc || 0 == cbDescMax )
                        {
                                strcpy( lpstmt->szSqlState, "S1090" );
                                lpstmt->pszSqlMsg =
                                    "[SQLColAttributes] NULL == rgbDesc";
                                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                                LogEntry( LOG_RETURNCODE,
                                          "[SQLColAttributes] SQL_ERROR (%s)",
                                          lpstmt->szSqlState );
                                return SQL_ERROR;
                        }
                        break;
                default:
                        strcpy( lpstmt->szSqlState, "S1091" );
                        lpstmt->pszSqlMsg =
                            "[SQLColAttributes] invalid fDescType value";
                        LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                        LogEntry( LOG_RETURNCODE,
                                  "[SQLColAttributes] SQL_ERROR (%s)",
                                  lpstmt->szSqlState );
                        return SQL_ERROR;
        } /* switch */

        /*
        ** check mode
        */
        switch( fDescType )
        {
                case SQL_COLUMN_AUTO_INCREMENT:
                case SQL_COLUMN_MONEY:
                        *pfDesc = FALSE;
                        break;
                case SQL_COLUMN_CASE_SENSITIVE:
                        switch( o_fColType )
                        {
                                case VARCHAR2_TYPE:
                                        *pfDesc = TRUE;
                                        break;
                                default:
                                        *pfDesc = FALSE;
                        }
                        break;
                case SQL_COLUMN_CATALOG_NAME:
                case SQL_COLUMN_DISTINCT_TYPE:
                case SQL_COLUMN_OWNER_NAME:
                case SQL_COLUMN_QUALIFIER_NAME:
                case SQL_COLUMN_SCHEMA_NAME:
                case SQL_COLUMN_TABLE_NAME:
                        *rgbDesc = '\0';
                        break;
                case SQL_COLUMN_COUNT:
                        *pfDesc = (SDWORD) lpstmt->ciCol;
                        break;
                case SQL_COLUMN_DISPLAY_SIZE:
                case SQL_COLUMN_PRECISION:
                        switch( o_fColType )
                        {
                                case VARCHAR2_TYPE:
                                case INT_TYPE:
                                case NUMBER_TYPE:
                                        *pfDesc = (SDWORD) o_cbColSize;
                                        break;
                                default:
                                        *pfDesc = (SDWORD) 0;
                        }
                        break;
                case SQL_COLUMN_LABEL:
                case SQL_COLUMN_NAME:
                        strncpy( rgbDesc, o_szColName, cbDescMax - 1 );
                        if( NULL != pcbDesc )
                        {
                                *pcbDesc = (SWORD) strlen( rgbDesc );
                        }
                        break;
                case SQL_COLUMN_LENGTH:
                        switch( o_fColType )
                        {
                                case VARCHAR2_TYPE:
                                case INT_TYPE:
                                case NUMBER_TYPE:
                                        *pfDesc = (SDWORD) o_cbColSize;
                                        break;
                                default:
                                        *pfDesc = (SDWORD) 0;
                        }
                        break;
                case SQL_COLUMN_NULLABLE:
                        *pfDesc = (0 != o_fColNullOk) ? SQL_NULLABLE
                                                      : SQL_NO_NULLS;
                        break;
                case SQL_COLUMN_SCALE:
                        *pfDesc = (SDWORD) o_cbColScale;
                        break;
                case SQL_COLUMN_SEARCHABLE:
                        switch( o_fColType )
                        {
                                case VARCHAR2_TYPE:
                                        *pfDesc = SQL_SEARCHABLE;
                                        break;
                                case INT_TYPE:
                                case NUMBER_TYPE:
                                        *pfDesc = SQL_ALL_EXCEPT_LIKE;
                                        break;
                                default:
                                        *pfDesc = SQL_UNSEARCHABLE;
                        }
                        break;
                case SQL_COLUMN_TYPE:
                        switch( o_fColType )
                        {
                                case VARCHAR2_TYPE:
                                        *pfDesc = SQL_CHAR;
                                        break;
                                case INT_TYPE:
                                        *pfDesc = SQL_INTEGER;
                                        break;
                                case NUMBER_TYPE:
                                        *pfDesc = SQL_REAL;
                                        break;
                                default:
                                        *pfDesc = 0;
                        }
                        break;
                case SQL_COLUMN_TYPE_NAME:
                        switch( o_fColType )
                        {
                                case VARCHAR2_TYPE:
                                        strncpy( rgbDesc, "CHARacter",
                                                 cbDescMax - 1 );
                                        break;
                                case INT_TYPE:
                                        strncpy( rgbDesc, "INTeger",
                                                 cbDescMax - 1 );
                                        break;
                                case NUMBER_TYPE:
                                        strncpy( rgbDesc, "REAL",
                                                 cbDescMax - 1 );
                                        break;
                                default:
                                        *rgbDesc = '\0';
                        }
                        if( NULL != pcbDesc )
                        {
                                *pcbDesc = (SWORD) strlen( rgbDesc );
                        }
                        break;
                case SQL_COLUMN_UNSIGNED:
                        switch( o_fColType )
                        {
                                case VARCHAR2_TYPE:
                                        *pfDesc = TRUE;
                                        break;
                                default:
                                        *pfDesc = FALSE;
                        }
                        break;
                case SQL_COLUMN_UPDATABLE:
                        *pfDesc = SQL_ATTR_READWRITE_UNKNOWN;
                        break;
                default:
                        break;
        } /* switch */

        /*
        ** display results
        */
        LogEntry( LOG_OUTPUT, "[SQLColAttributes] rgbDesc = '%s'\n",
                  (NULL == rgbDesc) ? (char *) "<NULL>" : (char *) rgbDesc );
        LogEntry( LOG_OUTPUT, "[SQLColAttributes] pcbDesc = %5d\n",
                  (NULL == pcbDesc) ? 0 : pcbDesc );
        LogEntry( LOG_OUTPUT, "[SQLColAttributes] pfDesc  = %5d\n",
                  (NULL == pfDesc) ? 0 : pfDesc );

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLColAttributes] SQL_SUCCESS" );
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Associate a user-supplied buffer with a database column.             */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLBindCol(
        LPSTMT  lpstmt,
        UWORD   icol,
        SWORD   fCType,
        PTR     rgbValue,
        SDWORD  cbValueMax,
        SDWORD FAR *pcbValue)
{                                                /*--------------------------*/
                                                 /*--------------------------*/
        /*
        ** check parameters
        */
        LogEntry( LOG_STATUS, "[SQLBindCol]  lpstmt     = $%08lX", lpstmt     );
        LogEntry( LOG_STATUS, "              icol       =  %5d",   icol       );
        LogEntry( LOG_STATUS, "              fCType     = $%08lX", fCType     );
        LogEntry( LOG_STATUS, "              rgbValue   = $%08lX", rgbValue   );
        LogEntry( LOG_STATUS, "              cbValueMax =  %5d",   cbValueMax );
        LogEntry( LOG_STATUS, "              pcbValue   = $%08lX", pcbValue   );
        if( NULL == lpstmt )
        {
                LogEntry( LOG_ERROR,      "[SQLBindCol] NULL == lpstmt" );
                LogEntry( LOG_RETURNCODE, "[SQLBindCol] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }

        /*
        ** set defaults
        */
        if( NULL != pcbValue )
                *pcbValue = 0;

        /*
        ** check query status
        */
        if( NULL == lpstmt->pszQuery )
        {
                strcpy( lpstmt->szSqlState, "S1010" );
                lpstmt->pszSqlMsg = "[SQLBindCol] no prepared statement";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLBindCol] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** bind column to variable (Oracle)
        */

        /*---| check C data type |---*/
        switch( fCType )
        {
                case SQL_C_DEFAULT:
                        fCType = VARCHAR2_TYPE;
                        break;
                case SQL_C_INTEGER:
                        fCType = INT_TYPE;
                        /* cbValueMax = sizeof( int ); */
                        break;
                case SQL_C_REAL:
                        fCType = FLOAT_TYPE;
                        /* cbValueMax = sizeof( float ); */
                        break;
                case SQL_C_CHAR:
                case SQL_C_DBCHAR:
                        fCType = STRING_TYPE;
                        break;
                default:
                        strcpy( lpstmt->szSqlState, "S1003" );
                        lpstmt->pszSqlMsg = "[SQLBindCol] unsupported C data type";
                        LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                        LogEntry( LOG_RETURNCODE,
                                  "[SQLBindCol] SQL_ERROR (%s)",
                                  lpstmt->szSqlState );
                        return SQL_ERROR;
        }

        /*---| initialize result variable |---*/
        if( NULL != pcbValue )
                *pcbValue = 0;

        /*---| bind column to variable |---*/
        if( odefin( &lpstmt->cda, icol, rgbValue, (sword) cbValueMax,
                    fCType, -1, 0, 0, -1, -1, (sb2 *) pcbValue, 0 ) )
        {
                strcpy( lpstmt->szSqlState, "07005" );
                lpstmt->pszSqlMsg = "[SQLBindCol] column not available";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLBindCol] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLBindCol] SQL_SUCCESS" );
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Returns data for bound columns in the current row                    */
/*      ("lpstmt->iCursor"), advances the cursor.                            */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLFetch(
        LPSTMT  lpstmt)
{                                                /*--------------------------*/
        RETCODE          iRC;                    /* return code              */
                                                 /*--------------------------*/
        /*
        ** check parameters
        */
        LogEntry( LOG_STATUS, "[SQLFetch]  lpstmt = $%08lX", lpstmt );
        if( NULL == lpstmt )
        {
                LogEntry( LOG_ERROR,      "[SQLFetch] NULL == lpstmt" );
                LogEntry( LOG_RETURNCODE, "[SQLFetch] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }

        /*
        ** check query status
        */
        if( NULL == lpstmt->pszQuery )
        {
                strcpy( lpstmt->szSqlState, "S1010" );
                lpstmt->pszSqlMsg = "[SQLFetch] no prepared statement";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLFetch] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** get next row
        */
        lpstmt->cda.rc = 0;
        if( ofetch(&lpstmt->cda) && 1403 != lpstmt->cda.rc
                                 && 1405 != lpstmt->cda.rc
                                 && 1406 != lpstmt->cda.rc )
        {
                strcpy( lpstmt->szSqlState, "58004" );
                lpstmt->pszSqlMsg = "[SQLFetch] can't get data";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLFetch] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }
        if( 1403 == lpstmt->cda.rc )
        {
                LogEntry( LOG_RETURNCODE, "[SQLFetch] SQL_NO_DATA_FOUND" );
                return SQL_NO_DATA_FOUND;
        }

        /*
        ** pre-set return value
        */
        iRC = SQL_SUCCESS;

        /*
        ** return result
        */
        if( SQL_SUCCESS != iRC )
                LogEntry( LOG_RETURNCODE, "[SQLFetch] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
        else
                LogEntry( LOG_RETURNCODE, "[SQLFetch] SQL_SUCCESS" );
        return iRC;
}


/*---------------------------------------------------------------------------*/
/*      Returns result data for a single column in the current row.          */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLGetData(
        LPSTMT  lpstmt,
        UWORD   icol,
        SWORD   fCType,
        PTR     rgbValue,
        SDWORD  cbValueMax,
        SDWORD FAR *pcbValue)
{                                                /*--------------------------*/
                                                 /*--------------------------*/
        /*
        ** check parameters
        */
        LogEntry( LOG_STATUS, "[SQLGetData]  lpstmt     = $%08lX", lpstmt );
        LogEntry( LOG_STATUS, "              icol       =  %5d",   icol       );
        LogEntry( LOG_STATUS, "              fCType     = $%08lX", fCType     );
        LogEntry( LOG_STATUS, "              rgbValue   = $%08lX", rgbValue   );
        LogEntry( LOG_STATUS, "              cbValueMax =  %5d",   cbValueMax );
        LogEntry( LOG_STATUS, "              pcbValue   = $%08lX", pcbValue   );
        if( NULL == lpstmt )
        {
                LogEntry( LOG_ERROR,      "[SQLGetData] NULL == lpstmt" );
                LogEntry( LOG_RETURNCODE, "[SQLGetData] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }

        /*
        ** check query status
        */
        if( NULL == lpstmt->pszQuery )
        {
                strcpy( lpstmt->szSqlState, "S1010" );
                lpstmt->pszSqlMsg = "[SQLGetData] no prepared statement";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLGetData] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** check column index
        */
        if( 0 >= icol || SQL_MAX_COLUMN <= icol )
        {
                strcpy( lpstmt->szSqlState, "S1002" );
                lpstmt->pszSqlMsg = "[SQLGetData] invalid column number";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLGetData] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** check availability of data in column
        */
#if 0
        /*
        ** !!! not implemented yet !!!
        */
        icol--;
        if( NULL == lpstmt->sRow[icol] )
        {
                LogEntry( LOG_SUMMARY, "[SQLGetData] NULL statement in column %d",
                          icol+1 );
                if( NULL != pcbValue )
                {
                        *pcbValue = SQL_NULL_DATA;
                }
                switch( fCType )
                {
                        case SQL_C_INTEGER:
                                LogEntry( LOG_SUMMARY,
                                          "           (SQL_C_INTEGER)" );
                                memset( rgbValue, 0, sizeof(int) );
                                break;
                        case SQL_C_REAL:
                                LogEntry( LOG_SUMMARY,
                                          "           (SQL_C_REAL)" );
                                memset( rgbValue, 0, sizeof(float) );
                                break;
                        case SQL_C_CHAR:
                        case SQL_C_DBCHAR:
                                LogEntry( LOG_SUMMARY,
                                          "           (SQL_C_(DB)CHAR)" );
                        default:
                                *((char *)rgbValue) = '\0';
                                /*------------------------------------
                                 * enable this in order to fill the
                                 * complete destination buffer with 0s
                                 *------------------------------------
                                 * memset( rgbValue, 0, cbValueMax );
                                 *------------------------------------
                                 */
                }
        }

        /*---| copy data |---*/
        else
        {
                /*
                 * all data can be treated equally since
                 * the mSQL server sends all data as ascii
                 * strings :-)
                 */
                LogEntry( LOG_OUTPUT, "[SQLGetData] col %d -> %s",
                          icol+1, lpstmt->sRow[icol] );
                switch( fCType )
                {
                        case SQL_C_INTEGER:
                                *((int *)rgbValue) =
                                        atoi(lpstmt->sRow[icol]);
                                if( NULL != pcbValue )
                                {
                                        *pcbValue = sizeof( int );
                                }
                                break;
                        case SQL_C_REAL:
                                sscanf( lpstmt->sRow[ci], "%g", rgbValue );
                                if( NULL != pcbValue )
                                {
                                        *pcbValue = sizeof( float );
                                }
                                break;
                        case SQL_C_CHAR:
                        case SQL_C_DBCHAR:
                                strncpy( rgbValue, lpstmt->sRow[icol],
                                         cbValueMax );
                                if( NULL != pcbValue )
                                {
                                         *pcbValue = strlen(rgbValue);
                                }
                                break;
                        default:
                                if( NULL != pcbValue )
                                {
                                         *pcbValue = SQL_NULL_DATA;
                                }
                }
        }
#endif

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLGetData] SQL_SUCCESS" );
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      This determines whether there are more results sets available for    */
/*      the "lpstmt".                                                        */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLMoreResults(
        LPSTMT  lpstmt)
{
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      This returns the number of rows associated with the database         */
/*      attached to "lpstmt".                                                */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLRowCount(
        LPSTMT  lpstmt,
        SDWORD FAR *pcrow)
{
        /*
        ** check parameter
        */
        LogEntry( LOG_STATUS, "[SQLRowCount]  lpstmt = $%08lX", lpstmt );
        LogEntry( LOG_STATUS, "              *pcrow  = $%08lX", pcrow  );
        if( NULL == lpstmt )
        {
                LogEntry( LOG_ERROR,      "[SQLRowCount] NULL == lpstmt" );
                LogEntry( LOG_RETURNCODE, "[SQLRowCount] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }
        if( NULL == pcrow )
        {
                strcpy( lpstmt->szSqlState, "S1009" );
                lpstmt->pszSqlMsg = "[SQLRowCount] pcrow must not be NULL";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLRowCount] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }
        *pcrow = 0;

        /*
        ** get number of result columns
        */
        *pcrow = lpstmt->cda.rpc;

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLRowCount] SQL_SUCCESS" );
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      This positions the cursor within a block of data.                    */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLSetPos(
        LPSTMT  lpstmt,
        UWORD   irow,
        BOOL    fRefresh,
        BOOL    fLock)
{
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      This fetchs a block of data (rowset).                                */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLExtendedFetch(
        LPSTMT  lpstmt,
        UWORD   fFetchType,
        SDWORD  irow,
        UDWORD FAR *pcrow,
        UWORD FAR *rgfRowStatus)
{
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Returns the next SQL error information.                              */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLError(
        LPENV   lpenv,
        LPDBC   lpdbc,
        LPSTMT  lpstmt,
        UCHAR FAR *szSqlState,
        SDWORD FAR *pfNativeError,
        UCHAR FAR *szErrorMsg,
        SWORD   cbErrorMsgMax,
        SWORD FAR *pcbErrorMsg)
{                                                /*--------------------------*/
        char    *pszState       = NULL;          /* pointer to status code   */
        char    *pszErrMsg      = NULL;          /* pointer to error message */
        char     szOraErr[512];                  /* Oracle error message     */
                                                 /*--------------------------*/
        /*
        ** check parameter
        */
        LogEntry( LOG_STATUS, "[SQLError] lpenv         = $%08lX", lpenv  );
        LogEntry( LOG_STATUS, "           lpdbc         = $%08lX", lpdbc  );
        LogEntry( LOG_STATUS, "           lpstmt        = $%08lX", lpstmt );
        LogEntry( LOG_STATUS, "           szSqlState    = $%08lX", szSqlState );
        LogEntry( LOG_STATUS, "           pfNativeError = $%08lX",
                  pfNativeError
                );
        LogEntry( LOG_STATUS, "           szErrorMsg    = $%08lX", szErrorMsg );
        LogEntry( LOG_STATUS, "           cbErrorMsgMax =  %d", cbErrorMsgMax );
        LogEntry( LOG_STATUS, "           pcbErrorMsg   = $%08lX", pcbErrorMsg);

        pszState        =
        pszErrMsg       = NULL;
        szOraErr[0]     = '\0';

        /*---| environment |---*/
        if( NULL != lpenv )
        {
                pszState          = &lpenv->szSqlState[0];
                pszErrMsg         =  lpenv->pszSqlMsg;
                lpenv->pszSqlMsg  = NULL;
        }

        /*---| database context |---*/
        if( NULL != lpdbc )
        {
                pszState          = &lpdbc->szSqlState[0];
                pszErrMsg         =  lpdbc->pszSqlMsg;
                lpdbc->pszSqlMsg  = NULL;
        }

        /*---| statement |---*/
        if( NULL != lpstmt )
        {
                pszState          = &lpstmt->szSqlState[0];
                pszErrMsg         =  lpstmt->pszSqlMsg;
                lpstmt->pszSqlMsg = NULL;
                if( NULL != lpstmt->pSqlDbc )
                {
                        oerhms( &lpstmt->pSqlDbc->lda, lpstmt->cda.rc,
                                &szOraErr[0], sizeof(szOraErr) );
                }
        }

        /*---| check pointers |---*/
        if( NULL == pszState )
        {
                LogEntry( LOG_ERROR, "[SQLError] no valid handle(s)" );
                LogEntry( LOG_RETURNCODE, "[SQLError] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }

        /*
        ** check status
        */
        if( NULL == pszErrMsg )
        {
                /*---| reset status code |---*/
                strcpy( pszState, "00000" );

                /*---| return status code |---*/
                if( NULL != szSqlState )
                {
                        strcpy( szSqlState, pszState );
                }

                /*---| return internal code |---*/
                if( NULL != pfNativeError )
                {
                        *pfNativeError = 0;
                }

                /*---| return message |---*/
                if( NULL != szErrorMsg && 0 < cbErrorMsgMax )
                {
                        *szErrorMsg = '\0';
                        if( NULL != pcbErrorMsg )
                        {
                                *pcbErrorMsg = 0;
                        }
                }

                /*---| return data |---*/
                LogEntry( LOG_RETURNCODE, "[SQLError] SQL_NO_DATA_FOUND" );
                return SQL_NO_DATA_FOUND;
        }

        /*
        ** return data on errors
        */
        /*---| return status code |---*/
        if( NULL != szSqlState )
        {
                strcpy( szSqlState, pszState );
        }

        /*---| return internal code |---*/
        if( NULL != pfNativeError )
        {
                *pfNativeError = atoi( pszState );
        }

        /*---| return message |---*/
        if( NULL != szErrorMsg && 0 < cbErrorMsgMax )
        {                                        /*--------------------------*/
                register int    ci;              /* counter variable         */
                                                 /*--------------------------*/
                /*---| initialize buffer |---*/
                ci  = 0;
                strncpy( szErrorMsg, "[Oracle]", cbErrorMsgMax );
                ci += 8;

                /*---| add message |---*/
                if( cbErrorMsgMax > ci )
                {
                        strncat( szErrorMsg, pszErrMsg, cbErrorMsgMax - ci );
                        ci += strlen( pszErrMsg );
                }

                /*---| additional info |---*/
                if( '\0' != szOraErr[0] )
                {
                        szErrorMsg[ci++] = ' ';
                        szErrorMsg[ci++] = '(';
                        strncat( szErrorMsg, szOraErr, cbErrorMsgMax - ci );
                        ci += strlen( szOraErr );
                        if( '\n' == szErrorMsg[ci-1] )
                                ci--;
                        szErrorMsg[ci++] = ')';
                        szErrorMsg[ci]   = '\0';
                }

                /*---| set string length |---*/
                if( NULL != pcbErrorMsg )
                {
                        *pcbErrorMsg = ci;
                }

        } /* if */

        /*---| return data |---*/
        LogEntry( LOG_RETURNCODE, "[SQLError] SQL_SUCCESS" );
        return SQL_SUCCESS;
}

/*===| end of file |=========================================================*/
