/*
** DLL.C - This is the module containing the code for ODBC for
** advanced driver setup
**
** (c) 1997 by Dirk Ohme - all rights reserved
*/

#include "cli.h"
#include "dll.h"

/*---| local routines |------------------------------------------------------*/
MRESULT EXPENTRY DlgAdvSetup( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 );
static int InitDlg( HWND hwnd, SQLPDriverList pDrvEntry );
static void SaveDlg( HWND hwnd, SQLPDriverList pDrvEntry );

/*---------------------------------------------------------------------------*/
/*      This function is an entry point for the iODBC driver manager         */
/*      to start the advanced driver setup.                                  */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLDriverSetup(
                             LPDBC               lpdbc,
                             HWND                hwnd,
                             HMODULE             hmod,
                             SQLPDriverList      pDrvEntry
                                )
{
        /*
        ** check handle and pointers
        */
        LogEntry( LOG_STATUS, "[SQLDriverSetup] lpdbc     = $%08lX", lpdbc     );
        LogEntry( LOG_STATUS, "                 hwnd      = %5d",    hwnd      );
        LogEntry( LOG_STATUS, "                 hmod      = %5d",    hmod      );
        LogEntry( LOG_STATUS, "                 pDrvEntry = $%08lX", pDrvEntry );

        if( SQL_NULL_HDBC == lpdbc || SQL_NULL_HENV == lpdbc->pSqlEnv ||
            NULL          == pDrvEntry
          )
        {
                LogEntry( LOG_ERROR, "[SQLDriverSetup] NULL handle(s)" );
                LogEntry( LOG_RETURNCODE,
                          "[SQLDriverSetup] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }

        /*
        ** run setup dialog
        */
        WinDlgBox( HWND_DESKTOP, hwnd, DlgAdvSetup, hmod,
                   WND_SETUP,  (PVOID) pDrvEntry );

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLDriverSetup] SQL_SUCCESS" );
        return SQL_SUCCESS;

} /* SQLDriverSetup() */

/*---------------------------------------------------------------------------*/
/*      Main dialog routine                                                  */
/*---------------------------------------------------------------------------*/
MRESULT EXPENTRY DlgAdvSetup( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{

        static SQLPDriverList    pDrvList = NULL;

        /*
        ** distinguish between the various incoming messages
        */
        switch( msg )
        {
                /*
                ** initialisation of a new dialog window
                */
                case WM_INITDLG:
                        /*---| get the pointer to pointer to |---*/
                        /*---| the anchor of the driver list |---*/
                        pDrvList = (SQLPDriverList) PVOIDFROMMP( mp2 );

                        /*---| initialize dialog |---*/
                        if( 0 > InitDlg(hwnd, pDrvList) )
                                WinPostMsg( hwnd, WM_COMMAND,
                                            MPFROM2SHORT(DID_CANCEL, 0), 0 );

                        /*---| return success |---*/
                        return FALSE;

                /*
                ** commands for the dialog window procedure
                */
                case WM_COMMAND:
                        /*---| check command |---*/
                        switch( SHORT1FROMMP(mp1) )
                        {
                                /*---| 'Close' button pressed |---*/
                                case DID_OK:
                                        /*---| save new settings |---*/
                                        SaveDlg( hwnd, pDrvList );

                                        /*---| return, saving enabled |---*/
                                        WinDismissDlg( hwnd, TRUE );
                                        return 0;

                                /*---| 'Abort' button pressed |---*/
                                case DID_CANCEL:
                                        /*---| return, saving disabled |---*/
                                        WinDismissDlg( hwnd, FALSE );
                                        return 0;

                                /*---| ???: not one of ours |---*/
                                default:
                                        break;
                        }

                        /*---| break |---*/
                        break;

                /*
                ** ???: unknown message
                */
                default:
                        break;

        }

        /*
        ** we can't handle this message? standard processing!
        */
        return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* DlgAdvSetup() */

/*---------------------------------------------------------------------------*/
/*      get settings from driver list                                        */
/*---------------------------------------------------------------------------*/
static int InitDlg( HWND hwnd, SQLPDriverList pDrvEntry )
{
        SQLPDrvOptList   pCurrOpt;
        char             szBuffer[SQL_MAX_DRV_ID_LENGTH],
                        *psz,
                        *psz1;

        /*
        ** check parameters
        */
        if( NULL == pDrvEntry || NULL == (pCurrOpt=pDrvEntry->pOptions) )
                return -1;

        /*
        ** get replacement DSN entry 'dsn'
        */
        psz = NULL;
        while( NULL != pCurrOpt )
        {
                /*---| check for 'dsn' |---*/
                if( '\0' != *pCurrOpt->szOptName &&
                      0  == strcmp(pCurrOpt->szOptName, "dsn") )
                {
                        /*---| set pointer to option value |---*/
                        psz = &(pCurrOpt->szOptValue[0]);
                        break;
                }

                /*---| next entry |---*/
                pCurrOpt = (SQLPDrvOptList) pCurrOpt->pNext;
        }

        /*
        ** exit if no data is available
        */
        if( NULL == psz )
                return 0;
        strcpy( szBuffer, psz );
        psz = &szBuffer[0];

        /*
        ** parse data
        */
        /*---| check for user name |---*/
        if( NULL != (psz1=strchr(psz, '.')) )
        {
                /*---| set user field |---*/
                *(psz1++) = '\0';
                WinSetDlgItemText( hwnd, WND_SETUP_USER, psz );
                psz       = psz1;
        }

        /*---| check for host name |---*/
        if( NULL != (psz1=strrchr(psz, '@')) )
        {
                /*---| set host field |---*/
                *(psz1++) = '\0';
                WinSetDlgItemText( hwnd, WND_SETUP_HOST, psz1 );
        }

        /*---| set database name |---*/
        WinSetDlgItemText( hwnd, WND_SETUP_DB, psz );

        /*
        ** return success
        */
        return 0;

} /* InitDlg() */

/*---------------------------------------------------------------------------*/
/*      save new data to driver list                                         */
/*---------------------------------------------------------------------------*/
static void SaveDlg( HWND hwnd, SQLPDriverList pDrvEntry )
{
        SQLPDrvOptList   pCurrOpt,
                         pOldOpt;
        char             szBuffer[SQL_MAX_DRV_ID_LENGTH],
                        *psz;
        /*
        ** check parameters
        */
        if( NULL == pDrvEntry || NULL == (pCurrOpt=pDrvEntry->pOptions) )
                return;

        /*
        ** get replacement DSN entry 'dsn'
        */
        psz     = NULL;
        pOldOpt = NULL;
        while( NULL != pCurrOpt )
        {
                /*---| check for 'dsn' |---*/
                if( '\0' != *pCurrOpt->szOptName &&
                      0  == strcmp(pCurrOpt->szOptName, "dsn") )
                {
                        /*---| set pointer to option value |---*/
                        psz = &(pCurrOpt->szOptValue[0]);
                        break;
                }

                /*---| next entry |---*/
                pOldOpt  = pCurrOpt;
                pCurrOpt = (SQLPDrvOptList) pCurrOpt->pNext;
        }

        /*
        ** create a new entry, if not found
        */
        if( NULL == pCurrOpt )
        {
                /*---| exit on memory problems |---*/
                if( NULL == (pOldOpt->pNext=malloc(sizeof(SQLDrvOptList))) )
                        return;
                memset(pOldOpt->pNext, 0, sizeof(SQLDrvOptList) );
                pCurrOpt = (SQLPDrvOptList) pOldOpt->pNext;

                /*---| set defaults |---*/
                strcpy( pCurrOpt, "dsn" );
                psz = &(pCurrOpt->szOptValue[0]);
        }

        /*
        ** insert data
        */
        *psz = '\0';

        /*---| check for user name |---*/
        WinQueryDlgItemText( hwnd, WND_SETUP_USER,
                             sizeof(szBuffer), szBuffer );
        if( '\0' != szBuffer[0] )
        {
                strcat( psz, szBuffer );
                strcat( psz, "." );
        }

        /*---| check for database name |---*/
        WinQueryDlgItemText( hwnd, WND_SETUP_DB,
                             sizeof(szBuffer), szBuffer );
        if( '\0' != szBuffer[0] )
        {
                strcat( psz, szBuffer );
        }

        /*---| check for host name |---*/
        WinQueryDlgItemText( hwnd, WND_SETUP_HOST,
                             sizeof(szBuffer), szBuffer );
        if( '\0' != szBuffer[0] )
        {
                strcat( psz, "@" );
                strcat( psz, szBuffer );
        }

} /* SaveDlg() */

/*===| end of file |=========================================================*/
