/*
** file.c
**
** file access to INI file
**
** (c) 1997 by Dirk Ohme, all rights reserved
*/

/*---| includes |------------------------------------------------------------*/
#define INCL_DOSERRORS
#define INCL_DOSMODULEMGR
#define INCL_DOSPROCESS
#include <os2.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <sqlcli_.h>
#include "iODBCmgr.h"

/*---| prototypes |----------------------------------------------------------*/
static char *getIniFilename( void );

/*-----------------------------------------------------------------------------
 * name:     getIniFilename
 * input:    --none--
 * output:   char *                    == NULL : error
 *                                     != NULL : ptr to string with path/name
 * function: returns absolute path and name of INI file
 *-----------------------------------------------------------------------------
 */
static char *getIniFilename( void )
{
        static char     szFilename[CCHMAXPATH]   = "";
        PPIB            ppib;
        PTIB            ptib;
        char           *psz;

#ifdef DEBUG
        fprintf( stderr, "> [%s] getIniFilename\n", __FILE__ );
#endif

        /*
        ** get path and name of EXE
        */
        DosGetInfoBlocks( &ptib, &ppib );
        if( NO_ERROR != DosQueryModuleName( ppib->pib_hmte,
                                            sizeof(szFilename),
                                            &szFilename[0] ) )
        {
                /*---| return failure |---*/
#ifdef DEBUG
                fprintf( stderr,
                         "! [%s] getIniFilename: can't get info blocks\n",
                         __FILE__ );
#endif
                return NULL;
        }

        /*
        ** truncate file name
        */
        if( NULL != (psz=strrchr(szFilename, '\\')) )
                *(++psz) = '\0';

        /*
        ** add INI file name, return string
        */
        strcat( szFilename, INI_FILE );
#ifdef DEBUG
        fprintf( stderr, "< [%s] getIniFilename -> '%s'\n",
                 __FILE__, szFilename );
#endif
        return &szFilename[0];

} /* getIniFilename() */

/*-----------------------------------------------------------------------------
 * name:     getDriverList
 * input:    --none--
 * output:   SQLPDriverList            == NULL : error
 *                                     != NULL : ptr to start of driver list
 * function: tries to read in the INI file -> driver list structure
 *-----------------------------------------------------------------------------
 */
SQLPDriverList getDriverList( void )
{
        FILE            *pfIniFile;
        SQLPDrvOptList   pOptStart,
                         pOptCurr;
        SQLPDriverList   pListStart,
                         pListCurr;
        char            *psz,
                        *psz1,
                        *psz2,
                         szBuffer[SQL_MAX_DRV_ID_LENGTH];

#ifdef DEBUG
        fprintf( stderr, "> [%s] getDriverList\n", __FILE__ );
#endif
        /*
        ** get name of INI file, open it
        */
        if( NULL == (psz=getIniFilename()) ||
            NULL == (pfIniFile=fopen(psz, "r")) )
        {
                /*---| return failure |---*/
#ifdef DEBUG
                fprintf( stderr,
                         "! [%s] getDriverList: can't read INI file (%s)\n",
                         __FILE__, (NULL==psz) ? "<NULL>" : psz );
#endif
                return NULL;
        }

        /*
        ** initialize, loop
        */
        pListStart =
        pListCurr  = NULL;
        while( !feof(pfIniFile) )
        {
                /*---| get next line |---*/
                if( NULL == fgets(szBuffer, sizeof(szBuffer), pfIniFile) )
                        continue;

                /*---| skip leading spaces, skip commentary lines |---*/
                psz = &szBuffer[0];
                while( ' ' == *psz || '\t' == *psz )
                        psz++;
                if( '#' == *psz || '\n' == *psz || '\r' == *psz )
                        continue;

                /*---| check for new entry |---*/
                if( '[' == *psz )
                {
                        /*---| skip leading bracket |---*/
                        psz++;

                        /*---| break on non-existing right bracket |---*/
                        if( NULL == strchr(psz, ']') )
                                break;
                        else
                                *(strchr(psz, ']')) = '\0';

                        /*---| create new entry |---*/
                        if( NULL == pListStart )
                        {
                                /*---| new list |---*/
                                if( NULL == (pListStart=malloc(sizeof(SQLDriverList))) )
                                        break;
                                pListCurr = pListStart;
                        }
                        else
                        {
                                /*---| next entry |---*/
                                if( NULL == (pListCurr->pNext=malloc(sizeof(SQLDriverList))) )
                                        break;
                                pListCurr = (SQLPDriverList) pListCurr->pNext;
                        }

                        /*---| set data |---*/
                        memset( pListCurr, 0, sizeof(SQLDriverList) );
                        strncpy( pListCurr->szName, psz,
                                 SQL_MAX_DRV_ID_LENGTH );
                        pOptStart = NULL;

                        /*---| loop ... |---*/
#ifdef DEBUG
                        fprintf( stderr,
                                 "+ [%s] getDriverList: new section '%s'\n",
                                 __FILE__, psz );
#endif
                        continue;
                }

                /*---| skip option outside of driver entries |---*/
                if( NULL == pListStart )
                        continue;

                /*---| must be an option -> add to option list |---*/
                /*---| create new entry |---*/
                if( NULL == pOptStart )
                {
                        /*---| new list |---*/
                        if( NULL == (pOptStart=malloc(sizeof(SQLDrvOptList))) )
                                break;
                        pListCurr->pOptions =
                        pOptCurr            = pOptStart;
                }
                else
                {
                        /*---| next entry |---*/
                        if( NULL == (pOptCurr->pNext=malloc(sizeof(SQLDrvOptList))) )
                                break;
                        pOptCurr = (SQLPDrvOptList) pOptCurr->pNext;
                }

                /*---| initialize structure |---*/
                memset( pOptCurr, 0, sizeof(SQLDrvOptList) );

                /*---| filter terminating blanks |---*/
                psz1 = psz;
                while( '\0' != *psz1 )
                        psz1++;
                do {
                        *(psz1--) = '\0';
                }
                while( ' '  == *psz1 || '\n' == *psz1 ||
                       '\r' == *psz1 || '\t' == *psz1 );

                /*---| check for '=' |---*/
                if( NULL == (psz1=strchr(psz, '=')) )
                {
                        /*---| set boolean entry |---*/
                        strlwr( psz );
                        strncpy( pOptCurr->szOptName, psz,
                                 SQL_MAX_DRV_ID_LENGTH );
#ifdef DEBUG
                        fprintf( stderr,
                                 "  [%s] getDriverList:   boolean <%s>\n",
                                 __FILE__, psz );
#endif
                        continue;
                }

                /*---| must be an assignment -> store name |---*/
                psz2 = psz;
                while( '\0' != *psz2 && ' ' != *psz2 &&
                       '\t' != *psz2 && '=' != *psz2 )
                        psz2++;
                *psz2 = '\0';
                strlwr( psz );
                strncpy( pOptCurr->szOptName, psz, SQL_MAX_DRV_ID_LENGTH );

                /*---| check value |---*/
                psz = ++psz1;
                while( ' ' == *psz || '\t' == *psz )
                        psz++;
                strncpy( pOptCurr->szOptValue, psz, SQL_MAX_DRV_ID_LENGTH );
#ifdef DEBUG
                fprintf( stderr,
                         "  [%s] getDriverList:   switch  <%s> := <%s>\n",
                         __FILE__, pOptCurr->szOptName,
                         pOptCurr->szOptValue );
#endif
        }

        /*
        ** return success
        */
        fclose( pfIniFile );
#ifdef DEBUG
        fprintf( stderr, "< [%s] getDriverList\n", __FILE__ );
#endif
        return pListStart;

} /* getDriverList() */

/*-----------------------------------------------------------------------------
 * name:     putDriverList
 * input:    SQLPDriverList pDrvList    - list with drivers and options
 * output:   int                       == 0 : okay
 *                                     != 0 : error
 * function: tries to write the INI file -> driver list structure
 *-----------------------------------------------------------------------------
 */
int putDriverList( SQLPDriverList pDrvList )
{
        FILE            *pfIniFile;
        SQLPDrvOptList   pOptStart,
                         pOptCurr;
        SQLPDriverList   pListCurr;
        char            *psz;

#ifdef DEBUG
        fprintf( stderr, "> [%s] putDriverList\n", __FILE__ );
#endif
        /*
        ** get name of INI file, open it
        */
        if( NULL == (psz=getIniFilename()) ||
            NULL == (pfIniFile=fopen(psz, "w")) )
        {
                /*---| return failure |---*/
#ifdef DEBUG
                fprintf( stderr,
                         "! [%s] getDriverList: can't write INI file (%s)\n",
                         __FILE__, (NULL==psz) ? "<NULL>" : psz );
#endif
                return -1;
        }

        /*
        ** loop
        */
        while( NULL != pDrvList )
        {
                /*---| write out new section |---*/
                if( NULL != pDrvList->szName )
                {
                        fprintf( pfIniFile, "[%s]\n", pDrvList->szName );
#ifdef DEBUG
                        fprintf( stderr,
                                 "+ [%s] putDriverList: new section '%s'\n",
                                 __FILE__, pDrvList->szName );
#endif
                }

                /*---| recurse through options |---*/
                pOptStart = pDrvList->pOptions;
                while( NULL != pOptStart )
                {
                        /*---| valid entry? |---*/
                        if( NULL != pOptStart->szOptName )
                        {
                                /*---| boolean entry? |---*/
                                if( NULL == pOptStart->szOptValue )
                                {
                                        fprintf( pfIniFile,
                                                 "        %s\n",
                                                 pOptStart->szOptName );
#ifdef DEBUG
                                        fprintf( stderr,
                                                 "  [%s] putDriverList:   boolean <%s>\n",
                                                 __FILE__,
                                                 pOptStart->szOptName );
#endif
                                }
                                else
                                {
                                        fprintf( pfIniFile,
                                                 "        %s = %s\n",
                                                 pOptStart->szOptName,
                                                 pOptStart->szOptValue );
#ifdef DEBUG
                                        fprintf( stderr,
                                                 "  [%s] putDriverList:   switch  <%s> := <%s>\n",
                                                 __FILE__,
                                                 pOptStart->szOptName,
                                                 pOptStart->szOptValue );
#endif
                                }
                        }

                        /*---| kill old entry, next |---*/
                        pOptCurr  = pOptStart;
                        pOptStart = (SQLPDrvOptList) pOptStart->pNext;
                        free( pOptCurr );
                }

                /*---| kill old entry, next |---*/
                pListCurr = pDrvList;
                pDrvList  = (SQLPDriverList) pDrvList->pNext;
                free( pListCurr );
        }

        /*
        ** return success
        */
        fclose( pfIniFile );
#ifdef DEBUG
        fprintf( stderr, "< [%s] putDriverList\n", __FILE__ );
#endif
        return 0;

} /* putDriverList() */

/*===| end of file |=========================================================*/
