/*
** dlg_set.c
**
** setup dialog box of the iODBC driver manager for OS/2
**
** (c) 1997 by Dirk Ohme, all rights reserved
*/

/*---| includes |------------------------------------------------------------*/
#define INCL_DOSERRORS
#define INCL_DOSMODULEMGR
#define INCL_PM
#define INCL_WIN
#include <os2.h>
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>

#include <sqlcli_.h>
#include "iODBCmgr.h"

/*---| local prototypes |----------------------------------------------------*/
static void InitDlgBox( HWND hwnd, SQLPDriverList pDrvEntry );
static void SaveDlgBox( HWND hwnd, SQLPDriverList pDrvEntry );
static int DisplayDriverInfo( HWND hwnd, char *pszDriverFile );
static void SearchForDriver( HWND hwnd );
static void AdvancedDriverSetup( HWND hwnd, SQLPDriverList pDrvEntry );

/*-----------------------------------------------------------------------------
 * name:     DlgSetup
 * input:    HWND hwnd                 : handle for dialog box window
 *           ULONG msg                 : incoming message
 *           MPARAM mp1                : message parameter 1
 *           MPARAM mp2                : message parameter 2
 * output:   MRESULT EXPENTRY          : result for Presentation Manager
 * function: setup dialog box for iODBC drivers
 *-----------------------------------------------------------------------------
 */
MRESULT EXPENTRY DlgSetup( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
        static SQLPDriverList    pDrvList = NULL;

        /*
        ** distinguish between the various incoming messages
        */
        switch( msg )
        {
                /*
                ** initialisation of a new dialog window
                */
                case WM_INITDLG:
                        /*---| get the pointer to the |---*/
                        /*---|  current driver entry  |---*/
                        pDrvList = (SQLPDriverList) PVOIDFROMMP( mp2 );

                        /*---| initialize dialog |---*/
                        InitDlgBox( hwnd, pDrvList );

                        /*---| return success |---*/
                        return FALSE;

                /*
                ** changes of control elements
                */
                case WM_CONTROL:
                        /*---| check for which control |---*/
                        /*---| element caused message  |---*/
                        switch( SHORT1FROMMP(mp1) )
                        {
                                /*---| selection list activities |---*/
                                case WND_SETUP_TRACE_STDERR:
                                        /*---| enable tracing |---*/
                                        WinSendDlgItemMsg( hwnd,
                                                WND_SETUP_TRACE,
                                                BM_SETCHECK,
                                                MPFROMSHORT(TRUE),
                                                0L );

                                        /*---| return success |---*/
                                        return 0;

                                case WND_SETUP_TRACE_STDOUT:
                                        /*---| enable tracing |---*/
                                        WinSendDlgItemMsg( hwnd,
                                                WND_SETUP_TRACE,
                                                BM_SETCHECK,
                                                MPFROMSHORT(TRUE),
                                                0L );

                                        /*---| return success |---*/
                                        return 0;

                                /*---| ???: not one of ours |---*/
                                default:
                                        break;
                        }

                        /*---| break |---*/
                        break;

                /*
                ** commands for the dialog window procedure
                */
                case WM_COMMAND:
                        /*---| check command |---*/
                        switch( SHORT1FROMMP(mp1) )
                        {
                                char     szBuffer[CCHMAXPATH];

                                /*---| 'Close' button pressed |---*/
                                case DID_OK:
                                        /*---| save settings |---*/
                                        SaveDlgBox( hwnd, pDrvList );

                                        /*---| return, saving enabled |---*/
                                        WinDismissDlg( hwnd, TRUE );
                                        return 0;

                                /*---| 'Abort' button pressed |---*/
                                case DID_CANCEL:
                                        /*---| return, saving disabled |---*/
                                        WinDismissDlg( hwnd, FALSE );
                                        return 0;

                                /*---| 'File List' button pressed |---*/
                                case WND_SETUP_FILE_SEARCH:
                                        /*---| start search |---*/
                                        SearchForDriver( hwnd );

                                        /*---| return to PM |---*/
                                        return 0;

                                /*---| 'Advanced' button pressed |---*/
                                case WND_SETUP_ADV:
                                        /*---| start advanced setup |---*/
                                        AdvancedDriverSetup( hwnd,
                                                             pDrvList );

                                        /*---| initialize dialog |---*/
                                        InitDlgBox( hwnd, pDrvList );

                                        /*---| return to PM |---*/
                                        return 0;

                                /*---| 'Info' button pressed |---*/
                                case WND_SETUP_INFO:
                                        /*---| get driver path and name |---*/
                                        WinQueryDlgItemText( hwnd,
                                                             WND_SETUP_FILE,
                                                             sizeof(szBuffer),
                                                             szBuffer );

                                        /*---| display driver info |---*/
                                        DisplayDriverInfo( hwnd, szBuffer );

                                        /*---| return to PM |---*/
                                        return 0;

                                /*---| ???: not one of ours |---*/
                                default:
                                        break;
                        }

                        /*---| break |---*/
                        break;

                /*
                ** ???: unknown message
                */
                default:
                        break;

        }

        /*
        ** we can't handle this message? standard processing!
        */
        return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* DlgSetup() */

/*-----------------------------------------------------------------------------
 * name:     InitDlgBox
 * input:    HWND hwnd                 : handle for dialog box window
 *           SQLPDriverList pDrvEntry  : pointer to an driver entry
 * output:   --nothing--
 * function: initializes the elements of the dialog box
 *-----------------------------------------------------------------------------
 */
static void InitDlgBox( HWND hwnd, SQLPDriverList pDrvEntry )
{
        SQLPDrvOptList   pCurrOpt;

        /*
        ** check options
        */
        if( NULL == pDrvEntry )
                return;

        /*
        ** set driver name
        */
        WinSetDlgItemText( hwnd, WND_SETUP_NAME, pDrvEntry->szName );

        /*
        ** set length in entry fields
        */
        WinSendDlgItemMsg( hwnd, WND_SETUP_NAME, EM_SETTEXTLIMIT,
                           MPFROMSHORT(60), 0ul );
        WinSendDlgItemMsg( hwnd, WND_SETUP_COMMENT, EM_SETTEXTLIMIT,
                           MPFROMSHORT(60), 0ul );
        WinSendDlgItemMsg( hwnd, WND_SETUP_FILE, EM_SETTEXTLIMIT,
                           MPFROMSHORT(CCHMAXPATH), 0ul );
        WinSendDlgItemMsg( hwnd, WND_SETUP_DSN, EM_SETTEXTLIMIT,
                           MPFROMSHORT(SQL_MAX_DSN_LENGTH + 1), 0ul );

        /*
        ** go through options
        */
        pCurrOpt = pDrvEntry->pOptions;
        while( NULL != pCurrOpt )
        {
                /*---| check for illegal values |---*/
                if( '\0' == *(pCurrOpt->szOptName) )
                {
                        /*---| get next entry, restart loop |---*/
                        pCurrOpt = (SQLPDrvOptList) pCurrOpt->pNext;
                        continue;
                }

                /*---| check for comment |---*/
                if( 0 == strcmp(pCurrOpt->szOptName, "comment") )
                {
                        WinSetDlgItemText( hwnd, WND_SETUP_COMMENT,
                                           pCurrOpt->szOptValue );
                }

                /*---| check for tracing |---*/
                else if( 0 == strcmp(pCurrOpt->szOptName, "trace") )
                {
                        /*---| check option |---*/
                        if( '\0' != *(pCurrOpt->szOptValue) &&
                            ( 0 == strcmp(pCurrOpt->szOptValue, "1") ||
                              0 == strcmp(pCurrOpt->szOptValue, "on") ||
                              0 == strcmp(pCurrOpt->szOptValue, "On") ||
                              0 == strcmp(pCurrOpt->szOptValue, "ON")) )
                        {
                                /*---| set check box |---*/
                                WinSendDlgItemMsg( hwnd, WND_SETUP_TRACE,
                                                   BM_SETCHECK,
                                                   MPFROMSHORT(TRUE),
                                                   0L );
                        }
                }

                /*---| check for tracing output |---*/
                else if( 0 == strcmp(pCurrOpt->szOptName, "tracefile") )
                {
                        if( '\0' != *(pCurrOpt->szOptValue) &&
                              0  == strcmp(pCurrOpt->szOptValue, "stdout") )
                        {
                                /*---| 'stdout' |---*/
                                WinSendDlgItemMsg( hwnd,
                                                   WND_SETUP_TRACE_STDOUT,
                                                   BM_SETCHECK,
                                                   MPFROMSHORT(TRUE),
                                                   0L );
                        }
                        else
                        {
                                /*---| 'stderr' |---*/
                                WinSendDlgItemMsg( hwnd,
                                                   WND_SETUP_TRACE_STDERR,
                                                   BM_SETCHECK,
                                                   MPFROMSHORT(TRUE),
                                                   0L );
                        }
                }

                /*---| check for driver file |---*/
                else if( 0 == strcmp(pCurrOpt->szOptName, "driver") )
                {
                        WinSetDlgItemText( hwnd, WND_SETUP_FILE,
                                           pCurrOpt->szOptValue );
                }
                /*---| check for replacement DSN |---*/
                else if( 0 == strcmp(pCurrOpt->szOptName, "dsn") )
                {
                        WinSetDlgItemText( hwnd, WND_SETUP_DSN,
                                           pCurrOpt->szOptValue );
                }

                /*---| next option |---*/
                pCurrOpt = (SQLPDrvOptList) pCurrOpt->pNext;
        }

} /* InitDlgBox() */

/*-----------------------------------------------------------------------------
 * name:     SaveDlgBox
 * input:    HWND hwnd                 : handle for dialog box window
 *           SQLPDriverList pDrvEntry  : pointer to an driver entry
 * output:   --nothing--
 * function: store results of list box in driver structure
 *-----------------------------------------------------------------------------
 */
static void SaveDlgBox( HWND hwnd, SQLPDriverList pDrvEntry )
{
        char             szBuffer[256];
        SQLPDrvOptList   pCurrOpt,
                         pTmpOpt;

        /*
        ** drop old options structure
        */
        if( NULL == pDrvEntry )
                return;
        pCurrOpt = pDrvEntry->pOptions;
        while( NULL != pCurrOpt )
        {
                /*---| free memory |---*/
                pTmpOpt  = pCurrOpt;
                pCurrOpt = (SQLPDrvOptList) pCurrOpt->pNext;
                free( pTmpOpt );
        }
        pDrvEntry->pOptions = NULL;

        /*
        ** set new driver name
        */
        WinQueryDlgItemText( hwnd, WND_SETUP_NAME,
                             SQL_MAX_DRV_ID_LENGTH,
                             pDrvEntry->szName );

        /*
        ** check for replacement DSN
        */
        WinQueryDlgItemText( hwnd, WND_SETUP_DSN,
                             sizeof(szBuffer), szBuffer );
        if( '\0' != szBuffer[0] &&
            NULL != (pTmpOpt=malloc(sizeof(SQLDrvOptList))) )
        {
                /*---| set new option |---*/
                memset( pTmpOpt, 0, sizeof(SQLDrvOptList) );
                strncpy( pTmpOpt->szOptName,  "dsn",    SQL_MAX_DRV_ID_LENGTH );
                strncpy( pTmpOpt->szOptValue, szBuffer, SQL_MAX_DRV_ID_LENGTH );

                /*---| set links |---*/
                pTmpOpt->pNext      = pDrvEntry->pOptions;
                pDrvEntry->pOptions = pTmpOpt;
        }

        /*
        ** check for driver file name
        */
        WinQueryDlgItemText( hwnd, WND_SETUP_FILE,
                             sizeof(szBuffer), szBuffer );
        if( NULL != (pTmpOpt=malloc(sizeof(SQLDrvOptList))) )
        {
                /*---| set new option |---*/
                memset( pTmpOpt, 0, sizeof(SQLDrvOptList) );
                strncpy( pTmpOpt->szOptName,  "driver", SQL_MAX_DRV_ID_LENGTH );
                strncpy( pTmpOpt->szOptValue, szBuffer, SQL_MAX_DRV_ID_LENGTH );

                /*---| set links |---*/
                pTmpOpt->pNext      = pDrvEntry->pOptions;
                pDrvEntry->pOptions = pTmpOpt;
        }

        /*
        ** check for trace tracing
        */
        if( WinSendDlgItemMsg( hwnd, WND_SETUP_TRACE,
                               BM_QUERYCHECK, NULL, NULL ) )
        {
                /*---| check for output to 'stdout' |---*/
                if( WinSendDlgItemMsg( hwnd, WND_SETUP_TRACE_STDOUT,
                                       BM_QUERYCHECK, NULL, NULL ) &&
                    NULL != (pTmpOpt=malloc(sizeof(SQLDrvOptList))) )
                {
                        /*---| set new option |---*/
                        memset( pTmpOpt, 0, sizeof(SQLDrvOptList) );
                        strncpy( pTmpOpt->szOptName,  "tracefile", SQL_MAX_DRV_ID_LENGTH );
                        strncpy( pTmpOpt->szOptValue, "stdout",    SQL_MAX_DRV_ID_LENGTH );

                        /*---| set links |---*/
                        pTmpOpt->pNext      = pDrvEntry->pOptions;
                        pDrvEntry->pOptions = pTmpOpt;
                }

                /*---| check for trace output to 'stderr' |---*/
                if( WinSendDlgItemMsg( hwnd, WND_SETUP_TRACE_STDERR,
                                       BM_QUERYCHECK, NULL, NULL ) &&
                    NULL != (pTmpOpt=malloc(sizeof(SQLDrvOptList))) )
                {
                        /*---| set new option |---*/
                        memset( pTmpOpt, 0, sizeof(SQLDrvOptList) );
                        strncpy( pTmpOpt->szOptName,  "tracefile", SQL_MAX_DRV_ID_LENGTH );
                        strncpy( pTmpOpt->szOptValue, "stderr",    SQL_MAX_DRV_ID_LENGTH );

                        /*---| set links |---*/
                        pTmpOpt->pNext      = pDrvEntry->pOptions;
                        pDrvEntry->pOptions = pTmpOpt;
                }

                /*---| set tracing entry |---*/
                if( NULL != (pTmpOpt=malloc(sizeof(SQLDrvOptList))) )
                {
                        /*---| set new option |---*/
                        memset( pTmpOpt, 0, sizeof(SQLDrvOptList) );
                        strncpy( pTmpOpt->szOptName,  "trace", SQL_MAX_DRV_ID_LENGTH );
                        strncpy( pTmpOpt->szOptValue, "on",    SQL_MAX_DRV_ID_LENGTH );

                        /*---| set links |---*/
                        pTmpOpt->pNext      = pDrvEntry->pOptions;
                        pDrvEntry->pOptions = pTmpOpt;
                }
        }
        else
        {
                /*---| disable tracing |---*/
                if( NULL != (pTmpOpt=malloc(sizeof(SQLDrvOptList))) )
                {
                        /*---| set new option |---*/
                        memset( pTmpOpt, 0, sizeof(SQLDrvOptList) );
                        strncpy( pTmpOpt->szOptName,  "trace", SQL_MAX_DRV_ID_LENGTH );
                        strncpy( pTmpOpt->szOptValue, "off",   SQL_MAX_DRV_ID_LENGTH );

                        /*---| set links |---*/
                        pTmpOpt->pNext      = pDrvEntry->pOptions;
                        pDrvEntry->pOptions = pTmpOpt;
                }
        }

        /*
        ** check for comment
        */
        WinQueryDlgItemText( hwnd, WND_SETUP_COMMENT,
                             sizeof(szBuffer), szBuffer );
        if( '\0' != szBuffer[0] &&
            NULL != (pTmpOpt=malloc(sizeof(SQLDrvOptList))) )
        {
                /*---| set new option |---*/
                memset( pTmpOpt, 0, sizeof(SQLDrvOptList) );
                strncpy( pTmpOpt->szOptName,  "comment", SQL_MAX_DRV_ID_LENGTH );
                strncpy( pTmpOpt->szOptValue, szBuffer,  SQL_MAX_DRV_ID_LENGTH );

                /*---| set links |---*/
                pTmpOpt->pNext      = pDrvEntry->pOptions;
                pDrvEntry->pOptions = pTmpOpt;
        }

} /* SaveDlgBox() */

/*-----------------------------------------------------------------------------
 * name:     DisplayDriverInfo
 * input:    HWND hwnd                 : handle for dialog box window
 *           char *pszDriverFile       : path and file name of driver
 * output:   int                == -1  : error
 *                              ==  0  : okay
 *                              ==  1  : okay, has driver setup on its own
 * function: checks a driver for correctness and level
 *-----------------------------------------------------------------------------
 */
static int DisplayDriverInfo( HWND hwnd, char *pszDriverFile )
{
        APIRET           rc;
        HMODULE          hModule = NULLHANDLE;
        PFN              pSQLAllocEnv,
                         pSQLAllocConnect,
                         pSQLDriverSetup,
                         pSQLGetInfo,
                         pSQLFreeConnect,
                         pSQLFreeEnv;
        SQLHENV          pSqlEnv;
        SQLHDBC          pSqlHdbc;
        char             szBuffer[256],
                        *psz;

        /*
        ** initialize
        */
        pSqlEnv  = SQL_NULL_HENV;
        pSqlHdbc = SQL_NULL_HDBC;

        /*
        ** check parameters
        */
        if( NULL == pszDriverFile || '\0' == *pszDriverFile )
        {
                /*---| display error text |---*/
                WinMessageBox( HWND_DESKTOP, hwnd,
                               "You must specify a correct path and file name",
                               "iODBCmgr Driver Info", 0,
                               MB_OK | MB_ERROR | MB_MOVEABLE );

                /*---| return failure |---*/
                return -1;
        }

        /*
        ** try to load the specified driver
        */
        if( NO_ERROR != (rc=DosLoadModule(szBuffer, sizeof(szBuffer),
                                          pszDriverFile, &hModule)) )
        {
                /*---| create error message |---*/
                sprintf( szBuffer, "Could not load driver '%s', rc=%d\n",
                         pszDriverFile, rc );

                /*---| display error text |---*/
                WinMessageBox( HWND_DESKTOP, hwnd,
                               szBuffer, "iODBCmgr Driver Info", 0,
                               MB_OK | MB_ERROR | MB_MOVEABLE );

                /*---| return failure |---*/
                return -1;
        }

        /*
        ** check for ODBC / CLI conformance
        */
        if( NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLAllocEnv",     &pSQLAllocEnv)     ||
            NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLAllocConnect", &pSQLAllocConnect) ||
            NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLGetInfo",      &pSQLGetInfo)      ||
            NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLFreeConnect",  &pSQLFreeConnect)  ||
            NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLFreeEnv",      &pSQLFreeEnv) )
        {
                /*---| free module on errors |---*/
                DosFreeModule( hModule );

                /*---| create error message |---*/
                sprintf( szBuffer, "File '%s' is no valid ODBC/CLI driver\n",
                         pszDriverFile );

                /*---| display error text |---*/
                WinMessageBox( HWND_DESKTOP, hwnd,
                               szBuffer, "iODBCmgr Driver Info", 0,
                               MB_OK | MB_ERROR | MB_MOVEABLE );

                /*---| return failure |---*/
                return -1;
        }

        /*
        ** get ODBC/CLI environment
        */
        strcpy( szBuffer, "ODBC level: " );
        psz = strchr( szBuffer, '\0' );
        if( SQL_SUCCESS != (rc=(*pSQLAllocEnv)(&pSqlEnv))               ||
            SQL_SUCCESS != (rc=(*pSQLAllocConnect)(pSqlEnv, &pSqlHdbc)) )
        {
                /*---| free statement handle |---*/
                if( SQL_NULL_HENV != pSqlEnv )
                        (*pSQLFreeEnv)( pSqlEnv );

                /*---| free module on errors |---*/
                DosFreeModule( hModule );

                /*---| create error message |---*/
                sprintf( szBuffer,
                         "Can't check ODBC conforming level on '%s' (rc=%d)\n",
                         pszDriverFile, rc );

                /*---| display error text |---*/
                WinMessageBox( HWND_DESKTOP, hwnd,
                               szBuffer, "iODBCmgr Driver Info", 0,
                               MB_OK | MB_ERROR | MB_MOVEABLE );

                /*---| return failure |---*/
                return -1;
        }

        /*
        ** check ODBC/CLI conforming level
        */
        if( SQL_SUCCESS != (rc=(*pSQLGetInfo)(pSqlHdbc, SQL_ODBC_VER,
                                              psz, sizeof(szBuffer) -
                                              (int) (psz - &szBuffer[0]),
                                              NULL))                    &&
            SQL_SUCCESS != (rc=(*pSQLGetInfo)(pSqlHdbc, SQL_DRIVER_VER,
                                              psz, sizeof(szBuffer) -
                                              (int) (psz - &szBuffer[0]),
                                              NULL))                    )
        {
                /*---| something went wrong ... |---*/
                strcat( psz, "can't detect ODBC level" );

        }

        /*
        ** free ODBC/CLI environment
        */
        if( SQL_NULL_HDBC != pSqlHdbc )
                (*pSQLFreeConnect)( pSqlHdbc );
        if( SQL_NULL_HENV != pSqlEnv )
                (*pSQLFreeEnv)( pSqlEnv );

        /*---| no ODBC level found -> 'core level' |---*/
        if( '\0' == *psz )
                strcat( psz, "core level" );

        /*
        ** check for driver setup
        */
        pSQLDriverSetup = NULL;
        if( NO_ERROR == DosQueryProcAddr(hModule, 0L, "SQLDriverSetup",
                                         &pSQLDriverSetup) )
        {
                /*---| update info |---*/
                strcat( szBuffer, ", has iODBC compatible driver setup" );
        }

        /*
        ** unload driver
        */
        DosFreeModule( hModule );

        /*
        ** display info
        */
        WinMessageBox( HWND_DESKTOP, hwnd,
                       szBuffer, "iODBCmgr Driver Info", 0,
                       MB_OK | MB_INFORMATION | MB_MOVEABLE );

        /*
        ** return success
        */
        return (NULL == pSQLDriverSetup) ? 0 : 1;

} /* DisplayDriverInfo() */

/*-----------------------------------------------------------------------------
 * name:     SearchForDriver
 * input:    HWND hwnd                 : handle for dialog box window
 * output:   --nothing--
 * function: presents a file dialog box for driver search
 *-----------------------------------------------------------------------------
 */
static void SearchForDriver( HWND hwnd )
{
        BOOL     fSuccess;
        FILEDLG  aFileDlg;
        HWND     hwndFileDlg;

        /*
        ** initialize structures
        */
        memset( &aFileDlg, 0, sizeof(aFileDlg) );
        aFileDlg.cbSize   = sizeof(aFileDlg);
        aFileDlg.fl       = FDS_CENTER | FDS_OPEN_DIALOG;
        aFileDlg.pszTitle = "ODBC driver selection";
        if( NO_ERROR != WinQueryDlgItemText(hwnd, WND_SETUP_FILE,
                                            sizeof(aFileDlg.szFullFile),
                                            aFileDlg.szFullFile) )
                aFileDlg.szFullFile[0] = '\0';
        if( '\0' == aFileDlg.szFullFile )
                strcpy( aFileDlg.szFullFile, "*.DLL" );

        /*
        ** loop until abort or success
        */
        fSuccess = FALSE;
        while( !fSuccess )
        {
                /*---| display file dialog |---*/
                hwndFileDlg = WinFileDlg( HWND_DESKTOP, hwnd, &aFileDlg );
                if( hwndFileDlg )
                {
                        if( DID_OK == aFileDlg.lReturn &&
                             0     <= DisplayDriverInfo(hwnd,
                                                        aFileDlg.szFullFile) )
                        {
                                /*---| set new driver file name |---*/
                                WinSetDlgItemText( hwnd, WND_SETUP_FILE,
                                                   aFileDlg.szFullFile );

                                /*---| set success flag |---*/
                                fSuccess = TRUE;
                        }
                        else if( DID_CANCEL == aFileDlg.lReturn );
                        {
                                /*---| set abort flag |---*/
                                fSuccess = TRUE;
                        }
                }
        }

} /* SearchForDriver() */

/*-----------------------------------------------------------------------------
 * name:     AdvancedDriverSetup
 * input:    HWND hwnd                 : handle for dialog box window
 *           SQLPDriverList pDrvEntry  : pointer to an driver entry
 * output:   --nothing--
 * function: starts advanced driver setup dialog from driver
 *-----------------------------------------------------------------------------
 */
static void AdvancedDriverSetup( HWND hwnd, SQLPDriverList pDrvEntry )
{
        APIRET                   rc;
        HMODULE                  hModule = NULLHANDLE;
        PFN                      pSQLAllocEnv,
                                 pSQLAllocConnect,
                                 pSQLDriverSetup,
                                 pSQLFreeConnect,
                                 pSQLFreeEnv;
        SQLHENV                  pSqlEnv;
        SQLHDBC                  pSqlHdbc;
        char                     szBuffer[256],
                                *psz;
        register SQLPDrvOptList  pCurrOpt;

        /*
        ** check parameters
        */
        psz      = NULL;
        if( NULL != pDrvEntry )
        {
                /*---| scan for file name |---*/
                pCurrOpt = pDrvEntry->pOptions;
                while( NULL != pCurrOpt )
                {
                        /*---| check for 'driver' entry |---*/
                        if( '\0' != pCurrOpt->szOptName &&
                              0  == strcmp(pCurrOpt->szOptName, "driver") )
                        {
                                /*---| save pointer to options |---*/
                                psz = &(pCurrOpt->szOptValue[0]);
                        }

                        /*---| next |---*/
                        pCurrOpt = (SQLPDrvOptList) pCurrOpt->pNext;
                }
        }

        /*
        ** check for file name
        */
        if( NULL == psz )
        {
                /*---| display error message |---*/
                WinMessageBox( HWND_DESKTOP, hwnd,
                               "Need a reference to the path and file name",
                               "iODBCmgr Advanced Driver Setup", 0,
                               MB_OK | MB_ERROR | MB_MOVEABLE );
                return;
        }

        /*
        ** try to load the specified driver
        */
        if( NO_ERROR != (rc=DosLoadModule(szBuffer, sizeof(szBuffer),
                                          psz, &hModule)) )
        {
                /*---| create error message |---*/
                sprintf( szBuffer, "Could not load driver '%s', rc=%d\n",
                         psz, rc );

                /*---| display error text |---*/
                WinMessageBox( HWND_DESKTOP, hwnd,
                               szBuffer, "iODBCmgr Advanced Driver Setup", 0,
                               MB_OK | MB_ERROR | MB_MOVEABLE );

                /*---| return |---*/
                return;
        }

        /*
        ** check for driver setup entry point
        */
        if( NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLAllocEnv",     &pSQLAllocEnv)     ||
            NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLAllocConnect", &pSQLAllocConnect) ||
            NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLDriverSetup",  &pSQLDriverSetup)  ||
            NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLFreeConnect",  &pSQLFreeConnect)  ||
            NO_ERROR != DosQueryProcAddr(hModule, 0L, "SQLFreeEnv",      &pSQLFreeEnv) )
        {
                /*---| free module on errors |---*/
                DosFreeModule( hModule );

                /*---| create error message |---*/
                sprintf( szBuffer,
                         "File '%s' has no driver setup routine of its own\n",
                         psz );

                /*---| display error text |---*/
                WinMessageBox( HWND_DESKTOP, hwnd,
                               szBuffer, "iODBCmgr Advanced Driver Setup", 0,
                               MB_OK | MB_ERROR | MB_MOVEABLE );

                /*---| return |---*/
                return;
        }

        /*
        ** call driver setup routine
        */
        if( SQL_SUCCESS != (rc=(*pSQLAllocEnv)(&pSqlEnv))               ||
            SQL_SUCCESS != (rc=(*pSQLAllocConnect)(pSqlEnv, &pSqlHdbc)) ||
            SQL_SUCCESS != (rc=(*pSQLDriverSetup)(pSqlHdbc, hwnd,
                                                  hModule, pDrvEntry))  ||
            SQL_SUCCESS != (rc=(*pSQLFreeConnect)(pSqlHdbc))            ||
            SQL_SUCCESS != (rc=(*pSQLFreeEnv)(pSqlEnv)) )
        {
                /*---| free module on errors |---*/
                DosFreeModule( hModule );

                /*---| create error message |---*/
                sprintf( szBuffer,
                         "Can't execute driver's setup on '%s' (rc=%d)\n",
                         psz, rc );

                /*---| display error text |---*/
                WinMessageBox( HWND_DESKTOP, hwnd,
                               szBuffer, "iODBCmgr Advanced Driver Setup", 0,
                               MB_OK | MB_ERROR | MB_MOVEABLE );

                /*---| return failure |---*/
                return -1;
        }


        /*
        ** unload driver
        */
        DosFreeModule( hModule );

        /*
        ** return
        */
        return;

} /* AdvancedDriverSetup() */

/*===| end of file |=========================================================*/
