/*
** dlg_main.c
**
** main dialog box of the iODBC driver manager for OS/2
**
** (c) 1997 by Dirk Ohme, all rights reserved
*/

/*---| includes |------------------------------------------------------------*/
#define INCL_PM
#define INCL_WIN
#include <os2.h>
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>

#include <sqlcli_.h>
#include "iODBCmgr.h"

/*---| local prototypes |----------------------------------------------------*/
static void FillListBox( HWND hwnd, SQLPDriverList pDrvList );
static void DelListBoxEntry( HWND hwnd, SQLPDriverList *ppDrvList, SHORT iDrvList );
static void AddListBoxEntry( HWND hwnd, SQLPDriverList *ppDrvList );
static void SetupListBoxEntry( HWND hwnd, SQLPDriverList pDrvList, SHORT iDrvList );

/*-----------------------------------------------------------------------------
 * name:     DlgMain
 * input:    HWND hwnd                 : handle for dialog box window
 *           ULONG msg                 : incoming message
 *           MPARAM mp1                : message parameter 1
 *           MPARAM mp2                : message parameter 2
 * output:   MRESULT EXPENTRY          : result for Presentation Manager
 * function: main dialog box with iODBC driver selection menu
 *-----------------------------------------------------------------------------
 */
MRESULT EXPENTRY DlgMain( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
        static SQLPDriverList   *ppDrvList = NULL;
        static SHORT             iDrvList  = 0;

        /*
        ** distinguish between the various incoming messages
        */
        switch( msg )
        {
                /*
                ** initialisation of a new dialog window
                */
                case WM_INITDLG:
                        /*---| get the pointer to pointer to |---*/
                        /*---| the anchor of the driver list |---*/
                        ppDrvList = (SQLPDriverList *) PVOIDFROMMP( mp2 );

                        /*---| set program version |---*/
                        WinSetDlgItemText( hwnd, WND_MAIN_VER, VERSION );

                        /*---| fill list box with data |---*/
                        FillListBox( hwnd, *ppDrvList );

                        /*---| select first entry of lsit box |---*/
                        WinSendDlgItemMsg( hwnd, WND_MAIN_LIST,
                                           LM_SELECTITEM,
                                           MPFROMSHORT(iDrvList),
                                           MPFROMSHORT(TRUE) );

                        /*---| return success |---*/
                        return FALSE;

                /*
                ** changes of control elements
                */
                case WM_CONTROL:
                        /*---| check for which control |---*/
                        /*---| element caused message  |---*/
                        switch( SHORT1FROMMP(mp1) )
                        {
                                /*---| list box activities |---*/
                                case WND_MAIN_LIST:
                                        /*---| get selected item index |---*/
                                        iDrvList = (SHORT)
                                                   WinSendDlgItemMsg( hwnd,
                                                        WND_MAIN_LIST,
                                                        LM_QUERYSELECTION,
                                                        0L, 0L );

                                        /*---| what action? |---*/
                                        switch( SHORT2FROMMP(mp1) )
                                        {
                                                /*---| selection |---*/
                                                case LN_SELECT:
                                                        return 0;

                                                /*---| double click |---*/
                                                case LN_ENTER:
                                                        /*---| setup |---*/
                                                        WinPostMsg( hwnd,
                                                                WM_COMMAND,
                                                                MPFROM2SHORT(WND_MAIN_SETUP, 0),
                                                                0 );
                                                        return 0;

                                                /*---| ???: unknown |---*/
                                                default:
                                                        break;
                                        }

                                        /*---| return success |---*/
                                        return 0;

                                /*---| ???: not one of ours |---*/
                                default:
                                        break;
                        }

                        /*---| break |---*/
                        break;

                /*
                ** commands for the dialog window procedure
                */
                case WM_COMMAND:
                        /*---| check command |---*/
                        switch( SHORT1FROMMP(mp1) )
                        {
                                /*---| 'Close' button pressed |---*/
                                case DID_OK:
                                        /*---| return, saving enabled |---*/
                                        WinDismissDlg( hwnd, TRUE );
                                        return 0;

                                /*---| 'Abort' button pressed |---*/
                                case DID_CANCEL:
                                        /*---| return, saving disabled |---*/
                                        WinDismissDlg( hwnd, FALSE );
                                        return 0;

                                /*---| 'Add' button pressed |---*/
                                case WND_MAIN_ADD:
                                        /*---| add new element to list |---*/
                                        AddListBoxEntry( hwnd, ppDrvList );

                                        /*---| fill list box with data |---*/
                                        FillListBox( hwnd, *ppDrvList );

                                        /*---| return to PM |---*/
                                        return 0;

                                /*---| 'Delete' button pressed |---*/
                                case WND_MAIN_DEL:
                                        /*---| perform deletion |---*/
                                        DelListBoxEntry( hwnd, ppDrvList, iDrvList );

                                        /*---| fill list box with data |---*/
                                        FillListBox( hwnd, *ppDrvList );

                                        /*---| return to PM |---*/
                                        return 0;

                                /*---| 'Setup' button pressed |---*/
                                case WND_MAIN_SETUP:
                                        /*---| perform setup |---*/
                                        SetupListBoxEntry( hwnd, *ppDrvList,
                                                           iDrvList );

                                        /*---| fill list box with data |---*/
                                        FillListBox( hwnd, *ppDrvList );

                                        /*---| select old entry |---*/
                                        WinSendDlgItemMsg( hwnd, WND_MAIN_LIST,
                                                           LM_SELECTITEM,
                                                           MPFROMSHORT(iDrvList),
                                                           MPFROMSHORT(TRUE) );

                                        /*---| return to PM |---*/
                                        return 0;

                                /*---| ???: not one of ours |---*/
                                default:
                                        break;
                        }

                        /*---| break |---*/
                        break;

                /*
                ** ???: unknown message
                */
                default:
                        break;

        }

        /*
        ** we can't handle this message? standard processing!
        */
        return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* DlgMain() */

/*-----------------------------------------------------------------------------
 * name:     FillListBox
 * input:    HWND hwnd                 : handle for dialog box window
 *           SQLPDriverList pDrvList   : pointer to start of driver list
 * output:   --nothing--
 * function: clears and redraws list box with driver entries
 *-----------------------------------------------------------------------------
 */
static void FillListBox( HWND hwnd, SQLPDriverList pDrvList )
{
        char             szBuffer[256],
                        *psz;
        SQLPDriverList   pCurr;
        SQLPDrvOptList   pCurrOpt;

        /*
        ** clear list box
        */
        WinSendDlgItemMsg( hwnd, WND_MAIN_LIST, LM_DELETEALL, NULL, NULL );

        /*
        ** re-fill list box
        */
        pCurr = pDrvList;
        while( NULL != pCurr )
        {
                /*---| check for options ... |---*/
                pCurrOpt = pCurr->pOptions;
                psz      = NULL;
                while( NULL != pCurrOpt )
                {
                        /*---| check for an comment |---*/
                        if( '\0' != pCurrOpt->szOptName &&
                              0  == strcmp(pCurrOpt->szOptName, "comment") )
                        {
                                /*---| store pointer to comment |---*/
                                psz = pCurrOpt->szOptValue;
                                break;
                        }

                        /*---| next entry |---*/
                        pCurrOpt = (SQLPDrvOptList) pCurrOpt->pNext;
                }

                /*---| create output string - watch for comments |---*/
                if( NULL == psz )
                        strcpy( szBuffer, pCurr->szName );
                else
                        sprintf( szBuffer, "%s - %s", pCurr->szName, psz );

                /*---| add string to end of list |---*/
                WinSendDlgItemMsg( hwnd, WND_MAIN_LIST, LM_INSERTITEM,
                                   MPFROM2SHORT(LIT_SORTASCENDING, 0),
                                   MPFROMP(szBuffer) );

                /*---| next entry |---*/
                pCurr = (SQLPDriverList) pCurr->pNext;
        }

} /* FillListBox() */

/*-----------------------------------------------------------------------------
 * name:     DelListBoxEntry
 * input:    HWND hwnd                 : handle for dialog box window
 *           SQLPDriverList *ppDrvList : pointer to anchor of driver list
 *           SHORT iDrvList            : idex of selected item
 * output:   --nothing--
 * function: deletes an entry from the driver list
 *-----------------------------------------------------------------------------
 */
static void DelListBoxEntry( HWND hwnd, SQLPDriverList *ppDrvList, SHORT iDrvList )
{
        char             szBuffer[256],
                        *psz;
        SQLPDriverList   pCurr,
                         pTmp;
        register ULONG   ulResponse;

        /*
        ** check for selection
        */
        if( NULL == ppDrvList || NULL == *ppDrvList || LIT_NONE == iDrvList )
                return;

        /*
        ** get string with driver name
        */
        WinSendDlgItemMsg( hwnd, WND_MAIN_LIST, LM_QUERYITEMTEXT,
                           MPFROM2SHORT(iDrvList, sizeof(szBuffer)),
                           MPFROMP(szBuffer) );

        /*
        ** get driver name, split comment
        */
        if( NULL != (psz=strstr(szBuffer, " - ")) )
                *psz = '\0';

        /*
        ** ask for deletion, abort if answer is 'no'
        */
        ulResponse = WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
                                    "Shall I really remove this driver"\
                                    " from the list?",
                                    szBuffer, 0,
                                    MB_ICONQUESTION | MB_YESNO |
                                    MB_DEFBUTTON2   | MB_MOVEABLE );

        if( MBID_YES != ulResponse )
                return;

        /*
        ** check for entry
        */
        pCurr = *ppDrvList;
        pTmp  = NULL;
        while( NULL != pCurr )
        {
                /*---| check |---*/
                if( '\0' != pCurr && 0 == strcmp(pCurr->szName, szBuffer) )
                        break;

                /*---| next entry |---*/
                pTmp  = pCurr;
                pCurr = (SQLPDriverList) pCurr->pNext;
        }

        /*
        ** display error, if not found, then abort
        */
        if( NULL == pCurr )
        {
                WinMessageBox( HWND_DESKTOP, HWND_DESKTOP,
                               "Driver could not be found in list!",
                               szBuffer, 0, MB_OK | MB_ERROR | MB_MOVEABLE );
                return;
        }

        /*
        ** check, if it is the anchor
        **      --> yes: change anchor
        **      --> no:  set new link
        */
        if( NULL == pTmp )
                *ppDrvList  = (SQLPDriverList) (*ppDrvList)->pNext;
        else
                pTmp->pNext = pCurr->pNext;

        /*
        ** remote entry
        */
        free( pCurr );

} /* DelListBoxEntry() */

/*-----------------------------------------------------------------------------
 * name:     AddListBoxEntry
 * input:    HWND hwnd                 : handle for dialog box window
 *           SQLPDriverList *ppDrvList : pointer to anchor of driver list
 * output:   --nothing--
 * function: adds a new entry to the driver list
 *-----------------------------------------------------------------------------
 */
static void AddListBoxEntry( HWND hwnd, SQLPDriverList *ppDrvList )
{
        SQLPDriverList   pTmp;

        /*
        ** create temporary entry
        */
        if( NULL == ppDrvList || NULL == (pTmp=malloc(sizeof(SQLDriverList))) )
                return;
        memset( pTmp, 0, sizeof(SQLDriverList) );

        /*
        ** run setup
        */
        if( !WinDlgBox( HWND_DESKTOP, hwnd,
                        DlgSetup,     NULLHANDLE,
                        WND_SETUP,    (PVOID) pTmp ) )
        {
                register SQLPDriverList  pCurr;
                register SQLPDrvOptList  pCurrOpt,
                                         pTmpOpt;

                /*---| abort -> do not store results |---*/
                pCurrOpt = pTmp->pOptions;
                while( NULL != pCurrOpt )
                {
                        /*---| free current option |---*/
                        pTmpOpt  = pCurrOpt;
                        pCurrOpt = (SQLPDrvOptList) pCurrOpt->pNext;
                        free( pTmpOpt );
                }

                /*---| free object |---*/
                free( pTmp );
                return;
        }

        /*---| add new driver to end of list |---*/
        if( NULL == *ppDrvList )
                *ppDrvList = pTmp;
        else
        {
                register SQLPDriverList  pCurr;

                /*---| step to end of list |---*/
                pCurr = *ppDrvList;
                while( NULL != pCurr->pNext )
                        pCurr = (SQLPDriverList) pCurr->pNext;

                /*---| add new entry |---*/
                pCurr->pNext = pTmp;
        }

} /* AddListBoxEntry() */

/*-----------------------------------------------------------------------------
 * name:     SetupListBoxEntry
 * input:    HWND hwnd                 : handle for dialog box window
 *           SQLPDriverList *ppDrvList : pointer to anchor of driver list
 *           SHORT iDrvList            : idex of selected item
 * output:   --nothing--
 * function: setup an entry from the driver list
 *-----------------------------------------------------------------------------
 */
static void SetupListBoxEntry( HWND hwnd, SQLPDriverList pDrvList, SHORT iDrvList )
{
        char             szBuffer[256],
                        *psz;

        /*
        ** check for selection
        */
        if( NULL == pDrvList || LIT_NONE == iDrvList )
                return;

        /*
        ** get string with driver name
        */
        WinSendDlgItemMsg( hwnd, WND_MAIN_LIST, LM_QUERYITEMTEXT,
                           MPFROM2SHORT(iDrvList, sizeof(szBuffer)),
                           MPFROMP(szBuffer) );

        /*
        ** get driver name, split comment
        */
        if( NULL != (psz=strstr(szBuffer, " - ")) )
                *psz = '\0';

        /*
        ** get entry
        */
        while( NULL != pDrvList )
        {
                /*---| check |---*/
                if( '\0' != pDrvList &&
                      0 == strcmp(pDrvList->szName, szBuffer) )
                        break;

                /*---| next entry |---*/
                pDrvList = (SQLPDriverList) pDrvList->pNext;
        }

        /*
        ** setup or abort, if not found
        */
        if( NULL == pDrvList )
                return 0;
        WinDlgBox( HWND_DESKTOP, hwnd, DlgSetup, NULLHANDLE,
                   WND_SETUP,  (PVOID) pDrvList );

} /* SetupListBoxEntry() */

/*===| end of file |=========================================================*/
