/*
 * MultiMail offline mail reader
 * some low-level routines common to both sides

 Copyright (c) 2000 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

/* Most non-ANSI, non-curses stuff is here. */

#include "interfac.h"
#include "error.h"

extern "C" {
#ifndef USE_FINDFIRST
#include <dirent.h>
#endif

#ifdef USE_SPAWNO
# include <spawno.h>
#endif

#ifdef LIMIT_MEM
# include <alloc.h>
#endif

#ifdef HAS_UNISTD
# include <unistd.h>
#endif

#ifdef USE_DIRH
# include <dir.h>
#endif

#include <sys/stat.h>

#ifdef HAS_UNAME
# include <sys/utsname.h>
#endif

#ifdef __EMX__
int _chdir2(const char *);
char *_getcwd2(char *, int);
#endif
}

#ifndef S_IREAD
# define S_IREAD S_IRUSR
#endif

#ifndef S_IWRITE
# define S_IWRITE S_IWUSR
#endif

#ifndef USE_FINDFIRST
static DIR *Dir;
#endif

void fatalError(const char *description);

char *myfgets(char *s, size_t size, FILE *stream)
{
	char *end = 0;

	if (!feof(stream) && fgets(s, size, stream)) {

		end = s + strlen(s) - 1;

		// Skip any leftovers:
		if (*end != '\n')
			while (!feof(stream) && (fgetc(stream) != '\n'));

	}
	return end;
}

int mysystem(const char *cmd)
{
	if (ui) {
		if (!isendwin())
			endwin();
#ifdef PDCURSKLUDGE
		// Restore original cursor
		PDC_set_cursor_mode(curs_start, curs_end);
#endif
	}

#ifdef USE_SPAWNO
	const char *tmp = getenv("TEMP");

	if (!tmp) {
		tmp = getenv("TMP");
		if (!tmp)
			tmp = error.getOrigDir();
	}

	int result = mm.resourceObject->getInt(swapOut) ?
		systemo(tmp, cmd) : -1;

	if (-1 == result)
		result = system(cmd);
#else
	int result = system(cmd);
#endif

#ifdef HAS_SLEEP
	// Non-zero result = error; pause so it can (maybe) be read
	if (result)
		sleep(2);
#endif

	if (ui) {
		keypad(stdscr, TRUE);
#ifdef NOTYPEAHEAD
		typeahead(-1);
#endif
	}

	return result;
}

void mytmpnam(char *name)
{
/* EMX, RSX/NT and Borland/Turbo C++ don't return an absolute pathname
   from tmpnam(), so we create one ourselves. Otherwise, use the system's
   version, and make sure it hasn't run out of names.
*/
#ifdef TEMP_RELATIVE
	const char *tmp = getenv("TEMP");

	if (!tmp) {
		tmp = getenv("TMP");
		if (!tmp)
			tmp = error.getOrigDir();
	}
	char end = tmp[strlen(tmp) - 1];
	sprintf(name, (('/' == end) || ('\\' == end)) ? "%s%s" : "%s/%s",
		tmp, tmpnam(0));
#else
	if (!tmpnam(name))
		fatalError("Out of temporary filenames");
#endif
}

void edit(const char *reply_filename)
{
        char command[512];

        sprintf(command, "%.255s %.255s", mm.resourceObject->get(editor),
                canonize(reply_filename));
        mysystem(command);
}

int mychdir(const char *pathname)
{
#ifdef USE_SETDISK
	if (':' == pathname[1])
		setdisk(toupper(pathname[0]) - 'A');
#endif
	return
#ifdef __EMX__
		_chdir2(pathname);
#else
		chdir(pathname);
#endif
}

int mymkdir(const char *pathname)
{
#ifdef HAS_UNISTD
	return mkdir(pathname, S_IRWXU);
#else
	return mkdir(pathname);
#endif
}

void myrmdir(const char *pathname)
{
	rmdir(pathname);
}

void mygetcwd(char *pathname)
{
#ifdef __EMX__
	_getcwd2(pathname, 255);
#else
	getcwd(pathname, 255);
#endif
}

// system name -- results of uname()
const char *sysname()
{
#ifdef HAS_UNAME
	static struct utsname buf;

	if (!buf.sysname[0])
		uname(&buf);

	return buf.sysname;
#else
# ifdef __WIN32__
	return "Win32";
# else
	return "XT";
# endif
#endif
}

bool myopendir(const char *dirname)
{
#ifdef USE_FINDFIRST
	return !mychdir(dirname);
#else
	return ((Dir = opendir((char *) dirname)) != 0) ?
		!mychdir(dirname) : false;
#endif
}

const char *myreaddir(mystat &st)
{
#ifdef USE_FINDFIRST
	static struct ffblk blk;
	static bool first = true;
	int result;

	if (first) {
		result = findfirst("*.*", &blk, FA_DIREC);
		first = false;
	} else
                result = findnext(&blk);

	if (result) {
		first = true;
		return 0;
	} else {
		st.init(blk.ff_fsize, ((long) blk.ff_ftime << 16) +
			(long) blk.ff_fdate, blk.ff_attrib);
		return blk.ff_name;
	}
#else
	static dirent *entry;
	const char *result = 0;

	entry = readdir(Dir);
	if (entry)
		result = entry->d_name;
	else
		closedir(Dir);

	if (result)
		st.init(result);
	return result;
#endif
}

void clearDirectory(const char *DirName)
{
	mystat st;
	const char *fname;

	if (myopendir(DirName))
		while ((fname = myreaddir(st)) != 0)
			if (!st.isdir())
				remove(fname);
}

#ifdef LIMIT_MEM

/* Constrain memory allocation according to maximum block size and free
   memory remaining. Currently used only in the 16-bit MS-DOS port.
*/

long limitmem(long wanted)
{
	long maxavail = (long) coreleft();

	// Give it a 25% margin
	maxavail -= (wanted >> 2);

	//if (maxavail > MAXBLOCK)
	//	maxavail = MAXBLOCK;

	if (wanted > maxavail)
		wanted = maxavail;

	return wanted;
}

#endif

#ifdef DOSNAMES

/* Convert pathnames to "canonical" form (change slashes to backslashes).
   The "nospace" stuff leaves any parameters unconverted.
   Don't call this twice in a row without first copying the result! D'oh!
*/

const char *canonize(const char *sinner)
{
	static char saint[256];
	int i;
	bool nospace = true;

	for (i = 0; sinner[i]; i++) {
		saint[i] = (nospace && (sinner[i] == '/')) ?
			'\\' : sinner[i];
		if (nospace && (saint[i] == ' '))
			nospace = false;
	}
	saint[i] = '\0';
	return saint;
}

#endif

#ifdef HAS_HOME

/* Recognize '~' as a substitute for the home directory path, on Unix-like
   systems.
*/

const char *homify(const char *raw)
{
	static const char *home = getenv("HOME");

	if (home && raw && (raw[0] == '~') &&
	    ((raw[1] == '/') || (raw[1] == '\0'))) {
		static char expanded[256];

		sprintf(expanded, "%s/%s", home, raw + 1);
		return expanded;
	} else
		return raw;
}

#endif

#ifdef USE_SHELL

/* Command shell routine -- currently only used in the DOSish ports */

Shell::Shell()
{
	const char *oldprompt = getenv("PROMPT");
	if (!oldprompt)
		oldprompt = "$p$g";

	int len = strlen(oldprompt) + 13;
	prompt = new char[len];

	sprintf(prompt, "PROMPT=%s[MM] ", oldprompt);
	putenv(prompt);
}

Shell::~Shell()
{
	delete[] prompt;
}

void Shell::out()
{
	mychdir(error.getOrigDir());
	touchwin(stdscr);
	refresh();
	mysystem(getenv("COMSPEC"));

	ui->redraw();
}

#endif

mystat::mystat(const char *fname)
{
	init(fname);
}

mystat::mystat()
{
	init();
}

bool mystat::init(const char *fname)
{
#ifdef USE_FINDFIRST
	struct ffblk blk;
	bool retval = !findfirst(fname, &blk, FA_DIREC);

	if (retval) {
		init(blk.ff_fsize, ((long) blk.ff_ftime << 16) +
			(long) blk.ff_fdate, blk.ff_attrib);
	} else
		init();
#else
	struct stat fileStat;
	bool retval = !stat((char *) fname, &fileStat);

	if (retval) {
		size = fileStat.st_size;
		date = fileStat.st_mtime;
		mode = fileStat.st_mode;
	} else
		init();
#endif
	return retval;
}

#ifdef USE_FINDFIRST

void mystat::init(long sizeA, long dateA, char ff_attrib)
{
	size = sizeA;
	date = mktime(getdostime(dateA));
	mode = S_IREAD | ((ff_attrib & FA_RDONLY) ? 0 : S_IWRITE) |
		((ff_attrib & FA_DIREC) ? S_IFDIR : 0);
}

#endif

void mystat::init()
{
	size = -1;
	date = (time_t) -1;
	mode = 0;
}

bool mystat::isdir()
{
	return !(!(mode & S_IFDIR));
}

bool mystat::readable()
{
	return !(!(mode & S_IREAD));
}

bool mystat::writeable()
{
	return !(!(mode & S_IWRITE));
}

off_t mystat::fsize()
{
	return size;
}

time_t mystat::fdate()
{
	return date;
}
