/*
 * MultiMail offline mail reader
 * message list

 Copyright (c) 1996 Kolossvary Tamas <thomas@tvnet.hu>
 Copyright (c) 1997 John Zero <john@graphisoft.hu>
 Copyright (c) 2000 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

#include "interfac.h"

LetterListWindow::LetterListWindow()
{
	lsorttype = mm.resourceObject->getInt(LetterSort);
}

void LetterListWindow::listSave()
{
	static const char *saveopts[] = { "Marked", "All", "This one",
		"Quit" };

	int marked = !(!mm.areaList->getNoOfMarked());

	int status = ui->WarningWindow("Save which?", saveopts +
		!marked, 3 + marked);
	if (status) {
		bool saveok = ui->letterwindow.Save(status);
		if ((status == 1) && saveok)
			Move(DOWN);
	}
}

void LetterListWindow::Next()
{
	do {
		Move(DOWN);
		mm.letterList->gotoActive(active);
	} while (mm.letterList->getRead() && ((active + 1) < NumOfItems()));
}

void LetterListWindow::FirstUnread()
{
	position = 0;
	active = 0;
}

void LetterListWindow::Prev()
{
	do {
		Move(UP);
		mm.letterList->gotoActive(active);
	} while (mm.letterList->getRead() && (active > 0));
}

int LetterListWindow::NumOfItems()
{
	return mm.letterList->noOfActive();
}

void LetterListWindow::oneLine(int i)
{
	mm.letterList->gotoActive(position + i);

	int st = mm.letterList->getStatus();

	char *p = list->lineBuf;
	p += sprintf(p, format, (st & MS_MARKED) ? 'M' : ' ',
	    (st & MS_REPLIED) ? '~' : ' ', (st & MS_READ) ? '*' : ' ',
		mm.letterList->getMsgNum(), mm.letterList->getFrom(),
		    mm.letterList->getTo(),
			stripre(mm.letterList->getSubject()));

	if (mm.areaList->isCollection()) {
		const char *origArea = mm.letterList->getNewsgrps();
		if (!origArea)
			origArea = mm.areaList->
			    getDescription(mm.letterList->getAreaID());
		if (origArea)
			sprintf(p - 15, " %-13.13s ", origArea);
	}

	coltype linecol = mm.letterList->isPersonal() ? C_LLPERSONAL :
		C_LISTWIN;

	letterconv_in(list->lineBuf);
	DrawOne(i, (st & MS_READ) ? noemph(linecol) : emph(linecol));
}

searchret LetterListWindow::oneSearch(int x, const char *item, int mode)
{
	const char *s;
	searchret retval;

	mm.letterList->gotoActive(x);

	s = searchstr(mm.letterList->getFrom(), item);
	if (!s) {
		s = searchstr(mm.letterList->getTo(), item);
		if (!s)
			s = searchstr(mm.letterList->getSubject(), item);
	}

	retval = s ? True : False;
	if (!s && (mode == s_fulltext)) {
		ui->changestate(letter);
		ui->letterwindow.setPos(-1);
		retval = ui->letterwindow.search(item);
		if (retval != True)
			ui->changestate(letterlist);
	}

	return retval;
}

void LetterListWindow::MakeActive()
{
	static const char *llmodes[] = {"All", "Unread", "Marked"},
		*llsorts[] = {"subject", "number", "from", "to"};
	char *topline, tformat[50];
	int tot, maxFromLen, maxToLen, maxSubjLen;

	topline = new char[COLS + 1];

	tot = COLS - 19;
	maxSubjLen = tot / 2;
	tot -= maxSubjLen;
	maxToLen = tot / 2;
	maxFromLen = tot - maxToLen;

	if (!mm.areaList->hasTo() || (mm.areaList->isCollection() &&
	    !mm.areaList->isReplyArea())) {
		maxSubjLen += maxToLen;
		maxToLen = 0;
	}

	if (mm.areaList->isReplyArea()) {
		maxSubjLen += maxFromLen;
		maxFromLen = 0;
	}

	sprintf(format, "%%c%%c%%c%%6ld  %%-%d.%ds %%-%d.%ds %%-%d.%ds",
		maxFromLen, maxFromLen, maxToLen, maxToLen,
			maxSubjLen, maxSubjLen);

	ui->areas.Select();

	list_max_y = (NumOfItems() < LINES - 11) ? NumOfItems() : LINES - 11;
	list_max_x = COLS - 6;
	top_offset = 2;

	bool too_many = (NumOfItems() > list_max_y);

	const char *modestr = llmodes[mm.letterList->getMode()];
	const char *sortstr = llsorts[lsorttype];
	int offset = strlen(modestr);
	int nwidth = COLS - (too_many ? 32 : 24) - offset -
		strlen(sortstr);

	sprintf(tformat, "%%s msgs in %%.%ds", nwidth);
	char *end = topline + sprintf(topline, tformat,
		modestr, mm.areaList->getDescription());

	char *newend = end + sprintf(end, ", by %s", sortstr);
	if (too_many)
		sprintf(newend, " (%d)", NumOfItems());

	areaconv_in(topline);

	borderCol = C_LLBBORD;

	list = new InfoWin(list_max_y + 3, COLS - 4, 2, borderCol,
		topline, C_LLTOPTEXT1);

	list->attrib(C_LLTOPTEXT2);

	*end = '\0';
	offset += 8;
	list->put(0, offset + 3, topline + offset);

	sprintf(tformat, "   Msg#  %%-%d.%ds %%-%d.%ds %%-%d.%ds",
		maxFromLen, maxFromLen, maxToLen, maxToLen,
			maxSubjLen, maxSubjLen);
	list->attrib(C_LLHEAD);
	sprintf(topline, tformat, "From", "To", "Subject");
	list->put(1, 3, topline);

	delete[] topline;

	if (mm.areaList->isCollection())
		list->put(1, COLS - 19, "Area");
	list->touch();

	DrawAll();
	Select();
}

void LetterListWindow::Select()
{
	mm.letterList->gotoActive(active);
}

void LetterListWindow::ResetActive()
{
        active = mm.letterList->getActive();
}

void LetterListWindow::Delete()
{
	delete list;
}

bool LetterListWindow::extrakeys(int key)
{
	Select();
	switch (key) {
	case 'U':
	case 'M':	// Toggle read/unread and marked from letterlist
		mm.letterList->setStatus(mm.letterList->getStatus() ^
			((key == 'U') ? MS_READ : MS_MARKED));
		ui->setAnyRead();
		Move(DOWN);
		Draw();
		break;
	case 'E':
		if (mm.areaList->isReplyArea())
			ui->letterwindow.KeyHandle('E');
		else
		    if (!(mm.areaList->getType() & (COLLECTION | READONLY))) {
			    if (mm.areaList->isEmail())
				ui->addressbook();
			    ui->letterwindow.set_Letter_Params(
				mm.areaList->getAreaNo(), 'E');
			    ui->letterwindow.EnterLetter();
		    } else
			    ui->nonFatalError("Cannot reply there");
		break;
	case 2:
	case 6:
	case MM_DEL:
	case 'K':
		if (mm.areaList->isReplyArea())
			ui->letterwindow.KeyHandle(key);
		break;
	case 'L':
		mm.letterList->relist();
		ResetActive();
		ui->redraw();
		break;
	case '$':
		mm.letterList->resort();
		//DrawAll();
		ui->redraw();
		break;
	case 'S':
		listSave();
		ui->redraw();
	}
	return false;
}
