/*
 *
 * literal.c
 * Translate a string with tokens in it according to several conventions:
 * 1.  standard c comment delineators
 * 2.  \x1b translates to char(0x1b)
 * 3.  \27  translates to char(27)
 * 4.  \"   translates to "
 * 5.  \'   translates to '
 * 6.  \\   translates to \
 * 7.  \r   translates to carriage return
 * 8.  \n   translates to linefeed
 * 9.  \b   translates to backspace
 * 10. \t   translates to tab
 * 11. \a   translates to bell
 *
 * Synopsis
 *    *s = "this\x20is\32a test of \\MSC\\CSM\7"
 *    literal(s);
 *
 *    ( s now equals "this is a test of \MSC\CSM")
 */

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#define HEX "0123456789ABCDEF"
#define DEC "0123456789"

int _fastcall index (char *s,char c);


char * _fastcall literal (char *fsource) {

  int wpos,w,wincomment,x;
  char *fdestin,*freeme,wchar;


  if(!fsource || !*fsource) return fsource;
  x = strlen(fsource) + 1;
  freeme = fdestin = malloc(x + 1);
  wincomment = 0;
  memset(fdestin,0,x);  /* start out with NULL string */

  w = 0;            /* set index to first character */
  while (fsource[w]) {      /* until end of string */
    switch(fsource[w]) {
      case '\\':
       if(!wincomment)
        switch(fsource[w + 1]) {
          case 'x' :                    /* Hexadecimal */
            wchar = 0;
            w += 2;            /* get past "\x" */
            if (index(HEX,(char)toupper(fsource[w])) != -1) {
              while ((wpos = index(HEX,(char)toupper(fsource[w]))) != -1) {
                wchar = (char)(wchar << 4) + (char)wpos;
                w++;
              }
            }
            else
              wchar = 'x';       /* just an x */
            w--;
            *fdestin++ = wchar;
            break;

          case '\\' :                 /* we want a "\" */
            w++;
            *fdestin++ = '\\';
            break;

          case 't' :                  /* tab char */
            w++;
             *fdestin++ = '\t';
            break;

          case 'n' :                  /* new line */
            w++;
            *fdestin++ = 10;
            break;

          case 'r' :                  /* carr return */
            w++;
            *fdestin++ = '\r';
            break;

          case 'b' :                  /* back space */
            w++;
            *fdestin++ = '\b';
            break;

          case 'a':                   /* bell */
            w++;
            *fdestin++ = '\07';
            break;

          case '\'' :                 /* single quote */
            w++;
            *fdestin++ = '\'';
            break;

          case '\"' :                 /* double quote */
            w++;
            *fdestin++ = '\"';
            break;

          default :             /* DECIMAL */
            w++;           /* get past "\" */
            wchar = 0;
            if (index(DEC,fsource[w]) != -1) {
              do {
                  wchar = (char)(wchar * 10 + (fsource[w++] - 48)); /* cnv to binary */
              } while (index(DEC,fsource[w]) != -1);
              w--;
            }
            else wchar = fsource[w];
            *fdestin ++ = wchar;
            break;
        }
        break;

      case '*' :
        if(wincomment) {
          if (fsource[w + 1] == '/') {
              wincomment--;            /* toggle the flag */
              w++;
          }
        }
        else *fdestin++ = fsource[w];
        break;

      case '/' :              /* beginning of comment perhaps */
        if(fsource[w + 1] == '*') {
            wincomment++;             /* toggle the flag */
            w++;
        }

      default :
        if(!wincomment) {
            *fdestin++ = fsource[w];
        }
        break;
   }
   w++;
  }
  *fdestin = 0;         /* terminate the string */

  strcpy(fsource,freeme);  /* swap 'em */
  free(freeme);

  return fsource;
}

int _fastcall index (char *s,char c) {

    char *p;

    p = strchr(s,c);
    if(p == NULL || !*p) return -1;
    return (int)(p - s);
}
