#include "doorskel.h"
#include "hydra.h"


#define H_DEBUG 0

/* HYDRA Some stuff to aid readability of the source and prevent typos ----- */
#define h_updcrc16(crc,c)  (crc16tab[(       crc ^ (c)) & 0xff] ^ ((crc >> 8) & 0x00ff))
#define h_updcrc32(crc,c)  (crc32tab[((char) crc ^ (c)) & 0xff] ^ ((crc >> 8) & 0x00ffffffL))
#define h_crc16poly        (0x8408)
#define h_crc32poly        (0xedb88320L)
#define h_crc16test(crc)   (((crc) == 0xf0b8     ) ? 1 : 0)
#define h_crc32test(crc)   (((crc) == 0xdebb20e3L) ? 1 : 0)
#define h_uuenc(c)         (((c) & 0x3f) + '!')
#define h_uudec(c)         (((c) - '!') & 0x3f)
#define h_long1(buf)       (*((long *) (buf)))
#define h_long2(buf)       (*((long *) ((buf) + ((int) sizeof (long)))))
#define h_long3(buf)       (*((long *) ((buf) + (2 * ((int) sizeof (long))))))
typedef long               h_timer;
#define h_timer_set(t)     (time(NULL) + (t))
#define h_timer_running(t) (t != 0L)
#define h_timer_expired(t) (time(NULL) > (t))
#define h_timer_reset()    (0L)

typedef struct {
  boolean originator;                     /* are we the orig side?     */
  int     batchesdone;                    /* No. HYDRA batches done    */
  boolean hdxlink;                        /* hdx link & not orig side  */
  dword   options;                        /* INIT options Hydra_init() */
  word    timeout;                        /* general timeout in secs   */
  char   *txbuf,         *rxbuf;          /* packet buffers            */
  dword   txoptions,      rxoptions;      /* HYDRA options (INIT seq)  */
  char    txpktprefix[H_PKTPREFIX + 1];   /* pkt prefix str they want  */
  long    txwindow,       rxwindow;       /* window size (0=streaming) */
  h_timer                 braindead;      /* braindead timer           */
  char   *txbufin;                        /* read data from disk here  */
  char    txlastc;                        /* last char put in txbuf    */
  char                    rxdle;          /* count of received H_DLEs  */
  char                    rxpktformat;    /* format of pkt receiving   */
  char                   *rxbufptr;       /* current position in rxbuf */
  char                   *rxbufmax;       /* highwatermark of rxbuf    */
  char    txfname[13],    rxfname[13];    /* fname of current files    */
  char                   *rxpathname;     /* pointer to rx pathname    */
  long    txftime,        rxftime;        /* file timestamp (UNIX)     */
  long    txfsize,        rxfsize;        /* file length               */
  int     txfd,           rxfd;           /* file handles              */
  word                    rxpktlen;       /* length of last packet     */
  word                    rxblklen;       /* len of last good data blk */
  char    txstate,        rxstate;        /* xmit/recv states          */
  long    txpos,          rxpos;          /* current position in files */
  word    txblklen;                       /* length of last block sent */
  word    txmaxblklen;                    /* max block length allowed  */
  long    txlastack;                      /* last dataack received     */
  long    txstart,        rxstart;        /* time we started this file */
  long    txoffset,       rxoffset;       /* offset in file we begun   */
  h_timer txtimer,        rxtimer;        /* retry timers              */
  word    txretries,      rxretries;      /* retry counters            */
  long                    rxlastsync;     /* filepos last sync retry   */
  long    txsyncid,       rxsyncid;       /* id of last resync         */
  word    txgoodneeded;                   /* to send before larger blk */
  word    txgoodbytes;                    /* no. sent at this blk size */
} PRIVHYDRADATA;

PRIVHYDRADATA *pvhd[MAXINSTANCES];
PUBHYDRADATA  *pbhd[MAXINSTANCES];

/* HYDRA's memory ---------------------------------------------------------- */
static  char    abortstr[] = { 24,24,24,24,24,24,24,24,8,8,8,8,8,8,8,8,8,8,0 };
static  char   *hdxmsg     = "Fallback to one-way xfer";
static  char   *pktprefix  = "";
static  char   *autostr    = "Hydra\r";
static  word   *crc16tab;                       /* CRC-16 table              */
static  dword  *crc32tab;                       /* CRC-32 table              */

struct _h_flags {
  char  *str;
  dword  val;
};

static struct _h_flags h_flags[] = {
  { "XON", HOPT_XONXOFF },
  { "TLN", HOPT_TELENET },
  { "CTL", HOPT_CTLCHRS },
  { "HIC", HOPT_HIGHCTL },
  { "HI8", HOPT_HIGHBIT },
  { "BRK", HOPT_CANBRK  },
  { "ASC", HOPT_CANASC  },
  { "UUE", HOPT_CANUUE  },
  { "C32", HOPT_CRC32   },
  { "DEV", HOPT_DEVICE  },
  { "FPT", HOPT_FPT     },
  { NULL , 0x0L         }
};


int DoHydra (USHORT cp,char *filespec,char *alias,long opts) {

  int            *save,oldx,oldy,ret = 0;
  char            middle[MAXDIR + 2];
  word            drive;
  char            dir[MAXDIR];

  save = malloc(DoorGetMaxx() * DoorGetMaxy() * sizeof(short));
  if(!save)
    return -1;
  drive = DoorGetDrive();
  getcwd (dir, MAXDIR);
  DoorCurrCursor(&oldx,&oldy);
  DoorReadScreen(save);
  DoorDeinitComm();
  DoorClearWindow(0,0,DoorGetRealMaxx() - 1,DoorGetRealMaxy() - 1,7);
  DoorCursor(1,1);

  ret = setjmp(Hydrajmpbuf);
  if(!ret) {

    char        path[HYDRAPATHLEN];
    char       *p;
    char        doafter;
    FILE       *fd;
    DOORFINDBUF f;
    int         sh = HDIR_SYSTEM;

    Hydra_init(opts);

    switch (*filespec) {
      case '^':
      case '~': doafter = *filespec++;
                break;
      default:  doafter = 0;
                break;
    }

    Hydrasplitpath(filespec,path,Hydrawork);
    if(!DoorFindFirst(&sh,filespec,&f,
                      FILE_NORMAL | FILE_ARCHIVED | FILE_READONLY)) {
      do {
        p = f.achName;
        Hydramergepath(Hydrawork,path,p);
        switch (Hydra(Hydrawork,alias)) {
          case XFER_ABORT:
            return (1);

          case XFER_SKIP:
            break;

          case XFER_OK:
            switch (doafter) {
              case '^':  /* Delete */
                if (unlink(Hydrawork))
                  Hydramessage(6,"!HSEND: Could not delete %s",Hydrawork);
                else
                  Hydramessage(1,"+Sent-H deleted %s",Hydrawork);
                break;

              case '~':  /* Truncate */
                if((fd = DoorFsopen(Hydrawork,"wt",SH_DENYNO)) == NULL) {
                  Hydramessage(6,"!HSEND: Error truncating %s",Hydrawork);
                  break;
                }
                fclose(fd);
                Hydramessage(1,"+Sent-H truncated %s",Hydrawork);
                break;

              default:
                Hydramessage(1,"+Sent-H %s",Hydrawork);
                break;
            }
            break;
        }
      } while(!DoorFindNext(sh,&f));
      DoorFindClose(sh);
    }
  }
  Hydra_deinit();

  DoorSetDrive(drive);
  strcpy (middle, "\\");
  strcat (middle, dir);
  chdir (middle);
  /* in case video mode changed */
  DoorSetScreensize(DoorGetNumStatLines());
  DoorANSIPuts("\x1b[0m\x1b[2J");
  DoorPrintStat ();
  DoorWriteScreen(save);
  free(save);
  DoorANSIPrintf("\x1b[%d;%dH",oldy,oldx);
  return ret;
}


void Hydraendprog (int x) {

  longjmp(Hydrajmpbuf,x);
}


/*---------------------------------------------------------------------------*/
static void Hydra_msgdev (char *data, word len)
{       /* text is already NUL terminated by calling func Hydra_devrecv() */
        len = len;
        Hydramessage(3,"*HMSGDEV: %s",data);
}/*Hydra_msgdev()*/


/*---------------------------------------------------------------------------*/
static  word    devtxstate;                     /* dev xmit state            */
static  h_timer devtxtimer;                     /* dev xmit retry timer      */
static  word    devtxretries;                   /* dev xmit retry counter    */
static  long    devtxid,        devrxid;        /* id of last devdata pkt    */
static  char    devtxdev[H_FLAGLEN + 1];        /* xmit device ident flag    */
static  char   *devtxbuf;                       /* ptr to usersupplied dbuf  */
static  word    devtxlen;                       /* len of data in xmit buf   */

struct _h_dev {
        char  *dev;
        void (*func) (char *data, word len);
};

static  struct _h_dev h_dev[] = {
  { "MSG", Hydra_msgdev },                /* internal protocol msg     */
  { "CON", NULL         },                /* text to console (chat)    */
  { "PRN", NULL         },                /* data to printer           */
  { "ERR", NULL         },                /* text to error output      */
  { NULL , NULL         }
};


/*---------------------------------------------------------------------------*/
boolean Hydra_devfree (void)
{
  if (devtxstate || !(txoptions & HOPT_DEVICE) || txstate >= HTX_END)
     return (false);                      /* busy or not allowed       */
  else
     return (true);                       /* allowed to send a new pkt */
}/*Hydra_devfree()*/


/*---------------------------------------------------------------------------*/
boolean Hydra_devsend (char *dev, char *data, word len)
{
  if (!dev || !data || !len || !Hydra_devfree())
     return (false);

  strncpy(devtxdev,dev,H_FLAGLEN);
  devtxdev[H_FLAGLEN] = '\0';
  strupr(devtxdev);
  devtxbuf = data;
  devtxlen = (len > H_MAXBLKLEN) ? H_MAXBLKLEN : len;

  devtxid++;
  devtxtimer   = h_timer_reset();
  devtxretries = 0;
  devtxstate   = HTD_DATA;

  /* special for chat, only prolong life if our side keeps typing! */
  if (Hydrachattimer > 0L && !strcmp(devtxdev,"CON") && txstate == HTX_REND)
     braindead = h_timer_set(H_BRAINDEAD);

  return (true);
}/*Hydra_devsend()*/


/*---------------------------------------------------------------------------*/
boolean Hydra_devfunc (char *dev, void (*func) (char *data, word len))
{
  register int i;

  for (i = 0; h_dev[i].dev; i++) {
      if (!strnicmp(dev,h_dev[i].dev,H_FLAGLEN)) {
         h_dev[i].func = func;
         return (true);
      }
  }

  return (false);
}/*Hydra_devfunc()*/


/*---------------------------------------------------------------------------*/
static void Hydra_devrecv (void)
{
  register char *p = (char *) rxbuf;
  register int   i;
  word len = rxpktlen;

  p += (int) sizeof (long);                       /* skip the id long  */
  len -= (int) sizeof (long);
  for (i = 0; h_dev[i].dev; i++) {                /* walk through devs */
      if (!strncmp(p,h_dev[i].dev,H_FLAGLEN)) {
         if (h_dev[i].func) {
            len -= ((int) strlen(p)) + 1;         /* sub devstr len    */
            p += ((int) strlen(p)) + 1;           /* skip devtag       */
            p[len] = '\0';                        /* NUL terminate     */
            (*h_dev[i].func)((char *) p,len);     /* call output func  */
         }
         break;
      }
  }
}/*Hydra_devrecv()*/


/*---------------------------------------------------------------------------*/
static void put_flags (char *buf, struct _h_flags flags[], long val)
{
  register char *p;
  register int   i;

  p = buf;
  for (i = 0; flags[i].val; i++) {
      if (val & flags[i].val) {
         if (p > buf) *p++ = ',';
         strcpy(p,flags[i].str);
         p += H_FLAGLEN;
      }
  }
  *p = '\0';
}/*put_flags()*/


/*---------------------------------------------------------------------------*/
static dword get_flags (char *buf, struct _h_flags flags[])
{
  register dword  val;
  register char  *p;
  register int    i;

  val = 0x0L;
  for (p = strtok(buf,","); p; p = strtok(NULL,",")) {
      for (i = 0; flags[i].val; i++) {
          if (!strcmp(p,flags[i].str)) {
             val |= flags[i].val;
             break;
          }
      }
  }

  return (val);
}/*get_flags()*/


/*---------------------------------------------------------------------------*/
static word crc16block (register char *buf, register word len)
{
  register word crc;

  for (crc = 0xffff; len > 0; len--)
      crc = h_updcrc16(crc,*buf++);
  return (crc);
}/*crc16block()*/


/*---------------------------------------------------------------------------*/
static dword crc32block (register char *buf, register word len)
{
  register dword crc;

  for (crc = 0xffffffffL; len > 0; len--)
      crc = h_updcrc32(crc,*buf++);
  return (crc);
}/*crc32block()*/


/*---------------------------------------------------------------------------*/
static char *put_binbyte (register char *p, register char c)
{
  register char n;

  n = c;
  if (txoptions & HOPT_HIGHCTL)
     n &= 0x7f;

  if (n == H_DLE ||
      ((txoptions & HOPT_XONXOFF) && (n == XON || n == XOFF)) ||
      ((txoptions & HOPT_TELENET) && n == '\r' && txlastc == '@') ||
      ((txoptions & HOPT_CTLCHRS) && (n < 32 || n == 127))) {
     *p++ = H_DLE;
     c ^= 0x40;
  }

  *p++ = c;
  txlastc = n;

  return (p);
}/*put_binbyte()*/


/*---------------------------------------------------------------------------*/
static void txpkt (register word len, int type)
{
  register char *in, *out;
  register word  c, n;
  boolean crc32 = false;
  char    format;
  static char hexdigit[] = "0123456789abcdef";

  txbufin[len++] = type;

  switch (type) {
         case HPKT_START:
         case HPKT_INIT:
         case HPKT_INITACK:
         case HPKT_END:
         case HPKT_IDLE:
              format = HCHR_HEXPKT;
              break;

         default:
              /* COULD do smart format selection depending on data and options! */
              if (txoptions & HOPT_HIGHBIT) {
                 if ((txoptions & HOPT_CTLCHRS) && (txoptions & HOPT_CANUUE))
                    format = HCHR_UUEPKT;
                 else if (txoptions & HOPT_CANASC)
                    format = HCHR_ASCPKT;
                 else
                    format = HCHR_HEXPKT;
              }
              else
                 format = HCHR_BINPKT;
              break;
  }

  if (format != HCHR_HEXPKT && (txoptions & HOPT_CRC32))
     crc32 = true;

#if H_DEBUG
if (Hydraloglevel==0) {
   char *s1, *s2, *s3, *s4;

   Hydramessage(0," -> PKT (format='%c'  type='%c'  crc=%d  len=%d)",
             format, type, crc32 ? 32 : 16, len - 1);

   switch (type) {
          case HPKT_START:    Hydramessage(0,"    <autostr>START");
                              break;
          case HPKT_INIT:     s1 = ((char *) txbufin) + ((int) strlen((char *) txbufin)) + 1;
                              s2 = s1 + ((int) strlen(s1)) + 1;
                              s3 = s2 + ((int) strlen(s2)) + 1;
                              s4 = s3 + ((int) strlen(s3)) + 1;
                              Hydramessage(0,"    INIT (appinfo='%s'  can='%s'  want='%s'  options='%s'  pktprefix='%s')",
                                      (char *) txbufin, s1, s2, s3, s4);
                              break;
          case HPKT_INITACK:  Hydramessage(0,"    INITACK");
                              break;
          case HPKT_FINFO:    Hydramessage(0,"    FINFO (%s)",txbufin);
                              break;
          case HPKT_FINFOACK: if (rxfd >= 0) {
                                 if (rxpos > 0L) s1 = "RES";
                                 else            s1 = "BOF";
                              }
                              else if (rxpos == -1L) s1 = "HAVE";
                              else if (rxpos == -2L) s1 = "SKIP";
                              else                   s1 = "EOB";
                              Hydramessage(0,"    FINFOACK (pos=%ld %s  rxstate=%d  rxfd=%d)",
                                      rxpos,s1,rxstate,rxfd);
                              break;
          case HPKT_DATA:     Hydramessage(0,"    DATA (ofs=%ld  len=%d)",
                                      Hydraintell(h_long1(txbufin)), len - 5);
                              break;
          case HPKT_DATAACK:  Hydramessage(0,"    DATAACK (ofs=%ld)",
                                      Hydraintell(h_long1(txbufin)));
                              break;
          case HPKT_RPOS:     Hydramessage(0,"    RPOS (pos=%ld%s  blklen=%ld  syncid=%ld)",
                                      rxpos, rxpos < 0L ? " SKIP" : "",
                                      Hydraintell(h_long2(txbufin)), rxsyncid);
                              break;
          case HPKT_EOF:      Hydramessage(0,"    EOF (ofs=%ld%s)",
                                      txpos, txpos < 0L ? " SKIP" : "");
                              break;
          case HPKT_EOFACK:   Hydramessage(0,"    EOFACK");
                              break;
          case HPKT_IDLE:     Hydramessage(0,"    IDLE");
                              break;
          case HPKT_END:      Hydramessage(0,"    END");
                              break;
          case HPKT_DEVDATA:  Hydramessage(0,"    DEVDATA (id=%ld  dev='%s'  len=%u)",
                                      devtxid, devtxdev, devtxlen);
                              break;
          case HPKT_DEVDACK:  Hydramessage(0,"    DEVDACK (id=%ld)",
                                      Hydraintell(h_long1(rxbuf)));
                              break;
          default:            /* This couldn't possibly happen! ;-) */
                              break;
   }
}
#endif

  if (crc32) {
     dword crc = ~crc32block(txbufin,len);

     txbufin[len++] = crc;
     txbufin[len++] = crc >> 8;
     txbufin[len++] = crc >> 16;
     txbufin[len++] = crc >> 24;
  }
  else {
     word crc = ~crc16block(txbufin,len);

     txbufin[len++] = crc;
     txbufin[len++] = crc >> 8;
  }

  in = txbufin;
  out = txbuf;
  txlastc = 0;
  *out++ = H_DLE;
  *out++ = format;

  switch (format) {
         case HCHR_HEXPKT:
              for (; len > 0; len--, in++) {
                  if (*in & 0x80) {
                     *out++ = '\\';
                     *out++ = hexdigit[((*in) >> 4) & 0x0f];
                     *out++ = hexdigit[(*in) & 0x0f];
                  }
                  else if (*in < 32 || *in == 127) {
                     *out++ = H_DLE;
                     *out++ = (*in) ^ 0x40;
                  }
                  else if (*in == '\\') {
                     *out++ = '\\';
                     *out++ = '\\';
                  }
                  else
                     *out++ = *in;
              }
              break;

         case HCHR_BINPKT:
              for (; len > 0; len--)
                  out = put_binbyte(out,*in++);
              break;

         case HCHR_ASCPKT:
              for (n = c = 0; len > 0; len--) {
                  c |= ((*in++) << n);
                  out = put_binbyte(out,c & 0x7f);
                  c >>= 7;
                  if (++n >= 7) {
                     out = put_binbyte(out,c & 0x7f);
                     n = c = 0;
                  }
              }
              if (n > 0)
                 out = put_binbyte(out,c & 0x7f);
              break;

         case HCHR_UUEPKT:
              for ( ; len >= 3; in += 3, len -= 3) {
                  *out++ = h_uuenc(in[0] >> 2);
                  *out++ = h_uuenc(((in[0] << 4) & 0x30) | ((in[1] >> 4) & 0x0f));
                  *out++ = h_uuenc(((in[1] << 2) & 0x3c) | ((in[2] >> 6) & 0x03));
                  *out++ = h_uuenc(in[2] & 0x3f);
              }
              if (len > 0) {
                 *out++ = h_uuenc(in[0] >> 2);
                 *out++ = h_uuenc(((in[0] << 4) & 0x30) | ((in[1] >> 4) & 0x0f));
                 if (len == 2)
                    *out++ = h_uuenc((in[1] << 2) & 0x3c);
              }
              break;
  }

  *out++ = H_DLE;
  *out++ = HCHR_PKTEND;

  if (type != HPKT_DATA && format != HCHR_BINPKT) {
     *out++ = '\r';
     *out++ = '\n';
  }

  for (in = (char *) txpktprefix; *in; in++) {
      switch (*in) {
             case 221: /* transmit break signal for one second */
                       break;
             case 222: { h_timer t = h_timer_set(2);
                         while (!h_timer_expired(t))
                               DosSleep(1L);
                       }
                       break;
             case 223: DoorComWriteByte(0);
                       break;
             default:  DoorComWriteByte(*in);
                       break;
      }
  }

  DoorComWriteBlk(txbuf,(int) (out - txbuf));
}/*txpkt()*/


/*---------------------------------------------------------------------------*/
static int rxpkt (void)
{
  register char *p, *q;
  register int   c, n, i;

  if (Hydrakeyabort())
     return (H_SYSABORT);
  if (!DoorComOnLine())
     return (H_CARRIER);

  p = rxbufptr;

  while ((c = DoorComReadByte(0L)) >= 0) {

        if (rxoptions & HOPT_HIGHBIT)
           c &= 0x7f;

        n = c;
        if (rxoptions & HOPT_HIGHCTL)
           n &= 0x7f;
        if (n != H_DLE &&
            (((rxoptions & HOPT_XONXOFF) && (n == XON || n == XOFF)) ||
             ((rxoptions & HOPT_CTLCHRS) && (n < 32 || n == 127))))
           continue;

        if (rxdle || c == H_DLE) {
           switch (c) {
                  case H_DLE:
                       if (++rxdle >= 5)
                          return (H_CANCEL);
                       break;

                  case HCHR_PKTEND:
                       rxbufptr = p;

                       switch (rxpktformat) {
                              case HCHR_BINPKT:
                                   q = rxbufptr;
                                   break;

                              case HCHR_HEXPKT:
                                   for (p = q = rxbuf; p < rxbufptr; p++) {
                                       if (*p == '\\' && *++p != '\\') {
                                          i = *p;
                                          n = *++p;
                                          if ((i -= '0') > 9) i -= ('a' - ':');
                                          if ((n -= '0') > 9) n -= ('a' - ':');
                                          if ((i & ~0x0f) || (n & ~0x0f)) {
                                             i = H_NOPKT;
                                             break;
                                          }
                                          *q++ = (i << 4) | n;
                                       }
                                       else
                                          *q++ = *p;
                                   }
                                   if (p > rxbufptr)
                                      c = H_NOPKT;
                                   break;

                              case HCHR_ASCPKT:
                                   n = i = 0;
                                   for (p = q = rxbuf; p < rxbufptr; p++) {
                                       i |= ((*p & 0x7f) << n);
                                       if ((n += 7) >= 8) {
                                          *q++ = (char) (i & 0xff);
                                          i >>= 8;
                                          n -= 8;
                                       }
                                   }
                                   break;

                              case HCHR_UUEPKT:
                                   n = (int) (rxbufptr - rxbuf);
                                   for (p = q = rxbuf; n >= 4; n -= 4, p += 4) {
                                       if (p[0] <= ' ' || p[0] >= 'a' ||
                                           p[1] <= ' ' || p[1] >= 'a' ||
                                           p[2] <= ' ' || p[2] >= 'a') {
                                          c = H_NOPKT;
                                          break;
                                       }
                                       *q++ = (char) ((h_uudec(p[0]) << 2) | (h_uudec(p[1]) >> 4));
                                       *q++ = (char) ((h_uudec(p[1]) << 4) | (h_uudec(p[2]) >> 2));
                                       *q++ = (char) ((h_uudec(p[2]) << 6) | h_uudec(p[3]));
                                   }
                                   if (n >= 2) {
                                      if (p[0] <= ' ' || p[0] >= 'a') {
                                         c = H_NOPKT;
                                         break;
                                      }
                                      *q++ = (char) ((h_uudec(p[0]) << 2) | (h_uudec(p[1]) >> 4));
                                      if (n == 3) {
                                         if (p[0] <= ' ' || p[0] >= 'a') {
                                            c = H_NOPKT;
                                            break;
                                         }
                                         *q++ = (char) ((h_uudec(p[1]) << 4) | (h_uudec(p[2]) >> 2));
                                      }
                                   }
                                   break;

                              default:   /* This'd mean internal fluke */
#if H_DEBUG
if (Hydraloglevel==0) {
   Hydramessage(0," <- <PKTEND> (pktformat='%c' dec=%d hex=%02x) ??",
             rxpktformat, rxpktformat, rxpktformat);
}
#endif
                                c = H_NOPKT;
                                break;
                    }

                    rxbufptr = NULL;

                    if (c == H_NOPKT)
                       break;

                    rxpktlen = (word) (q - rxbuf);
                    if (rxpktformat != HCHR_HEXPKT && (rxoptions & HOPT_CRC32)) {
                       if (rxpktlen < 5) {
                          c = H_NOPKT;
                          break;
                       }
                       n = h_crc32test(crc32block(rxbuf,rxpktlen));
                       rxpktlen -= (int) sizeof (long);  /* remove CRC-32 */
                    }
                    else {
                       if (rxpktlen < 3) {
                          c = H_NOPKT;
                          break;
                       }
                       n = h_crc16test(crc16block(rxbuf,rxpktlen));
                       rxpktlen -= (int) sizeof (word);  /* remove CRC-16 */
                    }

                    rxpktlen--;                     /* remove type  */

                    if (n) {
#if H_DEBUG
if (Hydraloglevel==0) {
   char *s1, *s2, *s3, *s4;

   Hydramessage(0," <- PKT (format='%c'  type='%c'  len=%d)",
           rxpktformat, (int) rxbuf[rxpktlen], rxpktlen);

   switch (rxbuf[rxpktlen]) {
          case HPKT_START:    Hydramessage(0,"    START");
                              break;
          case HPKT_INIT:     s1 = ((char *) rxbuf) + ((int) strlen((char *) rxbuf)) + 1;
                              s2 = s1 + ((int) strlen(s1)) + 1;
                              s3 = s2 + ((int) strlen(s2)) + 1;
                              s4 = s3 + ((int) strlen(s3)) + 1;
                              Hydramessage(0,"    INIT (appinfo='%s'  can='%s'  want='%s'  options='%s'  pktprefix='%s')",
                                      (char *) rxbuf, s1, s2, s3, s4);
                              break;
          case HPKT_INITACK:  Hydramessage(0,"    INITACK");
                              break;
          case HPKT_FINFO:    Hydramessage(0,"    FINFO ('%s'  rxstate=%d)",rxbuf,rxstate);
                              break;
          case HPKT_FINFOACK: Hydramessage(0,"    FINFOACK (pos=%ld  txstate=%d  txfd=%d)",
                                      Hydraintell(h_long1(rxbuf)), txstate, txfd);
                              break;
          case HPKT_DATA:     Hydramessage(0,"    DATA (rxstate=%d  pos=%ld  len=%u)",
                                      rxstate, Hydraintell(h_long1(rxbuf)),
                                      (word) (rxpktlen - ((int) sizeof (long))));
                              break;
          case HPKT_DATAACK:  Hydramessage(0,"    DATAACK (rxstate=%d  pos=%ld)",
                                      rxstate, Hydraintell(h_long1(rxbuf)));
                              break;
          case HPKT_RPOS:     Hydramessage(0,"    RPOS (pos=%ld%s  blklen=%u->%ld  syncid=%ld%s  txstate=%d  txfd=%d)",
                                      Hydraintell(h_long1(rxbuf)),
                                      Hydraintell(h_long1(rxbuf)) < 0L ? " SKIP" : "",
                                      txblklen, Hydraintell(h_long2(rxbuf)),
                                      Hydraintell(h_long3(rxbuf)),
                                      Hydraintell(h_long3(rxbuf)) == rxsyncid ? " DUP" : "",
                                      txstate, txfd);
                              break;
          case HPKT_EOF:      Hydramessage(0,"    EOF (rxstate=%d  pos=%ld%s)",
                                      rxstate, Hydraintell(h_long1(rxbuf)),
                                      Hydraintell(h_long1(rxbuf)) < 0L ? " SKIP" : "");
                              break;
          case HPKT_EOFACK:   Hydramessage(0,"    EOFACK (txstate=%d)", txstate);
                              break;
          case HPKT_IDLE:     Hydramessage(0,"    IDLE");
                              break;
          case HPKT_END:      Hydramessage(0,"    END");
                              break;
          case HPKT_DEVDATA:  s1 = ((char *) rxbuf) + ((int) sizeof (long));
                              Hydramessage(0,"    DEVDATA (id=%ld  dev=%s  len=%u",
                                      Hydraintell(h_long1(rxbuf)), s1,
                                      rxpktlen - (((int) sizeof (long)) + ((int) strlen(s1)) + 1));
                              break;
          case HPKT_DEVDACK:  Hydramessage(0,"    DEVDACK (devtxstate=%d  id=%ld)",
                                      devtxstate, Hydraintell(h_long1(rxbuf)));
                              break;
          default:            Hydramessage(0,"    Unkown pkttype %d (txstate=%d  rxstate=%d)",
                                      (int) rxbuf[rxpktlen], txstate, rxstate);
                              break;
   }
}
#endif
                                return ((int) rxbuf[rxpktlen]);
                             }/*goodpkt*/

#if H_DEBUG
if (Hydraloglevel==0)
   Hydramessage(0," Bad CRC (format='%c'  type='%c'  len=%d)",
             rxpktformat, (int) rxbuf[rxpktlen], rxpktlen);
#endif
                             break;

                        case HCHR_BINPKT: 
                        case HCHR_HEXPKT: 
                        case HCHR_ASCPKT: 
                        case HCHR_UUEPKT:
#if H_DEBUG
if (Hydraloglevel==0)
   Hydramessage(0," <- <PKTSTART> (pktformat='%c')",c);
#endif
                             rxpktformat = c;
                             p = rxbufptr = rxbuf;
                             rxdle = 0;
                             break;

                        default:
                             if (p) {
                                if (p < rxbufmax)
                                   *p++ = (char) (c ^ 0x40);
                                else {
#if H_DEBUG
if (Hydraloglevel==0)
   Hydramessage(0," <- Pkt too long - discarded");
#endif
                                   p = NULL;
                                }
                             }
                             rxdle = 0;
                             break;
                 }
              }
              else if (p) {
                 if (p < rxbufmax)
                    *p++ = (char) c;
                 else {
#if H_DEBUG
if (Hydraloglevel==0)
   Hydramessage(0," <- Pkt too long - discarded");
#endif
                    p = NULL;
                 }
              }
        }

        rxbufptr = p;

        if (h_timer_running(braindead) && h_timer_expired(braindead)) {
#if H_DEBUG
if (Hydraloglevel==0)
   Hydramessage(0," <- BrainDead (timer=%08lx  time=%08lx)",
           braindead,time(NULL));
#endif
           return (H_BRAINTIME);
        }
        if (h_timer_running(txtimer) && h_timer_expired(txtimer)) {
#if H_DEBUG
if (Hydraloglevel==0)
   Hydramessage(0," <- TxTimer (timer=%08lx  time=%08lx)",
           txtimer,time(NULL));
#endif
           return (H_TXTIME);
        }
        if (h_timer_running(devtxtimer) && h_timer_expired(devtxtimer)) {
#if H_DEBUG
if (Hydraloglevel==0)
   Hydramessage(0," <- DevTxTimer (timer=%08lx  time=%08lx)",
           devtxtimer,time(NULL));
#endif
           return (H_DEVTXTIME);
        }

        DosSleep(1L);
        return (H_NOPKT);
}/*rxpkt()*/


/*---------------------------------------------------------------------------*/
static void Hydra_status (boolean xmit)
{
  long pos    = xmit ? txpos    : rxpos,
       fsize  = xmit ? txfsize  : rxfsize;

  Hydra_gotoxy(46,xmit ? 1 : 2);
  if (pos >= fsize)
     Hydra_printf("%ld/%ld (EOF)",pos,fsize);
  else {
     int left = (int) ((((fsize - pos) / 128L) * 1340L) / DoorGetBaudVar());
     char *p = "";

     if (xmit) {
        if      (txstate == HTX_DATAACK) p = "ACK ";
        else if (txstate == HTX_XWAIT)   p = "WAIT ";
     }
     Hydra_printf("%ld/%ld (%s%d:%02d min left)",
                  pos, fsize, p, left / 60, left % 60);
  }
  Hydra_clreol();
}/*Hydra_status()*/


/*---------------------------------------------------------------------------*/
static void Hydra_pct (boolean xmit)
{
  long offset = xmit ? txoffset : rxoffset,
       fsize  = xmit ? txfsize  : rxfsize,
       start  = xmit ? txstart  : rxstart,
       elapsed, bytes, cps, pct;

  elapsed = time(NULL) - start;
  bytes = fsize - offset;
  if (bytes < 1024L || elapsed == 0L)
     return;
  cps = bytes / elapsed;
  pct = (cps * 1000L) / ((long) DoorGetBaudVar());
  Hydramessage(2,"+%s-H CPS: %ld (%ld bytes), %d:%02d min.  Eff: %ld%%",
          xmit ? "Sent" : "Rcvd", cps, bytes,
          (int) (elapsed / 60), (int) (elapsed % 60), pct);
}/*Hydra_pct()*/


/*---------------------------------------------------------------------------*/
void Hydra_badxfer (void)
{
  if (rxfd >= 0) {
     close(rxfd);
     rxfd = -1;
     if (Hydraxfer_bad())
        Hydramessage(1,"+HRECV: Bad xfer recovery-info saved");
     else
        Hydramessage(0,"-HRECV: Bad xfer - file deleted");
  }
}/*Hydra_badxfer()*/


#ifdef OS2
  static DCBINFO dcborig;
#endif

/*---------------------------------------------------------------------------*/
void Hydra_init (dword want_options) {

  register word i, j;
  word  crc16;
  dword crc32;

#ifdef OS2
  DosDevIOCtl(&dcborig,0L,0x73,1,DoorGetCommHandle());
#endif
  txbuf    = (char *)  malloc(H_BUFLEN);
  rxbuf    = (char *)  malloc(H_BUFLEN);
  crc16tab = (word *)  malloc(256 * ((int) sizeof (word)));
  crc32tab = (dword *) malloc(256 * ((int) sizeof (dword)));
  if (!txbuf || !rxbuf || !crc16tab || !crc32tab) {
     Hydramessage(6,"!HYDRA: Can't allocate buffers!");
     Hydraendprog(2);
  }
  txbufin  = txbuf + ((H_MAXBLKLEN + H_OVERHEAD + 5) * 2);
  rxbufmax = rxbuf + H_MAXPKTLEN;

  for (i = 0; i < 256; i++) {
      crc16 = i;
      crc32 = i;
      for (j = 8; j > 0; j--) {
          if (crc16 & 1) crc16 = (crc16 >> 1) ^ h_crc16poly;
          else           crc16 >>= 1;
          if (crc32 & 1) crc32 = (crc32 >> 1) ^ h_crc32poly;
          else           crc32 >>= 1;
      }
      crc16tab[i] = crc16;
      crc32tab[i] = crc32;
  }

  batchesdone = 0;

  originator = Hydranooriginator ? false : true;

  if (originator)
     hdxlink = false;
  else if (Hydrahdxsession)
     hdxlink = true;

  options = (want_options & HCAN_OPTIONS) & ~HUNN_OPTIONS;

  timeout = (word) (40960L / DoorGetBaudVar());
  if      (timeout < H_MINTIMER) timeout = H_MINTIMER;
  else if (timeout > H_MAXTIMER) timeout = H_MAXTIMER;

  txmaxblklen = (DoorGetBaudVar() / 300) * 128;
  if      (txmaxblklen < 256)         txmaxblklen = 256;
  else if (txmaxblklen > H_MAXBLKLEN) txmaxblklen = H_MAXBLKLEN;

  rxblklen = txblklen = (DoorGetBaudVar() < 2400U) ? 256 : 512;

  txgoodbytes  = 0;
  txgoodneeded = 1024;

  txstate = HTX_DONE;

  Hydramessage(-1,"+HYDRA session (%s-directional mode)",
          hdxlink ? "Uni" : "Bi");
  DoorCursor(1,6);
  DoorANSIPrintf("\x1b[K Port=%u   LineSpeed=%u",
           DoorGetCommHandle(), DoorGetBaudVar());
  DoorCursor(48,7);
  DoorANSIPuts("Press ESC to abort Hydra session");
  DoorCursor(1,10);
  for (i = 0; i < DoorGetMaxx(); i++)
    DoorANSIPutc(177);
  DoorCursor(1,10);
  DoorANSIPuts(" Remote ");
  DoorCursor(1,18);
  for (i = 0; i < DoorGetMaxx(); i++)
    DoorANSIPutc(177);
  DoorCursor(1,18);
  DoorANSIPuts(" Local (Press Alt-C to start/end chat mode) ");
  DoorCursor(2,8);
  DoorANSIPuts("\x1b[0;1;33mSending  :");
  DoorCursor(2,9);
  DoorANSIPuts("Receiving:\x1b[0m");
  Hydrafile_x = Hydrafile_y = Hydraremote_x = Hydraremote_y = Hydralocal_x = Hydralocal_y = 1;
  Hydralog_y = 0;

  Hydra_devfunc("CON",Hydrarem_chat);

  Hydrachatfill  = 0;
  Hydrachattimer = -1L;
  Hydralasttimer = 0L;

  Hydramessage(2,"+HYDRA session (%s-directional mode)",
            hdxlink ? "Uni" : "Bi");
}/*Hydra_init()*/


/*---------------------------------------------------------------------------*/
void Hydra_deinit (void)
{
  Hydrafile_x = 0;

  free(txbuf);
  free(rxbuf);
  free(crc16tab);
  free(crc32tab);
#ifdef OS2
  DosDevIOCtl(0L,&dcborig,0x53,1,DoorGetCommHandle());
#endif

}/*Hydra_deinit()*/


/*---------------------------------------------------------------------------*/
int Hydra (char *txpathname, char *txalias)
{
  int   res;
  int   pkttype;
  char *p, *q;
  int   i;
  struct stat f;

  /*-------------------------------------------------------------------*/
  if (txstate == HTX_DONE) {
     txstate        = HTX_START;
     Hydra_gotoxy(13,1);
     Hydra_printf("Init");
     txoptions      = HTXI_OPTIONS;
     txpktprefix[0] = '\0';

     rxstate   = HRX_INIT;
     Hydra_gotoxy(13,2);
     Hydra_printf("Init");
     rxoptions = HRXI_OPTIONS;
     rxfd      = -1;
     rxdle     = 0;
     rxbufptr  = NULL;
     rxtimer   = h_timer_reset();

     devtxid    = devrxid = 0L;
     devtxtimer = h_timer_reset();
     devtxstate = HTD_DONE;

     braindead = h_timer_set(H_BRAINDEAD);
  }
  else
     txstate = HTX_FINFO;

  txtimer   = h_timer_reset();
  txretries = 0;

  /*-------------------------------------------------------------------*/
  if (txpathname) {
     stat(txpathname,&f);
     txfsize = f.st_size;
     txftime = f.st_mtime;

     if ((txfd = sopen(txpathname,O_RDWR | O_BINARY,SH_DENYRW,S_IWRITE)) < 0) {
        Hydramessage(3,"-HSEND: Unable to open %s",txpathname);
        return (XFER_SKIP);
     }

     strupr(txpathname);
     for (p=txpathname, q=txfname; *p; p++) {
         if (*q=*p, *p=='\\' || *p==':' || *p=='/')
            q=txfname;
         else q++;
     }
     *q = '\0';

     if (txalias)
        strupr(txalias);

     txstart  = 0L;
     txsyncid = 0L;
  }
  else {
     txfd = -1;
     strcpy(txfname,"");
  }

  /*-------------------------------------------------------------------*/
  do {
     /*----------------------------------------------------------------*/
     switch (devtxstate) {
            /*---------------------------------------------------------*/
            case HTD_DATA:
                 if (txstate > HTX_RINIT) {
                    h_long1(txbufin) = Hydraintell(devtxid);
                    p = ((char *) txbufin) + ((int) sizeof(long));
                    strcpy(p,devtxdev);
                    p += H_FLAGLEN + 1;
                    memcpy(p,devtxbuf,devtxlen);
                    txpkt(((int) sizeof (long)) + H_FLAGLEN + 1 + devtxlen,HPKT_DEVDATA);
                    devtxtimer = h_timer_set(timeout);
                    devtxstate = HTD_DACK;
                 }
                 break;

            /*---------------------------------------------------------*/
            default:
                 break;

            /*---------------------------------------------------------*/
     }

     /*----------------------------------------------------------------*/
     switch (txstate) {
            /*---------------------------------------------------------*/
            case HTX_START:
                 DoorComWriteBlk((char *) autostr,(int) strlen(autostr));
                 txpkt(0,HPKT_START);
                 txtimer = h_timer_set(H_START);
                 txstate = HTX_SWAIT;
                 break;

            /*---------------------------------------------------------*/
            case HTX_INIT:
                 p = (char *) txbufin;
                 sprintf(p,"%08lxDOORSKEL,%d.%02d OS/2-DOS",
                           H_REVSTAMP,"DoorSkel",DRSK_MAJVER,DRSK_MINVER);
                 p += ((int) strlen(p)) + 1;/* our app info & HYDRA rev. */
                 put_flags(p,h_flags,HCAN_OPTIONS);    /* what we CAN  */
                 p += ((int) strlen(p)) + 1;
                 put_flags(p,h_flags,options);         /* what we WANT */
                 p += ((int) strlen(p)) + 1;
                 sprintf(p,"%08lx%08lx",               /* TxRx windows */
                           Hydra_txwindow,Hydra_rxwindow);
                 p += ((int) strlen(p)) + 1;
                 strcpy(p,pktprefix);     /* pkt prefix string we want */
                 p += ((int) strlen(p)) + 1;

                 txoptions = HTXI_OPTIONS;
                 txpkt((word) (((char *) p) - txbufin), HPKT_INIT);
                 txoptions = rxoptions;
                 txtimer = h_timer_set(timeout / 2);
                 txstate = HTX_INITACK;
                 break;

            /*---------------------------------------------------------*/
            case HTX_FINFO:
                 if (txfd >= 0) {
                    if (!txretries) {
                       Hydra_gotoxy(13,1);
                       Hydra_printf(txfname);
                       if (txalias) {
                          Hydra_gotoxy(25,1);
                          Hydra_printf("  ->  %s",txalias);
                       }
                       Hydra_clreol();
                       Hydramessage(2,"+HSEND: %s%s%s (%ldb), %d min.",
                               txpathname, txalias ? " -> " : "", txalias ? txalias : "",
                               txfsize, (int) (txfsize * 10L / DoorGetBaudVar() + 27L) / 54L);

                       strlwr(txfname);
                    }
                    sprintf((char *) txbufin,"%08lx%08lx%08lx%08lx%08lx%s",
                            txftime, txfsize, 0L, 0L, 0L,
                            txalias ? txalias : txfname);
                 }
                 else {
                    if (!txretries) {
                       Hydra_gotoxy(13,1);
                       Hydra_printf("End of batch");
                       Hydra_clreol();
                       Hydramessage(1,"+HSEND: End of batch");
                    }
                    strcpy((char *) txbufin,txfname);
                 }
                 txpkt(((int) strlen((char *) txbufin)) + 1,HPKT_FINFO);
                 txtimer = h_timer_set(txretries ? timeout / 2 : timeout);
                 txstate = HTX_FINFOACK;
                 break;

            /*---------------------------------------------------------*/
            case HTX_XDATA:
                 if (DoorComOutWaiting() > txmaxblklen)
                    break;

                 if (txpos < 0L)
                    i = -1;                                    /* Skip */
                 else {
                    h_long1(txbufin) = Hydraintell(txpos);
                    if ((i = read(txfd,txbufin + ((int) sizeof (long)),txblklen)) < 0) {
                       Hydramessage(6,"!HSEND: File read error");
                       close(txfd);
                       txfd = -1;
                       txpos = -2L;                            /* Skip */
                    }
                 }

                 if (i > 0) {
                    txpos += i;
                    txpkt(((int) sizeof (long)) + i, HPKT_DATA);

                    if (txblklen < txmaxblklen &&
                        (txgoodbytes += i) >= txgoodneeded) {
                       txblklen <<= 1;
                       if (txblklen >= txmaxblklen) {
                          txblklen = txmaxblklen;
                          txgoodneeded = 0;
                       }
                       txgoodbytes = 0;
                    }

                    if (txwindow && (txpos >= (txlastack + txwindow))) {
                       txtimer = h_timer_set(txretries ? timeout / 2 : timeout);
                       txstate = HTX_DATAACK;
                    }

                    if (!txstart)
                       txstart = time(NULL);
                    Hydra_status(true);
                    break;
                 }

                 /* fallthrough to HTX_EOF */

            /*---------------------------------------------------------*/
            case HTX_EOF:
                 h_long1(txbufin) = Hydraintell(txpos);
                 txpkt((int) sizeof (long),HPKT_EOF);
                 txtimer = h_timer_set(txretries ? timeout / 2 : timeout);
                 txstate = HTX_EOFACK;
                 break;

            /*---------------------------------------------------------*/
            case HTX_END:
                 txpkt(0,HPKT_END);
                 txpkt(0,HPKT_END);
                 txtimer = h_timer_set(timeout / 2);
                 txstate = HTX_ENDACK;
                 break;

            /*---------------------------------------------------------*/
            default:
                 break;

            /*---------------------------------------------------------*/
     }

     /*----------------------------------------------------------------*/
     while (txstate && (pkttype = rxpkt()) != H_NOPKT) {
           /*----------------------------------------------------------*/
           switch (pkttype) {
                  /*---------------------------------------------------*/
                  case H_CARRIER:
                  case H_CANCEL:
                  case H_SYSABORT:
                  case H_BRAINTIME:
                       switch (pkttype) {
                              case H_CARRIER:   p = "Carrier lost";          break;
                              case H_CANCEL:    p = "Aborted by other side"; break;
                              case H_SYSABORT:  p = "Aborted by operator";   break;
                              case H_BRAINTIME: p = "Other end died";        break;
                       }
                       Hydramessage(3,"-HYDRA: %s",p);
                       txstate = HTX_DONE;
                       res = XFER_ABORT;
                       break;

                  /*---------------------------------------------------*/
                  case H_TXTIME:
                       if (txstate == HTX_XWAIT || txstate == HTX_REND) {
                          txpkt(0,HPKT_IDLE);
                          txtimer = h_timer_set(H_IDLE);
                          break;
                       }

                       if (++txretries > H_RETRIES) {
                          Hydramessage(3,"-HSEND: Too many errors");
                          txstate = HTX_DONE;
                          res = XFER_ABORT;
                          break;
                       }

                       Hydramessage(0,"-HSEND: Timeout - Retry %u",txretries);

                       txtimer = h_timer_reset();

                       switch (txstate) {
                              case HTX_SWAIT:    txstate = HTX_START; break;
                              case HTX_INITACK:  txstate = HTX_INIT;  break;
                              case HTX_FINFOACK: txstate = HTX_FINFO; break;
                              case HTX_DATAACK:  txstate = HTX_XDATA; break;
                              case HTX_EOFACK:   txstate = HTX_EOF;   break;
                              case HTX_ENDACK:   txstate = HTX_END;   break;
                       }
                       break;

                  /*---------------------------------------------------*/
                  case H_DEVTXTIME:
                       if (++devtxretries > H_RETRIES) {
                          Hydramessage(3,"-HDEVTX: Too many errors");
                          txstate = HTX_DONE;
                          res = XFER_ABORT;
                          break;
                       }

                       Hydramessage(0,"-HDEVTX: Timeout - Retry %u",devtxretries);

                       devtxtimer = h_timer_reset();
                       devtxstate = HTD_DATA;
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_START:
                       if (txstate == HTX_START || txstate == HTX_SWAIT) {
                          txtimer = h_timer_reset();
                          txretries = 0;
                          txstate = HTX_INIT;
                          braindead = h_timer_set(H_BRAINDEAD);
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_INIT:
                       if (rxstate == HRX_INIT) {
                          p = (char *) rxbuf;
                          p += ((int) strlen(p)) + 1;
                          q = p + ((int) strlen(p)) + 1;
                          rxoptions  = options | HUNN_OPTIONS;
                          rxoptions |= get_flags(q,h_flags);
                          rxoptions &= get_flags(p,h_flags);
                          rxoptions &= HCAN_OPTIONS;
                          if (rxoptions < (options & HNEC_OPTIONS)) {
                             Hydramessage(6,"!HYDRA: Incompatible on this link");
                             txstate = HTX_DONE;
                             res = XFER_ABORT;
                             break;
                          }
                          p = q + ((int) strlen(q)) + 1;
                          rxwindow = txwindow = 0L;
                          sscanf(p,"%08lx%08lx", &rxwindow,&txwindow);
                          if (rxwindow < 0L) rxwindow = 0L;
                          if (Hydra_rxwindow &&
                              (!rxwindow || Hydra_rxwindow < rxwindow))
                             rxwindow = Hydra_rxwindow;
                          if (txwindow < 0L) txwindow = 0L;
                          if (Hydra_txwindow &&
                              (!txwindow || Hydra_txwindow < txwindow))
                             txwindow = Hydra_txwindow;
                          p += ((int) strlen(p)) + 1;
                          strncpy(txpktprefix,p,H_PKTPREFIX);
                          txpktprefix[H_PKTPREFIX] = '\0';

                          if (!batchesdone) {
                             long revstamp;

                             p = (char *) rxbuf;
                             sscanf(p,"%08lx",&revstamp);
                             Hydramessage(0,"*HYDRA: Other's HydraRev=%s",
                                     Hydrah_revdate(revstamp));
                             p += 8;
                             if ((q = strchr(p,',')) != NULL) *q = ' ';
                             if ((q = strchr(p,',')) != NULL) *q = '/';
                             Hydramessage(0,"*HYDRA: Other's App.Info '%s'",p);
                             put_flags((char *) rxbuf,h_flags,rxoptions);
                             Hydramessage(1,"*HYDRA: Using link options '%s'",rxbuf);
                             if (txwindow || rxwindow)
                                Hydramessage(0,"*HYDRA: Window tx=%ld rx=%ld",
                                          txwindow,rxwindow);
                          }

                          Hydrachattimer = (rxoptions & HOPT_DEVICE) ? 0L : -2L;

                          txoptions = rxoptions;
                          rxstate = HRX_FINFO;
                       }

                       txpkt(0,HPKT_INITACK);
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_INITACK:
                       if (txstate == HTX_INIT || txstate == HTX_INITACK) {
                          braindead = h_timer_set(H_BRAINDEAD);
                          txtimer = h_timer_reset();
                          txretries = 0;
                          txstate = HTX_RINIT;
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_FINFO:
                       if (rxstate == HRX_FINFO) {
                          braindead = h_timer_set(H_BRAINDEAD);
                          if (!rxbuf[0]) {
                             Hydra_gotoxy(13,2);
                             Hydra_printf("End of batch");
                             Hydra_clreol();
                             Hydramessage(1,"*HRECV: End of batch");
                             rxpos = 0L;
                             rxstate = HRX_DONE;
                             batchesdone++;
                          }
                          else {
                             long diskfree;

                             rxfsize = rxftime = 0L;
                             rxfname[0] = '\0';
                             sscanf((char *) rxbuf,"%08lx%08lx%*08lx%*08lx%*08lx%s",
                                    &rxftime, &rxfsize, rxfname);
                             strupr(rxfname);

                             Hydra_gotoxy(13,2);
                             Hydra_printf(rxfname);
                             Hydra_clreol();

                             rxpathname = Hydraxfer_init(rxfname,rxfsize,rxftime);

                             diskfree = DoorFreeSpace(rxpathname);

                             if (!rxpathname) {   /* Already have file */
                                if (Hydrasingle_done) {
                                   Hydra_gotoxy(29,2);
                                   Hydra_printf("Skipping additional files");
                                   Hydramessage(1,"+HRECV: Skipping additional files (file %s)",rxfname);
                                   rxpos = -2L;
                                }
                                else {
                                   Hydra_gotoxy(29,2);
                                   Hydra_printf("Already have file");
                                   Hydramessage(1,"+HRECV: Already have %s",rxfname);
                                   rxpos = -1L;
                                }
                             }
                             else if (rxfsize + 10240L > diskfree) {
                                Hydra_gotoxy(29,2);
                                Hydra_printf("Not enough diskspace");
                                Hydramessage(6,"!HRECV: %s not enough diskspace: %ld > %ld",
                                        rxfname, rxfsize + 10240L, diskfree);
                                rxpos = -2L;
                             }
                             else {
                                if (Hydrafexist(rxpathname)) { /* Resuming? */
                                   if ((rxfd = sopen(rxpathname,O_RDWR | O_BINARY,SH_DENYRW,S_IWRITE)) < 0) {
                                      Hydramessage(6,"!HRECV: Unable to re-open %s",rxpathname);
                                      rxpos = -2L;
                                   }
                                }
                                else if ((rxfd = sopen(rxpathname,O_CREAT | O_RDWR | O_BINARY,SH_DENYRW,S_IWRITE | S_IREAD)) < 0) {
                                   Hydramessage(6,"!HRECV: Unable to create %s",rxpathname);
                                   rxpos = -2L;
                                }

                                if (rxfd >= 0) {
                                   Hydramessage(2,"+HRECV: %s (%ldb), %d min.",
                                           rxfname, rxfsize,
                                           (int) (rxfsize * 10L / DoorGetBaudVar() + 27L) / 54L);
                                   if (lseek(rxfd,0L,SEEK_END) < 0L) {
                                      Hydramessage(6,"!HRECV: File seek error");
                                      Hydra_badxfer();
                                      rxpos = -2L;
                                   }
                                   else {
                                      rxoffset = rxpos = tell(rxfd);
                                      if (rxpos < 0L) {
                                         Hydramessage(6,"!HRECV: File tell error");
                                         Hydra_badxfer();
                                         rxpos = -2L;
                                      }
                                      else {
                                         rxstart = 0L;
                                         rxtimer = h_timer_reset();
                                         rxretries = 0;
                                         rxlastsync = 0L;
                                         rxsyncid = 0L;
                                         Hydra_status(false);
                                         if (rxpos > 0L) {
                                            Hydra_gotoxy(46,2);
                                            Hydra_printf("%ld/%ld",rxpos,rxfsize);
                                            Hydramessage(1,"+HRECV: Resuming from offset %ld (%d min. to go)",
                                                    rxpos, (int) ((rxfsize - rxoffset) * 10L / DoorGetBaudVar() + 27L) / 54L);
                                         }
                                         rxstate = HRX_DATA;
                                      }
                                   }
                                }
                             }
                          }
                       }
                       else if (rxstate == HRX_DONE)
                          rxpos = (!rxbuf[0]) ? 0L : -2L;

                       h_long1(txbufin) = Hydraintell(rxpos);
                       txpkt((int) sizeof (long),HPKT_FINFOACK);
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_FINFOACK:
                       if (txstate == HTX_FINFO || txstate == HTX_FINFOACK) {
                          braindead = h_timer_set(H_BRAINDEAD);
                          txretries = 0;
                          if (!txfname[0]) {
                             txtimer = h_timer_set(H_IDLE);
                             txstate = HTX_REND;
                          }
                          else {
                             txtimer = h_timer_reset();
                             txpos = Hydraintell(h_long1(rxbuf));
                             if (txpos >= 0L) {
                                txoffset = txpos;
                                txlastack = txpos;
                                Hydra_status(true);
                                if (txpos > 0L) {
                                   Hydramessage(1,"+HSEND: Transmitting from offset %ld (%d min. to go)",
                                           txpos, (int) ((txfsize - txoffset) * 10L / DoorGetBaudVar() + 27L) / 54L);
                                   if (lseek(txfd,txpos,SEEK_SET) < 0L) {
                                      Hydramessage(6,"!HSEND: File seek error");
                                      close(txfd);
                                      txfd = -1;
                                      txpos = -2L;
                                      txstate = HTX_EOF;
                                      break;
                                   }
                                }
                                txstate = HTX_XDATA;
                             }
                             else {
                                close(txfd);
                                if (txpos == -1L) {
                                   Hydra_gotoxy(29,1);
                                   Hydra_printf("They already have file");
                                   Hydramessage(1,"+HSEND: They already have %s",txfname);
                                   return (XFER_OK);
                                }
                                else {  /* (txpos < -1L) file NOT sent */
                                   Hydra_gotoxy(29,1);
                                   Hydra_printf("Skipping");
                                   Hydramessage(1,"+HSEND: Skipping %s",txfname);
                                   return (XFER_SKIP);
                                }
                             }
                          }
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_DATA:
                       if (rxstate == HRX_DATA) {
                          if (Hydraintell(h_long1(rxbuf)) != rxpos ||
                              Hydraintell(h_long1(rxbuf)) < 0L) {
                             if (Hydraintell(h_long1(rxbuf)) <= rxlastsync) {
                                rxtimer = h_timer_reset();
                                rxretries = 0;
                             }
                             rxlastsync = Hydraintell(h_long1(rxbuf));

                             if (!h_timer_running(rxtimer) ||
                                 h_timer_expired(rxtimer)) {
                                if (rxretries > 4) {
                                   if (txstate < HTX_REND &&
                                       !originator && !hdxlink) {
                                      hdxlink = true;
                                      rxretries = 0;
                                   }
                                }
                                if (++rxretries > H_RETRIES) {
                                   Hydramessage(3,"-HRECV: Too many errors");
                                   txstate = HTX_DONE;
                                   res = XFER_ABORT;
                                   break;
                                }
                                if (rxretries == 1)
                                   rxsyncid++;

                                rxblklen /= 2;
                                i = rxblklen;
                                if      (i <=  64) i =   64;
                                else if (i <= 128) i =  128;
                                else if (i <= 256) i =  256;
                                else if (i <= 512) i =  512;
                                else               i = 1024;
                                Hydramessage(0,"-HRECV: Bad pkt at %ld - Retry %u (newblklen=%u)",
                                        rxpos,rxretries,i);
                                h_long1(txbufin) = Hydraintell(rxpos);
                                h_long2(txbufin) = Hydraintell((long) i);
                                h_long3(txbufin) = Hydraintell(rxsyncid);
                                txpkt(3 * ((int) sizeof(long)),HPKT_RPOS);
                                rxtimer = h_timer_set(timeout);
                             }
                          }
                          else {
                             braindead = h_timer_set(H_BRAINDEAD);
                             rxpktlen -= (int) sizeof (long);
                             rxblklen = rxpktlen;
                             if (write(rxfd,rxbuf + ((int) sizeof (long)),rxpktlen) < 0) {
                                Hydramessage(6,"!HRECV: File write error");
                                Hydra_badxfer();
                                rxpos = -2L;
                                rxretries = 1;
                                rxsyncid++;
                                h_long1(txbufin) = Hydraintell(rxpos);
                                h_long2(txbufin) = Hydraintell(0L);
                                h_long3(txbufin) = Hydraintell(rxsyncid);
                                txpkt(3 * ((int) sizeof(long)),HPKT_RPOS);
                                rxtimer = h_timer_set(timeout);
                                break;
                             }
                             rxretries = 0;
                             rxtimer = h_timer_reset();
                             rxlastsync = rxpos;
                             rxpos += rxpktlen;
                             if (rxwindow) {
                                h_long1(txbufin) = Hydraintell(rxpos);
                                txpkt((int) sizeof(long),HPKT_DATAACK);
                             }
                             if (!rxstart)
                                rxstart = time(NULL) -
                                          ((rxpktlen * 10) / DoorGetBaudVar());
                             Hydra_status(false);
                          }/*badpkt*/
                       }/*rxstate==HRX_DATA*/
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_DATAACK:
                       if (txstate == HTX_XDATA || txstate == HTX_DATAACK ||
                           txstate == HTX_XWAIT ||
                           txstate == HTX_EOF || txstate == HTX_EOFACK) {
                          if (txwindow && Hydraintell(h_long1(rxbuf)) > txlastack) {
                             txlastack = Hydraintell(h_long1(rxbuf));
                             if (txstate == HTX_DATAACK &&
                                 (txpos < (txlastack + txwindow))) {
                                txstate = HTX_XDATA;
                                txretries = 0;
                                txtimer = h_timer_reset();
                             }
                          }
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_RPOS:
                       if (txstate == HTX_XDATA || txstate == HTX_DATAACK ||
                           txstate == HTX_XWAIT ||
                           txstate == HTX_EOF || txstate == HTX_EOFACK) {
                          if (Hydraintell(h_long3(rxbuf)) != txsyncid) {
                             txsyncid = Hydraintell(h_long3(rxbuf));
                             txretries = 1;
                             txtimer = h_timer_reset();
                             txpos = Hydraintell(h_long1(rxbuf));
                             if (txpos < 0L) {
                                if (txfd >= 0) {
                                   Hydra_gotoxy(29,1);
                                   Hydra_printf("Skipping");
                                   Hydramessage(1,"+HSEND: Skipping %s",txfname);
                                   close(txfd);
                                   txfd = -1;
                                   txstate = HTX_EOF;
                                }
                                txpos = -2L;
                                break;
                             }

                             if (txblklen > Hydraintell(h_long2(rxbuf)))
                                txblklen = (word) Hydraintell(h_long2(rxbuf));
                             else
                                txblklen >>= 1;
                             if      (txblklen <=  64) txblklen =   64;
                             else if (txblklen <= 128) txblklen =  128;
                             else if (txblklen <= 256) txblklen =  256;
                             else if (txblklen <= 512) txblklen =  512;
                             else                      txblklen = 1024;
                             txgoodbytes = 0;
                             txgoodneeded += 1024;
                             if (txgoodneeded > 8192)
                                txgoodneeded = 8192;

                             Hydra_status(true);
                             Hydramessage(0,"+HSEND: Resending from offset %ld (newblklen=%u)",
                                     txpos,txblklen);
                             if (lseek(txfd,txpos,SEEK_SET) < 0L) {
                                Hydramessage(6,"!HSEND: File seek error");
                                close(txfd);
                                txfd = -1;
                                txpos = -2L;
                                txstate = HTX_EOF;
                                break;
                             }

                             if (txstate != HTX_XWAIT)
                                txstate = HTX_XDATA;
                          }
                          else {
                             if (++txretries > H_RETRIES) {
                                Hydramessage(3,"-HSEND: Too many errors");
                                txstate = HTX_DONE;
                                res = XFER_ABORT;
                                break;
                             }
                          }
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_EOF:
                       if (rxstate == HRX_DATA) {
                          if (Hydraintell(h_long1(rxbuf)) < 0L) {
                             Hydra_badxfer();
                             Hydra_gotoxy(29,2);
                             Hydra_printf("Skipping");
                             Hydra_clreol();
                             Hydramessage(1,"+HRECV: Skipping %s",rxfname);
                             rxstate = HRX_FINFO;
                             braindead = h_timer_set(H_BRAINDEAD);
                          }
                          else if (Hydraintell(h_long1(rxbuf)) != rxpos) {
                             if (Hydraintell(h_long1(rxbuf)) <= rxlastsync) {
                                rxtimer = h_timer_reset();
                                rxretries = 0;
                             }
                             rxlastsync = Hydraintell(h_long1(rxbuf));

                             if (!h_timer_running(rxtimer) ||
                                 h_timer_expired(rxtimer)) {
                                if (++rxretries > H_RETRIES) {
                                   Hydramessage(3,"-HRECV: Too many errors");
                                   txstate = HTX_DONE;
                                   res = XFER_ABORT;
                                   break;
                                }
                                if (rxretries == 1)
                                   rxsyncid++;

                                rxblklen /= 2;
                                i = rxblklen;
                                if      (i <=  64) i =   64;
                                else if (i <= 128) i =  128;
                                else if (i <= 256) i =  256;
                                else if (i <= 512) i =  512;
                                else               i = 1024;
                                Hydramessage(0,"-HRECV: Bad EOF at %ld - Retry %u (newblklen=%u)",
                                        rxpos,rxretries,i);
                                h_long1(txbufin) = Hydraintell(rxpos);
                                h_long2(txbufin) = Hydraintell((long) i);
                                h_long3(txbufin) = Hydraintell(rxsyncid);
                                txpkt(3 * ((int) sizeof(long)),HPKT_RPOS);
                                rxtimer = h_timer_set(timeout);
                             }
                          }
                          else {
                             rxfsize = rxpos;
                             close(rxfd);
                             rxfd = -1;
                             Hydra_pct(false);

                             p = Hydraxfer_okay();
                             if (p) {
                                Hydra_gotoxy(25,2);
                                Hydra_printf("  -> %s",p);
                                Hydramessage(1,"+HRECV: Dup file renamed: %s",p);
                             }

                             Hydra_status(false);
                             Hydramessage(1,"+Rcvd-H %s",p ? p : rxfname);
                             Hydraresultlog(false,p ? p : rxfname,rxfsize - rxoffset,time(NULL) - rxstart);
                             rxstate = HRX_FINFO;
                             braindead = h_timer_set(H_BRAINDEAD);
                          }/*skip/badeof/eof*/
                       }/*rxstate==HRX_DATA*/

                       if (rxstate == HRX_FINFO)
                          txpkt(0,HPKT_EOFACK);
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_EOFACK:
                       if (txstate == HTX_EOF || txstate == HTX_EOFACK) {
                          braindead = h_timer_set(H_BRAINDEAD);
                          if (txfd >= 0) {
                             txfsize = txpos;
                             close(txfd);
                             Hydra_pct(true);
                             Hydraresultlog(true,txpathname,txfsize - txoffset,time(NULL) - txstart);
                             return (XFER_OK);
                          }
                          else
                             return (XFER_SKIP);
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_IDLE:
                       if (txstate == HTX_XWAIT) {
                          hdxlink = false;
                          txtimer = h_timer_reset();
                          txretries = 0;
                          txstate = HTX_XDATA;
                       }
                       else if (txstate >= HTX_FINFO && txstate < HTX_REND)
                          braindead = h_timer_set(H_BRAINDEAD);
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_END:
                       /* special for chat, other side wants to quit */
                       if (Hydrachattimer > 0L && txstate == HTX_REND) {
                          Hydrachattimer = -3L;
                          break;
                       }

                       if (txstate == HTX_END || txstate == HTX_ENDACK) {
                          txpkt(0,HPKT_END);
                          txpkt(0,HPKT_END);
                          txpkt(0,HPKT_END);
                          Hydramessage(1,"+HYDRA: Completed");
                          txstate = HTX_DONE;
                          res = XFER_OK;
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_DEVDATA:
                       if (devrxid != Hydraintell(h_long1(rxbuf))) {
                          Hydra_devrecv();
                          devrxid = Hydraintell(h_long1(rxbuf));
                       }
                       h_long1(txbufin) = Hydraintell(devrxid);
                       txpkt((int) sizeof (long),HPKT_DEVDACK);
                       break;

                  /*---------------------------------------------------*/
                  case HPKT_DEVDACK:
                       if (devtxstate && (devtxid == Hydraintell(h_long1(rxbuf)))) {
                          devtxtimer = h_timer_reset();
                          devtxstate = HTD_DONE;
                       }
                       break;

                  /*---------------------------------------------------*/
                  default:  /* unknown packet types: IGNORE, no error! */
                       break;

                  /*---------------------------------------------------*/
           }/*switch(pkttype)*/

           /*----------------------------------------------------------*/
           switch (txstate) {
                  /*---------------------------------------------------*/
                  case HTX_START:
                  case HTX_SWAIT:
                       if (rxstate == HRX_FINFO) {
                          txtimer = h_timer_reset();
                          txretries = 0;
                          txstate = HTX_INIT;
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HTX_RINIT:
                       if (rxstate == HRX_FINFO) {
                          txtimer = h_timer_reset();
                          txretries = 0;
                          txstate = HTX_FINFO;
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HTX_XDATA:
                       if (rxstate && hdxlink) {
                          Hydramessage(3,"*HYDRA: %s",hdxmsg);
                          Hydra_devsend("MSG",(char *) hdxmsg,(int) strlen(hdxmsg));

                          txtimer = h_timer_set(H_IDLE);
                          txstate = HTX_XWAIT;
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HTX_XWAIT:
                       if (!rxstate) {
                          txtimer = h_timer_reset();
                          txretries = 0;
                          txstate = HTX_XDATA;
                       }
                       break;

                  /*---------------------------------------------------*/
                  case HTX_REND:
                       if (!rxstate && !devtxstate) {
                          /* special for chat, braindead will protect */
                          if (Hydrachattimer > 0L) break;
                          if (Hydrachattimer == 0L) Hydrachattimer = -3L;

                          txtimer = h_timer_reset();
                          txretries = 0;
                          txstate = HTX_END;
                       }
                       break;

                  /*---------------------------------------------------*/
           }/*switch(txstate)*/
     }/*while(txstate&&pkttype)*/
  } while (txstate);

  if (txfd >= 0)
     close(txfd);
  Hydra_badxfer();

  if (res == XFER_ABORT) {
     DoorComPurgeOut();
     if (DoorComOnLine()) {
        DoorComWriteBlk((char *) abortstr,(int) strlen(abortstr));
        DoorComFlushOut();
     }
     DoorComPurgeIn();
  }
  else
     DoorComFlushOut();

  return (res);
}/*Hydra()*/


/* end of Hydra.c */
