/*
**  EVAL.C - A simple mathematical expression evaluator in C
**
**  operators supported: (
**                       )
**                       +
**                       -
**                       *
**                       /
**                       %
**                       ^
**                       &
**                       |
**
**
**  limitations: No precedence rules are implemented
**
**  Original Copyright 1991 by Bob Stout as part of
**  the MicroFirm Function Library (MFL)
**
**  This subset version is hereby donated to the public domain.
**
**  Hacked by M. Kimes to use longs instead of doubles
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

typedef enum {D_ERROR = -2, ERROR, SUCCESS} STATUS;

static char   delims[]   = "+-*/^)(%&|";        /* Tokens               */
static char   op_stack[256];                    /* Operator stack       */
static long   arg_stack[256];                   /* Argument stack       */
static char   token[256];                       /* Token buffer         */
static int    op_sptr,                          /* op_stack pointer     */
              arg_sptr,                         /* arg_stack pointer    */
              parens,                           /* Nesting level        */
              state = 0;                        /* 0 = Initialized
                                                   1 = Awaiting expression
                                                   2 = Awaiting operator
                                                */

int _fastcall mevaluate(char *, long *);

static int     _fastcall   do_op(void);
static int     _fastcall   do_paren(void);
static void    _fastcall   push_op(char);
static void    _fastcall   push_arg(long  );
static STATUS  _fastcall   pop_arg(long   *);
static STATUS  _fastcall   pop_op(int *);
static char *  _fastcall   getexp(char *);
static char *  _fastcall   getop(char *);
static void    _fastcall   pack(char *);


/*
**  Evaluate a mathematical expression
*/

int _fastcall mevaluate(char *line, long *val)
{
      long arg;
      char *ptr = line, *str, *endptr;
      int  ercode;


      pack(line);
      state = 0;

      while (*ptr)
      {
            switch (state)
            {
            case 0:
            case 2:
                  if (NULL == (str = getop(ptr))) return ERROR;

                  if (strchr(delims, *str))
                  {
                        if (')' == *str)
                        {
                              if (SUCCESS > (ercode = do_paren()))
                                return ercode;
                              state = 2;
                        }
                        else
                        {
                              push_op(*str);
                              state = 1;
                        }

                        ptr += strlen(str);
                  }
                  else
                  {
                        if (state) return ERROR;
                        else  state = 1;
                  }

                  break;

            case 1:
                  if (NULL != (str = getexp(ptr)))
                  {
                        if ('(' == *str)
                        {
                              push_op(*str);
                              ptr += strlen(str);
                              break;
                        }

                        if (0L == (arg = strtol(str,&endptr,0)) &&
                              NULL == strchr(str, '0'))
                        {
                              return ERROR;
                        }
                        push_arg(arg);
                        ptr += strlen(str);
                  }
                  else return ERROR;

                  state = 2;
                  break;
            }
      }

      while (1 < arg_sptr)
      {
            if (SUCCESS > (ercode = do_op())) return ercode;
      }
      if (!op_sptr) {
        return pop_arg(val);
      }
      else  return ERROR;
}

/*
**  Evaluate stacked arguments and operands
*/

static int _fastcall do_op(void)
{
      long arg1, arg2;
      int  op;


      if (ERROR == pop_op(&op)) return ERROR;

      pop_arg(&arg1);
      pop_arg(&arg2);

      switch (op)
      {
      case '+':
            push_arg(arg2 + arg1);
            break;

      case '-':
            push_arg(arg2 - arg1);
            break;

      case '*':
            push_arg(arg2 * arg1);
            break;

      case '/':
            if (0L == arg1) return D_ERROR;
            push_arg(arg2 / arg1);
            break;

      case '^':
            push_arg(arg2 ^ arg1);
            break;

      case '%':
            push_arg(arg2 % arg1);
            break;

      case '&':
            push_arg(arg2 & arg1);
            break;

      case '|':
            push_arg(arg2 | arg1);
            break;

      case '(':
            arg_sptr += 2;
            break;

      default:
            return ERROR;
      }
      if (1 > arg_sptr) return ERROR;
      else  return op;
}

/*
**  Evaluate one level
*/

static int _fastcall do_paren(void)
{
      int op;


      if (1 > parens--) return ERROR;
      do
      {
          if (SUCCESS > (op = do_op())) break;
      } while ('(' != op);
      return op;
}

/*
**  Stack operations
*/

static void _fastcall push_op(char op)
{
      if ('(' == op) ++parens;
      op_stack[op_sptr++] = op;
}

static void _fastcall push_arg(long arg)
{
      arg_stack[arg_sptr++] = arg;
}

static STATUS _fastcall pop_arg(long *arg)
{
      *arg = arg_stack[--arg_sptr];
      if (0 > arg_sptr) return ERROR;
      else return SUCCESS;
}

static STATUS _fastcall pop_op(int *op)
{
      if (!op_sptr) return ERROR;
      *op = op_stack[--op_sptr];
      return SUCCESS;
}

/*
**  Get an expression
*/

static char * _fastcall getexp(char *str)
{
      char *ptr = str, *tptr = token;


      while (*ptr)
      {
            if (strchr(delims, *ptr))
            {
                  if ('-' == *ptr)
                  {
                        if (str != ptr && 'E' != ptr[-1]) break;
                  }

                  else if (str == ptr) return getop(str);

                  else if ('E' == *ptr)
                  {
                        if (!isdigit(ptr[1]) && '-' != ptr[1]) return NULL;
                  }
                  else break;
            }

            *tptr++ = *ptr++;
      }
      *tptr = 0;

      return token;
}

/*
**  Get an operator
*/

static char * _fastcall getop (char *str)
{
      *token = *str;
      token[1] = 0;
      return token;
}

/*
**  Remove whitespace & capitalize
*/

static void _fastcall pack(char *str)
{
      char *ptr = str, *p;


//      strupr(str);        /* why? */

      for ( ; *ptr; ++ptr)
      {
            p = ptr;
            while (*p && isspace(*p)) ++p;
            if (ptr != p) strcpy(ptr, p);
      }
} 
