/* config.c
**
** released into the PUBLIC DOMAIN 30 jul 1990 by jim nutt
** Changes released into the PUBLIC DOMAIN 10, jul 1994 by John Dennis
**
** Finds & reads config files, initializes everything.
**
*/

#define  TEXTLEN  512
#define  NDEBUG   1
#define AREAS_BBS 0

#include "msged.h"
#include "main.h"
#include "config.h"
#include "nshow.h"
#include "bmg.h"

/*#include <direct.h>*/


/* non-local prototypes */

void   e_assignkey(unsigned int key, char *label);
void   r_assignkey(unsigned int key, char *label);


/* local variables */

static GROUP *group            = NULL;
static int    areas_type       = 0,
              num_groups       = 0,
              check_area_files = 0;

static char *cfgverbs[] =
{
    "name",          "address",         "privatenet",       "alias",
    "outfile",       "lastread",        "tosslog",          "userlist",
    "swappath",      "nodepath",        "nodelist",         "helpfile",
    "xxxxx",         "areafile",        "useroffset",       "quote",
    "attribution",   "forwardline",     "redirline",        "followup",
    "replyarea",     "replyextra",      "alterfunc",        "switch",
    "qquotes",       "sqllast",         "video",            "color",
    "right",         "quoteright",      "tabsize",          "curstart",
    "curend",        "fido",            "squish",           "uucp",
    "domain",        "gate",            "origin",           "readkey",
    "editkey",       "function",        "include",          "speller",
    "videoseg",      "maxx",             "maxy",            "template",
    "uucpname",      "group",            "colour",          "editor",
    NULL
   };

static struct colorverb colortable[] =
{
    {"black",     0x00}, {"blue",      0x01}, {"green",     0x02},
    {"cyan",      0x03}, {"red",       0x04}, {"magenta",   0x05},
    {"brown",     0x06}, {"lgrey",     0x07}, {"dgrey",     0x08},
    {"lblue",     0x09}, {"lgreen",    0x0A}, {"lcyan",     0x0B},
    {"lred",      0x0C}, {"lmagenta",  0x0D}, {"yellow",    0x0E},
    {"white",     0x0F}, {"intense",   0x08}, {"_black",    0x00},
    {"_blue",     0x10}, {"_green",    0x20}, {"_cyan",     0x30},
    {"_red",      0x40}, {"_magenta",  0x50}, {"_brown",    0x60},
    {"_lgrey",    0x70}, {"_dgrey",    0x80}, {"_lblue",    0x90},
    {"_lgreen",   0xA0}, {"_lcyan",    0xB0}, {"_lred",     0xC0},
    {"_lmagenta", 0xD0}, {"_yellow",   0xE0}, {"_white",    0xF0},
    {"_blink",    0x80}, {NULL, 0x00}
    };

static char *cfgcolors[] =
{
    "mainnorm",    "mainquote",    "mainkludge",   "maintempl",   "maininfo",
    "maindivide",  "mainheader",   "mainhdrtxt",   "mainblock",   "mainedit",
    "mainwarn",    "mainnetinf",   "menuborder",   "menunorm",    "menuselect",
    "helpborder",  "helptitle",    "helpnorm",     "helphilight", "infoborder",
    "infonorm",    "inputborder",  "inputnorm",    "inputedit",   "dlgborder",
    "dlgnorm",     "dlgchnorm",    "dlgchsel",     "dlgennorm",   "dlgensel",
    "dlgbutsel",   "dlgbutnorm",   "dlgbutshadow", "listbdr",     "listtitle",
    "listnorm",    "listinfo",     "listselect",    NULL
};

static char *cfgswitches[] =
{
    "hardquote", "savecc", "rawcc", "seen-bys", "shownotes",
    "confirm", "showaddr", "msgids", "opusdate", "datearvd",
    "chopquote", "qquotes", "uselastr", "showcr", "showeol",  
    "realmsgn", "usemouse", "editcronly", "usepid", "soteot",
    "tearline", "showsystem", NULL
};


/*
**
** Strips leading whitespaces from a string.
**
*/

char *  striplwhite(char *s)
{
    while (*s && isspace((unsigned char)*s))
        s++;
    return(s);
}


/*
**
** Strips trailing spaces from a string.
**
*/

char *  striptwhite(char *s)
{
    char *c = s;

    if (!s)
        return NULL;

    s = c + strlen(c) - 1;  /* dunno why I reversed it */
    while (((s > c) && *s && isspace((unsigned char)*s)) 
           || (*s == '\\') || (*s == '/'))
    {
        *s = '\0'; s--;
    }
    return((c) ? c : NULL);
}


/*
**
** Skips from current position to first blank found.
**
*/

char *  skip_to_blank(char *str)
{
    while (str && *str && !isspace((unsigned char)*str)) str++;

    return str;
}


/*
**
** Kills a trailing slash from a path.
**
*/

void  kill_trail_slash(char *s)
{
    char *c = s + strlen(s) - 1;

    if (*c == '\\' || *c == '/')
        *c = '\0';
}


/*
**
** Returns the number of the passed verb in relation to
** the cfgverbs array, or -1 if the passed verb is not
** in the array.
**
*/

int GetVerbNum(char *verb)
{
    int i = 0;

    while (cfgverbs[i] != NULL) {
        if (strcmp(verb, cfgverbs[i]) == 0)
            return i;
        i++;
    }
    return -1;
}


/*
**
** Assigns the passed named area to the global colour array.
**
*/

void AssignColor(char *name, int color)
{
    int i = 0;

    while (cfgcolors[i] != NULL)
    {
        if (stricmp(name, cfgcolors[i]) == 0)
        {
            cm[i] = color;
            return;
        }
        i++;
    }
	printf("\nunknown color argument: %s\n", name);
}


/*
**
** Assigns the passed switch name to on or off.
**
*/

void AssignSwitch(char *swtch, int OnOff)
{
    int i = 0;

    while (cfgswitches[i] != NULL)
    {
        if (stricmp(swtch, cfgswitches[i]) == 0)
            break;
        i++;
    }

    if (cfgswitches[i] == NULL)
    {
		printf("\nunknown switch: %s\n", swtch);
        return;
	}

    switch (i)
    {
        case 0:  SW->hardquote     = OnOff; break;
        case 1:  SW->savecc        = OnOff; break;
        case 2:  SW->rawcc         = OnOff; break;
        case 3:  SW->seenbys       = OnOff; break;
        case 4:  SW->shownotes     = OnOff; break;
        case 5:  SW->confirmations = OnOff; break;
        case 6:  SW->showaddr      = OnOff; break;
        case 7:  SW->msgids        = OnOff; break;
        case 8:  SW->opusdate      = OnOff; break;
        case 9:  SW->datearrived   = OnOff; break;
        case 10: SW->chopquote     = OnOff; break;
        case 11: SW->qquote        = OnOff; break;
        case 12: SW->use_lastr     = OnOff; break;
        case 13: SW->showcr        = OnOff; break;
        case 14: SW->showeol       = OnOff; break;
        case 15: SW->showrealmsgn  = OnOff; break;
        case 16: SW->usemouse      = OnOff; break;
        case 17: SW->editcronly    = OnOff; break;
        case 18: SW->usepid        = OnOff; break;
        case 19: SW->soteot        = OnOff; break;
        case 20: SW->tearline      = OnOff; break;
        case 21: SW->showsystem    = OnOff; break;
        default: printf("\nunknown switch: %s\n", swtch); break;
    }
}


/*
** Searches through the colortable for the matching
** colorname and returns the color, or -1 if not found.
*/

int GetColor(char *color)
{
    int i = 0;

    while (colortable[i].name != NULL)
    {
        if (stricmp(color, colortable[i].name) == 0)
            return colortable[i].color;
        i++;
    }
    printf("\nunknown color: %s\n", color);
    return -1;
}


/*
**
** Determines if the passed char is a token seperator.
**
*/

int toksep(char ch)
{
    switch ((int) ch)
    {
        case ' '  :
        case ','  :
        case '\r' :
        case '\t' :
        case '\n' :
            return TRUE;
        default   :
            return FALSE;
    }
}


/*
**
** Gets num tokens from a string, seperated by a default set of tokens.
** Handles stuff bewtween quotes as one token.
**
*/

void parse_tokens(char *str, char *tokens[], int num)
{
    int   i    = 0;
    char *s    = str;
    int   done = 0;

    while (!done && i < num)
    {
        if (toksep(*s))
            while (*s && toksep(*s)) s++;

        if (*s == ';')
        {
            /*
            ** We hit a comment... dinna process anymore.
            */

            tokens[i] = NULL;
            break;
        }

        if (*s == '\"')
        {
            tokens[i++] = ++s;

            while (*s && *s != '\"') s++;
        }
        else
        {
            tokens[i++] = s;
            while (*s && !toksep(*s)) s++;
        }
        if (*s == '\0')
        {
            tokens[i] = NULL;
            done      = TRUE;
        }
        else
            *s++ = '\0';
    }
    tokens[i] = NULL;
}


/*
**
** Opens cfn first, and returns if successful, else it tries
** env.  env can be considered the default name.
**
*/

static FILE *  fileopen(char *env, char *cfn)
{
    FILE    *fp;

    if (cfn)
    {
        if ((fp = fopen(cfn, "r")) != NULL)
            return fp;
    }

    fp = fopen(env, "r");

    return fp;
}


/*
**
** Parses a macro.
**
*/

static unsigned int *  parse_macro(char *macro)
{
    unsigned int *m;
    unsigned int *t;
    int l;
    char tmp[5];

    t = m = (unsigned int *) calloc(strlen(macro) * 2,sizeof(int));

    if ((t == (void *) NULL) || (macro == NULL))
        return(NULL);

    while (*macro)
    {
        if (*macro == '^')
        {
            *t++ = (unsigned int) (*(macro + 1) == '^') ? '^' : (toupper(*(macro + 1)) - 64);
            macro += 2;
        }
        else if (*macro == '\\')
        {
            if (*(macro + 1) == '\\')
            {
                *t++   = (unsigned int) '\\';
                macro += 2;
            }
            else
            {
                memset(tmp,0,sizeof tmp);
                strncpy(tmp,macro+1,4);
                *t++   = (unsigned int) strtol(tmp,NULL,0) << 8;
                macro += 5;
            }
        }
        else
            *t++ = (unsigned int) *macro++;
    }
    *t = 0;
    l  = (int) (t - m) + 1;
    
    t = realloc(m, l * sizeof (int));

    if (t)
        return(t);

    return(m);
}


/*
**
** Sets the username and the template for an area.
**
*/

static void SetAreaInfo(AREA *a)
{
    int i;

    for (i = 0; i < num_groups; i++)
    {
        if (bmg_find(a->description, group[i].search) != NULL)
            break;
    }

    if (i != num_groups)
    {
        a->username = group[i].username;
        a->template = group[i].template;
    }
}


/*
**
** Adds a new area to the area-list.
**
*/

static void  AddArea(AREA *a)
{
    int i;
    char *d, *t, *p;

    for (i = 0; i < SW->areas; i++)
        if ((arealist[i].path != NULL) && (stricmp(a->path, arealist[i].path) == 0))
            break;

    if (i == SW->areas)
    {
        /*
        ** It's a new area, so we add it to the end of the list.
        */

        SW->areas++;
        SW->area = SW->areas - 1;
        arealist = realloc(arealist, SW->areas * sizeof(struct _area));

        if (arealist == NULL)
            outamemory();

        memset(&(CurArea), 0, sizeof(struct _area));

        CurArea = *a;

        if (a->addr.domain)
            CurArea.addr.domain = strdup(a->addr.domain);

        CurArea.description = strdup(a->description);
        CurArea.tag         = strdup(a->tag);
        CurArea.path        = strdup(a->path);

        kill_trail_slash(CurArea.path);
    }
    else
    {
        /*
        ** We redefine the area to the new defaults,
        ** with the exception of the path, dag and desc.
        */

        release(arealist[i].addr.domain);
        p = arealist[i].path;
        t = arealist[i].tag;
        d = arealist[i].description;

        arealist[i]             = *a;
        arealist[i].path        = p;
        arealist[i].tag         = t;
        arealist[i].description = d;
        if (a->addr.domain)
            arealist[i].addr.domain = strdup(a->addr.domain);
    }
    SetAreaInfo(&arealist[i]);

    /*
    ** Clean up after us...
    */

    release(a->tag);
    release(a->description);
    release(a->path);
    release(a->addr.domain);
}


/*
**
** Checks a file in the areas.bbs format and gets all
** the area definitions. First attemps to open the file
** specified on the commandline, and if that fails, tries
** for the default name "areas.bbs".
**
*/

static void  checkareas(char *areafile)
{
    AREA    a;                      /* current area */
    char    buffer[TEXTLEN];        /* line buffer */
    FILE   *fp   = NULL;            /* file handle */
    char   *s    = NULL;            /* pointer */
    int     flag = 1;               /* found host name? */
    int     sq   = 0;               /* a squish base? */
    char   *tokens[12];

    if ((fp = fileopen("areas.bbs", areafile)) == NULL)
        return;

    if (fp != NULL)
    {
        while (fgets(buffer, TEXTLEN, fp) != NULL)
        {
            if ((*buffer == '\r') || (*buffer == '\n'))
                continue;

            s = striplwhite(buffer);

            if ((*s == ';') || (*s == '-') || (*s == '#'))
                continue;

            if ((strchr(buffer,'!')) && flag)
            {
                char *p = strrchr(buffer,'!');
                if (p != NULL)
                {
                    *p = '\0';

                    release(ST->origin);
                    ST->origin = strdup(buffer);
                }
                flag = 0;
                continue;
            }

            if (*s == '$')
            {
                sq = 1;
                s++;
            }
            else
                sq = 0;

            /*
            ** Grab the tokens on the line.
            */

            memset(tokens, 0, sizeof(tokens));
            parse_tokens(s, tokens, 4);

            if (tokens[0] == NULL || tokens[1] == NULL)
                continue;

            memset(&a, 0, sizeof(AREA));

            a.echomail      = 1;
            a.msgtype       = FIDO;
            a.path          = strdup(tokens[0]);
            a.tag           = strdup(tokens[1]);
            a.description   = strdup(tokens[1]);
            a.addr.notfound = 1;

            kill_trail_slash(a.path);

            if (tokens[2] != NULL && strlen(tokens[2]) >= 7)
            {
                s = tokens[2];
                if (*s == '-' && toupper(*(s + 1)) == 'P')
                    a.addr = parsenode(s + 2);
            }

            if (a.addr.notfound)
            {
                a.addr = thisnode;
                if (thisnode.domain)
                    a.addr.domain = strdup(thisnode.domain);
            }

            strlwr(a.path);
            strupr(a.tag);

            if (sq)
                a.msgtype = SQUISH;

            AddArea(&a);
        }
    }
    if (fp != NULL)
        fclose(fp);
}


/*
**
** Reads the areas in a squish config file.
**
*/

static void  check_squish(char *areafile)
{
    AREA   a;
    char   buffer[TEXTLEN];
    char  *tokens[20];
    FILE  *fp = NULL;
    int    i  = 0;
    char  *s;

    if ((fp = fileopen("squish.cfg", areafile)) == NULL)
        return;

    if (fp != NULL)
    {
        while (fgets(buffer, TEXTLEN, fp) != NULL)
        {
            if ((*buffer == '\r') || (*buffer == '\n'))
                continue;

            s = striplwhite(buffer);

            if ((*s == ';') || (*s == '-') || (*s == '#'))
                continue;

            memset(tokens, 0, sizeof(tokens));
            parse_tokens(s, tokens, 15);

            if (tokens[0] == NULL)
                continue;

            if (!stricmp(tokens[0],     "echoarea") ||
                    !stricmp(tokens[0], "netarea")  ||
                    !stricmp(tokens[0], "badarea")  ||
                    !stricmp(tokens[0], "dupearea"))
            {
                if (tokens[1] == NULL || tokens[2] == NULL)
                    continue;

                memset(&a, 0, sizeof(AREA));

                if (!stricmp(tokens[0], "echoarea"))
                    a.echomail = 1;
                else
                {
                    a.netmail = 1;
                    a.priv = 1;
                }

                a.msgtype       = FIDO;
                a.addr.notfound = 1;
                a.tag           = strdup(tokens[1]);
                a.path          = strdup(tokens[2]);
                a.description   = strdup(tokens[1]);

                strupr(a.tag);
                strlwr(a.path);

                i = 3;
                while ((s = tokens[i]) != NULL)
                {
                    if (*s == ';')
                    {
                        i = 0;
                        break;
                    }
                    else if (*s == '-' && *(s + 1) && *(s + 1) == '$')
                    {
                        a.msgtype = SQUISH;
                    }
                    else if (*s == '-' && *(s + 1) && toupper(*(s + 1)) == 'P')
                    {
                        /*
                        ** An address is specified... get it.
                        */

                        release(a.addr.domain);
                        a.addr = parsenode(s + 2);
                    }
                    else if (*s == '-' && *(s + 1) && *(s + 1) == '0')
                    {
                        /*
                        ** A passthru area... ignore it.
                        */

                        release(a.tag);
                        release(a.description);
                        release(a.path);
                        release(a.addr.domain);
                        i = 0;
                        break;
                    }
                    i++;
                }

                if (i == 0)
                    continue;

                if (a.addr.notfound)
                {
                    a.addr = thisnode;
                    if (thisnode.domain)
                        a.addr.domain = strdup(thisnode.domain);
                }
                AddArea(&a);
            }
        }
    }
    if (fp != NULL)
        fclose(fp);
}


/*
**
** Reads in an area definition in the msged.cfg format.
**
*/

static void  parsemail(char *keyword, char *value)
{
    char  *tokens[20];
    char  *s;
    AREA   a;

    memset(&a, 0, sizeof a);
    memset(tokens, 0, sizeof tokens);

    switch (tolower(*keyword))        /* one letter is enough for now */
    { 
        default  :
        case 'f' : a.msgtype = FIDO;   break;
        case 's' : a.msgtype = SQUISH; break;
    }

    parse_tokens(value, tokens, 5);

    if (tokens[2] == NULL)
        return;

    s = tokens[0];

    while (*s)
    {
        switch (tolower(*s))               /* one letter is enough */
        {
            case 'n' : a.news     = 1; break;
            case 'u' : a.uucp     = 1; break;
            case 'm' : a.netmail  = 1; break;
            case 'e' : a.echomail = 1; break;
            case 'p' : a.priv     = 1; break;
            case 'h' : a.hold     = 1; break;
            case 'd' : a.direct   = 1; break;
            case 'c' : a.crash    = 1; break;
            case 'k' : a.killsent = 1; break;
            case 'l' : a.local    = 1; break;
            default  : break;
        }
        s++;
    }

    if (!a.echomail && !a.netmail && !a.uucp && !a.news)
        a.local = 1;

    a.description   = strdup(tokens[1]);
    a.path          = strdup(tokens[2]);
    a.addr.notfound = 1;

    if (a.echomail)
    {
        if (tokens[3] != NULL)
        {
            a.tag = strdup(tokens[3]);
            if (tokens[4] != NULL)
                a.addr = parsenode(tokens[4]);
        }
        else
            a.tag = strdup(a.description);
    }
    else
    {
        a.tag = strdup(a.description);
        if (tokens[3] != NULL)
            a.addr = parsenode(tokens[3]);
    }

    if (a.addr.notfound)
    {
        a.addr = thisnode;
        if (thisnode.domain)
            a.addr.domain = strdup(thisnode.domain);
    }
    AddArea(&a);
}


/*
**
** Parses an alias.
**
**
*/

static void  parse_alias(char *value)
{
    ALIAS  a;
    char  *tokens[6];
    char  *s;

    memset(&a, 0, sizeof a);
    memset(tokens, 0, sizeof(tokens));

    parse_tokens(value, tokens, 5);

    if (tokens[2] == NULL)
        return;

    a.alias = strdup(tokens[0]);
    a.name  = strdup(tokens[1]);

    if (!stricmp(tokens[2], "uucp"))
        a.addr.internet = 1;
    else
        a.addr  = parsenode(tokens[2]);

    if ((s = tokens[3]) != NULL)
    {
        a.attrib.local = 1;
        a.attr         = 1;
        while (*s)
        {
            switch (tolower(*s))
            {
                case 'p' : a.attrib.private  = 1; break;
                case 'h' : a.attrib.hold     = 1; break;
                case 'd' : a.attrib.direct   = 1; break;
                case 'c' : a.attrib.crash    = 1; break;
                case 'k' : a.attrib.killsent = 1; break;
                case 'n' : a.attr = 0;            break;
                default  : break;
            }
            s++;
        }
        if (tokens[4] != NULL)
            a.subj = strdup(tokens[4]);
    }

    aliaslist = realloc(aliaslist, (++SW->otheraliases) * sizeof(struct _alias));

    if (aliaslist == NULL)
        outamemory();

    aliaslist[SW->otheraliases - 1] = a;
}


/*
**
** Handles the areafiles...
**
*/

void  do_areafile(char *value)
{
    char *tokens[3];

    memset(tokens, 0, sizeof(tokens));
    parse_tokens(value, tokens, 2);

    if (tokens[0] == NULL)
        return;

    areas_type = AREAS_BBS;

    if (toupper(*tokens[0]) == 'S')
        areas_type = SQUISH;

    if (tokens[1])
    {
        if (areas_type == AREAS_BBS)
            checkareas(tokens[1]);
        else
            check_squish(tokens[1]);
    }
    else
    {
        /*
        ** If no area-file was specified, then we want to check later.
        */

        check_area_files = 1;
    }
}


/*
**
** Hanldes an entire config file.
**
**
*/

static void  parseconfig(FILE *fp)
{
    char    progress_indicators[4] = {'-', '\\', '|', '/'};
    char    buffer[TEXTLEN];
    char   *keyword = NULL;
    char   *value   = NULL;
    char   *s       = NULL;
    char   *tokens[20];
    int     i, line_num = 0;

    memset(buffer, 0, TEXTLEN);

    while (!feof(fp))
    {
        line_num++;
        printf("%c\b", progress_indicators[line_num % 4]);
        
        if (fgets(buffer, TEXTLEN, fp) == NULL)
            return;

        keyword = strtok(buffer, " \t\n\r");

        if (keyword)
            strlwr(keyword);
        else
            continue;

        if ((*keyword) == ';')
            continue;

        value = strtok(NULL, ";\n\r");
        value = striptwhite(value);

        /* clear the pointers in the array */

        memset(tokens, 0, sizeof(tokens));

        while (value && *value && isspace((unsigned char)*value))
            if ((*value == '\n') || (*value == ';'))
                break;
            else
                value++;

        switch(GetVerbNum(keyword))
        {
            case -1:
                printf("\nunknown config verb: %s\n", keyword);
                break;

            case  0:
                parse_tokens(value, tokens, 3);
                if (tokens[0] != NULL)
                {
                    for (i = 0; i < 11; i++)
                    {
                        if (user_list[i].name == NULL)
                            break;
                    }
                    if (i < 11)
                    {
                        user_list[i].name = strdup(tokens[0]);

                        if (tokens[1] != NULL)
                            user_list[i].lastread = strdup(tokens[1]);

                        if (tokens[2] != NULL)
                            user_list[i].offset = atol(tokens[2]);

                        if (i == 0)
                        {
                            release(ST->username);
                            ST->username = strdup(user_list[i].name);

                            if (user_list[i].lastread)
                            {
                                release(ST->lastread);
                                ST->lastread = strdup(user_list[i].lastread);
                            }
                            SW->useroffset = user_list[i].offset;
                        }
                    }
                }
                break;

            case 1:
                alias = realloc(alias, (++SW->aliascount) * sizeof(ADDRESS));
                alias[SW->aliascount - 1] = parsenode(value);
                break;

            case 2:
                SW->pointnet = (int) strtol(value, NULL, 0);
                break;

            case 3:
                parse_alias(value);
                break;

            case 4:
                release(ST->outfile);
                ST->outfile = strdup(value);
                break;

            case 5:
                if (user_list[0].name == NULL)
                {
                    release(ST->lastread);
                    ST->lastread          = strdup(value);
                    user_list[0].lastread = strdup(value);
                }
                break;

            case 6:
                release(ST->confmail);
                ST->confmail = strdup(value);
                break;

            case 7:
                ST->fidolist = strdup(strtok(value, ",\n"));
                if ((ST->userlist = strtok(NULL,",\n")) != NULL)
                    ST->userlist = strdup(ST->userlist);
                break;
#ifndef __OS2__
            case 8:
                release(ST->swap_path);
                kill_trail_slash(value);
                ST->swap_path = strdup(value);
                break;
#endif
            case 9:
                release(ST->nodepath);
                ST->nodepath = strdup(value);
                break;

            case 10:
                parse_tokens(value, tokens, 3);
                if (tokens[2] != NULL)
                {
                    node_lists = realloc(node_lists, (++SW->nodelists) * sizeof(D_LIST));

                    if (node_lists == NULL)
                        outamemory();

                    node_lists[SW->nodelists - 1].name      = strdup(tokens[0]);
                    node_lists[SW->nodelists - 1].base_name = strdup(tokens[1]);
                    node_lists[SW->nodelists - 1].sysop     = strdup(tokens[2]);

                    if (SW->nodelists == 1)
                    {
                        ST->nodebase = node_lists[SW->nodelists - 1].base_name;
                        ST->sysop    = node_lists[SW->nodelists - 1].sysop;
                    }
                }
                break;

            case 11:
                release(ST->helpfile);
                ST->helpfile = strdup(value);
                break;

            case 12:
                break;

            case 13:
                do_areafile(value);
                break;

            case 14:
                SW->useroffset = atoi(value);
                break;

            case 15:
                release(ST->quotestr);
                striptwhite(value);
                s = ST->quotestr = strdup(value);
                while (*s)
                {
                    *s = (*s == (char) '_') ? (char) ' ' : *s;
                    s++;
                }
                break;

            case 22:
                parse_tokens(value, tokens, 2);

                if (tokens[0] && tokens[1])
                {
                    int var = 0;
                    char *c;

                    c = striplwhite(tokens[0]);
                    s = striplwhite(tokens[1]);

                    if ((*s != '\0') && (*c != '\0'))
                    {
                        while (*c && !isspace((unsigned char)*c))
                        {
                            switch (toupper(*c))
                            {
                                case 'Q' :
                                    if (!(var & MT_REP) && !(var & MT_NEW))
                                        var |= MT_QUO;
                                    break;

                                case 'R' :
                                    if (!(var & MT_QUO) && !(var & MT_NEW))
                                        var |= MT_REP;
                                    break;

                                case 'N' :
                                    if (!(var & MT_QUO) && !(var & MT_REP))
                                        var |= MT_NEW;
                                    break;

                                case 'A' : var |= MT_ARC; break;
                                case 'F' : var |= MT_FOL;  break;
                                default  : break;
                            }
                            c++;
                        }
                        if (!stricmp("replyquote", s))
                            SW->rquote = var;
                        else if (!stricmp("replyotherarea", s))
                            SW->rotharea = var;
                        else if (!stricmp("replyfollow", s))
                            SW->rfollow = var;
                        else if (!stricmp("replyextra", s))
                            SW->rextra = var;
                    }
                }
                break;

            case 23:
                parse_tokens(value, tokens, 2);
                if (tokens[1] != NULL)
                    AssignSwitch(value, (stricmp(tokens[1], "on") == 0) ? 1 : 0);
                break;

            case 24:
                SW->qquote = 1;
                break;

            case 25:
                SW->use_lastr = 1;
                break;

            case 26:
                /* do nothing */
                break;

            case 27:
            case 50:
                parse_tokens(value, tokens, 3);
                if (tokens[2] != NULL)
                {
                    int fcol, bcol;

                    fcol = GetColor(tokens[1]);
                    bcol = GetColor(tokens[2]);

                    if (fcol == -1 || bcol == -1)
                        continue;

                    AssignColor(tokens[0], fcol|bcol);
                }
                break;

            case 28:
                SW->rm = (int) strtol(value,NULL,0);
                break;

            case 29:
                SW->qm = (int) strtol(value,NULL,0);
                break;

            case 30:
                SW->tabsize = (int) strtol(value,NULL,0);
                break;

            case 31:
                cur_start = atoi(value);
                break;

            case 32:
                cur_end = atoi(value);
                break;

            case 33:
            case 34:
                parsemail(keyword, value);
                break;

            case 35:
                uucp_gate = parsenode(value);
                break;

            case 36:
                domain_list = realloc(domain_list, (++SW->domains) * sizeof(ADDRESS));

                if (domain_list == NULL)
                    outamemory();

                domain_list[SW->domains - 1] = parsenode(value);
                break;

            case 37:
                if (value) strlwr(value);
                if (strncmp(value, "zones",5) == 0)
                    SW->gate = GZONES;
                else if (strncmp(value, "domains",7) == 0)
                    SW->gate = GDOMAINS;
                else if (strncmp(value, "both",4) == 0)
                    SW->gate = BOTH;
                else if (strncmp(value, "none",4) == 0)
                    SW->gate = 0;
                break;

            case 38:
                if (value != NULL)
                {
                    release(ST->origin);
                    ST->origin = strdup(value);
                }
                break;

            case 39:
            case 40:
                if (value)
                {
                    strlwr(value);
                    i     = (int) strtol(value,&value,0);
                    value = striplwhite(value);
                }
                if (*keyword == 'e')
                    e_assignkey(i, value);
                else
                    r_assignkey(i, value);
                break;

            case 41:
                i = (int) strtol(value,&s,0);
                s = striplwhite(s);

                if ((i >= 0) && (i <= 40))
                    macros[i] = parse_macro(s);
                break;

            case 42:
            {
                FILE *ifp;
                if ((ifp = fopen(value,"r")) != NULL)
                {
                    parseconfig(ifp);
                    fclose(ifp);
                }
            }
            break;

            case 45:
                maxx = (int) strtol(value,NULL,0);
                break;

            case 46:
                maxy = (int) strtol(value,NULL,0);
                break;

            case 47:
                templates = realloc(templates, (++SW->numtemplates) * sizeof(char *));

                if (templates == NULL)
                    outamemory();

                templates[SW->numtemplates - 1] = strdup(value);

                if (SW->numtemplates == 1)
                {
                    ST->template = strdup(templates[SW->numtemplates - 1]);
                }
                break;

            case 48:
                release(ST->uucpgate);
                ST->uucpgate = strdup(value);

            case 49:
                group = realloc(group, (++num_groups) * sizeof(GROUP));

                if (group == NULL)
                    outamemory();

                parse_tokens(value, tokens, 3);

                if (tokens[2] != NULL)
                {
                    group[num_groups - 1].search   = strdup(strupr(tokens[0]));
                    group[num_groups - 1].username = atoi(tokens[1]);
                    /* mod PE 1995-04-12 to protect against wild user */
                    if ((group[num_groups - 1].username >= MAXUSERS)
                        || (group[num_groups - 1].username >= MAXUSERS)
                        || (user_list[group[num_groups - 1].username].name 
                            == NULL))
                    {
                        group[num_groups - 1].username = 0;
                    }                    
                    group[num_groups - 1].template = atoi(tokens[2]);
                    /* mod PE 1995-04-08 to protect against wild template */
                    if (group[num_groups - 1].template >= SW->numtemplates)
                    {
                        group[num_groups - 1].template = 0;
                    }
                }
                break;
                
            case 51:
                ST->editorName = strdup(value);
                break;

            default :
                printf("\nunknown config verb: %s\n", keyword);
                break;
        }
        memset(buffer, 0, TEXTLEN);
    }
}


/*
**
** Handles the entire configuration process.
**
**
*/

void  opening(char *cfgfile, char *areafile)
{
    WND    *hCurr, *hWnd;
    FILE   *fp        = NULL;
    int     count     = 0,i;
    char    cfnname[] = "msged.cfg";
    char    tmp[PATHLEN];

    InitVars();

    printf("Msgedsq %s; FTSC & Squish compatible Mail Editor", VERSION CLOSED);

    if ((fp = fileopen(cfnname, cfgfile)) == NULL)
    {
        if (cfgfile == NULL)
            cfgfile = cfnname;

        printf("\n\nCould not find config file %s", cfgfile);
        exit(-1);
    }

    printf("\n\nReading configuration file: %s => ", (cfgfile == NULL) ? cfnname : cfgfile);

    if (fp)
    {
        parseconfig(fp);
    }

    for (i = 0; i < 40; i++)
        count += (macros[i] != (void *) NULL) ? 1 : 0;

    if (fp) fclose(fp);

    if (check_area_files)
    {
        if (areas_type == AREAS_BBS)
            checkareas(areafile);
        else
            check_squish(areafile);
    }

    if (!arealist)
    {
        printf("\n\nError! At least one message area must be defined.");
        exit(-1);
    }

    printf("\b Finished.\n");

    InitScreen();

    cursor(0);

    SW->rm = (SW->rm > maxx)   ? maxx : SW->rm;
    SW->qm = (SW->qm > SW->rm) ? SW->rm - strlen(ST->quotestr) : SW->qm;

    hCurr = Wtop();
    hWnd  = WPopUp(60, 15, SBDR|SHADOW, cm[IN_BTXT], cm[IN_NTXT]);

    if (!hWnd) outamemory();

    WTitle(" System Information ", cm[IN_BTXT]);
    WPutsCen(1, cm[IN_NTXT], "Msgedsq FTS & Squish Compatible Mail Editor");
    WPutsCen(2, cm[IN_NTXT], "v"VERSION CLOSED "; Authors Paul Edwards, John Dennis, Jim Nutt");
    WPutsCen(3, cm[IN_NTXT], "modifications by Bill Bond");
    WPutsCen(4, cm[IN_NTXT], "Compiled : "__DATE__" " __TIME__);

    sprintf(tmp, "Screen of %d by %d ", maxx, maxy);
    WPutsCen(5, cm[IN_NTXT], tmp);
    sprintf(tmp, "User is %s at %s%s", ST->username, show_address(&thisnode), (SW->aliascount > 1) ? " (primary)" : "");
    WPutsCen(6, cm[IN_NTXT], tmp);

    if (ST->origin != NULL)
        WPutsCen(7, cm[IN_NTXT], ST->origin);
    else
        WPutsCen(7, cm[IN_NTXT], "No \"Origin:\" line!");

    sprintf(tmp, "%d Message areas found", SW->areas);
    WPutsCen(8, cm[IN_NTXT], tmp);
    sprintf(tmp, "%d Macros defined",count);
    WPutsCen(9, cm[IN_NTXT], tmp);

    mygetcwd(tmp,PATHLEN);
    strlwr(tmp);
    ST->home = strdup(tmp);

    sprintf(tmp, "home directory is %s", ST->home);
    WPutsCen(10, cm[IN_NTXT], tmp);
    WPutsCen(12, cm[IN_NTXT], "Press <enter> to continue");

    GetKey();

    WClose(hWnd);
    WCurr(hCurr);
}

/* end */
