/*:ts=8*/
/*****************************************************************************
 * FIDOGATE --- Gateway UNIX Mail/News <-> FIDO NetMail/EchoMail
 *
 * $Id: msgid.c,v 3.8.0.4 1994/09/12 21:05:55 mj Exp mj $
 *
 * MSGID <-> Message-ID conversion handling. See also README.MSGID.
 *
 *****************************************************************************
 * Copyright (C) 1990, 1993, 1994
 *  _____ _____
 * |	 |___  |   Martin Junius	     FIDO:	2:2452/110.1
 * | | | |   | |   Republikplatz 3	     Internet:	mj@sungate.fido.de
 * |_|_|_|@home|   D-52072 Aachen, Germany   Phone:	++49-241-86931 (voice)
 *
 * This file is part of FIDOGATE.
 *
 * FIDOGATE is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * FIDOGATE is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with FIDOGATE; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/

#include "fidogate.h"
#include "shuffle.h"



/*
 * Convert FIDO ^AMSGID/REPLY to RFC Message-ID/References
 */

char *msgid_fido_to_rfc(msgid)
    char *msgid;
{
    char *system;
    char *id;
    Node idnode;
    long xtol();

    SHUFFLEBUFFERS;
    
    /*
     * Seperate system string
     */
    for(system=msgid; *msgid && !isspace(*msgid); msgid++);
    if(!*msgid)
	return NULL;
    *msgid++ = 0;
    
    /*
     * Seperate id string
     */
    for(; *msgid && isspace(*msgid); msgid++);
    if(!*msgid)
	return NULL;
    for(id=msgid; *msgid && isxdigit(*msgid); msgid++);
    if(!*id)
	return NULL;
    *msgid = 0;

    /*
     * Strip leading `0's from id string
     */
    while(*id == '0')
	id++;

    /*
     * Try to interprete system as FIDO node
     */
    if(asc_to_node(system, &idnode, TRUE) != ERROR)
    {
	/*
	 * This is a FIDO address, convert to RFC style:
	 *     Z:N/F.P@DOMAIN abcd1234
	 * ->  <abcd1234%DOMAIN@pP.fF.nN.zZ.fidonet.org>
	 * (DOMAIN is optional)
	 */
	if(idnode.domain[0])		    /* With DOMAIN part */
	    sprintf(tcharp, "<%s%%%s@%s%s>",
		    id, idnode.domain, node_to_pfnz(&idnode, TRUE),
		    FIDODOMAIN);
	else
	    sprintf(tcharp, "<%s@%s%s>",
		    id, node_to_pfnz(&idnode, TRUE), FIDODOMAIN);
    }
    else
    {
	/*
	 * Not a FIDO address, so use the system part of ^AMSGID as an
	 * RFC address.
	 */
	sprintf(tcharp, "<%s@%s>", id, system);
#if 0
	/*
	 * Special kludge for handling MAUS message ids: convert axxxxx id to
	 * uppercase to be compatible with MAUS gateway.
	 */
	if(wildmat(tcharp, "<a*@*.maus.de>")) 
	    tcharp[1] = 'A';
#endif	    
    }
    
    return tcharp;
}



/*
 * Generate ID for FIDO messages without ^AMSGID, using date and CRC over
 * From, To and Subject.
 */

char *msgid_default(node, msg_from, msg_to, msg_subj, msg_date)
    Node *node;
    char *msg_from;
    char *msg_to;
    char *msg_subj;
    time_t msg_date;
{
    SHUFFLEBUFFERS;

    /*
     * Compute CRC for strings from, to, subject
     */
    crc32_init();
    crc32_compute(msg_from, strlen(msg_from));
    crc32_compute(msg_to  , strlen(msg_to  ));
    crc32_compute(msg_subj, strlen(msg_subj));

    sprintf(tcharp, "<NOMSGID-%s-%08lx@%s%s>",
	    date("%y%m%d-%H%M%S", &msg_date),
	    crc32_value(), node_to_pfnz(node, TRUE), FIDODOMAIN);
    
    return tcharp;
}



/*
 * Convert RFC Message-ID/References to FIDO ^AMSGID/^AREPLY
 */

char *msgid_rfc_to_fido(origid_flag, message_id, part, split)
    int *origid_flag;			/* Flag for ^AORIGID */
    char *message_id;			/* Original RFC-ID */
    int part;				/* part number */
    int split;				/* != 0 # of parts */
{
    char *id, *host, *p;
    char *savep;
    Node node;
    int hexflag, fidoflag, i, len, dlen, id_flag;
    char hexid[9];
    unsigned long crc32;

    id_flag = FALSE;

    savep = strsave(message_id);
    /*
     * Format of message_id is "<identification@host.domain> ..."
     * We want the the last one in the chain, which is the message id
     * of the article replied to.
     */
    id = strrchr(savep, '<');
    if(!id)
    {
	free(savep);
	return NULL;
    }
    id++;
    host = strchr(id, '@');
    if(!host)
    {
	free(savep);
	return NULL;
    }
    *host++ = 0;
    p = strchr(host, '>');
    if(!p)  
    {
	free(savep);
	return NULL;
    }
    *p = 0;

#if 0
    /*
     * Special kludge for handling MAUS message ids, see also above
     */
    if(wildmat(host, "*.maus.de") && id[0]=='A') 
	id[0] = 'a';
#endif	    

    /*
     * Don't convert <funpack....@...> and <NOMSGID-...@...> IDs
     * generated by FIDOGATE
     */
    if(!strncmp(id, "funpack", 7) || !strncmp(id, "NOMSGID-", 8))
    {
	free(savep);
	return NULL;
    }

    /*
     * First check ID. A FIDO Message-ID is a hex number with an optional
     * %Domain string. The hex number must not start with `0' and it's length
     * must be no more then 8 digits.
     */
    node.domain[0] = 0;
    p = id;
    hexflag = isxdigit(*p) && *p!='0';
    for(p++, i=0; i<7 && *p && *p!='%'; i++, p++)
	if(!isxdigit(*p))
	    hexflag = FALSE;
    if(hexflag && *p=='%')		/* Domain part follows */
    {
	*p++ = 0;
	strncpy0(node.domain, p, MAX_DOMAIN);
    }
    else if(*p)
	hexflag = FALSE;
    /*
     * If this is a valid hex number, pad it with leading zero
     */
    if(hexflag) {
	strcpy(hexid, "00000000");
	strcpy(hexid + 8 - strlen(id), id);
    }
     
    /*
     * Try to interpret host.domain as a FIDO address. In this case the
     * domain must be FIDODOMAIN, i.e. ".fidonet.org".
     */
    /**FIXME: other domains than fidonet.org!**/
    len	 = strlen(host);
    dlen = strlen(FIDODOMAIN);
    if(len > dlen  &&  !stricmp(host + len - dlen, FIDODOMAIN))
    {
	char *host_pfnz;
	/* Found it! */
	host_pfnz = strsave(host);
	host_pfnz[len - dlen] = 0;
	if( ( fidoflag = (pfnz_to_node(host_pfnz, &node, TRUE) != ERROR) ) )
	    host = node_to_asc(&node, TRUE);
	free(host_pfnz);
    }

    /*
     * Generate string for ^AMSGID/REPLY kludge
     */
    SHUFFLEBUFFERS;
    
    if(hexflag && !split)
    {
	/* ID is a hex number and can be use directly */
	sprintf(tcharp, "%s %s", host, hexid);
	id_flag = FALSE;
    }
    else
    {
	/* Use CRC32 over ID part for hex number */
	crc32 = compute_crc32((unsigned char *)id, strlen(id));
	if(split)
	    /* ID for splitted message is incremented by one for every part */
	    sprintf(tcharp, "%s %08lx", host, crc32 + part - 1);
	else
	    sprintf(tcharp, "%s %08lx", host, crc32);
	id_flag = TRUE;
    }

    free(savep);
    if(origid_flag)
	*origid_flag = id_flag;
    return tcharp;
}



/*
 * Generate string for ^AORIGID/^AORIGREF kludge
 */

char *msgid_rfc_to_origid(message_id, part, split)
char *message_id;			/* Original RFC-ID */
int part;				/* part number */
int split;				/* != 0 # of parts */
{
    char *id, *p;
    char *savep;

    savep = strsave(message_id);
    /*
     * Format of message_id is "<identification@host.domain> ..."
     * We want the the last one in the chain, which is the message id
     * of the article replied to.
     */
    id = strrchr(savep, '<');
    if(!id)
    {
	free(savep);
	return NULL;
    }
    p = strchr(id, '>');
    if(!p)  
    {
	free(savep);
	return NULL;
    }
    *++p = 0;

    SHUFFLEBUFFERS;
    
    if(split)
	sprintf(tcharp, "%s %d/%d", id, part, split);
    else
	sprintf(tcharp, "%s", id);

    free(savep);
    return tcharp;
}



/*
 * Extract Message-ID from ^AORIGID/^AORIGREF with special handling for
 * splitted messages (appended " i/n"). Returns NULL for invalid ^AORIGID.
 */
char *msgid_convert_origid(origid, part_flag)
char *origid;
int part_flag;
{
    char *s, *p, *id, *part;
    
    s    = strsave(origid);

    id   = s;
    part = NULL;
    p    = strrchr(s, '>');
    if(!p)
    {
	free(s);
	debug(1, "Invalid ^AORIGID: %s", origid);
	return NULL;
    }

    p++;
    if(isspace(*p))
    {
	/*
	 * Indication of splitted message " p/n" follows ...
	 */
	*p++ = 0;
	while(isspace(*p)) p++;
	part = p;
    }

    /*
     * Message-IDs must NOT contain white spaces
     */
    if(strchr(id, ' ') || strchr(id, '\t'))
    {
	free(s);
	debug(1, "Invalid ^AORIGID: %s", origid);
	return NULL;
    }

    SHUFFLEBUFFERS;
    
    if(part && part_flag)
    {
	/*
	 * Prepend "i/n" as "Parti.n-" to id
	 */
	char *p;
	
	for(p=part; *p; p++)
	    switch(*p)
	    {
	    case '/':
		*p = '.';
		break;
	    }
	
	sprintf(tcharp, "<Part%s-%s", part, id+1);
    }
    else
	/*
	 * Return just the id
	 */
	sprintf(tcharp, "%s", id);
    
    free(s);
    
    return tcharp;
}
