/*:ts=8*/
/*****************************************************************************
 * FIDOGATE --- Gateway UNIX Mail/News <-> FIDO NetMail/EchoMail
 *
 * $Id: ftntoss.c,v 3.8.0.7 1994/10/04 07:04:45 mj Exp mj $
 *
 * Toss FTN NetMail/EchoMail
 *
 *****************************************************************************
 * Copyright (C) 1990, 1993, 1994
 *  _____ _____
 * |     |___  |   Martin Junius             FIDO:      2:2452/110.1
 * | | | |   | |   Republikplatz 3           Internet:  mj@sungate.fido.de
 * |_|_|_|@home|   D-52072 Aachen, Germany   Phone:     ++49-241-86931 (voice)
 *
 * This file is part of FIDOGATE.
 *
 * FIDOGATE is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * FIDOGATE is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with FIDOGATE; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/

#include "fidogate.h"
#include "getopt.h"

#include <fcntl.h>



#define PROGRAM "ftntoss"
#define VERSION "$Revision: 3.8.0.7 $"

#define MY_CONFIG "%L/config.toss"
#define ROUTING   "%L/routing"



/*
 * Prototypes
 */
void	lon_to_kludge		P((Textlist *, char *, LON *));
void	lon_to_kludge_sorted	P((Textlist *, char *, LON *));
int	toss_echomail		P((Message *, MsgBody *, LON *, LON *, LON *));
int	kludge_to_lon		P((Textlist *, LON *));
int	do_seenby		P((LON *, LON *, LON *));
int	do_path			P((LON *));
int	do_echomail		P((Packet *, Message *, MsgBody *));
void	add_via			P((Textlist *, Node *));
void	change_addr		P((Node *, Node *));
void	do_rewrite		P((Message *));
void	do_remap		P((Message *));
int	do_netmail		P((Packet *, Message *, MsgBody *));
int	unpack			P((FILE *, Packet *));
int	unpack_file		P((char *));

void	short_usage		P((void));
void	usage			P((void));



/*
 * Command line options
 */
int g_flag = 'n';			/* Processing grade */
int t_flag = FALSE;			/* Insecure tossing enabled */
int n_flag = FALSE;			/* Accept EchoMail messages not */
					/* addressed to our AKA         */
int s_flag = FALSE;			/* Strip CRASH, HOLD attribute */
int maxmsg = 0;				/* Process maxmsg messages */
int x_flag = FALSE;			/* Exit after maxmsg messages */

static char in_dir [MAXPATH];		/* Input directory */


static int must_exit = FALSE;		/* Flag for -x operation */
static int msg_count = 0;		/* Counter for -m, -x operation */


/*
 * Global stat counts
 */
static long msgs_in       = 0;		/* Input messages */
static long msgs_netmail  = 0;		/* Output messages NetMail */
static long msgs_echomail = 0;		/* Output messages NetMail */



/*
 * Create new SEEN-BY or ^APATH line from LON
 */
#define MAX_LENGTH 76

void lon_to_kludge(tl, text, lon)
    Textlist *tl;
    char *text;
    LON *lon;
{
    LNode *p;
    Node old;
    char *s = NULL;
    
    strncpy0(buffer, text, sizeof(buffer));
    node_invalid(&old);
    old.zone = cf_zone();

    for(p=lon->first; p; p=p->next)
	if(p->node.point == 0)			/* No 4D Points !!! */
	{
	    s = node_to_asc_diff(&p->node, &old);
	    old = p->node;
	    
	    if(strlen(buffer)+strlen(s)+1 > MAX_LENGTH)
	    {
		strncat0(buffer, "\r\n", sizeof(buffer));
		Textlist_append(tl, buffer);
		
		node_invalid(&old);
		old.zone = cf_zone();
		
		s = node_to_asc_diff(&p->node, &old);
		old = p->node;
		strncpy0(buffer, text, sizeof(buffer));
		strncat0(buffer, " ", sizeof(buffer));
		strncat0(buffer, s  , sizeof(buffer));
	    }
	    else
	    {
		strncat0(buffer, " ", sizeof(buffer));
		strncat0(buffer, s  , sizeof(buffer));
	    }
	}
    
    strncat0(buffer, "\r\n", sizeof(buffer));
    Textlist_append(tl, buffer);
}


void lon_to_kludge_sorted(tl, text, lon)
    Textlist *tl;
    char *text;
    LON *lon;
{
    Node old;
    char *s = NULL;
    int i;

    lon_sort(lon, 0);
    
    strncpy0(buffer, text, sizeof(buffer));
    node_invalid(&old);
    old.zone = cf_zone();

    for(i=0; i<lon->size; i++)
	if(lon->sorted[i]->point == 0)		/* No 4D Points !!! */
	{
	    s = node_to_asc_diff(lon->sorted[i], &old);
	    old = *lon->sorted[i];
	    
	    if(strlen(buffer)+strlen(s)+1 > MAX_LENGTH)
	    {
		strncat0(buffer, "\r\n", sizeof(buffer));
		Textlist_append(tl, buffer);
		
		node_invalid(&old);
		old.zone = cf_zone();
		
		s = node_to_asc_diff(lon->sorted[i], &old);
		old = *lon->sorted[i];
		strncpy0(buffer, text, sizeof(buffer));
		strncat0(buffer, " ", sizeof(buffer));
		strncat0(buffer, s  , sizeof(buffer));
	    }
	    else
	    {
		strncat0(buffer, " ", sizeof(buffer));
		strncat0(buffer, s  , sizeof(buffer));
	    }
	}
    
    strncat0(buffer, "\r\n", sizeof(buffer));
    Textlist_append(tl, buffer);
}

    

/*
 * Toss EchoMail, writing message to packet for each downlink
 */
int toss_echomail(msg, body, seenby, path, nodes)
    Message *msg;
    MsgBody *body;
    LON *seenby;
    LON *path;
    LON *nodes;
{
    LNode *p;
    FILE *fp;

    for(p=nodes->first; p; p=p->next)
    {
	/* Rewrite message header */
	msg->node_from = cf_n_fake();
	msg->node_to   = p->node;
	/* Open packet file, EchoMail, Normal */
	fp = outpkt_open(&p->node, g_flag, 'e', 'n');
	if(fp == NULL)
	    return ERROR;
	/* Write message header and body */
	if( pkt_put_msg_hdr(fp, msg, FALSE) != OK )
	    return ERROR;
	if( msg_put_msgbody(fp, body, TRUE) != OK )
	    return ERROR;

	msgs_echomail++;
    }
    
    return OK;
}



/*
 * Convert SEEN-BY or ^APATH lines to LON
 */
int kludge_to_lon(tl, lon)
    Textlist *tl;
    LON *lon;
{
    Node old, node;
    Textline *l;
    char *p;

    for(l=tl->first; l; l=l->next)
    {
	p = l->line;
	/* Skip SEEN-BY:, ^APATH: or whatever, copy to buffer[] */
	while(*p && !isspace(*p) && *p!=':')
	    p++;
	if(*p == ':')
	    p++;
	while(*p && isspace(*p))
	    p++;
	strncpy0(buffer, p, sizeof(buffer));
	strip_crlf(buffer);
	/* Tokenize and convert to node addresses */
	node_invalid(&old);
	old.zone = cf_zone();
	for(p=strtok(buffer, " \t"); p; p=strtok(NULL, " \t"))
	    if(asc_to_node_diff(p, &node, &old) == OK)
	    {
		lon_add(lon, &node);
		old = node;
	    }
    }
	
    return OK;
}



/*
 * Add nodes to SEEN-BY
 */
int do_seenby(seenby, nodes, new)
    LON *seenby;				/* Nodes in SEEN-BY lines */
    LON *nodes;					/* Nodes in AREAS.BBS */
    LON *new;					/* New nodes added */
{
    LNode *p;
    
    for(p=nodes->first; p; p=p->next)
	if( ! lon_search(seenby, &p->node) )
	{
	    lon_add(seenby, &p->node);
	    lon_add(new   , &p->node);
	}
    
    return OK;
}



/*
 * Add current address to ^APATH
 */
int do_path(path)
    LON *path;
{
    if(path->last && node_eq(&path->last->node, cf_addr()))
	/* Already there */
	return OK;
    
    lon_add(path, cf_addr());
    return OK;
}



/*
 * Process EchoMail message
 */
int do_echomail(pkt, msg, body)
    Packet *pkt;
    Message *msg;
    MsgBody *body;
{
    AreasBBS *area;
    LON seenby, path, new;
    int ret;

    if(!n_flag)
    {
	/*
	 * Check that this message is addressed to one of our AKAs
	 */
    }
    
    /*
     * Lookup area and set zone
     */
    strncpy0(buffer, body->area + strlen("AREA:"), sizeof(buffer));
    strip_crlf(buffer);
    debug(2, "EchoMail AREA: %s", buffer);
    
    if( (area = areasbbs_lookup(buffer)) == NULL )
    {
	/* Unknown area: log it, dump it. ;-) */
	log("unknown area %s", buffer);
	return OK;
    }
    cf_set_zone(area->zone);

    if(!t_flag)
    {
	/*
	 * Check that origin is listed in AREAS.BBS
	 */
    }
    
    /*
     * Convert SEEN-BY and ^APATH to LON
     */
    lon_init(&seenby);
    lon_init(&path);
    lon_init(&new);
    kludge_to_lon(&body->seenby, &seenby);
    kludge_to_lon(&body->path  , &path);
    Textlist_clear(&body->seenby);
    Textlist_clear(&body->path);
    /*
     * If from 4D point, add address to SEEN-BY to prevent sending
     * message back to this point. Won't show up in output SEEN-BY
     * because all point address are stripped then.
     */
    if(msg->node_from.point)
	lon_add(&seenby, &msg->node_from);

    lon_debug(6, "SEEN-BY: ", &seenby, TRUE);
    lon_debug(6, "Path   : ", &path,   TRUE);
    
    /*
     * Add nodes not already in SEEN-BY to seenby and new.
     */
    do_seenby(&seenby, &area->nodes, &new);
    
    /*
     * Add our address to end of ^APATH, if not already there.
     */
    do_path(&path);

    lon_debug(6, "SEEN-BY: ", &seenby, TRUE);
    lon_debug(6, "Path   : ", &path,   TRUE);
    lon_debug(6, "New    : ", &new,    TRUE);

    /*
     * Add new SEEN-BY and ^APATH lines
     */
    lon_to_kludge_sorted(&body->seenby, "SEEN-BY:", &seenby);
    lon_to_kludge       (&body->path  , "\001PATH:", &path );

    /*
     * Send message to all downlinks in new
     */
    ret = toss_echomail(msg, body, &seenby, &path, &new);

    lon_delete(&seenby);
    lon_delete(&path);
    lon_delete(&new);

    return ret;
}



/*
 * Add our ^AVia line
 */
void add_via(list, gate)
    Textlist *list;
    Node *gate;
{
    Textlist_appendf(list, "\001Via %s %s, %s\r\n",
		     PROGRAM, node_to_asc(gate, FALSE),
		     date("%a %b %d %Y at %H:%M:%S %Z", NULL)  );
}



/*
 * Change address according to rewrite pattern
 */
void change_addr(node, newpat)
    Node *node;
    Node *newpat;
{
    if(newpat->zone  != -1)
	node->zone  = newpat->zone;
    if(newpat->net   != -1)
	node->net   = newpat->net;
    if(newpat->node  != -1)
	node->node  = newpat->node;
    if(newpat->point != -1)
	node->point = newpat->point;
}



/*
 * Perform REWRITE commands
 */
void do_rewrite(msg)
    Message *msg;
{
    Rewrite *r;
    Node node;
    
    /* From */
    for(r=rewrite_first; r; r=r->next)
	if(node_match(&msg->node_from, &r->from))
	{
	    node = msg->node_from;
	    change_addr(&msg->node_from, &r->to);
	    log("rewrite: %s -> %s",
		node_to_asc(&node, TRUE), node_to_asc(&msg->node_from, TRUE) );
	    break;
	}

    /* To */
    for(r=rewrite_first; r; r=r->next)
	if(node_match(&msg->node_to, &r->from))
	{
	    node = msg->node_to;
	    change_addr(&msg->node_to, &r->to);
	    log("rewrite: %s -> %s",
		node_to_asc(&node, TRUE), node_to_asc(&msg->node_to, TRUE) );
	    break;
	}
}



/*
 * Perform REMAP commands
 */
void do_remap(msg)
    Message *msg;
{
    Remap *r;
    Node node;
    
    for(r=remap_first; r; r=r->next)
	if(node_match(&msg->node_to, &r->from) &&
	   wildmat(msg->name_to, r->name)         )
	{
	    node = msg->node_to;
	    change_addr(&msg->node_to, &r->to);
	    log("remap: %s @ %s -> %s", msg->name_to,
		node_to_asc(&node, TRUE), node_to_asc(&msg->node_to, TRUE) );
	    break;
	}
}



/*
 * Process NetMail message
 */
int do_netmail(pkt, msg, body)
    Packet *pkt;
    Message *msg;
    MsgBody *body;
{
    FILE *fp;
    int flav;

    /*
     * Rewrite from/to addresses according to ROUTING rules
     */
    do_rewrite(msg);
    
    /*
     * Remap to address according to ROUTING rules
     */
    do_remap(msg);
    
    /*
     * Write to output packet
     */
    cf_set_zone(msg->node_to.zone);

    /* Get outbound flavor from msg attributes */
    flav = 'n';
    if(!s_flag)
    {
	if(msg->attr & MSG_HOLD)
	    flav = 'h';
	if(msg->attr & MSG_CRASH)
	    flav = 'c';
    }

    /* Open output packet */
    fp = outpkt_open(&msg->node_to, g_flag, 'n', flav);
    if(fp == NULL)
	return ERROR;

    /* Add ftnroute ^AVia line */
    add_via(&body->via, cf_addr());

    /* Write message header and body */
    if( pkt_put_msg_hdr(fp, msg, TRUE) != OK )
	return ERROR;
    if( msg_put_msgbody(fp, body, TRUE) != OK )
	return ERROR;

    msgs_netmail++;
    
    return OK;
}



/*
 * Read and process FTN packets
 */
int unpack(pkt_file, pkt)
    FILE *pkt_file;
    Packet *pkt;
{
    Message msg;			/* Message header */
    Textlist tl;			/* Textlist for message body */
    MsgBody body;			/* Message body of FTN message */
    int type;
    
    /*
     * Initialize
     */
    Textlist_init(&tl);
    msg_body_init(&body);


    /*
     * Read packet
     */
    type = pkt_get_int16(pkt_file);
    while(type == MSG_TYPE)
    {
	/*
	 * Read message header
	 */
	msg.node_from = pkt->from;
	msg.node_to   = pkt->to;
	if(pkt_get_msg_hdr(pkt_file, &msg) == ERROR)
	{
	    log("%s: error reading input packet", PROGRAM);
	    return ERROR;
	}
	
	/*
	 * Read message body
	 */
	type = pkt_get_body(pkt_file, &tl);
	if(type == ERROR)
	{
	    log("%s: error reading input packet", PROGRAM);
	    return ERROR;
	}
	msgs_in++;
	msg_count++;
	
	/*
	 * Parse message body
	 */
	if( msg_body_parse(&tl, &body) == -2 )
	    log("%s: error parsing message body", PROGRAM);
	/* Retrieve address information from kludges for NetMail */
	if(body.area == NULL)
	{
	    /* Retrieve complete address from kludges */
	    kludge_pt_intl(&body, &msg);
	    msg.node_orig = msg.node_from;

	    debug(2, "NetMail: %s -> %s",
		  node_to_asc(&msg.node_from, TRUE),
		  node_to_asc(&msg.node_to  , TRUE) );
	    if(do_netmail(pkt, &msg, &body) == ERROR)
		return ERROR;
	}
	else 
	{
	    msg.node_orig = msg.node_from;

	    debug(2, "EchoMail: %s -> %s",
		  node_to_asc(&msg.node_from, TRUE),
		  node_to_asc(&msg.node_to  , TRUE) );
	    if(do_echomail(pkt, &msg, &body) == ERROR)
		return ERROR;
	}

	/*
	 * Check for number of messages exceeding maxmsg
	 */
	if(maxmsg  &&  msg_count >= maxmsg)
	{
	    if(x_flag)
		must_exit = TRUE;
	    else
		outpkt_close();
	    msg_count = 0;
	}

    } /**while(type == MSG_TYPE)**/

    return OK;
}



/*
 * Unpack one packet file
 */
int unpack_file(pkt_name)
    char *pkt_name;
{
    Packet pkt;
    FILE *pkt_file;

    debug(1, "Unpacking FTN packet %s", pkt_name);

    /*
     * Open packet and read header
     */
    pkt_file = fopen(pkt_name, R_MODE);
    if(!pkt_file) {
	log("$Can't open packet %s", pkt_name);
	return ERROR;
    }
    if(pkt_get_hdr(pkt_file, &pkt) == ERROR)
    {
	log("Error reading header from %s", pkt_name);
	fclose(pkt_file);
	return ERROR;
    }
    
    /*
     * Unpack it
     */
    if(unpack(pkt_file, &pkt) == ERROR) 
    {
	log("Error in unpacking %s", pkt_name);
	fclose(pkt_file);
	return ERROR;
    }
    
    fclose(pkt_file);

    if (unlink(pkt_name)) {
	log("$Could not unlink packet %s", pkt_name);
	return ERROR;
    }

    return OK;
}



/*
 * Usage messages
 */
void short_usage()
{
    fprintf(stderr, "usage: %s [-options] [packet ...]\n", PROGRAM);
    fprintf(stderr, "       %s --help  for more information\n", PROGRAM);
}


void usage()
{
    fprintf(stderr, "FIDOGATE %s  %s %s\n\n",
	    version_global(), PROGRAM, version_local(VERSION) );
    
    fprintf(stderr, "usage:   %s [-options] [packet ...]\n\n", PROGRAM);
    fprintf(stderr, "\
options: -g --grade G                 processing grade\n\
         -I --in-dir name             set input packet directory\n\
         -O --out-dir name            set output packet directory\n\
         -l --lock-file               create lock file while processing\n\
         -t --insecure                insecure tossing (no AREAS.BBS check)\n\
         -n --toss-all                toss all EchoMail messages\n\
         -r --routing-file NAME       read routing file\n\
         -s --strip-attribute         strip crash, hold message attribute\n\
         -m --maxmsg N                close output after N msgs\n\
         -x --maxmsg-exit N           close output and exit after N msgs\n\
         -M --maxopen N               set max # of open packet files\n\
\n\
	 -v --verbose                 more verbose\n\
	 -h --help                    this help\n\
         -c --config name             read config file (\"\" = none)\n\
	 -L --lib-dir name            set lib directory\n\
	 -S --spool-dir name          set spool directory\n\
	 -a --addr Z:N/F.P            set FTN address\n\
	 -f --fake-addr Z:N/F.P       set FTN fakenet address\n\
	 -u --uplink-addr Z:N/F.P     set FTN uplink address\n\
         -z --zone Z                  select aka for zone Z\n"  );
}



/***** main() ****************************************************************/

int main(argc, argv)
    int argc;
    char *argv[];
{
    struct dirent *dir;
    DIR *dp;
    int c, ret;
    int l_flag = FALSE;
    char *I_flag=NULL, *O_flag=NULL, *r_flag=NULL;
    char *c_flag=NULL;
    char *S_flag=NULL, *L_flag=NULL;
    char *a_flag=NULL, *f_flag=NULL, *u_flag=NULL;
    char *z_flag=NULL;
    char pkt_name[MAXPATH];

    int option_index;
    static struct option long_options[] =
    {
	{ "grade",        1, 0, 'g'},	/* grade */
	{ "in-dir",       1, 0, 'I'},	/* Set inbound packets directory */
	{ "lock-file",    0, 0, 'l'},	/* Create lock file while processing */
	{ "out-dir",      1, 0, 'O'},	/* Set packet directory */
	{ "insecure",     0, 0, 't'},	/* Insecure */
	{ "toss-all",     0, 0, 'n'},	/* Toss all EchoMail */
	{ "routing-file", 1, 0, 'r'},	/* Set routing file */
	{ "strip-attribute",0,0,'s'},	/* Strip attribute */
	{ "maxmsg",       1, 0, 'm'},	/* Close after N messages */
	{ "maxmsg-exit",  1, 0, 'x'},	/* Exit after N messages */
	{ "maxopen",      1, 0, 'M'},	/* Set max # open packet files */

	{ "verbose",      0, 0, 'v'},	/* More verbose */
	{ "help",         0, 0, 'h'},	/* Help */
	{ "config",       1, 0, 'c'},	/* Config file */
	{ "spool-dir",    1, 0, 'S'},	/* Set FIDOGATE spool directory */
	{ "lib-dir",      1, 0, 'L'},	/* Set FIDOGATE lib directory */
	{ "addr",         1, 0, 'a'},	/* Set FIDO address */
	{ "fake-addr",    1, 0, 'f'},	/* Set FIDO fakenet address */
	{ "uplink-addr",  1, 0, 'u'},	/* Set FIDO uplink address */
	{ "zone",         1, 0, 'z'},	/* Select aka for zone */
	{ 0,              0, 0, 0  }
    };

    log_program(PROGRAM);
    
    /* Init configuration */
    cf_initialize();


    while ((c = getopt_long(argc, argv, "g:O:I:ltnr:sm:x:M:vhc:S:L:a:f:u:z:",
			    long_options, &option_index     )) != EOF)
	switch (c) {
	/***** ftntoss options *****/
	case 'g':
	    g_flag = *optarg;
	    break;
	case 'I':
	    I_flag = optarg;
	    break;
        case 'l':
	    l_flag = TRUE;
            break;
	case 'O':
	    O_flag = optarg;
	    break;
	case 't':
	    t_flag = TRUE;
	    break;
	case 'n':
	    n_flag = TRUE;
	    break;
	case 'r':
	    r_flag = optarg;
	    break;
	case 's':
	    s_flag = TRUE;
	    break;
	case 'm':
	    maxmsg = atoi(optarg);
	    break;
	case 'x':
	    maxmsg = atoi(optarg);
	    x_flag = TRUE;
	    break;
	case 'M':
	    outpkt_set_maxopen(atoi(optarg));
	    break;
	    
	/***** Common options *****/
	case 'v':
	    verbose++;
	    break;
	case 'h':
	    usage();
	    exit(0);
	    break;
	case 'c':
	    c_flag = optarg;
	    break;
	case 'S':
	    S_flag = optarg;
	    break;
	case 'L':
	    L_flag = optarg;
	    break;
	case 'a':
	    a_flag = optarg;
	    break;
	case 'f':
	    f_flag = optarg;
	    break;
	case 'u':
	    u_flag = optarg;
	    break;
	case 'z':
	    z_flag = optarg;
	    break;
	default:
	    short_usage();
	    exit(EX_USAGE);
	    break;
	}

    /*
     * Read config file
     */
    if(L_flag)				/* Must set libdir beforehand */
	cf_set_libdir(L_flag);
    cf_read_config_file(c_flag ? c_flag : MY_CONFIG);

    /*
     * Process config options
     */
    if(L_flag)
	cf_set_libdir(L_flag);
    if(S_flag)
	cf_set_spooldir(S_flag);
    if(a_flag)
	cf_set_addr(a_flag);
    if(f_flag)
	cf_set_fake(f_flag);
    if(u_flag)
	cf_set_uplink(u_flag);

    cf_debug();

    if(z_flag)
	cf_set_zone(atoi(z_flag));
    
    /*
     * Process local options
     */
    if(I_flag)
	strncpy0(in_dir, I_flag, sizeof(in_dir));
    else 
	sprintf(in_dir, "%s/%s", cf_spooldir(), TOSS_IN);
    if(O_flag)
	pkt_outdir(O_flag, NULL);
    else
	pkt_outdir(cf_spooldir(), TOSS_TMP);

    routing_init(r_flag ? r_flag : ROUTING);
    
    /*
     * Read AREAS.BBS
     */
    areasbbs_init(cf_areas_bbs());

    ret = EXIT_OK;
    
    if(optind >= argc)
    {
	/*
	 * No packet file args, process in directory
	 */
	dp = opendir(in_dir);
	
	if(!dp)
	{
	    log("$Unable to open inbound directory %s", in_dir);
	    exit(EX_OSERR);
	}
    
	/* Lock file */
	if(l_flag)
	    if(lock_program(PROGRAM, FALSE) == ERROR)
		/* Already busy */
		exit(EXIT_BUSY);
	
	while((dir = readdir(dp)))
	{
	    if(wildmat(dir->d_name, "*.[pP][kK][tT]"))
	    {
		sprintf(pkt_name, "%s/%s", in_dir, dir->d_name);
		if(unpack_file(pkt_name) == ERROR)
		{
		    ret = EXIT_ERROR;
		    break;
		}
		if(must_exit)
		{
		    ret = EXIT_CONTINUE;
		    break;
		}
	    }
	}
	
	closedir(dp);

	/* Lock file */
	if(l_flag)
	    unlock_program(PROGRAM);
    }
    else
    {
	/* Lock file */
	if(l_flag)
	    if(lock_program(PROGRAM, FALSE) == ERROR)
		/* Already busy */
		exit(EXIT_BUSY);
	
	/*
	 * Process packet files on command line
	 */
	for(; optind<argc; optind++)
	{
	    if(unpack_file(argv[optind]) == ERROR)
	    {
		ret = EXIT_ERROR;
		break;
	    }
	    if(must_exit)
	    {
		ret = EXIT_CONTINUE;
		break;
	    }
	}
	
	/* Lock file */
	if(l_flag)
	    unlock_program(PROGRAM);
    }
    
    outpkt_close();

    if(msgs_in)
	log("msgs processed: %ld in, %ld out (%ld mail, %ld echo)",
	    msgs_in, msgs_netmail+msgs_echomail, msgs_netmail, msgs_echomail);
    
    exit(ret);

    /**NOT REACHED**/
    return 1;
}
