/*:ts=8*/
/*****************************************************************************
 * FIDOGATE --- Gateway UNIX Mail/News <-> FIDO NetMail/EchoMail
 *
 * $Id: ftnroute.c,v 3.8.0.7 1994/10/04 07:04:45 mj Exp mj $
 *
 * Route FTN NetMail/EchoMail
 *
 *****************************************************************************
 * Copyright (C) 1990, 1993, 1994
 *  _____ _____
 * |     |___  |   Martin Junius             FIDO:      2:2452/110.1
 * | | | |   | |   Republikplatz 3           Internet:  mj@sungate.fido.de
 * |_|_|_|@home|   D-52072 Aachen, Germany   Phone:     ++49-241-86931 (voice)
 *
 * This file is part of FIDOGATE.
 *
 * FIDOGATE is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * FIDOGATE is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with FIDOGATE; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/

#include "fidogate.h"
#include "getopt.h"

#include <fcntl.h>




#define PROGRAM "ftnroute"
#define VERSION "$Revision: 3.8.0.7 $"

#define MY_CONFIG "%L/config.toss"
#define ROUTING   "%L/routing"



/*
 * Prototypes
 */
int	do_routing		P((char *, FILE *, Packet *));
void	do_cmd			P((PktDesc *, Routing *));
int	do_packet		P((FILE *, Packet *, PktDesc *));
void	add_via			P((Textlist *, Node *));
int	do_file			P((char *));
void	short_usage		P((void));
void	usage			P((void));



/*
 * Command line options
 */
int g_flag = 0;

static char in_dir [MAXPATH];



/*
 * Route packet
 */
int do_routing(name, fp, pkt)
    char *name;
    FILE *fp;
    Packet *pkt;
{
    PktDesc *desc;
    Routing *r;
    LNode *p;
    
    desc = parse_pkt_name(name, &pkt->to);
    if(desc == NULL)
	return ERROR;

    debug(1, "Source packet: node=%s grade=%c type=%c flav=%c",
	  node_to_asc(&desc->node, TRUE), desc->grade, desc->type, desc->flav);

    /*
     * Search for matching routing commands
     */
    for(r=routing_first; r; r=r->next)
	if(desc->type == r->type)
	    for(p=r->nodes.first; p; p=p->next)
		if(node_match(&desc->node, &p->node))
		{
		    debug(4, "routing: type=%c cmd=%c flav=%c flav_new=%c",
			  r->type, r->cmd, r->flav, r->flav_new            );
		    lon_debug(4, "routing: nodes=", &r->nodes, TRUE);
		    
		    do_cmd(desc, r);
		    
		    break;			/* Inner for loop */
		}

    /*
     * Write contents of this packet to output packet
     */
    debug(1, "Target packet: node=%s grade=%c type=%c flav=%c",
	  node_to_asc(&desc->node, TRUE), desc->grade, desc->type, desc->flav);

    return do_packet(fp, pkt, desc);
}



/*
 * Exec routing command
 */
void do_cmd(desc, r)
    PktDesc *desc;
    Routing *r;
{
    switch(r->cmd)
    {
    case CMD_SEND:
	if(desc->flav == FLAV_NORMAL)
	{
	    debug(2, "send %c %s", r->flav, node_to_asc(&desc->node, TRUE));
	    desc->flav = r->flav;
	}
	break;
	
    case CMD_ROUTE:
	if(desc->flav == FLAV_NORMAL)
	{
	    debug(2, "route %c %s -> %s", r->flav,
		  node_to_asc(&desc->node, TRUE),
		  node_to_asc(&r->nodes.first->node, TRUE) );
	    desc->flav = r->flav;
	    desc->node = r->nodes.first->node;
	}
	break;

    case CMD_CHANGE:
	if(desc->flav == r->flav)
	{
	    debug(2, "change %c -> %c %s", r->flav, r->flav_new,
		  node_to_asc(&desc->node, TRUE)                );
	    desc->flav = r->flav_new;
	}
	break;
	
    case CMD_HOSTROUTE:
	if(desc->flav == FLAV_NORMAL)
	{
	    debug(2, "hostroute %c %s", r->flav,
		  node_to_asc(&desc->node, TRUE) );
	    desc->flav = r->flav;
	    desc->node.node  = 0;
	    desc->node.point = 0;
	}
	break;
	
    case CMD_HUBROUTE:
	debug(2, "hubroute not yet implemented");
	break;
	
    default:
	debug(2, "unknown routing command, strange");
	break;
    }

    /*
     * Set all -1 values to 0
     */
    if(desc->node.zone  == -1)
	desc->node.zone  = 0;
    if(desc->node.net   == -1)
	desc->node.net   = 0;
    if(desc->node.node  == -1)
	desc->node.node  = 0;
    if(desc->node.point == -1)
	desc->node.point = 0;
}



/*
 * Add our ^AVia line
 */
void add_via(list, gate)
    Textlist *list;
    Node *gate;
{
    Textlist_appendf(list, "\001Via %s %s, %s\r\n",
		     PROGRAM, node_to_asc(gate, FALSE),
		     date("%a %b %d %Y at %H:%M:%S %Z", NULL)  );
}



/*
 * Read and process packets, writing messages to output packet
 */
int do_packet(pkt_file, pkt, desc)
    FILE *pkt_file;
    Packet *pkt;
    PktDesc *desc;
{
    Message msg;			/* Message header */
    Textlist tl;			/* Textlist for message body */
    MsgBody body;			/* Message body of FTN message */
    int type;
    FILE *fp;
    
    /*
     * Initialize
     */
    Textlist_init(&tl);
    msg_body_init(&body);

    /*
     * Open output packet
     */
    cf_set_zone(desc->node.zone);
    fp = outpkt_open(&desc->node, desc->grade, desc->type, desc->flav);
    if(fp == NULL)
	return ERROR;

    /*
     * Read message from input packet and write to output packet
     */
    type = pkt_get_int16(pkt_file);
    while(type == MSG_TYPE)
    {
	/*
	 * Read message header
	 */
	msg.node_from = pkt->from;
	msg.node_to   = pkt->to;
	if(pkt_get_msg_hdr(pkt_file, &msg) == ERROR)
	{
	    log("error reading input packet");
	    return ERROR;
	}
	
	/*
	 * Read message body
	 */
	type = pkt_get_body(pkt_file, &tl);
	if(type == ERROR)
	{
	    log("error reading input packet");
	    return ERROR;
	}

	/*
	 * Parse message body
	 */
	if( msg_body_parse(&tl, &body) == -2 )
	    log("error parsing message body");

	if(body.area == NULL)
	{
	    /*
	     * Special processing for NetMail
	     */
	    /* Retrieve address from kludges */
	    kludge_pt_intl(&body, &msg);
	    /* Add ftnroute ^AVia line */
	    add_via(&body.via, cf_addr());
	    
	    /* Write message header and body */
	    if( pkt_put_msg_hdr(fp, &msg, TRUE) != OK )
		return ERROR;
	    if( msg_put_msgbody(fp, &body, TRUE) != OK )
		return ERROR;
	}
	else
	{
	    /*
	     * EchoMail
	     */
	    /* Write message header and body */
	    if( pkt_put_msg_hdr(fp, &msg, FALSE) != OK )
		return ERROR;
	    if( msg_put_msgbody(fp, &body, TRUE) != OK )
		return ERROR;
	}

    } /**while(type == MSG_TYPE)**/

    return OK;
}



/*
 * Process one packet file
 */
int do_file(pkt_name)
    char *pkt_name;
{
    Packet pkt;
    FILE *pkt_file;

    debug(1, "FTN packet %s", pkt_name);

    /*
     * Open packet and read header
     */
    pkt_file = fopen(pkt_name, R_MODE);
    if(!pkt_file) {
	log("$can't open packet %s", pkt_name);
	return ERROR;
    }
    if(pkt_get_hdr(pkt_file, &pkt) == ERROR)
    {
	log("error reading header from %s", pkt_name);
	return ERROR;
    }

    /*
     * Route it
     */
    if(do_routing(pkt_name, pkt_file, &pkt) == ERROR) 
    {
	log("error in processing %s", pkt_name);
	return ERROR;
    }

    if(fclose(pkt_file) == ERROR)
    {
	log("$could not close packet %s", pkt_name);
	return ERROR;
    }

    if (unlink(pkt_name)) {
	log("$could not unlink packet %s", pkt_name);
	return ERROR;
    }

    return OK;
}



/*
 * Usage messages
 */
void short_usage()
{
    fprintf(stderr, "usage: %s [-options] [packet ...]\n", PROGRAM);
    fprintf(stderr, "       %s --help  for more information\n", PROGRAM);
}


void usage()
{
    fprintf(stderr, "FIDOGATE %s  %s %s\n\n",
	    version_global(), PROGRAM, version_local(VERSION) );
    
    fprintf(stderr, "usage:   %s [-options] [packet ...]\n\n", PROGRAM);
    fprintf(stderr, "\
options: -g --grade G                 processing grade\n\
         -I --in-dir NAME             set input packet directory\n\
         -O --out-dir NAME            set output packet directory\n\
         -l --lock-file               create lock file while processing\n\
         -r --routing-file NAME       read routing file\n\
         -M --maxopen N               set max # of open packet files\n\
\n\
	 -v --verbose                 more verbose\n\
	 -h --help                    this help\n\
         -c --config NAME             read config file (\"\" = none)\n\
	 -L --lib-dir NAME            set lib directory\n\
	 -S --spool-dir NAME          set spool directory\n\
	 -a --addr Z:N/F.P            set FTN address\n\
	 -f --fake-addr Z:N/F.P       set FTN fakenet address\n\
	 -u --uplink-addr Z:N/F.P     set FTN uplink address\n\
         -z --zone Z                  select aka for zone Z\n"  );
}



/***** main() ****************************************************************/

int main(argc, argv)
    int argc;
    char *argv[];
{
    struct dirent *dir;
    DIR *dp;
    int c, ret;
    int l_flag = FALSE;
    char *I_flag=NULL, *O_flag=NULL, *r_flag=NULL;
    char *c_flag=NULL;
    char *S_flag=NULL, *L_flag=NULL;
    char *a_flag=NULL, *f_flag=NULL, *u_flag=NULL;
    char *z_flag=NULL;
    char pkt_name[MAXPATH];
    char pattern[16];
    
    int option_index;
    static struct option long_options[] =
    {
	{ "grade",        1, 0, 'g'},	/* grade */
	{ "in-dir",       1, 0, 'I'},	/* Set inbound packets directory */
	{ "lock-file",    0, 0, 'l'},	/* Create lock file while processing */
	{ "out-dir",      1, 0, 'O'},	/* Set packet directory */
	{ "routing-file", 1, 0, 'r'},	/* Set routing file */
	{ "maxopen",      1, 0, 'M'},	/* Set max # open packet files */

	{ "verbose",      0, 0, 'v'},	/* More verbose */
	{ "help",         0, 0, 'h'},	/* Help */
	{ "config",       1, 0, 'c'},	/* Config file */
	{ "spool-dir",    1, 0, 'S'},	/* Set FIDOGATE spool directory */
	{ "lib-dir",      1, 0, 'L'},	/* Set FIDOGATE lib directory */
	{ "addr",         1, 0, 'a'},	/* Set FIDO address */
	{ "fake-addr",    1, 0, 'f'},	/* Set FIDO fakenet address */
	{ "uplink-addr",  1, 0, 'u'},	/* Set FIDO uplink address */
	{ "zone",         1, 0, 'z'},	/* Select aka for zone */
	{ 0,              0, 0, 0  }
    };

    log_program(PROGRAM);
    
    /* Init configuration */
    cf_initialize();


    while ((c = getopt_long(argc, argv, "g:O:I:lr:M:vhc:S:L:a:f:u:z:",
			    long_options, &option_index     )) != EOF)
	switch (c) {
	/***** ftnroute options *****/
	case 'g':
	    g_flag = *optarg;
	    break;
	case 'I':
	    I_flag = optarg;
	    break;
        case 'l':
	    l_flag = TRUE;
            break;
	case 'O':
	    O_flag = optarg;
	    break;
	case 'r':
	    r_flag = optarg;
	    break;
	case 'M':
	    outpkt_set_maxopen(atoi(optarg));
	    break;
	    
	/***** Common options *****/
	case 'v':
	    verbose++;
	    break;
	case 'h':
	    usage();
	    exit(0);
	    break;
	case 'c':
	    c_flag = optarg;
	    break;
	case 'S':
	    S_flag = optarg;
	    break;
	case 'L':
	    L_flag = optarg;
	    break;
	case 'a':
	    a_flag = optarg;
	    break;
	case 'f':
	    f_flag = optarg;
	    break;
	case 'u':
	    u_flag = optarg;
	    break;
	case 'z':
	    z_flag = optarg;
	    break;
	default:
	    short_usage();
	    exit(EX_USAGE);
	    break;
	}

    /*
     * Read config file
     */
    if(L_flag)				/* Must set libdir beforehand */
	cf_set_libdir(L_flag);
    cf_read_config_file(c_flag ? c_flag : MY_CONFIG);

    /*
     * Process config options
     */
    if(L_flag)
	cf_set_libdir(L_flag);
    if(S_flag)
	cf_set_spooldir(S_flag);
    if(a_flag)
	cf_set_addr(a_flag);
    if(f_flag)
	cf_set_fake(f_flag);
    if(u_flag)
	cf_set_uplink(u_flag);

    cf_debug();

    if(z_flag)
	cf_set_zone(atoi(z_flag));
    
    /*
     * Process local options
     */
    if(I_flag)
	strncpy0(in_dir, I_flag, sizeof(in_dir));
    else 
	sprintf(in_dir, "%s/%s", cf_spooldir(), TOSS_TMP);
    if(O_flag)
	pkt_outdir(O_flag, NULL);
    else
	pkt_outdir(cf_spooldir(), TOSS_OUT);

    routing_init(r_flag ? r_flag : ROUTING);
    

    ret = EXIT_OK;
    
    if(optind >= argc)
    {
	strncpy0(pattern, "????????.?ut", sizeof(pattern));
	if(g_flag)
	    pattern[0] = g_flag;
	
	/*
	 * No packet file args, process in directory
	 */
	dp = opendir(in_dir);
	
	if(!dp)
	{
	    log("$Unable to open inbound directory %s", in_dir);
	    exit(EX_OSERR);
	}

	/* Lock file */
	if(l_flag)
	    if(lock_program(PROGRAM, FALSE) == ERROR)
		/* Already busy */
		exit(EXIT_BUSY);
	
	while((dir = readdir(dp)))
	{
	    if(wildmat(dir->d_name, pattern))
	    {
		sprintf(pkt_name, "%s/%s", in_dir, dir->d_name);
		if(do_file(pkt_name) == ERROR)
		{
		    ret = EXIT_ERROR;
		    break;
		}
	    }
	}

	closedir(dp);

	/* Lock file */
	if(l_flag)
	    unlock_program(PROGRAM);
    }
    else
    {
	/* Lock file */
	if(l_flag)
	    if(lock_program(PROGRAM, FALSE) == ERROR)
		/* Already busy */
		exit(EXIT_BUSY);
	
	/*
	 * Process packet files on command line
	 */
	for(; optind<argc; optind++)
	    if(do_file(argv[optind]) == ERROR)
	    {
		ret = EXIT_ERROR;
		break;
	    }

	/* Lock file */
	if(l_flag)
	    unlock_program(PROGRAM);
    }
    
    outpkt_close();

    exit(ret);

    /**NOT REACHED**/
    return 1;
}
