/*:ts=8*/
/*****************************************************************************
 * FIDOGATE --- Gateway UNIX Mail/News <-> FTN NetMail/EchoMail
 *
 * $Id: ftnaf.c,v 3.8.0.5 1994/09/18 17:23:13 mj Exp mj $
 *
 * Areafix-like AREAS.BBS EchoMail distribution manager. Commands somewhat
 * conforming to FSC-0057.
 *
 *****************************************************************************
 * Copyright (C) 1990, 1993, 1994
 *  _____ _____
 * |     |___  |   Martin Junius             FIDO:      2:2452/110.1
 * | | | |   | |   Republikplatz 3           Internet:  mj@sungate.fido.de
 * |_|_|_|@home|   D-52072 Aachen, Germany   Phone:     ++49-241-86931 (voice)
 *
 * This file is part of FIDOGATE.
 *
 * FIDOGATE is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * FIDOGATE is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with FIDOGATE; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/

#include "fidogate.h"
#include "getopt.h"



#define PROGRAM		"ftnaf"
#define VERSION		"$Revision: 3.8.0.5 $"

#define MY_CONFIG	"%L/config.af"
#define MY_HELP		"help.af"
#define MY_NAME		"Areafix"
#define MY_CONTEXT	"af"

#define MAILER		"/usr/lib/sendmail -t"



/*
 * Number of old AREAS.BBS to keep as AREAS.Onn
 */
#define N_HISTORY	10



/*
 * Prototypes
 */
FILE   *mailer_open		P((char *));
int     mailer_close		P((FILE *));
int	rewrite_areas_bbs	P((void));
int	match_area		P((char *, char *));
int	do_mail			P((void));
int	do_command		P((Node *, char *));
int	cmd_list		P((Node *));
int	cmd_listall		P((Node *));
int	cmd_query		P((Node *));
int	cmd_unlinked		P((Node *));
int	cmd_add			P((Node *, char *));
int	cmd_remove		P((Node *, char *));
int	cmd_help		P((Node *));
int	cmd_passwd		P((Node *, char *));

void	short_usage		P((void));
void	usage			P((void));



/*
 * Command line options
 */
static int   m_flag = FALSE;
static int   r_flag = FALSE;
static int   n_flag = FALSE;

static int   areas_bbs_changed = FALSE;
static int   authorized = FALSE;

static FILE *output = stdout;



/*
 * Open mailer for sending reply
 */
FILE *mailer_open(to)
    char *to;
{
    FILE *fp;
    
    fp = popen(MAILER, W_MODE);
    if(!fp)
    {
	log("$can't open pipe to %s", MAILER);
	return NULL;
    }
    
    fprintf(fp, "From: %s@%s (%s)\n", MY_NAME, cf_fqdn(), MY_NAME);
    fprintf(fp, "To: %s, %s@%s\n", to, POSTMASTER, cf_fqdn());
/*    fprintf(fp, "Errors-To: %s@%s\n", POSTMASTER, cf_fqdn()); */
    fprintf(fp, "Subject: Your %s request\n", MY_NAME);
    fprintf(fp, "\n");
    
    return fp;
}



/*
 * Close mailer
 */
int mailer_close(fp)
    FILE *fp;
{
    return pclose(fp);
}



/*
 * Rewrite AREAS.BBS if changed
 */
int rewrite_areas_bbs()
{
    char old[MAXPATH], new[MAXPATH];
    int i, ovwr;
    FILE *fp;
    
    if(!areas_bbs_changed)
    {
	debug(2, "AREAS.BBS not changed");
	return OK;
    }

    /*
     * Base name
     */
    strncpy0(buffer, cf_areas_bbs(), MAXPATH);
    ovwr = strlen(buffer) - 3;		/* 3 = extension "bbs" */
    if(ovwr < 0)			/* Just to be sure */
	ovwr = 0;

    /*
     * Write new one as AREAS.NEW
     */
    strcpy(new, buffer);
    strcpy(new+ovwr, "new");
    debug(2, "Writing %s", new);
    if(!n_flag)
    {
	if( (fp = fopen(new, W_MODE)) == NULL )
	{
	    log("$can't open %s for writing AREAS.BBS", new);
	    return ERROR;
	}
	if( areasbbs_print(fp) == ERROR )
	{
	    log("$error while writing to %s", new);
	    fclose(fp);
	    unlink(new);
	    return ERROR;
	}
	if( fclose(fp) == ERROR )
	{
	    log("$error closing %s", new);
	    unlink(new);
	    return ERROR;
	}
    }

    /*
     * Renumber saved AREAS.Onn
     */
    strcpy(old, buffer);
    sprintf(old+ovwr, "o%02d", N_HISTORY);
    debug(2, "Removing %s", old);
    if(!n_flag)
	unlink(old);
    for(i=N_HISTORY-1; i>=1; i--)
    {
	strcpy(old, buffer);
	sprintf(old+ovwr, "o%02d", i);
	strcpy(new, buffer);
	sprintf(new+ovwr, "o%02d", i+1);
	debug(2, "Renaming %s -> %s", old, new);
	if(!n_flag)
	    rename(old, new);
    }
    
    /*
     * Rename AREAS.BBS -> AREAS.O01
     */
    strcpy(old, buffer);
    strcpy(old+ovwr, "bbs");
    strcpy(new, buffer);
    strcpy(new+ovwr, "o01");
    debug(2, "Renaming %s -> %s", old, new);
    if(!n_flag)
	rename(old, new);
    
    /*
     * Rename AREAS.NEW -> AREAS.BBS
     */
    strcpy(old, buffer);
    strcpy(old+ovwr, "new");
    strcpy(new, buffer);
    strcpy(new+ovwr, "bbs");
    debug(2, "Renaming %s -> %s", old, new);
    if(!n_flag)
	rename(old, new);
    
    return OK;
}



/*
 * Match area against pattern with shell-style wildcards
 */
int match_area(pat, area)
    char *pat;
    char *area;
{
    char *p;
    int ret;
    
    pat  = strsave(pat);
    area = strsave(area);

    for(p=pat; *p; p++)
	if(isupper(*p))
	    *p = tolower(*p);
    for(p=area; *p; p++)
	if(isupper(*p))
	    *p = tolower(*p);

    ret =  wildmat(area, pat);
    
    free(pat);
    free(area);
    
    return ret;
}



/*
 * Process request message in stdin
 */
int do_mail()
{
    RFCAddr from;
    char *pfrom, *p;
    char *passwd;
    Node node, *n;
    Passwd *pwd;

    node_invalid(&node);
    
    /*
     * Read message header from stdin
     */
    header_delete();
    header_read(stdin);

    pfrom = header_get("From");
    if(!pfrom)
	return EX_UNAVAILABLE;
    debug(3, "From: %s", pfrom);

    /*
     * Open mailer
     */
    if(r_flag)
    {
	output = mailer_open(pfrom);
	if(!output)
	    return EX_OSERR;
    }

    /*
     * Check From / X-FTN-From for FTN address
     */
    n = NULL;
    if((p = header_get("X-FTN-From")))
    {
	debug(3, "X-FTN-From: %s", p);
	if((p = strchr(p, '@')))
	{
	    p++;
	    while(*p && isspace(*p))
		p++;
	    if(asc_to_node(p, &node, FALSE) == OK)
		n = &node;
	}
    }
    else
    {
	from = rfcaddr_from_rfc(pfrom);
	n    = inet_to_ftn(from.addr);
    }
    
    if(n)
    {
	debug(3, "FTN address: %s", node_to_asc(n, TRUE));
	node = *n;
	cf_set_zone(n->zone);
    }
    else
	node_invalid(&node);

    /*
     * Check password in Subject
     */
    if( (p = header_get("Subject")) )
    {
	passwd = strtok(p, " \t");
	debug(3, "Subject passwd: %s", passwd);
	pwd    = passwd_lookup(MY_CONTEXT, &node);
	debug(3, "PASSWD entry: %s", pwd ? pwd->passwd : "-NONE-");
	
	if(passwd && pwd && !stricmp(passwd, pwd->passwd))
	{
	    debug(3, "password check o.k. - authorized");
	    authorized = TRUE;
	}
    }

    /*
     * Execute commands in message body
     */
    while(fgets(buffer, sizeof(buffer), stdin))
    {
	strip_crlf(buffer);			/* Strip CR/LF */
	strip_space(buffer);			/* Strip spaces */
	if(!strncmp(buffer, " * ", 3))		/* Skip " * blah" lines */
	    continue;
	for(p=buffer; *p && isspace(*p); p++) ;	/* Skip white space */
	if(!*p)					/* Skip empty lines */
	    continue;

	do_command(&node, p);
    }

    if(!authorized)
	/* Trouble? Give some more help ... */
	fprintf(output, "\n\
Your %s request\n\
    From: %s\n\
    Node: %s\n\
didn't made it, because authorization failed. You may try using the\n\
PASSWORD command to set address and password, if there is an address\n\
problem (the Node: address above doesn't seem to be the one you intended).\n\
\n\
Additional help on the usage of %s may be requested with the HELP command.\n",
		PROGRAM, pfrom,
		node.zone!=-1 ? node_to_asc(&node, TRUE) : "no FTN address",
		PROGRAM);

    return r_flag ? mailer_close(output) : EX_OK;
}



/*
 * Process command line
 */
#define CMD_LIST	1
#define CMD_QUERY	2
#define CMD_UNLINKED	3
#define CMD_ADD		4
#define CMD_REMOVE	5
#define CMD_HELP	6
#define CMD_PASSWD	7
#define CMD_LISTALL     8

int do_command(node, line)
    Node *node;
    char *line;
{
    int cmd;
    char *arg;
    char buf[32];
    int i, ret;
    int percent = FALSE;
    
    if(line[0] == '+')
    {
	cmd = CMD_ADD;
	arg = line + 1;
    }
    else if(line[0] == '-')
    {
	cmd = CMD_REMOVE;
	arg = line + 1;
    }
    else
    {
	if(line[0] == '%')
	{
	    percent = TRUE;
	    line++;
	}
	for(i=0; line[i] && !isspace(line[i]) && i<sizeof(buf)-1; i++)
	    buf[i] = line[i];
	buf[i] = 0;
	arg = line + i;
	
	if     (!stricmp(buf, "list"))
	    cmd = CMD_LIST;
	else if(!stricmp(buf, "query"))
	    cmd = CMD_QUERY;
	else if(!stricmp(buf, "unlinked"))
	    cmd = CMD_UNLINKED;
	else if(!stricmp(buf, "add"))
	    cmd = CMD_ADD;
	else if(!stricmp(buf, "remove"))
	    cmd = CMD_REMOVE;
	else if(!stricmp(buf, "delete"))
	    cmd = CMD_REMOVE;
	else if(!stricmp(buf, "del"))
	    cmd = CMD_REMOVE;
	else if(!stricmp(buf, "help"))
	    cmd = CMD_HELP;
	else if(!stricmp(buf, "passwd"))
	    cmd = CMD_PASSWD;
	else if(!stricmp(buf, "password"))
	    cmd = CMD_PASSWD;
	else if(!stricmp(buf, "pass"))
	    cmd = CMD_PASSWD;
	else if(!stricmp(buf, "listall"))
	    cmd = CMD_LISTALL;
	else
	{
	    if(percent)
	    {
		fprintf(output, "Unknown command %%%s\n", buf);
		return OK;
	    }
	    else
	    {
		/* Interpret line as area to add */
		cmd = CMD_ADD;
		arg = line;
	    }
	}
    }

    while(*arg && isspace(*arg))
	arg++;

    switch(cmd)
    {
    case CMD_LIST:
	cmd_list(node);
	break;
    case CMD_QUERY:
	cmd_query(node);
	break;
    case CMD_UNLINKED:
	cmd_unlinked(node);
	break;
    case CMD_ADD:
	cmd_add(node, arg);
	break;
    case CMD_REMOVE:
	cmd_remove(node, arg);
	break;
    case CMD_HELP:
	cmd_help(node);
	break;
    case CMD_PASSWD:
	cmd_passwd(node, arg);
	break;
    case CMD_LISTALL:
	cmd_listall(node);
	break;
    }	
    
    return ret;
}



/*
 * ListAll command
 */
int cmd_listall(node)
    Node *node;
{
    AreasBBS *p;
    LON *l;
    
    if(!authorized)
    {
	fprintf(output, "Command LIST: not authorized.\n");
	return OK;
    }
    
    fprintf(output, "\nAll available areas:\n\n");
    
    for(p=areasbbs_first(); p; p=p->next)
    {
	l = &p->nodes;

	fprintf(output, lon_search(l, node) ? "* " : "  ");
	fprintf(output, "Z%-3d %s\n", p->zone, p->area);
    }
    
    fprintf(output, "\n* = linked to this area\n\n");

    return OK;
}



/*
 * List command
 */
int cmd_list(node)
    Node *node;
{
    AreasBBS *p;
    LON *l;
    
    if(!authorized)
    {
	fprintf(output, "Command LIST: not authorized.\n");
	return OK;
    }
    
    fprintf(output, "\nAreas available to %s:\n\n", node_to_asc(node, FALSE));
    
    for(p=areasbbs_first(); p; p=p->next)
    {
	l = &p->nodes;

	/* Check permissions */

	/* Check zone */
	if(p->zone != node->zone)
	    continue;
	
	fprintf(output, lon_search(l, node) ? "* " : "  ");
	fprintf(output, "%s\n", p->area);
    }
    
    fprintf(output, "\n* = linked to this area\n\n");
    
    return OK;
}



/*
 * Query command
 */
int cmd_query(node)
    Node *node;
{
    AreasBBS *p;
    LON *l;
    
    if(!authorized)
    {
	fprintf(output, "Command QUERY: not authorized.\n");
	return OK;
    }
    
    fprintf(output, "\n%s is linked to the following areas:\n\n",
	    node_to_asc(node, FALSE));
    
    for(p=areasbbs_first(); p; p=p->next)
    {
	l = &p->nodes;

	/* Check permissions */

	/* Check zone */
	if(p->zone != node->zone)
	    continue;
	
	if(lon_search(l, node))
	    fprintf(output, "  %s\n", p->area);
    }
    
    fprintf(output, "\n");
    
    return OK;
}



/*
 * Unlinked command
 */
int cmd_unlinked(node)
    Node *node;
{
    AreasBBS *p;
    LON *l;
    
    if(!authorized)
    {
	fprintf(output, "Command UNLINKED: not authorized.\n");
	return OK;
    }
    
    fprintf(output, "\n%s is not linked to the following available areas:\n\n",
	    node_to_asc(node, FALSE));
    
    for(p=areasbbs_first(); p; p=p->next)
    {
	l = &p->nodes;

	/* Check permissions */

	/* Check zone */
	if(p->zone != node->zone)
	    continue;
	
	if(! lon_search(l, node))
	    fprintf(output, "  %s\n", p->area);
    }
    
    fprintf(output, "\n");

    return OK;
}



/*
 * Add command
 */
int cmd_add(node, area)
    Node *node;
    char *area;
{
    AreasBBS *p;
    LON *l;
    int match = FALSE;
    
    if(!authorized)
    {
	fprintf(output, "Command ADD: not authorized.\n");
	return OK;
    }
    
    for(p=areasbbs_first(); p; p=p->next)
    {
	l = &p->nodes;

	if(match_area(area, p->area))
	{
	    match = TRUE;
	    fprintf(output, "%-39s: ", p->area);

	    /* Check permissions */

	    /* Check zone */
	    if(p->zone != node->zone)
	    {
		fprintf(output, "different zone (Z%d), not added\n", p->zone);
		continue;
	    }
	    
	    if(lon_search(l, node))
		fprintf(output, "already active\n");
	    else 
	    {
		lon_add(l, node);
		areas_bbs_changed = TRUE;
		fprintf(output, "added\n");
	    }
	}
    }

    if(!match)
	fprintf(output, "%s: no such area, or no area matching pattern\n",
		area);
	
    return OK;
}



/*
 * Remove command
 */
int cmd_remove(node, area)
    Node *node;
    char *area;
{
    AreasBBS *p;
    LON *l;
    int match = FALSE;

    if(!authorized)
    {
	fprintf(output, "Command REMOVE: not authorized.\n");
	return OK;
    }
    
    for(p=areasbbs_first(); p; p=p->next)
    {
	l = &p->nodes;

	if(match_area(area, p->area))
	{
	    match = TRUE;
	    fprintf(output, "%-39s: ", p->area);

	    if(!lon_search(l, node))
		fprintf(output, "not active\n");
	    else 
	    {
		lon_remove(l, node);
		areas_bbs_changed = TRUE;
		fprintf(output, "removed\n");
	    }
	}
    }
    
    if(!match)
	fprintf(output, "%s: no such area, or no area matching pattern\n",
		area);
	
    return OK;
}



/*
 * Help command
 */
int cmd_help(node)
    Node *node;
{
    fprintf(output, "\n\
Help for %s, FIDOGATE %s  %s %s.\n\n",
	    MY_NAME, version_global(), PROGRAM, version_local(VERSION) );

    fprintf(output, "\n\
Send mail\n\
\n\
    To:      %s@%s\n\
or  To:      %s @ %s\n\
    Subject: PASSWORD\n",
	    MY_NAME, cf_fqdn(), MY_NAME, node_to_asc(cf_addr(), FALSE) );

    fprintf(output, "\n\
Commands in message body:\n\
\n\
    listall                      list all available areas\n\
    list                         list areas available to node\n\
    query                        list subscribed areas\n\
    unlinked                     list unsubscribed areas\n\
    passwd Z:N/F.P PASSWORD      set address and password\n\
    password Z:N/F.P PASSWORD\n\
    pass Z:N/F.P PASSWORD\n\
    add AREA                     subscribe to area\n\
    +AREA\n\
    AREA\n\
    remove AREA                  unsubscribe to area\n\
    delete AREA\n\
    del AREA\n\
    -AREA\n\
    help                         this help\n\
\n\
AREA names are not case-sensitive and support shell-style wildcards\n\
* ? [a-z] [abc], e.g. COMP.OS.*, [A-D]*.GER\n\n"                       );

    return OK;
}



/*
 * Passwd command
 */
int cmd_passwd(node, arg)
    Node *node;
    char *arg;
{
    char *p;
    Node n;
    Passwd *pwd;
    
    p = strtok(arg, " \t");			/* Node address */
    if(!p)
    {
	fprintf(output, "Command PASSWORD: missing Z:N/F.P address.\n");
	return OK;
    }	
    if( asc_to_node(p, &n, FALSE) == ERROR )
    {
	fprintf(output, "Command PASSWORD: illegal address %s.\n", p);
	return OK;
    }
    *node = n;
    cf_set_zone(node->zone);
    authorized = FALSE;
    
    p = strtok(NULL, " \t");			/* Password */
    if(!p)
    {
	fprintf(output, "Command PASSWORD: no password given!\n");
	authorized = FALSE;
	return OK;
    }

    pwd = passwd_lookup(MY_CONTEXT, node);
    debug(3, "PASSWD entry: %s", pwd ? pwd->passwd : "-NONE-");
    if(!pwd)
    {
	fprintf(output, "Command PASSWORD: address %s not authorized.\n",
		node_to_asc(node, TRUE));
	authorized = FALSE;
	return OK;
    }	
    if(stricmp(p, pwd->passwd))
    {
	fprintf(output, "Command PASSWORD: wrong password for address %s.\n",
		node_to_asc(node, TRUE));
	authorized = FALSE;
	return OK;
    }	

    debug(3, "password check o.k. - authorized");
    authorized = TRUE;

    return OK;
}



/*
 * Usage messages
 */
void short_usage()
{
    fprintf(stderr, "usage: %s [-options] [Z:N/F.P command]\n", PROGRAM);
    fprintf(stderr, "       %s --help  for more information\n", PROGRAM);
    exit(EX_USAGE);
}


void usage()
{
    fprintf(stderr, "FIDOGATE %s  %s %s\n\n",
	    version_global(), PROGRAM, version_local(VERSION) );
    
    fprintf(stderr, "usage:   %s [-options] [Z:N/F.P command]\n\n", PROGRAM);
    fprintf(stderr, "\
options: -m --mail                    process Areafix mail on stdin\n\
         -r --no-reply                don't send reply via mail\n\
         -n --no-rewrite              don't rewrite AREAS.BBS\n\
         -b --areas-bbs NAME          use alternate AREAS.BBS\n\
\n\
         -v --verbose                 more verbose\n\
	 -h --help                    this help\n\
         -c --config name             read config file (\"\" = none)\n\
	 -L --lib-dir name            set lib directory\n\
	 -S --spool-dir name          set spool directory\n\
	 -a --addr Z:N/F.P            set FTN address\n\
	 -f --fake-addr Z:N/F.P       set FTN fakenet address\n\
	 -u --uplink-addr Z:N/F.P     set FTN uplink address\n\
         -z --zone Z                  select aka for zone Z\n"  );
    fprintf(stderr, "\n\
command: listall                      list all available areas\n\
         list                         list areas available to node\n\
         query                        list subscribed areas\n\
         unlinked                     list unsubscribed areas\n\
         passwd Z:N/F.P PASSWORD      set address and password\n\
         password Z:N/F.P PASSWORD\n\
         pass Z:N/F.P PASSWORD\n\
         add AREA                     subscribe to area\n\
         +AREA\n\
         AREA\n\
         remove AREA                  unsubscribe to area\n\
         delete AREA\n\
         del AREA\n\
         -AREA\n\
         help                         help for Areafix mails\n");
}



/***** main() ****************************************************************/

int main(argc, argv)
    int argc;
    char *argv[];
{
    int c;
    char *b_flag=NULL;
    char *c_flag=NULL;
    char *S_flag=NULL, *L_flag=NULL;
    char *a_flag=NULL, *f_flag=NULL, *u_flag=NULL;
    char *z_flag=NULL;
    Node node;
    int ret;
    
    int option_index;
    static struct option long_options[] =
    {
	{ "mail",         0, 0, 'm'},
	{ "no-reply",     0, 0, 'r'},
	{ "no-rewrite",   0, 0, 'n'},
        { "areas-bbs",	  1, 0, 'b'},

	{ "verbose",      0, 0, 'v'},	/* More verbose */
	{ "help",         0, 0, 'h'},	/* Help */
	{ "config",       1, 0, 'c'},	/* Config file */
	{ "spool-dir",    1, 0, 'S'},	/* Set FIDOGATE spool directory */
	{ "lib-dir",      1, 0, 'L'},	/* Set FIDOGATE lib directory */
	{ "addr",         1, 0, 'a'},	/* Set FIDO address */
	{ "fake-addr",    1, 0, 'f'},	/* Set FIDO fakenet address */
	{ "uplink-addr",  1, 0, 'u'},	/* Set FIDO uplink address */
	{ "zone",         1, 0, 'z'},	/* Select aka for zone */
	{ 0,              0, 0, 0  }
    };

    log_program(PROGRAM);
    
    /* Init configuration */
    cf_initialize();


    while ((c = getopt_long(argc, argv, "mrnb:vhc:S:L:a:f:u:z:",
			    long_options, &option_index     )) != EOF)
	switch (c) {
	/***** ftnaf options *****/
	case 'm':
	    m_flag = TRUE;
	    r_flag = TRUE;
	    break;
	case 'r':
	    r_flag = FALSE;
	    break;
	case 'n':
	    n_flag = TRUE;
	    break;
	case 'b':
	    b_flag = optarg;
	    break;
		
	/***** Common options *****/
	case 'v':
	    verbose++;
	    break;
	case 'h':
	    usage();
	    exit(0);
	    break;
	case 'c':
	    c_flag = optarg;
	    break;
	case 'S':
	    S_flag = optarg;
	    break;
	case 'L':
	    L_flag = optarg;
	    break;
	case 'a':
	    a_flag = optarg;
	    break;
	case 'f':
	    f_flag = optarg;
	    break;
	case 'u':
	    u_flag = optarg;
	    break;
	case 'z':
	    z_flag = optarg;
	    break;
	default:
	    short_usage();
	    exit(EX_USAGE);
	    break;
	}

    /*
     * Read config file
     */
    if(L_flag)				/* Must set libdir beforehand */
	cf_set_libdir(L_flag);
    cf_read_config_file(c_flag ? c_flag : MY_CONFIG);

    /*
     * Process config options
     */
    if(L_flag)
	cf_set_libdir(L_flag);
    if(S_flag)
	cf_set_spooldir(S_flag);
    if(a_flag)
	cf_set_addr(a_flag);
    if(f_flag)
	cf_set_fake(f_flag);
    if(u_flag)
	cf_set_uplink(u_flag);

    cf_debug();
    
    if(z_flag)
	cf_set_zone(atoi(z_flag));

    /*
     * Process local options
     */
    if(b_flag)
	cf_set_areas_bbs(b_flag);

    /* Read PASSWD */
    passwd_init();
    /* Read HOSTS */
    hosts_init();
    
    ret = 0;
    
    if(m_flag)
    {
	/*
	 * Process stdin as mail request for Areafix
	 */
	if(bink_bsy_create_all(WAIT) == ERROR)
	    ret = EX_OSERR;
	else
	{
	    if(areasbbs_init(cf_areas_bbs()) == ERROR)
		ret = EX_OSFILE;
	    else
		ret = do_mail();
	    if(ret == 0)
		if( rewrite_areas_bbs() == ERROR )
		    ret = EX_CANTCREAT;
	}
	bink_bsy_delete_all();
    }
    else
    {
	/*
	 * Process command on command line
	 */
	/* Node */
	if(optind >= argc)
	{
	    fprintf(stderr, "%s: expecting Z:N/F.P node\n", PROGRAM);
	    short_usage();
	}
	if( asc_to_node(argv[optind], &node, FALSE) == ERROR )
	{
	    fprintf(stderr, "%s: invalid node %s\n", PROGRAM, argv[optind]);
	    short_usage();
	}
	optind++;

	/*
	 * Execute command, always authorized if command line
	 */
	authorized = TRUE;

	if(areasbbs_init(cf_areas_bbs()) == ERROR)
	    exit(EX_OSFILE);

	/* Command is rest of args on command line */
	buffer[0] = 0;
	for(; optind<argc; optind++)
	{
	    strncat0(buffer, argv[optind], sizeof(buffer));
	    if(optind < argc-1)
		strncat0(buffer, " ", sizeof(buffer));
	}

	if(do_command(&node, buffer) == ERROR)
	    ret = 1;
	if(ret == 0)
	    if( rewrite_areas_bbs() == ERROR )
		ret = EX_CANTCREAT;
    }

    exit(ret);

    /**NOT REACHED**/
    return 1;
}
