/*:ts=8*/
/*****************************************************************************
 * FIDOGATE --- Gateway UNIX Mail/News <-> FIDO NetMail/EchoMail
 *
 * $Id: ftn2rfc.c,v 3.8.0.6 1994/09/25 14:39:55 mj Exp mj $
 *
 * Convert FTN mail packet to RFC messages (mail and news batches)
 *
 *****************************************************************************
 * Copyright (C) 1990, 1993, 1994
 *  _____ _____
 * |     |___  |   Martin Junius             FIDO:      2:2452/110.1
 * | | | |   | |   Republikplatz 3           Internet:  mj@sungate.fido.de
 * |_|_|_|@home|   D-52072 Aachen, Germany   Phone:     ++49-241-86931 (voice)
 *
 * This file is part of FIDOGATE.
 *
 * FIDOGATE is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * FIDOGATE is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with FIDOGATE; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/

#include "fidogate.h"
#include "getopt.h"

#include <pwd.h>
#include <fcntl.h>
#include <sys/wait.h>



#define PROGRAM "ftn2rfc"
#define VERSION "$Revision: 3.8.0.6 $"



/*
 * Prototypes
 */
void	check_chrs		P((char *));
char   *get_from		P((Textlist *));
char   *get_to			P((Textlist *));
char   *get_cc			P((Textlist *));
char   *get_bcc			P((Textlist *));
int 	check_origin		P((char *, Node *));
Area   *news_msg		P((char *));
int	unpack			P((FILE *, Packet *));
int	unpack_file		P((char *));

void	short_usage		P((void));
void	usage			P((void));



/*
 * Command line options
 */
int t_flag = FALSE;

char in_dir[MAXPATH];
char news_dir[MAXPATH];
char news_name[MAXPATH];
char news_tmp[MAXPATH];
FILE *news_file;



/*
 * check_chrs() --- Check for ^ACHARSET / ^ACHRS
 */
void check_chrs(buf)
    char *buf;
{
    /*
     * Look for `^ACHARSET ...' or `^ACHRS ...'
     */
    if(!strncmp(buf+1, "CHARSET: ", 9))
	charset_set(buf + 9);
    else if(!strncmp(buf+1, "CHRS: ", 6))
	charset_set(buf + 6);
}



/*
 * Get header for
 *   - From/Reply-To/UUCPFROM
 *   - To
 *   - Cc
 *   - Bcc
 */
char *get_from(tl)
    Textlist *tl;
{
    char *p;
    
    p = rfcheader_get(tl, "Reply-To");
    if(!p)
	p = rfcheader_get(tl, "From");
    if(!p)
	p = rfcheader_get(tl, "UUCPFROM");

    return p;
}    

char *get_to(tl)
    Textlist *tl;
{
    return rfcheader_get(tl, "To");
}    

char *get_cc(tl)
    Textlist *tl;
{
    return rfcheader_get(tl, "Cc");
}    

char *get_bcc(tl)
    Textlist *tl;
{
    return rfcheader_get(tl, "Bcc");
}    



/*
 * check_origin() --- Analyse ` * Origin: ...' line in FIDO message
 *		      body and parse address in ()s into Node structure
 *
 * Origin line is checked for the rightmost occurence of
 * ([text] z:n/n.p).
 */
int check_origin(buffer, node)
    char *buffer;
    Node *node;
{
    char *left, *right;
    char *buf;

    if(buffer == NULL)
	return ERROR;
    
    buf   = strsave(buffer);
    right = strrchr(buf, ')');
    if(!right) {
	free(buf);
	return ERROR;
    }
    left  = strrchr(buf, '(');
    if(!left) {
	free(buf);
	return ERROR;
    }

    *right = 0;
    *left++ = 0;

    /*
     * Parse node info
     */
    while(*left && !isdigit(*left))
	left++;
    if(asc_to_node(left, node, FALSE) != OK)
	/* Not a valid FIDO address */
	node->zone = node->net = node->node = node->point = -1;

    free(buf);
    return node->zone != -1 ? OK : ERROR;
}



/*
 * Test for EchoMail, return Area entry.
 */
Area *news_msg(line)
    char *line;
{
    char *p;
    Area *pa;
    static Area area;
    
    if(line)
    {
	/* Message is FIDO EchoMail */
	strip_crlf(line);
	
	for(p=line+strlen("AREA:"); *p && isspace(*p); p++);
	debug(3, "FIDO Area: %s", p);
	pa = areas_lookup(p, NULL);
	if(pa)
	{
	    debug(5, "Found: %s %s Z%d", pa->area, pa->group, pa->zone);
	    return pa;
	}

	/* Area not found */
	area.next  = NULL;
	area.area  = p;
	area.group = NULL;
	area.zone  = 0;
	return &area;
    }

    /* Message is FIDO NetMail */
    return NULL;
}



/*
 * Read and convert FTN mail packet
 */
int unpack(pkt_file, pkt)
    FILE *pkt_file;
    Packet *pkt;
{
    Message msg;			/* Message header */
    RFCAddr addr_from, addr_to;		/* Representation of FTN addresses */
    Textlist tl;			/* Textlist for message body */
    MsgBody body;			/* Message body of FTN message */
    int lines;				/* Lines in message body */
    Area *area;				/* Area entry for EchoMail */
    int type;
    Textline *pl;
    char *p;
    char *msgbody_rfc_from;		/* RFC header From/Reply-To */
    char *msgbody_rfc_to;		/* RFC header To */
    char *msgbody_rfc_cc;		/* RFC header Cc */
    char *msgbody_rfc_bcc;		/* RFC header Bcc */
    char mail_to[MAXINETADDR];		/* Addressee of mail */
    char x_orig_to[MAXINETADDR];
    char *from_line, *to_line;		/* From:, To: */
    char *id_line, *ref_line;		/* Message-ID:, References: */
    char *cc_line, *bcc_line;		/* Cc:, Bcc: */
    char *thisdomain, *uplinkdomain;	/* FQDN addr this node, uplink,  */
    char *origindomain;			/*           node in Origin line */
    Textlist theader;			/* RFC headers */
    Textlist tbody;    			/* RFC message body */
    int uucp_flag;			/* To == UUCP or GATEWAY */


    /*
     * Initialize
     */
    Textlist_init(&tl);
    Textlist_init(&theader);
    Textlist_init(&tbody);
    msg_body_init(&body);
    news_file = NULL;

    /*
     * Read packet
     */
    type = pkt_get_int16(pkt_file);
    while(type == MSG_TYPE)
    {
	x_orig_to[0] = 0;
	Textlist_clear(&theader);
	Textlist_clear(&tbody);
	charset_reset();			/* Reset ^ACHRS char set */

	/*
	 * Read message header
	 */
	msg.node_from = pkt->from;
	msg.node_to   = pkt->to;
	pkt_get_msg_hdr(pkt_file, &msg);
	if (!*msg.subject)
	    strcpy(msg.subject, "(no subject)");
	/* Strip spaces at end of line */
	strip_space(msg.name_from);
	strip_space(msg.name_to);
	strip_space(msg.subject);
	
	/*
	 * Read message body
	 */
	type = pkt_get_body(pkt_file, &tl);
	if(type == ERROR)
	{
	    log("error reading input packet");
	    return ERROR;
	}

	/*
	 * Parse message body
	 */
	if( msg_body_parse(&tl, &body) == -2 )
	    log("error parsing message body");
	/* Retrieve address information from kludges for NetMail */
	if(body.area == NULL)
	{
	    /* Retrieve complete address from kludges */
	    kludge_pt_intl(&body, &msg);
	    msg.node_orig = msg.node_from;
	}
	else 
	{
	    /* Retrieve address information from * Origin line */
	    if(check_origin(body.origin, &msg.node_orig) == ERROR)
		/* No * Origin line address, use header */
		msg.node_orig = msg.node_from;
	}
	debug(2, "FIDO From: %s", node_to_asc(&msg.node_orig, TRUE));

	/*
	 * strip_crlf() all kludge and RFC lines
	 */
	for(pl=body.kludge.first; pl; pl=pl->next)
	    strip_crlf(pl->line);
	for(pl=body.rfc.first; pl; pl=pl->next)
	    strip_crlf(pl->line);
	
	/*
	 * Convert message body
	 */
	lines = 0;
	if( (p = kludge_get(&body.kludge, "CHRS", NULL)) )
	    charset_set(p);
	else if( (p = kludge_get(&body.kludge, "CHARSET", NULL)) )
	    charset_set(p);
	
	for(pl=body.body.first; pl; pl=pl->next)
	{
	    p = pl->line;
	    if(*p == '\001')			/* Kludge in message body */
		check_chrs(p);			/* ^ACHRS / ^ACHARSET */
	    else				/* Normal line */
	    {
		msg_xlate_line(buffer, sizeof(buffer), p);
		lines += msg_format_buffer(buffer, &tbody);
	    }
	}

	/*
	 * Check for mail or news.
	 *
	 * area == NULL  -  NetMail
	 * area != NULL  -  EchoMail
	 */
	if( (area = news_msg(body.area)) )
	{
	    /*
	     * Set AKA according to area's zone
	     */
	    cf_set_zone(area->zone);
	    
	    if(!news_file)
	    {
		long n = sequencer(SEQ_NEWS);

		sprintf(news_tmp,  "%s/%08ld.tmp", news_dir, n);
		sprintf(news_name, "%s/%08ld.msg", news_dir, n);
		news_file = fopen(news_tmp, W_MODE);
		if(!news_file)
		{
		    log("$can't create output file %s", news_tmp);
		    return ERROR;
		}
	    }
	}
	else
	{
	    /*
	     * Set AKA according to sender's zone
	     */
	    cf_set_zone(msg.node_orig.zone);
	}
	

	/*
	 * Convert FTN from/to addresses to RFCAddr struct
	 */
	addr_from = rfcaddr_from_ftn(msg.name_from, &msg.node_orig);
	addr_to   = rfcaddr_from_ftn(msg.name_to,   &msg.node_to  );

	uucp_flag = FALSE;
	if(!stricmp(addr_to.real, "UUCP")    ||
	   !stricmp(addr_to.real, "GATEWAY")   )
	{
	    /* Don't output UUCP or GATEWAY as real name */
	    uucp_flag = TRUE;
	}

	/*
	 * RFC address headers from text body
	 */
	msgbody_rfc_from = get_from(&body.rfc);
	msgbody_rfc_to   = get_to  (&body.rfc);
	msgbody_rfc_cc   = get_cc  (&body.rfc);
	msgbody_rfc_bcc  = get_bcc (&body.rfc);

	/*
	 * If -g flag is set for area and message seems to comme from
	 * another gateway, skip it.
	 */
	if(area && (area->flags & AREA_NOGATE) && msgbody_rfc_from)
	{
	    log("skipping message from gateway, origin=%s",
		node_to_asc(&msg.node_orig, TRUE));
	    continue;
	}
	
#ifdef NO_BRAIN_DEAD_GATEWAYS
	/*
	 * Skip messages from gateway software known to cause trouble.
	 */
	/* GIGO */
	if( (p = kludge_get(&body.kludge, "PID", NULL))  &&
	    !strnicmp(p, "GIGO", 4)                         )
	{
	    log("skipping message from GIGO, origin=%s",
		node_to_asc(&msg.node_orig, TRUE));
	    continue;
	}
#endif

	/*
	 * Do alias checking on both from and to names
	 */
	if(!msgbody_rfc_from)
	{
	    Alias *a;

	    debug(4, "Checking for alias: %s",
		  rfcaddr_to_asc(&addr_from, TRUE));
	    a = alias_lookup(&msg.node_orig, NULL, addr_from.real);
	    if(a)
	    {
		debug(3, "Alias found: %s %s %s", a->username,
		      node_to_asc(&a->node, FALSE), a->fullname);
		strncpy0(addr_from.user, a->username, sizeof(addr_from.user));
#ifdef ALIASES_ARE_LOCAL
		strncpy0(addr_from.addr, cf_fqdn(), sizeof(addr_from.user));
#endif
	    }
	}
	if(!msgbody_rfc_to)
	{
	    Alias *a;
	    
	    debug(4, "Checking for alias: %s",
		  rfcaddr_to_asc(&addr_to, TRUE));
	    a = alias_lookup(cf_addr(), NULL, addr_to.real);
	    if(a)
	    {
		debug(3, "Alias found: %s %s %s", a->username,
		      node_to_asc(&a->node, FALSE), a->fullname);
		strcpy(mail_to, a->username);
	    }
	    else
		strcpy(mail_to, addr_to.user);
	}

	/*
	 * Special handling for -t flag (insecure):
	 *   Messages with To line will be bounced
	 */
	if(area==NULL && t_flag && msgbody_rfc_to)
	{
	    debug(1, "Insecure message with To line");
	    log("bouncing mail from %s", rfcaddr_to_asc(&addr_from, TRUE));
	    bounce_mail("insecure", &addr_from, &msg, msgbody_rfc_to, &tbody);
	    continue;
	}
	    
	/*
	 * There are message trackers out there in FIDONET. Obviously
	 * they can't handling addressing the gateway so we send mail
	 * from "MsgTrack..." etc. to postmaster.
	 */
	if(    !strnicmp(addr_from.user, "MsgTrack", 8)
	    || !strnicmp(addr_from.user, "Reflex_Netmail_Policeman", 24)
	    || !strnicmp(addr_from.user, "TrackM", 6)
	    || !strnicmp(addr_from.user, "ITrack", 6)
	    || !strnicmp(addr_from.user, "O/T-Track", 9)
	 /* || whatever ... */		    )
	{
	    debug(1, "Mail from FIDO message tracker");
	    strcpy(x_orig_to, mail_to);
	    strcpy(mail_to, POSTMASTER);
	    /* Reset uucp_flag to avoid bouncing of these messages */
	    uucp_flag = FALSE;
	}


	thisdomain   = ftn_to_inet(cf_addr(),      TRUE);
	uplinkdomain = ftn_to_inet(&msg.node_from, TRUE);
	origindomain = ftn_to_inet(&msg.node_orig, TRUE);

#ifdef HOSTS_RESTRICTED
	/*
	 * Bounce mail from nodes not within domain .fido.de
	 */
	if(area==NULL && msgbody_rfc_to)
	    if(hosts_lookup(&msg.node_orig, NULL) == NULL)
	    {
		/*
		 * Not registered in HOSTS
		 */
		debug(1, "Not a Fido.DE node: %s",
		      node_to_asc(&msg.node_orig, FALSE));
		log("bouncing mail from %s", rfcaddr_to_asc(&addr_from, TRUE));
		bounce_mail("restricted", &addr_from, &msg,
			    msgbody_rfc_to, &tbody);
		continue;
	    }
#endif /**HOSTS_RESTRICTED**/

	/*
	 * Check mail messages' user name
	 */
	if(area==NULL && !msgbody_rfc_to &&
	   !strchr(mail_to, '@') && !strchr(mail_to, '%') &&
	   !strchr(mail_to, '!')			    )
	{
	    if(uucp_flag) 
	    {
		/*
		 * Addressed to `UUCP' or `GATEWAY', but no To: line
		 */
		debug(1, "Message to `UUCP' or `GATEWAY' without To line");
		log("bouncing mail from %s", rfcaddr_to_asc(&addr_from, TRUE));
		bounce_mail("noto", &addr_from, &msg, msgbody_rfc_to, &tbody);
		continue;
	    }
#ifdef ADD_DOMAIN
	    /*
	     * Add `@host.domain' to local address
	     */
	    strcat(mail_to, "@");
	    strcat(mail_to, cf_fqdn());
#endif
	}

	/*
	 * Construct string for From: header line
	 */
	if(msgbody_rfc_from) {
	    if(strchr(msgbody_rfc_from, '(') || strchr(msgbody_rfc_from, '<'))
		from_line = buf_sprintf("%s", msgbody_rfc_from);
	    else
		from_line = buf_sprintf("%s (%s)", msgbody_rfc_from,
					addr_from.real);
	}
	else
	{
	    from_line = rfcaddr_to_asc(&addr_from, TRUE);
	}

	/*
	 * Construct string for To:/X-Comment-To: header line
	 */
	if(msgbody_rfc_to) {
	    if(strchr(msgbody_rfc_to, '(') || strchr(msgbody_rfc_to, '<') ||
	       !*addr_to.real || uucp_flag                                   )
		to_line = buf_sprintf("%s", msgbody_rfc_to);
	    else
		to_line = buf_sprintf("%s (%s)", msgbody_rfc_to, addr_to.real);
	}
	else {
	    if(area)
	    {
		if(!strcmp(addr_to.user, "All")  ||
		   !strcmp(addr_to.user, "Alle") ||
		   !strcmp(addr_to.user, "*.*")  ||
		   !strcmp(addr_to.user, "*")      )
		    to_line = NULL;
		else
		    to_line = buf_sprintf("(%s)", addr_to.real);
	    }
	    else
		to_line = buf_sprintf("%s", mail_to);
	}

	/*
	 * Construct Cc/Bcc header lines
	 */
	cc_line  = msgbody_rfc_cc;
	bcc_line = msgbody_rfc_bcc;
	
	/*
	 * Construct Message-ID and References header lines
	 */
	id_line  = NULL;
	ref_line = NULL;
	
	if( (p = kludge_get(&body.kludge, "ORIGID", NULL)) )
	    id_line = msgid_convert_origid(p, TRUE);
	else if( (p = kludge_get(&body.kludge, "Message-Id", NULL)) )
	    id_line = msgid_convert_origid(p, TRUE);
	if(!id_line)
	{
	    if( (p = kludge_get(&body.kludge, "MSGID", NULL)) )
		id_line = msgid_fido_to_rfc(p);
	    if(id_line == NULL)
		id_line = msgid_default(&msg.node_orig, msg.name_from,
					msg.name_to,
					msg.subject, msg.date);
	}

	if( (p = kludge_get(&body.kludge, "ORIGREF", NULL)) )
	    ref_line = msgid_convert_origid(p, FALSE);
	if(!ref_line)
	    if( (p = kludge_get(&body.kludge, "REPLY", NULL)) )
		ref_line = msgid_fido_to_rfc(p);

	/*
	 * Output RFC mail/news header
	 */

	/* Different header for mail and news */
	if(area==NULL) {			/* Mail */
	    log("%s  %s -> %s", node_to_asc(&msg.node_orig, TRUE),
		from_line, to_line);
	    
	    if(mail_open() == ERROR)
		return ERROR;

	    Textlist_appendf(&theader,
			     "From %s %s\n", rfcaddr_to_asc(&addr_from, FALSE),
			     date("%a %b %d %H:%M:%S %Y", NULL) );
	    Textlist_appendf(&theader,
		"Received: by %s (FIDOGATE %s)\n",
		thisdomain, version_global()                            );
	    Textlist_appendf(&theader,
		"\tid AA%05d; %s\n",
		getpid(), date(NULL, NULL) );
	}
	else 					/* News */
	{
	    Textlist_appendf(&theader,
			     "Path: %s!%s!%s!not-for-mail\n",
			     thisdomain, uplinkdomain, origindomain );
	}

	/* Common header */
	Textlist_appendf(&theader, "Date: %s\n", date(NULL, &msg.date));
	Textlist_appendf(&theader, "From: %s\n", from_line);
	Textlist_appendf(&theader, "Subject: %s\n", msg.subject);
	Textlist_appendf(&theader, "Message-ID: %s\n", id_line);
	
	/* Different header for mail and news */
	if(area==NULL) {			/* Mail */
	    Textlist_appendf(&theader, "To: %s\n", to_line);
	    if(*x_orig_to)
		Textlist_appendf(&theader, "X-Orig-To: %s\n", x_orig_to);
#ifdef ERRORS_TO
	    Textlist_appendf(&theader, "Errors-To: %s\n", ERRORS_TO);
#endif
	    /* FTN ReturnReceiptRequest -> Return-Receipt-To */
	    if(msg.attr & MSG_RRREQ)
		Textlist_appendf(&theader, "Return-Receipt-To: %s\n",
				 rfcaddr_to_asc(&addr_from, FALSE)   );
	}
	else 					/* News */
	{
	    if(ref_line)
		Textlist_appendf(&theader, "References: %s\n", ref_line);
	    Textlist_appendf(&theader, "Newsgroups: %s\n",
			     area->group ? area->group : FTN_JUNK);
	    if(area->distribution)
		Textlist_appendf(&theader, "Distribution: %s\n",
				 area->distribution               );
	    if(to_line)
		Textlist_appendf(&theader, "X-Comment-To: %s\n", to_line);
	}
	
	/* Common header */
	Textlist_appendf(&theader, "Organization: %s\n", cf_organization() );
	Textlist_appendf(&theader, "Lines: %d\n", lines);
	Textlist_appendf(&theader, "X-Gateway: FIDOGATE %s\n",
			 version_global()		);

#ifdef X_FTN_FROMTO
	if(area==NULL)
	{
	    Textlist_appendf(&theader, "X-FTN-From: %s @ %s\n",
			   addr_from.real, node_to_asc(&msg.node_orig, TRUE));
	    Textlist_appendf(&theader, "X-FTN-To: %s @ %s\n",
			     addr_to.real, node_to_asc(&msg.node_to, TRUE));
	}
#endif /**X_FTN_FROMTO**/

#ifdef X_FTN
	if(body.tear && !strncmp(body.tear, "--- ", 4))
	{
	    strip_crlf(body.tear);
	    msg_xlate_line(buffer, sizeof(buffer), body.tear);
	    Textlist_appendf(&theader, "X-FTN-Tearline: %s\n", buffer+4);
	}
	if(body.origin)
	{
	    strip_crlf(body.origin);
	    msg_xlate_line(buffer, sizeof(buffer), body.origin);
	    Textlist_appendf(&theader, "X-FTN-Origin: %s\n", buffer+11);
	}
	for(pl=body.via.first; pl; pl=pl->next)
	{
	    p = pl->line;
	    strip_crlf(p);
	    msg_xlate_line(buffer, sizeof(buffer), p+1);
	    Textlist_appendf(&theader, "X-FTN-Via: %s\n", buffer+4);
	}
	Textlist_appendf(&theader, "X-FTN-Domain: Z%d@%s\n",
			 cf_zone(), cf_zones_ftn_domain(cf_zone()));
	for(pl=body.seenby.first; pl; pl=pl->next)
	{
	    p = pl->line;
	    strip_crlf(p);
	    Textlist_appendf(&theader, "X-FTN-Seen-By: %s\n", p + 9);
	}
	for(pl=body.path.first; pl; pl=pl->next)
	{
	    p = pl->line;
	    strip_crlf(p);
	    Textlist_appendf(&theader, "X-FTN-Path: %s\n", p + 7);
	}
#endif /**X_FTN**/

	Textlist_appendf(&theader, "\n");

	/*
	 * Write header and message body to output file
	 */
	if(area) {
	    /* News batch */
	    fprintf(news_file, "#! rnews %ld\n", Textlist_size(&theader) +
						 Textlist_size(&tbody)	);
	    Textlist_print(&theader, news_file);
	    Textlist_print(&tbody,   news_file);
	}
	else
	{
	    /* Mail message */
	    Textlist_print(&theader, mail_file());
	    Textlist_print(&tbody,   mail_file());
	    /* Close mail */
	    mail_close();
	}

    } /**while(type == MSG_TYPE)**/

    if(news_file) 
    {
	/* Close news file */
	fclose(news_file);
	/* Rename .tmp -> .msg */
	if(rename(news_tmp, news_name) == -1)
	    log("$can't rename %s to %s", news_tmp, news_name);
    }
    
    return OK;
}



/*
 * Unpack one packet file
 */
int unpack_file(pkt_name)
    char *pkt_name;
{
    Packet pkt;
    FILE *pkt_file;

    debug(1, "Unpacking FTN packet %s", pkt_name);

    /*
     * Open packet and read header
     */
    pkt_file = fopen(pkt_name, R_MODE);
    if(!pkt_file) {
	log("$can't open packet %s", pkt_name);
	exit(EX_OSERR);
    }
    if(pkt_get_hdr(pkt_file, &pkt) == ERROR)
    {
	log("error reading header from %s", pkt_name);
	exit(EX_OSERR);
    }
    
    /*
     * Unpack it
     */
    if(unpack(pkt_file, &pkt) == -1) 
    {
	log("error in unpacking %s", pkt_name);
	exit(EX_OSERR);
    }
    
    fclose(pkt_file);
    
    if (unlink(pkt_name)) {
	log("$could not unlink packet %s", pkt_name);
	exit(EX_OSERR);
    }

    return OK;
}



/*
 * Usage messages
 */
void short_usage()
{
    fprintf(stderr, "usage: %s [-options] [packet ...]\n", PROGRAM);
    fprintf(stderr, "       %s --help  for more information\n", PROGRAM);
}


void usage()
{
    fprintf(stderr, "FIDOGATE %s  %s %s\n\n",
	    version_global(), PROGRAM, version_local(VERSION) );
    
    fprintf(stderr, "usage:   %s [-options] [packet ...]\n\n", PROGRAM);
    fprintf(stderr, "\
options: -t --insecure                toss insecure packets\n\
         -I --inbound-dir name        set inbound packets directory\n\
         -l --lock-file name          create lock file while processing\n\
         -x --exec-program name       exec program after processing\n\
\n\
	 -v --verbose                 more verbose\n\
	 -h --help                    this help\n\
         -c --config name             read config file (\"\" = none)\n\
	 -L --lib-dir name            set lib directory\n\
	 -S --spool-dir name          set spool directory\n\
	 -a --addr Z:N/F.P            set FTN address\n\
	 -f --fake-addr Z:N/F.P       set FTN fakenet address\n\
	 -u --uplink-addr Z:N/F.P     set FTN uplink address\n\
         -z --zone Z                  select aka for zone Z\n"  );
}



/***** main() ****************************************************************/

int main(argc, argv)
    int argc;
    char *argv[];
{
    struct dirent *dir;
    DIR *dp;
    int c;
    char *lockname = NULL;
    char *execprog = NULL;
    char *I_flag=NULL;
    char *c_flag=NULL;
    char *S_flag=NULL, *L_flag=NULL;
    char *a_flag=NULL, *f_flag=NULL, *u_flag=NULL;
    char *z_flag=NULL;
    char pkt_name[MAXPATH];

    int option_index;
    static struct option long_options[] =
    {
	{ "insecure",     0, 0, 't'},	/* Toss insecure packets */
	{ "inbound-dir",  1, 0, 'I'},	/* Set inbound packets directory */
	{ "lock-file",    1, 0, 'l'},	/* Create lock file while processing */
	{ "exec-program", 1, 0, 'x'},	/* Exec program after processing */

	{ "verbose",      0, 0, 'v'},	/* More verbose */
	{ "help",         0, 0, 'h'},	/* Help */
	{ "config",       1, 0, 'c'},	/* Config file */
	{ "spool-dir",    1, 0, 'S'},	/* Set FIDOGATE spool directory */
	{ "lib-dir",      1, 0, 'L'},	/* Set FIDOGATE lib directory */
	{ "addr",         1, 0, 'a'},	/* Set FIDO address */
	{ "fake-addr",    1, 0, 'f'},	/* Set FIDO fakenet address */
	{ "uplink-addr",  1, 0, 'u'},	/* Set FIDO uplink address */
	{ "zone",         1, 0, 'z'},	/* Select aka for zone */
	{ 0,              0, 0, 0  }
    };

    log_program(PROGRAM);
    
    /* Init configuration */
    cf_initialize();


    while ((c = getopt_long(argc, argv, "tI:l:x:vhc:S:L:a:f:u:z:",
			    long_options, &option_index     )) != EOF)
	switch (c) {
	/***** ftn2rfc options *****/
	case 't':
	    /* Insecure */
	    t_flag = TRUE;
	    break;
	case 'I':
	    /* Inbound packets directory */
	    I_flag = optarg;
	    break;
        case 'l':
            /* Lock file */
            lockname = optarg;
            break;
        case 'x':
            /* Exec program after unpack */
            execprog = optarg;
            break;

	/***** Common options *****/
	case 'v':
	    verbose++;
	    break;
	case 'h':
	    usage();
	    exit(0);
	    break;
	case 'c':
	    c_flag = optarg;
	    break;
	case 'S':
	    S_flag = optarg;
	    break;
	case 'L':
	    L_flag = optarg;
	    break;
	case 'a':
	    a_flag = optarg;
	    break;
	case 'f':
	    f_flag = optarg;
	    break;
	case 'u':
	    u_flag = optarg;
	    break;
	case 'z':
	    z_flag = optarg;
	    break;
	default:
	    short_usage();
	    exit(EX_USAGE);
	    break;
	}

    /*
     * Read config file
     */
    if(L_flag)				/* Must set libdir beforehand */
	cf_set_libdir(L_flag);
    cf_read_config_file(c_flag);

    /*
     * Process config options
     */
    if(L_flag)
	cf_set_libdir(L_flag);
    if(S_flag)
	cf_set_spooldir(S_flag);
    if(a_flag)
	cf_set_addr(a_flag);
    if(f_flag)
	cf_set_fake(f_flag);
    if(u_flag)
	cf_set_uplink(u_flag);

    cf_debug();

    if(z_flag)
	cf_set_zone(atoi(z_flag));
    
    /*
     * Process local options
     */
    if(I_flag)
    {
	strncpy0(in_dir, I_flag, sizeof(in_dir));
	sprintf(mail_dir, "%s/%s", in_dir, INBOUND_MAIL);
	sprintf(news_dir, "%s/%s", in_dir, INBOUND_NEWS);
    }
    else 
    {
	/* Default inbound directory */
	if(t_flag)
	    sprintf(in_dir, "%s/%s", cf_spooldir(), INSECURE);
	else
	    sprintf(in_dir, "%s/%s", cf_spooldir(), INBOUND);
	sprintf(mail_dir, "%s/%s/%s", cf_spooldir(), INBOUND, INBOUND_MAIL);
	sprintf(news_dir, "%s/%s/%s", cf_spooldir(), INBOUND, INBOUND_NEWS);
    }
    
    /*
     * Init various modules
     */
    areas_init();
    hosts_init();
    alias_init();
#ifdef MAUS_DE
    maus_init();
#endif

    /*
     * If called with -l lock option, try to create lock FILE
     */
    if(lockname)
    {
	int fd;

	/*
	 * Use open() with flag O_EXCL, this will fail if the
	 * lock file already exists
	 */
	fd = open(buf_sprintf("%s/%s", cf_spooldir(), lockname),
		  O_RDWR | O_CREAT | O_EXCL, 0666);
	debug(1, "lock %s", fd==-1 ? "failed" : "succeeded");
	if(fd == -1)			    /* Failed */
	    exit(2);
	close(fd);
    }

    if(optind >= argc)
    {
	/*
	 * No packet file args, process in directory
	 */
	dp = opendir(in_dir);
	
	if(!dp)
	{
	    log("$unable to open inbound directory %s", in_dir);
	    if(lockname)
		unlink(lockname);
	    exit(EX_OSERR);
	}
    
	while((dir = readdir(dp)))
	{
	    if(wildmat(dir->d_name, "*.[pP][kK][tT]"))
	    {
		sprintf(pkt_name, "%s/%s", in_dir, dir->d_name);
		unpack_file(pkt_name);
	    }
	}
	
	closedir(dp);
    }
    else
    {
	/*
	 * Process packet files on command line
	 */
	for(; optind<argc; optind++)
	    unpack_file(argv[optind]);
    }
    

    /*
     * Execute given command, if option -x set.
     */
    if(execprog)
    {
	char buf[128];
	
	sprintf(buf, "%s/%s", cf_libdir(), execprog);
	system(buf);
    }
    
    if(lockname)
	if(unlink(buf_sprintf("%s/%s", cf_spooldir(), lockname)))
	    log("$error removing lockfile");
    
    exit(EX_OK);

    /**NOT REACHED**/
    return 1;
}
