/*  Module Header  */
/*    Name:                  PSnSApi.H                                          */
/*    Description:                                                              */
/*    Part version:          1.0                                                */
/*    Used in PSnS version:  5.1                                                */
/*    Last update:           29/Aug/1997                                        */
/*    Author:                James Austin                                       */
/*    Owner:                 Michael Fabianski                                  */
/*  */
/*  OCO SOURCE MATERIALS                                                        */
/*  IBM CONFIDENTIAL                                                            */
/*  5622884                                                                    */
/*  Copyright (c) International Business Machines Corporation 1995, 1997        */
/*                        All Rights Reserved                                   */
/*  */
/*    Change history:                                                           */
/*          29/Aug/97  JA  Initial version created                              */
/*          22/Dec/97  MF  Increase buffer size to 256                          */
/*          05/Mar/98  MF  Add support for removable drive source drive type    */
/*          12/Sep/98  MF  Add macros to convert ULONG64 to/from                */
/*                          unsigned long long                                  */
/*  */
/*  Description: PSnS API header file                                           */
/*  */
/*  Dependencies:                                                               */
/*  */
#ifndef _PSNSAPI_H_
#define _PSNSAPI_H_

#include <os2def.h>
#include <bsedos.h>

#ifdef __cplusplus
extern "C" {
#endif

/*___________________________________________________________________________*/
/*..................................................................... sizes*/
#define PSNS_NAME_SIZE         31
#define PSNS_VOLUME_NAME_SIZE  256                                       
#define PSNS_DESCRIPTION_SIZE  127
#define PSNS_DEVICE_NAME       63

/*___________________________________________________________________________*/
/*................................................................ PsnsHandle*/
typedef unsigned long PSNSHANDLE;

/*___________________________________________________________________________*/
/*................................................................... ULONG64*/
typedef struct _ULONG64 {
   unsigned long high, low;
} ULONG64, *PULONG64;

/*___________________________________________________________________________*/
/*.................................... convert ULONG64 to unsigned long long */
#define UL64TOULL(ul64) \
   ((((unsigned long long)(ul64).high)<<32)|((ul64).low))

/*___________________________________________________________________________*/
/*.................................... convert unsigned long long to ULONG64 */
#define UL64FROMULL(ul64, ull) \
   (ul64).high = (ull)>>32, (ul64).low = (ull)

/*___________________________________________________________________________*/
/*.................................................. examples of macro usage */
/* #include "stdio.h"                                                        */
/* #include <psnsapi.h>                                                      */
/*                                                                           */
/* void showvalues(ULONG64 ul64, unsigned long long ull)                     */
/* {                                                                         */
/*    printf("\nULONG64 representation = {0x%08x, 0x%08x};" \                */
/*           "\nunsigned long long     = {0x%08x, 0x%08x} %llu\n",           */
/*           ul64.high, ul64.low,                                            */
/*           (unsigned long)(ull>>32), (unsigned long)ull, ull);             */
/* }                                                                         */
/*                                                                           */
/* int main(int argc, char **argv)                                           */
/* {                                                                         */
/*    ULONG64 ul64 = {0xffffffff, 0x00000000};                               */
/*    unsigned long long ull;                                                */
/*                                                                           */
/*    ull = UL64TOULL(ul64);                                                 */
/*    showvalues(ul64, ull);                                                 */
/*                                                                           */
/*    UL64FROMULL(ul64, ull);                                                */
/*    showvalues(ul64, ull);                                                 */
/*                                                                           */
/*    ull = 0x00000000ffffffff;                                              */
/*                                                                           */
/*    UL64FROMULL(ul64, ull);                                                */
/*    showvalues(ul64, ull);                                                 */
/*                                                                           */
/*    ull = UL64TOULL(ul64);                                                 */
/*    showvalues(ul64, ull);                                                 */
/*                                                                           */
/*    return 0;                                                              */
/* }                                                                         */
/*___________________________________________________________________________*/

/*___________________________________________________________________________*/
/*....................................................... Files Folders Bytes*/
typedef struct _PSNSFFB {
   unsigned long files, folders;
   ULONG64 bytes;
} PSNSFFB, *PPSNSFFB;

/*___________________________________________________________________________*/
/*................................................................. Estimates*/
typedef struct _ESTIMATE {
   PSNSFFB total;
   unsigned long time;
} ESTIMATE, *PESTIMATE;

/*___________________________________________________________________________*/
/*................................................................. PsnsStats*/
typedef struct _PSNSSTATS {
   PSNSFFB total, processed, failed;
   unsigned long timeRemaining, timeElapsed, timeEstimate;
   unsigned long dataRate;
   unsigned long averageDataRate;
   ULONG64 spaceLeft;
} PSNSSTATS, *PPSNSSTATS;

/*___________________________________________________________________________*/
/*................................................................. HPsnsList*/
typedef int HPSNSLIST, *PHPSNSLIST;

/*___________________________________________________________________________*/
/*............................................................... PsnsStatsFn*/
typedef void (*PSNSSTATSFN)(PPSNSSTATS pPsnsStats, char *message);

/*___________________________________________________________________________*/
/*........................................................... PsnsCompression*/
typedef enum {
   PSNS_NOCOMPRESSION,
   PSNS_COMPRESSION,
   PSNS_DEFAULT
} PSNSCOMPRESSION;

/*___________________________________________________________________________*/
/*............................................................... Init & Term*/
typedef enum {
   PSNS_PIPE,
   PSNS_TCPIP
} PSNSCONNECTION;

typedef struct _PSNSPIPEINFO {
   char serverName[CCHMAXPATH];
   char pipeName[CCHMAXPATH];
} PSNSPIPEINFO, *PPSNSPIPEINFO;

typedef struct _PSNSTCPIPINFO {
   char hostName[CCHMAXPATH];
   SHORT port;
} PSNSTCPIPINFO, *PPSNSTCPIPINFO;

int APIENTRY PsnsInit(PSNSHANDLE *handle, PSNSCONNECTION psnsConnection, PVOID data);
int APIENTRY PsnsTerm(PSNSHANDLE handle);

/*___________________________________________________________________________*/
/*................................................... Backup Method Functions*/
typedef struct _BACKUPMETHODINFO {
   char name[PSNS_NAME_SIZE+1];
   char description[PSNS_DESCRIPTION_SIZE+1];
   BOOL allFiles;
   char drive;
   char directory[CCHMAXPATH];
   BOOL subdirectories;
   PSNSCOMPRESSION compression;
   int generations;
   BOOL useRulebook;
   char rulebook[PSNS_NAME_SIZE+1];
   BOOL useFileFilter;
   char fileFilter[PSNS_NAME_SIZE+1];
   BOOL changedFilesOnly;
   BOOL preview;
   char backupSet[PSNS_NAME_SIZE+1];
} BACKUPMETHODINFO, *PBACKUPMETHODINFO;

#define BM_NONE                    0x00000000
#define BM_NAME                    0x00000001
#define BM_DESCRIPTION             0x00000002
#define BM_ALLFILES                0x00000004
#define BM_DRIVE                   0x00000008
#define BM_DIRECTORY               0x00000010
#define BM_SUBDIRECTORIES          0x00000020
#define BM_COMPRESSION             0x00000040
#define BM_GENERATIONS             0x00000080
#define BM_USERULEBOOK             0x00000100
#define BM_RULEBOOK                0x00000200
#define BM_USEFILEFILTER           0x00000400
#define BM_FILEFILTER              0x00000800
#define BM_CHANGEDFILESONLY        0x00001000
#define BM_PREVIEW                 0x00002000
#define BM_BACKUPSET               0x00004000
#define BM_ALL                     0x00007FFF

int APIENTRY PsnsCreateBackupMethod(PSNSHANDLE handle, const char *name);
int APIENTRY PsnsDeleteBackupMethod(PSNSHANDLE handle, const char *name);
int APIENTRY PsnsRenameBackupMethod(PSNSHANDLE handle, const char *name, const char *newName);
int APIENTRY PsnsCopyBackupMethod(PSNSHANDLE handle, const char *name, const char *newName);
int APIENTRY PsnsBackupMethodInfo(PSNSHANDLE handle, const char *name, PBACKUPMETHODINFO pBackupMethodInfo, long mask);
int APIENTRY PsnsListBackupMethods(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PBACKUPMETHODINFO pBackupMethodInfo, long mask);
int APIENTRY PsnsEstimateBackupMethod(PSNSHANDLE handle, const char *name, PESTIMATE pEstimate);
int APIENTRY PsnsRunBackupMethod(PSNSHANDLE handle, const char *name, PSNSSTATSFN psnsStatsFn);

/*___________________________________________________________________________*/
/*.................................................. Restore Method Functions*/
typedef struct _RESTOREMETHODINFO {
   char name[PSNS_NAME_SIZE+1];
   char description[PSNS_DESCRIPTION_SIZE+1];
   char backupSet[PSNS_NAME_SIZE+1];
   BOOL allFiles;
   char drive;
   char directory[CCHMAXPATH];
   char pattern[CCHMAXPATH];
   BOOL subdirectories;
   BOOL byDate;
   FDATE date;
   FTIME time;
   BOOL preview;
   BOOL prompt;
   BOOL originalLocation;
   char destinationDrive;
   char destinationPath[CCHMAXPATH];
} RESTOREMETHODINFO, *PRESTOREMETHODINFO;

#define RM_NONE                    0x00000000
#define RM_NAME                    0x00000001
#define RM_DESCRIPTION             0x00000002
#define RM_BACKUPSET               0x00000004
#define RM_ALLFILES                0x00000008
#define RM_DRIVE                   0x00000010
#define RM_DIRECTORY               0x00000020
#define RM_PATTERN                 0x00000040
#define RM_SUBDIRECTORIES          0x00000080
#define RM_BYDATE                  0x00000100
#define RM_DATE                    0x00000200
#define RM_TIME                    0x00000400
#define RM_PREVIEW                 0x00000800
#define RM_PROMPT                  0x00001000
#define RM_ORIGINALLOCATION        0x00002000
#define RM_DESTINATIONDRIVE        0x00004000
#define RM_DESTINATIONPATH         0x00008000
#define RM_ALL                     0x0000FFFF

int APIENTRY PsnsCreateRestoreMethod(PSNSHANDLE handle, const char *name);
int APIENTRY PsnsDeleteRestoreMethod(PSNSHANDLE handle, const char *name);
int APIENTRY PsnsRenameRestoreMethod(PSNSHANDLE handle, const char *name, const char *newName);
int APIENTRY PsnsCopyRestoreMethod(PSNSHANDLE handle, const char *name, const char *newName);
int APIENTRY PsnsRestoreMethodInfo(PSNSHANDLE handle, const char *name, PRESTOREMETHODINFO pRestoreMethodInfo, long mask);
int APIENTRY PsnsListRestoreMethods(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PRESTOREMETHODINFO pRestoreMethodInfo, long mask);
int APIENTRY PsnsEstimateRestoreMethod(PSNSHANDLE handle, const char *name, PESTIMATE pEstimate);
int APIENTRY PsnsRunRestoreMethod(PSNSHANDLE handle, const char *name, PSNSSTATSFN psnsStatsFn, BOOL base, BOOL incremental);

/*___________________________________________________________________________*/
/*...................................................... Backup Set Functions*/
typedef struct _BACKUPSETINFO {
   char name[PSNS_NAME_SIZE+1];
   char description[PSNS_DESCRIPTION_SIZE+1];
   char storageDevice[PSNS_DEVICE_NAME+1];
   char incStorageDevice[PSNS_DEVICE_NAME+1];
   BOOL transferredIn;
} BACKUPSETINFO, *PBACKUPSETINFO;

#define BS_NONE                    0x00000000
#define BS_NAME                    0x00000001
#define BS_DESCRIPTION             0x00000002
#define BS_STORAGEDEVICE           0x00000004
#define BS_INCSTORAGEDEVICE        0x00000008
#define BS_TRANSFERREDIN           0x00000010
#define BS_ALL                     0x000000FF

int APIENTRY PsnsCreateBackupSet(PSNSHANDLE handle, const char *name, const char *storageDevice, const char *config, const char *incStorageDevice, const char *incConfig);
int APIENTRY PsnsDeleteBackupSet(PSNSHANDLE handle, const char *name);
int APIENTRY PsnsBackupSetInfo(PSNSHANDLE handle, const char *name, PBACKUPSETINFO pBackupSetInfo, long mask);
int APIENTRY PsnsSetBackupSetConfig(PSNSHANDLE handle, const char *name, const char *config, const char *incConfig);
int APIENTRY PsnsGetBackupSetConfig(PSNSHANDLE handle, const char *name, char *config, ULONG *length, char *incConfig, ULONG *incLength);
int APIENTRY PsnsListBackupSets(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PBACKUPSETINFO pBackupSetInfo, long mask);
int APIENTRY PsnsTransferBackupSet(PSNSHANDLE handle, const char *name, BOOL action, BOOL base);
int APIENTRY PsnsTransferIn(PSNSHANDLE handle, const char *storageDevice, const char *backupSet);
int APIENTRY PsnsGetLogFile(PSNSHANDLE handle, const char *backupSet, char *buffer);
int APIENTRY PsnsEmptyBackupSet(PSNSHANDLE handle, const char *name);

/*___________________________________________________________________________*/
/*.................................................. Storage Device Functions*/
typedef struct _STORAGEDEVICEINFO {
   char name[PSNS_DEVICE_NAME+1];
   char dllName[CCHMAXPATH];
} STORAGEDEVICEINFO, *PSTORAGEDEVICEINFO;

#define SD_NONE                    0x00000000
#define SD_NAME                    0x00000001
#define SD_DLLNAME                 0x00000002
#define SD_CREATABLE               0x00000004
#define SD_ALL                     0x00000007

int APIENTRY PsnsCreateStorageDevice(PSNSHANDLE handle, const char *type, const char *config, char *name);
int APIENTRY PsnsDeleteStorageDevice(PSNSHANDLE handle, const char *name);
int APIENTRY PsnsStorageDeviceInfo(PSNSHANDLE handle, const char *name, PSTORAGEDEVICEINFO pStorageDevice, long mask);
int APIENTRY PsnsSetStorageDeviceConfig(PSNSHANDLE handle, const char *name, const char *config);
int APIENTRY PsnsGetStorageDeviceConfig(PSNSHANDLE handle, const char *name, char *config, ULONG *length);
int APIENTRY PsnsListStorageDevices(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PSTORAGEDEVICEINFO pStorageDeviceInfo, long mask);
int APIENTRY PsnsRefreshStorageDevices(PSNSHANDLE handle);

/*___________________________________________________________________________*/
/*.......................................................... Volume Functions*/
typedef struct _VOLUMEINFO {
   ULONG id;
   char name[PSNS_VOLUME_NAME_SIZE+1];
   char storageDevice[PSNS_DEVICE_NAME+1];
   BOOL transferredOut;
   BOOL available;
   char backupSets;
   ULONG unitMultiplier;
   ULONG totalCapacity;
   ULONG psnsCapacity;
   ULONG psnsUsed;
   ULONG psnsFree;
   ULONG otherFree;
   ULONG otherUsed;
   ULONG totalBad;
   char backupSet[PSNS_NAME_SIZE+1];
   LONG indexNo;
} VOLUMEINFO, *PVOLUMEINFO;

typedef struct _BACKUPSETVOLUME {
   ULONG id;
   char backupSet[PSNS_NAME_SIZE+1];
   ULONG spaceUsed;
} BACKUPSETVOLUME, *PBACKUPSETVOLUME;

#define VOL_NONE                   0x00000000
#define VOL_ID                     0x00000001
#define VOL_NAME                   0x00000002
#define VOL_STORAGEDEVICE          0x00000004
#define VOL_TRANSFERREDOUT         0x00000008
#define VOL_AVAILABLE              0x00000010
#define VOL_BACKUPSETS             0x00000020
#define VOL_TOTALCAPACITY_LESS     0x00000040
#define VOL_TOTALCAPACITY_MORE     0x00000080
#define VOL_PSNSCAPACITY_LESS      0x00000100
#define VOL_PSNSCAPACITY_MORE      0x00000200
#define VOL_PSNSUSED_LESS          0x00000400
#define VOL_PSNSUSED_MORE          0x00000800
#define VOL_PSNSFREE_LESS          0x00001000
#define VOL_PSNSFREE_MORE          0x00002000
#define VOL_OTHERFREE_LESS         0x00004000
#define VOL_OTHERFREE_MORE         0x00008000
#define VOL_OTHERUSED_LESS         0x00010000
#define VOL_OTHERUSED_MORE         0x00020000
#define VOL_TOTALBAD_LESS          0x00040000
#define VOL_TOTALBAD_MORE          0x00080000
#define VOL_BACKUPSET              0x00100000
#define VOL_INDEXNO                0x00200000
#define VOL_ALL                    0x003FFFFF

int APIENTRY PsnsCreateVolume(PSNSHANDLE handle, const char *storageDevice, ULONG *voluemID);
int APIENTRY PsnsDeleteVolume(PSNSHANDLE handle, ULONG volumeID);
int APIENTRY PsnsVolumeInfo(PSNSHANDLE handle, ULONG volumeID, PVOLUMEINFO pVolumeInfo, long mask);
int APIENTRY PsnsListVolumeBackupSets(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, const char *storageDevice, PBACKUPSETVOLUME pBackupSetVolume);
int APIENTRY PsnsListVolumes(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PVOLUMEINFO pVolumeInfo, long mask);
int APIENTRY PsnsActivateVolume(PSNSHANDLE handle, ULONG volumeID, BOOL activate);
int APIENTRY PsnsAssociateVolume(PSNSHANDLE handle, ULONG volumeID, const char *backupSet);

/*___________________________________________________________________________*/
/*........................................................ Schedule Functions*/
typedef enum {
   EV_REGULAR  =1,
   EV_DAILY    =2,
   EV_STARTUP  =3,
   EV_NAMEDDAYS=4,
   EV_MONTHLY  =5
} EVENTTYPE;

typedef enum {
   EVENT_NO       =-1,
   EVENT_AS_METHOD=0,
   EVENT_YES      =+1
} EVENTOPTION;

typedef struct _EVENTINFO {
   USHORT id;
   EVENTTYPE type;
   char backupMethod[PSNS_NAME_SIZE+1];
   union {
      struct {
         int hours, minutes;
      } regular;
      struct {
         int hours, minutes;
      } daily;
      struct {
         int day, hours, minutes;
      } monthly;
      struct {
         int hours, minutes;
         char days, weeks;
      } namedDays;
      struct {
         int hours, minutes;
      } startup;
   } info;
   EVENTOPTION changedFilesOnly;
   EVENTOPTION showPreview;
   BOOL active;
   DATETIME lastRun;
   DATETIME nextRun;
} EVENTINFO, *PEVENTINFO;

#define EV_NONE                    0x00000000
#define EV_ID                      0x00000001
#define EV_TYPE                    0x00000002
#define EV_BACKUPMETHOD            0x00000004
#define EV_HOURS                   0x00000008
#define EV_MINUTES                 0x00000010
#define EV_DAY                     0x00000020
#define EV_DAYS                    0x00000040
#define EV_WEEKS                   0x00000080
#define EV_TIMEMASK                0x000000F8
#define EV_CHANGEDFILESONLY        0x00000100
#define EV_SHOWPREVIEW             0x00000200
#define EV_ACTIVE                  0x00000400
#define EV_LASTRUN_BEFORE          0x00000800
#define EV_LASTRUN_AFTER           0x00001000
#define EV_NEXTRUN_BEFORE          0x00002000
#define EV_NEXTRUN_AFTER           0x00004000
#define EV_ALL                     0x00007FFF

int APIENTRY PsnsCreateEvent(PSNSHANDLE handle, EVENTTYPE type, const char *bkmethod, USHORT *eventID);
int APIENTRY PsnsDeleteEvent(PSNSHANDLE handle, USHORT eventID);
int APIENTRY PsnsCopyEvent(PSNSHANDLE handle, USHORT eventID, USHORT *newEventID);
int APIENTRY PsnsEventInfo(PSNSHANDLE handle, USHORT eventID, PEVENTINFO pEventInfo, long mask);
int APIENTRY PsnsListEvents(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PEVENTINFO pEventInfo, long mask);
int APIENTRY PsnsGetNextEvent(PSNSHANDLE handle, PEVENTINFO pEventInfo);
int APIENTRY PsnsActivateEvent(PSNSHANDLE handle, USHORT eventID, BOOL activate);

/*___________________________________________________________________________*/
/*..................................................... File Filter Functions*/
typedef enum {
   FILTER_TREE=1,
   FILTER_LIST=2
} FILEFILTERTYPE;

typedef struct _FILEFILTERINFO {
   FILEFILTERTYPE type;
   char name[PSNS_NAME_SIZE+1];
   char description[PSNS_DESCRIPTION_SIZE+1];
   int numRules;
   char searchFile[CCHMAXPATH];
} FILEFILTERINFO, *PFILEFILTERINFO;

typedef struct _FILEFILTERRULE {
   char drive;
   char directory[CCHMAXPATH];
   char pattern[CCHMAXPATH];
   BOOL include;
   BOOL subdirectories;
} FILEFILTERRULE, *PFILEFILTERRULE;

#define FF_NONE                    0x00000000
#define FF_TYPE                    0x00000001
#define FF_NAME                    0x00000002
#define FF_DESCRIPTION             0x00000004
#define FF_SEARCHFILE              0x00000008
#define FF_DRIVE                   0x00000010
#define FF_DIRECTORY               0x00000020
#define FF_PATTERN                 0x00000040
#define FF_INCLUDE                 0x00000080
#define FF_SUBDIRECTORIES          0x00000100
#define FF_ALL                     0x000001FF

int APIENTRY PsnsCreateFileFilter(PSNSHANDLE handle, const char *name, FILEFILTERTYPE fileFilterType);
int APIENTRY PsnsDeleteFileFilter(PSNSHANDLE handle, const char *name);
int APIENTRY PsnsRenameFileFilter(PSNSHANDLE handle, const char *name, const char *newName);
int APIENTRY PsnsCopyFileFilter(PSNSHANDLE handle, const char *name, const char *newName);
int APIENTRY PsnsFileFilterInfo(PSNSHANDLE handle, const char *name, PFILEFILTERINFO pFileFilterInfo, long mask);
int APIENTRY PsnsAddFileFilterRule(PSNSHANDLE handle, const char *fileFilter, PFILEFILTERRULE pFileFilterRule, int position);
int APIENTRY PsnsDeleteFileFilterRule(PSNSHANDLE handle, const char *fileFilter, int which);
int APIENTRY PsnsFileFilterRuleInfo(PSNSHANDLE handle, const char *fileFilter, int which, PFILEFILTERRULE pFileFilterRule, long mask);
int APIENTRY PsnsListFileFilters(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PFILEFILTERINFO pFileFilterInfo, long mask);

/*___________________________________________________________________________*/
/*........................................................ Rulebook Functions*/
typedef struct _RULEBOOKINFO {
   char name[PSNS_NAME_SIZE+1];
   char description[PSNS_DESCRIPTION_SIZE+1];
   int numRules;
} RULEBOOKINFO, *PRULEBOOKINFO;

typedef struct _RULEBOOKRULE {
   char drive;
   char directory[CCHMAXPATH];
   char pattern[CCHMAXPATH];
   BOOL subdirectories;
   int generations;
   PSNSCOMPRESSION compression;
} RULEBOOKRULE, *PRULEBOOKRULE;

#define RB_NONE                    0x00000000
#define RB_NAME                    0x00000001
#define RB_DESCRIPTION             0x00000002
#define RB_DRIVE                   0x00000004
#define RB_DIRECTORY               0x00000008
#define RB_PATTERN                 0x00000010
#define RB_GENERATIONS             0x00000020
#define RB_COMPRESSION             0x00000040
#define RB_SUBDIRECTORIES          0x00000080
#define RB_ALL                     0x000000FF

int APIENTRY PsnsCreateRulebook(PSNSHANDLE handle, const char *name);
int APIENTRY PsnsDeleteRulebook(PSNSHANDLE handle, const char *name);
int APIENTRY PsnsRenameRulebook(PSNSHANDLE handle, const char *name, const char *newName);
int APIENTRY PsnsCopyRulebook(PSNSHANDLE handle, const char *name, const char *newName);
int APIENTRY PsnsRulebookInfo(PSNSHANDLE handle, const char *name, PRULEBOOKINFO pRuleBookInfo, long mask);
int APIENTRY PsnsAddRulebookRule(PSNSHANDLE handle, const char *rulebook, PRULEBOOKRULE pRulebookRule, int position);
int APIENTRY PsnsDeleteRulebookRule(PSNSHANDLE handle, const char *rulebook, int which);
int APIENTRY PsnsRulebookRuleInfo(PSNSHANDLE handle, const char *rulebook, int which, PRULEBOOKRULE pRulebookRule, long mask);
int APIENTRY PsnsListRulebooks(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PRULEBOOKINFO pRulebookInfo, long mask);

/*___________________________________________________________________________*/
/*.................................................... Source Drive Functions*/
typedef enum {
   DRI_HARDDISK =1,
   DRI_DISKETTE =2,
   DRI_LANDRIVE =3,
   DRI_RWOPTICAL=4,
   DRI_CDROM    =5,
   DRI_PRM      =6,
   DRI_OTHER    =7
} DRIVETYPE;

typedef struct _SOURCEDRIVEINFO {
   char letter;
   DRIVETYPE type;
   BOOL selected;
} SOURCEDRIVEINFO, *PSOURCEDRIVEINFO;

#define DR_NONE                       0x00000000
#define DR_LETTER                     0x00000001
#define DR_TYPE                       0x00000002
#define DR_SELECTED                   0x00000004
#define DR_ALL                        0x00000007

int APIENTRY PsnsSelectSourceDrive(PSNSHANDLE handle, char drive, BOOL use);
int APIENTRY PsnsSourceDriveType(PSNSHANDLE handle, char drive, DRIVETYPE driveType);
int APIENTRY PsnsListSourceDrives(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PSOURCEDRIVEINFO pSourceDriveInfo, long mask);
int APIENTRY PsnsRefreshSourceDrives(PSNSHANDLE handle);

/*___________________________________________________________________________*/
/*........................................... Settings and Defaults Functions*/
typedef enum {
   NORMAL=1,
   HIGH  =2
} PSNSPRIORITY;

typedef struct _PSNSDEFAULTS {
   char fileFilter[PSNS_NAME_SIZE+1];
   char rulebook[PSNS_NAME_SIZE+1];
   char backupSet[PSNS_NAME_SIZE+1];
   BOOL compression;
   int generations;
} PSNSDEFAULTS, *PPSNSDEFAULTS;

typedef struct _PSNSSETTINGS {
   PSNSPRIORITY manualPriority;
   PSNSPRIORITY autoPriority;
   char tempPath[CCHMAXPATH];
   long flags;
   ULONG logFileLimit;
   char backupExit[CCHMAXPATH];
   char restoreExit[CCHMAXPATH];
   BOOL enableACLBackup;
   char uadFilename[CCHMAXPATH];
} PSNSSETTINGS, *PPSNSSETTINGS;

#define SET_NONE                    0x00000000
#define SET_FILEFILTER              0x00000001
#define SET_RULEBOOK                0x00000002
#define SET_BACKUPSET               0x00000004
#define SET_COMPRESSION             0x00000008
#define SET_GENERATIONS             0x00000010
#define SET_ACLENABLE               0x00000020
#define SET_ACLFILENAME             0x00000040
#define SET_MANUALPRIORITY          0x00000080
#define SET_AUTOPRIORITY            0x00000100
#define SET_TEMPPATH                0x00000200
#define SET_FLAGS                   0x00000400
#define SET_LOGFILELIMIT            0x00000800
#define SET_BACKUPEXIT              0x00001000
#define SET_RESTOREEXIT             0x00002000
#define SET_ALL                     0x00003FFF

#define SET_LOG                     0x00000001
#define SET_LOG_SUCCESSES           0x00000002
#define SET_LOG_FAILURES            0x00000004
#define SET_BACKUP_EXIT             0x00000008
#define SET_RESTORE_EXIT            0x00000010

int APIENTRY PsnsDefaults(PSNSHANDLE handle, PPSNSDEFAULTS pPsnsDefaults, long mask);
int APIENTRY PsnsSettings(PSNSHANDLE handle, PPSNSSETTINGS pPsnsSettings, long mask);

/*___________________________________________________________________________*/
/*..........................................................Archive Functions*/
typedef struct _PSNSFILEINFO {
   char backupSet[PSNS_NAME_SIZE+1];
   char filename[CCHMAXPATH];
   short generations;
   short thisGeneration;
   FDATE creationDate;
   FTIME creationTime;
   FDATE writeDate;
   FTIME writeTime;
   FDATE backupDate;
   FTIME backupTime;
   ULONG size;
   ULONG attribs;
   ULONG volumeID;
} PSNSFILEINFO, *PPSNSFILEINFO;

#define FILE_NONE                  0x00000000
#define FILE_BACKUPSET             0x00000001
#define FILE_FILENAME              0x00000002
#define FILE_GENERATIONS_LESS      0x00000004
#define FILE_GENERATIONS_MORE      0x00000008
#define FILE_CREATIONDATE_BEFORE   0x00000010
#define FILE_CREATIONDATE_ON       0x00000020
#define FILE_CREATIONDATE_AFTER    0x00000040
#define FILE_CREATIONTIME_BEFORE   0x00000080
#define FILE_CREATIONTIME_AFTER    0x00000100
#define FILE_WRITEDATE_BEFORE      0x00000200
#define FILE_WRITEDATE_ON          0x00000400
#define FILE_WRITEDATE_AFTER       0x00000800
#define FILE_WRITETIME_BEFORE      0x00001000
#define FILE_WRITETIME_AFTER       0x00002000
#define FILE_BACKUPDATE_BEFORE     0x00004000
#define FILE_BACKUPDATE_ON         0x00008000
#define FILE_BACKUPDATE_AFTER      0x00010000
#define FILE_BACKUPTIME_BEFORE     0x00020000
#define FILE_BACKUPTIME_AFTER      0x00040000
#define FILE_SIZE_LESS             0x00080000
#define FILE_SIZE_MORE             0x00100000
#define FILE_ATTRIBS_EXACT         0x00200000
#define FILE_ATTRIBS_ALLOF         0x00400000
#define FILE_ATTRIBS_SOMEOF        0x00800000
#define FILE_ATTRIBS_NOTSOMEOF     0x01000000
#define FILE_ATTRIBS_NONEOF        0x02000000
#define FILE_VOLUMEID              0x04000000
#define FILE_ALL                   0x07FFFFFF
#define FILE_ALLGENERATIONS        0x08000000

#define ALL_GENERATIONS            0xFFFF

int APIENTRY PsnsListFiles(PSNSHANDLE handle, PHPSNSLIST pHPsnsList, PPSNSFILEINFO pPsnsFileInfo, long mask);

/*___________________________________________________________________________*/
/*................................................ Higher-Level API Functions*/
int APIENTRY PsnsAddFile(PSNSHANDLE handle, const char *filename, short gen);
int APIENTRY PsnsBackupFiles(PSNSHANDLE handle, PBACKUPMETHODINFO pBackupMethodInfo, PSNSSTATSFN stats);
int APIENTRY PsnsRestoreFiles(PSNSHANDLE handle, PRESTOREMETHODINFO pRestoreMethodInfo, PSNSSTATSFN stats);
int APIENTRY PsnsDropFiles(PSNSHANDLE handle, const char *backupSet, PSNSSTATSFN stats);
int APIENTRY PsnsPurgeFiles(PSNSHANDLE handle);

int APIENTRY PsnsCloseList(PSNSHANDLE handle, PHPSNSLIST pHPsnsList);

#define PSNSM_OK                      0x00000001L
#define PSNSM_RETRY                   0x00000002L
#define PSNSM_IGNORE                  0x00000004L
#define PSNSM_CANCEL                  0x00000008L
#define PSNSM_YES                     0x00000010L
#define PSNSM_NO                      0x00000020L
#define PSNSM_ABORT                   0x00000040L
#define PSNSM_DEFAULT                 0x00000100L

typedef ULONG (*PSNSMESSAGEFN)(const char *msg, ULONG style, void *privates);

int APIENTRY PsnsMessageCallback(PSNSHANDLE handle, PSNSMESSAGEFN fn, void *privates);
int APIENTRY PsnsGetMessageText(PSNSHANDLE handle, int rc, char *message, ULONG *length);
int APIENTRY PsnsGetLastError(PSNSHANDLE handle, char *message, ULONG *length);

/*___________________________________________________________________________*/

#ifdef __cplusplus
}
#endif

#endif
/*___________________________________________________________________________*/
