UNIT synth;
{$APPTYPE CONSOLE}
{$IFDEF FPC}
{$MODE DELPHI}
{$HINTS OFF}
{$ENDIF}
{$UNDEF CLEAN}

INTERFACE

USES Windows,MMSystem;

TYPE PSINGLE=^SINGLE;
     PBYTE=^BYTE;

     TWAVEHDR=WAVEHDR;

CONST Max4kChannels=16;

TYPE T4kFileData=PACKED RECORD
      WaveForm:ARRAY[0..Max4kChannels-1] OF BYTE;
      Link:ARRAY[0..Max4kChannels-1] OF BYTE;
      LinkFactor:ARRAY[0..Max4kChannels-1] OF SINGLE;
      OutFactor:ARRAY[0..Max4kChannels-1] OF SINGLE;
      StartVolume:ARRAY[0..Max4kChannels-1] OF SINGLE;
      StartVolumeFactor:ARRAY[0..Max4kChannels-1] OF SINGLE;
      StartPhaseFactor:ARRAY[0..Max4kChannels-1] OF SINGLE;
      Panning:ARRAY[0..Max4kChannels-1] OF BYTE;
      EventOffset:ARRAY[0..Max4kChannels-1] OF INTEGER;
      NoteOffset:ARRAY[0..Max4kChannels-1] OF INTEGER;
      VolumeOffset:ARRAY[0..Max4kChannels-1] OF INTEGER;
     END;

     T4kFilter=PACKED RECORD
      FD1,FD2,FD3,FD4:SINGLE;
      FBLP,FBHP:SINGLE;
      CLP,CHP:SINGLE;
      RLP,RHP:SINGLE;
     END;

     T4kRealTimeData=PACKED RECORD
      LinkValues:ARRAY[0..Max4kChannels-1] OF SINGLE;
      Volume:ARRAY[0..Max4kChannels-1] OF SINGLE;
      VolumeValue:ARRAY[0..Max4kChannels-1] OF SINGLE;
      VolumeFactor:ARRAY[0..Max4kChannels-1] OF SINGLE;
      Phase:ARRAY[0..Max4kChannels-1] OF SINGLE;
      PhaseIncrement:ARRAY[0..Max4kChannels-1] OF SINGLE;
      PhaseFactor:ARRAY[0..Max4kChannels-1] OF SINGLE;
      Filter:ARRAY[0..Max4kChannels-1] OF T4kFilter;
      TickCounter:ARRAY[0..Max4kChannels-1] OF LONGINT;
      EventOffset:ARRAY[0..Max4kChannels-1] OF PBYTE;
      NoteOffset:ARRAY[0..Max4kChannels-1] OF PBYTE;
      VolumeOffset:ARRAY[0..Max4kChannels-1] OF PBYTE;
      BPMSamples:LONGWORD;
      CurrentBPMSamples:LONGWORD;
     END;

     T4kEventData=ARRAY[0..$1FFFFF] OF BYTE;

     T4kSynthData=PACKED RECORD
      FileData:T4kFileData;
      EventData:T4kEventData;
      RealtimeData:T4kRealtimeData;
     END;

CONST SampleRate=44100;
      SampleRateFactor=440/SampleRate;
      PortamentoFactor=SampleRateFactor/65536;
      Div8=1/8;
      Div12=1/12;
      Div64=1/64;
      Div127=1/127;
      Div255=1/255;
      Div256=1/256;
      Div65536=1/65536;
      CutOffLowPassStartValue=255*Div256;
      BufferSize=4096;

CONST WaveFormat:TWaveFormatEx=(wFormatTag:3;nChannels:2;nSamplesPerSec:SampleRate;
                                nAvgBytesPerSec:SampleRate*8;nBlockAlign:8;
                                wBitsPerSample:32;cbSize:0);

VAR WaveOutHandle:LONGWORD;
    WaveHandler:ARRAY[0..3] OF TWAVEHDR;
    BufferCounter:LONGWORD;
    WhiteNoiseSeed:LONGWORD;
    Buffers:ARRAY[0..3,0..BufferSize,0..1] OF SINGLE;
    SynthData:T4kSynthData;

PROCEDURE SynthFillBuffer(VAR SynthData:T4kSynthData;Buffer:PSINGLE); STDCALL;
PROCEDURE SynthInit(TrackData:POINTER;TrackSize:INTEGER); STDCALL;
PROCEDURE SynthPoll(TrackData:POINTER;TrackSize:INTEGER); STDCALL;
PROCEDURE SynthDone(TrackData:POINTER;TrackSize:INTEGER); STDCALL;

IMPLEMENTATION

FUNCTION F_POWER(Number,Exponent:SINGLE):SINGLE; ASSEMBLER; STDCALL;
ASM
 FLD Exponent
 FLD Number
 FYL2X
 FLD1
 FLD ST(1)
 FPREM
 F2XM1
 FADDP ST(1),ST
 FSCALE
 FSTP ST(1)
END;

FUNCTION WhiteNoiseRandom:SINGLE; STDCALL;
VAR WhiteNoiseValue:LONGWORD;
BEGIN
 WhiteNoiseSeed:=(WhiteNoiseSeed*$524281)+$3133731;
 WhiteNoiseValue:=(WhiteNoiseSeed AND $7FFFFF) OR $40000000;
 RESULT:=SINGLE(POINTER(@WhiteNoiseValue)^)-3;
END;

FUNCTION FRAC(X:SINGLE):SINGLE; STDCALL;
VAR Y:SINGLE;
BEGIN
 Y:=X;
 ASM
  FLD DWORD PTR Y
  FRNDINT
  FSTP DWORD PTR Y
 END;
 RESULT:=X-Y;
END;

PROCEDURE SynthReinit(VAR SynthData:T4kSynthData); STDCALL; {$IFDEF FPC}[public, alias : 'SynthReinit'];{$ENDIF}
VAR Channel:INTEGER;
BEGIN
 FOR Channel:=0 TO Max4kChannels-1 DO BEGIN
  IF SynthData.FileData.EventOffset[Channel]>=0 THEN BEGIN
   SynthData.RealtimeData.EventOffset[Channel]:=POINTER(LONGWORD(@SynthData.EventData)+LONGWORD(SynthData.FileData.EventOffset[Channel]));
   SynthData.RealtimeData.NoteOffset[Channel]:=POINTER(LONGWORD(@SynthData.EventData)+LONGWORD(SynthData.FileData.NoteOffset[Channel]));
   SynthData.RealtimeData.VolumeOffset[Channel]:=POINTER(LONGWORD(@SynthData.EventData)+LONGWORD(SynthData.FileData.VolumeOffset[Channel]));
   SynthData.RealtimeData.TickCounter[Channel]:=0;
  END;
 END;
END;

PROCEDURE SynthRecalcFilter(VAR Filter:T4kFilter); STDCALL; {$IFDEF FPC}[public, alias : 'SynthRecalcFilter'];{$ENDIF}
BEGIN
 WITH Filter DO BEGIN
  FBLP:=RLP+RLP/(1-CLP);
  FBHP:=RHP+RHP/(1-CLP);
 END;
END;

PROCEDURE SynthInitData(VAR SynthData:T4kSynthData); STDCALL; {$IFDEF FPC}[public, alias : 'SynthInitData'];{$ENDIF}
VAR Channel:INTEGER;
BEGIN
 SynthReinit(SynthData);
 FOR Channel:=0 TO Max4kChannels-1 DO BEGIN
  SynthData.RealtimeData.Filter[Channel].CLP:=CutOffLowPassStartValue;
  SynthRecalcFilter(SynthData.RealtimeData.Filter[Channel]);
 END;
END;

PROCEDURE SynthFillBuffer(VAR SynthData:T4kSynthData;Buffer:PSINGLE); STDCALL; {$IFDEF FPC}[public, alias : 'SynthFillBuffer'];{$ENDIF}
VAR Position,Channel,Count,WaveForm:INTEGER;
    Phase,Left,Right,PanningValue,OscValue,Value:SINGLE;
    PhaseCasted:LONGWORD ABSOLUTE Phase;
    Note,Volume:BYTE;
BEGIN
 FOR Position:=1 TO BufferSize DO BEGIN
  Count:=Max4kChannels;
  Left:=0;
  Right:=0;
  FOR Channel:=0 TO Max4kChannels-1 DO BEGIN
   IF SynthData.RealtimeData.CurrentBPMSamples=0 THEN BEGIN
    WHILE ASSIGNED(SynthData.RealtimeData.EventOffset[Channel]) AND (SynthData.RealtimeData.TickCounter[Channel]>=SynthData.RealtimeData.EventOffset[Channel]^) DO BEGIN
     SynthData.RealtimeData.TickCounter[Channel]:=0;
     Note:=SynthData.RealtimeData.NoteOffset[Channel]^;
     Volume:=SynthData.RealtimeData.VolumeOffset[Channel]^;
     INC(SynthData.RealtimeData.EventOffset[Channel]);
     INC(SynthData.RealtimeData.NoteOffset[Channel]);
     INC(SynthData.RealtimeData.VolumeOffset[Channel]);
     IF (Note<>0) AND (Note<=$81) THEN BEGIN
      IF Note<$80 THEN BEGIN
       SynthData.RealtimeData.PhaseIncrement[Channel]:=F_POWER(2,(Note-49)*Div12)*SampleRateFactor;
       SynthData.RealtimeData.PhaseFactor[Channel]:=SynthData.FileData.StartPhaseFactor[Channel];
       SynthData.RealtimeData.Volume[Channel]:=SynthData.FileData.StartVolume[Channel];
       SynthData.RealtimeData.VolumeFactor[Channel]:=SynthData.FileData.StartVolumeFactor[Channel];
      END ELSE IF Note=$80 THEN BEGIN
       SynthData.RealtimeData.PhaseIncrement[Channel]:=0;
       SynthData.RealtimeData.PhaseFactor[Channel]:=1;
      END;
      IF Volume<=64 THEN BEGIN
       SynthData.RealtimeData.VolumeValue[Channel]:=Volume*Div64;
      END;
     END ELSE BEGIN
      Value:=Volume*Div256;
      CASE Note OF
       $82:SynthData.RealtimeData.BPMSamples:=(SampleRate*5*128) DIV (Volume SHL 8);
       $83:SynthData.RealtimeData.EventOffset[Channel]:=NIL;
       $84,$85:BEGIN
        OscValue:=Volume*PortamentoFactor;
        IF Note=$84 THEN OscValue:=-OscValue;
        SynthData.RealtimeData.PhaseIncrement[Channel]:=SynthData.RealtimeData.PhaseIncrement[Channel]*(1+OscValue);
       END;
       $86:SynthData.FileData.Panning[Channel]:=Volume;
       $87:BEGIN
        SynthData.RealtimeData.Filter[Channel].CLP:=Value;
        SynthRecalcFilter(SynthData.RealtimeData.Filter[Channel]);
       END;
       $88:SynthData.RealtimeData.Filter[Channel].CHP:=Value;
       $89:SynthData.RealtimeData.Filter[Channel].RLP:=Value;
       $90:SynthData.RealtimeData.Filter[Channel].RHP:=Value;
      END;
     END;
    END;
    IF NOT ASSIGNED(SynthData.RealtimeData.EventOffset[Channel]) THEN DEC(Count);
    INC(SynthData.RealtimeData.TickCounter[Channel]);
   END;
   Phase:=FRAC(SynthData.RealtimeData.Phase[Channel]+SynthData.RealtimeData.LinkValues[Channel]);
   WaveForm:=SynthData.FileData.WaveForm[Channel];
   CASE WaveForm OF
    0:OscValue:=SIN(Phase*2*PI); // Sinus
    1:OscValue:=ABS((Phase-0.5)*4)-1; // Triangle
    2:BEGIN // Square
     Phase:=Phase-0.5;
     OscValue:=1-((PhaseCasted SHR 31) SHL 1);
    END;
    3,4:OscValue:=((Phase-0.5)*2)*INTEGER(1-((WaveForm-3)*2)); // Sawtooth Up/Down
    5:OscValue:=WhiteNoiseRandom; // White Noise
    ELSE OscValue:=0; // Nothing ;-)
   END;
   OscValue:=OscValue*SynthData.RealtimeData.Volume[Channel]*SynthData.RealtimeData.VolumeValue[Channel];
   WITH SynthData.RealtimeData.Filter[Channel] DO BEGIN
    FD1:=FD1+CLP*(OscValue-FD1+FBLP*(FD1-FD2));
    FD2:=FD2+CLP*(FD1-FD2);
    FD3:=FD3+CHP*(FD2-FD3+FBHP*(FD3-FD4));
    FD4:=FD4+CHP*(FD3-FD4);
    OscValue:=FD2-FD4;
   END;
   IF (SynthData.FileData.Link[Channel] AND $7F)<Max4kChannels THEN BEGIN
    SynthData.RealtimeData.LinkValues[SynthData.FileData.Link[Channel] AND $7F]:=OscValue*SynthData.FileData.LinkFactor[Channel];
   END;
   IF (SynthData.FileData.Link[Channel] AND $80)<>0 THEN BEGIN
    Value:=OscValue*SynthData.FileData.OutFactor[Channel];
    PanningValue:=SynthData.FileData.Panning[Channel]*Div255;
    Left:=Left+(Value*PanningValue);
    Right:=Right+(Value*(1-PanningValue));
   END;
   SynthData.RealtimeData.Phase[Channel]:=FRAC(SynthData.RealtimeData.Phase[Channel]+SynthData.RealtimeData.PhaseIncrement[Channel]);
   SynthData.RealtimeData.PhaseIncrement[Channel]:=SynthData.RealtimeData.PhaseIncrement[Channel]*SynthData.RealtimeData.PhaseFactor[Channel];
   SynthData.RealtimeData.Volume[Channel]:=SynthData.RealtimeData.Volume[Channel]*SynthData.RealtimeData.VolumeFactor[Channel];
  END;
  IF SynthData.RealtimeData.CurrentBPMSamples=0 THEN BEGIN
   IF Count=0 THEN SynthReinit(SynthData);
   SynthData.RealtimeData.CurrentBPMSamples:=SynthData.RealtimeData.BPMSamples;
  END;
  DEC(SynthData.RealtimeData.CurrentBPMSamples);
  Buffer^:=Left*Div8;
  INC(Buffer);
  Buffer^:=Right*Div8;
  INC(Buffer);
 END;
END;

PROCEDURE SynthInit(TrackData:POINTER;TrackSize:INTEGER); STDCALL; {$IFDEF FPC}[public, alias : 'SynthInit'];{$ENDIF}
CONST cwChop:WORD=$F7B;
VAR Counter:INTEGER;
BEGIN
 ASM
  FLDCW cwChop
 END;

 FILLCHAR(SynthData,SIZEOF(T4kSynthData),#0);
 MOVE(TrackData^,SynthData,TrackSize);
 SynthInitData(SynthData);

 WhiteNoiseSeed:=$12345678;

 waveOutOpen(@WaveOutHandle,WAVE_MAPPER,@WaveFormat,0,0,0);

 FOR Counter:=0 TO 3 DO BEGIN
  WaveHandler[Counter].dwFlags:=WHDR_DONE;
  WaveHandler[Counter].lpData:=@Buffers[Counter];
  WaveHandler[Counter].dwBufferLength:=BufferSize*8;
 END;

 BufferCounter:=0;
END;

PROCEDURE SynthPoll(TrackData:POINTER;TrackSize:INTEGER); STDCALL; {$IFDEF FPC}[public, alias : 'SynthPoll'];{$ENDIF}
BEGIN
 IF (WaveHandler[BufferCounter].dwFlags AND WHDR_DONE)<>0 THEN BEGIN
  IF waveOutUnprepareHeader(WaveOutHandle,@WaveHandler[BufferCounter],SIZEOF(TWAVEHDR))<>WAVERR_STILLPLAYING THEN BEGIN
   WaveHandler[BufferCounter].dwFlags:=WaveHandler[BufferCounter].dwFlags AND NOT WHDR_DONE;
   SynthFillBuffer(SynthData,@WaveHandler[BufferCounter].lpData^);
   waveOutPrepareHeader(WaveOutHandle,@WaveHandler[BufferCounter],SIZEOF(TWAVEHDR));
   waveOutWrite(WaveOutHandle,@WaveHandler[BufferCounter],SIZEOF(TWAVEHDR));
   BufferCounter:=(BufferCounter+1) MOD 4;
  END;
 END;
END;

PROCEDURE SynthDone(TrackData:POINTER;TrackSize:INTEGER); STDCALL; {$IFDEF FPC}[public, alias : 'SynthDone'];{$ENDIF}
VAR Counter:INTEGER;
BEGIN
 FOR Counter:=0 TO 3 DO BEGIN
  WHILE waveOutUnprepareHeader(WaveOutHandle,@WaveHandler[Counter],SIZEOF(TWAVEHDR))=WAVERR_STILLPLAYING DO BEGIN
  END;
 END;
 waveOutReset(WaveOutHandle);
 waveOutClose(WaveOutHandle);
END;

END.
