/*****************************************************************************

        TestMidiFile.cpp
        Author: Laurent de Soras, 2022

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law.You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://www.wtfpl.net/ for more details.

*Tab=3***********************************************************************/



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include "fstb/fnc.h"
#include "test/TestMidiFile.h"
#include "midi.h"
#include "MidiFile.h"

#include <stdexcept>
#include <vector>

#include <cassert>
#include <cstdint>
#include <cstdio>



/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



int	TestMidiFile::perform_test ()
{
	int            ret_val = 0;

	printf ("Testing MidiFile...\n");
	fflush (stdout);

	std::vector <uint8_t> content;
	if (ret_val == 0)
	{
		ret_val = load_file (
			content,
			"../../test_material/jbhak/2022-05-12/MIDI - ligne clavier.mid"
		);
	}

	if (ret_val == 0)
	{
		try
		{
			MidiFile    mf (
				content.data (), content.data () + content.size (),
				MidiFileTrack::NoteRepeat::_accumulate
			);

			const auto     format     = mf.get_format ();
			const auto     time_info  = mf.get_time_info ();
			const int      nbr_tracks = mf.get_nbr_tracks ();
			printf ("Format: %d\n", int (format));
			printf ("Time  : ");
			if (time_info._smpte)
			{
				printf (
					"SMPTE %d/%d fps, %d ticks/frame\n",
					time_info._smpte->_fps_num,
					time_info._smpte->_fps_den,
					time_info._nbr_ticks
				);
			}
			else
			{
				printf ("%d ticks/quarter\n", time_info._nbr_ticks);
			}
			printf ("Tracks: %d\n", nbr_tracks);

			for (int trk_idx = 0; trk_idx < nbr_tracks; ++trk_idx)
			{
				printf ("--- Track %d ---\n", trk_idx);
				const auto &   track = mf.use_track (trk_idx);
				print_track (track);
			}
		}
		catch (std::exception &e)
		{
			printf ("Exception: %s\n", e.what ());
			ret_val = -1;
		}
		catch (...)
		{
			printf ("Exception (unknown)\n");
			ret_val = -1;
		}

	}

	printf ("Done.");
	fflush (stdout);

	return ret_val;
}



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



int	TestMidiFile::load_file (std::vector <uint8_t> &content, const char filename_0 [])
{
	int            ret_val = 0;

	auto           f_ptr = fstb::fopen_utf8 (filename_0, "rb");
	if (f_ptr == nullptr)
	{
		ret_val = -1;
	}

	if (ret_val == 0 && fseek (f_ptr, 0, SEEK_END) != 0)
	{
		ret_val = -1;
	}
	long           len = 0;
	if (ret_val == 0)
	{
		len = ftell (f_ptr);
		if (fseek (f_ptr, 0, SEEK_SET) != 0)
		{
			ret_val = -1;
		}
	}

	if (ret_val == 0)
	{
		content.resize (size_t (len));
		if (fread (content.data (), len, 1, f_ptr) != 1)
		{
			ret_val = -1;
		}
	}

	if (f_ptr != nullptr)
	{
		fclose (f_ptr);
	}

	return ret_val;
}



void	TestMidiFile::print_track (const MidiFileTrack &track)
{
	const auto &   tempo_map = track.use_tempo_map ();
	if (! tempo_map.empty ())
	{
		printf ("Tempo list:\n");
		for (const auto &p : tempo_map)
		{
			printf ("%6d: %f quarter/min\n", int (p.first), p.second);
		}
	}

	const auto &   time_sig_map = track.use_time_sig_map ();
	if (! time_sig_map.empty ())
	{
		printf ("Time signatures:\n");
		for (const auto &p : time_sig_map)
		{
			printf ("%6d: %d/%d\n", int (p.first), p.second._num, p.second._den);
		}
	}

	const auto     note_arr = track.get_notes_all_chn ();
	for (const auto &note : note_arr)
	{
		printf (
			"%6d: note %3d, velo %3d, duration %d\n",
			int (note._timestamp),
			int (note._note),
			int (note._velo),
			int (note._duration)
		);
	}
}



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
