/*****************************************************************************

        CharData.cpp
        Author: Laurent de Soras, 2008

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*Tab=3***********************************************************************/



#if defined (_MSC_VER)
	#pragma warning (1 : 4130 4223 4705 4706)
	#pragma warning (4 : 4355 4786 4800)
#endif



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include "fstb/txt/unicode/Categ.h"
#include "fstb/txt/unicode/CharData.h"
#include "fstb/def.h"

#include <cassert>



namespace fstb
{
namespace txt
{
namespace unicode
{



/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



Categ	CharData::get_categ (char32_t c)
{
	const CharDataInfo &	info = use_info (c);

	return (static_cast <Categ> (info._categ));
}



char32_t	CharData::to_upper (char32_t c)
{
	const CharDataInfo &	info = use_info (c);
	const char32_t c2 = info._upper;
	if (c2 != 0)
	{
		c = c2;
	}

	return (c);
}



char32_t	CharData::to_lower (char32_t c)
{
	const CharDataInfo &	info = use_info (c);
	const char32_t c2 = info._lower;
	if (c2 != 0)
	{
		c = c2;
	}

	return (c);
}



char32_t	CharData::to_title (char32_t c)
{
	const CharDataInfo &	info = use_info (c);
	const char32_t c2 = info._title;
	if (c2 != 0)
	{
		c = c2;
	}

	return (c);
}



GraphemeBreakProp	CharData::get_grapheme_break_prop (char32_t c)
{
	const CharDataInfo &	info = use_info (c);

	return (static_cast <GraphemeBreakProp> (info._grapheme_break_prop));
}



LineBreakProp	CharData::get_line_break_prop (char32_t c)
{
	const CharDataInfo &	info = use_info (c);

	return (static_cast <LineBreakProp> (info._line_break_prop));
}



WordBreakProp	CharData::get_word_break_prop (char32_t c)
{
	const CharDataInfo &	info = use_info (c);

	return (static_cast <WordBreakProp> (info._word_break_prop));
}



BidiClass	CharData::get_bidi_class (char32_t c)
{
	const CharDataInfo &	info = use_info (c);

	return (static_cast <BidiClass> (info._bidi_class));
}



/*
==============================================================================
Name: is_grapheme_cluster_boundary
Description:

	Indicates if there is a grapheme cluster boundary between the two adjascent
	submitted characters.

	Algorithm from:

	Unicode Standard Annex #29: Unicode Text Segmentation
	Section 3: Grapheme Cluster Boundaries
	http://www.unicode.org/reports/tr29/

Input parameters:
	- c_prev: First character
	- c_next: Second character
Returns: true if there is a grapheme boudary.
Throws: Nothing
==============================================================================
*/

bool	CharData::is_grapheme_cluster_boundary (char32_t c_prev, char32_t c_next)
{
	const GraphemeBreakProp	gbp_prev = get_grapheme_break_prop (c_prev);
	const GraphemeBreakProp	gbp_next = get_grapheme_break_prop (c_next);

	const bool		break_flag = _table_break_grapheme [gbp_prev] [gbp_next];

	return (break_flag);
}



/*
==============================================================================
Name: find_line_breaks
Description:

	Computes line breaks actions following each character of a string.

	The string should have been previously processed in order to submit to the
	function an array of line break properties (each element correspounding to
	a character).

	The function returns an array of break actions, indicating which things can
	or must be done after the correspounding character.

	The function stops at the first mandatory line break encountered. The end
	of the string is always treated as a mandatory line break.

	Algorithm from:

	Unicode Standard Annex #14: Unicode Line Breaking Algorithm
	Section 7: Pair Table-Based Implementation
	http://www.unicode.org/reports/tr14/

	http://www.unicode.org/Public/PROGRAMS/LineBreakSampleCpp/5.1.0/

Input parameters:
	- prop_arr: Line break properties which has been previously extracted for
		each character of the string. The function does not requires the string
		to be null-terminated.
	- len: Number of character in the string, > 0
	- find_complex_line_breaks: optional custom function for handling script-
		dependent complexe cases.
Output parameters:
	- action_arr: Array giving for each processed character of the string, the
		opportunity to insert a line break after it. The array should have the
		same length as the input string; memory should have been reserved
		consequently. The number of valid output positions depends on the value
		returned by the function.
Returns: The number of processed characters
Throws: Nothing
==============================================================================
*/

size_t	CharData::find_line_breaks (LineBreakAction action_arr [], const LineBreakProp prop_arr [], size_t len, FindCplxBreakFnc &find_complex_line_breaks)
{
	assert (action_arr != nullptr);
	assert (prop_arr != nullptr);
	assert (len > 0);

	LineBreakProp	cls = prop_arr [0];

	// handle case where input starts with an LF
	if (cls == LineBreakProp_LF)
	{
		cls = LineBreakProp_BK;
	}

	// treat NL like BK
	if (cls == LineBreakProp_NL)
	{
		cls = LineBreakProp_BK;
	}

	// treat SP at start of input as if it followed WJ
	if (cls == LineBreakProp_SP)
	{
		cls = LineBreakProp_WJ;
	}

	// loop over all pairs in the string up to a hard break or CRLF pair
	size_t			ich;
	for (ich = 1
	;	   (ich < len)
		&& (cls != LineBreakProp_BK)
		&& (cls != LineBreakProp_CR || prop_arr [ich] == LineBreakProp_LF)
	;	++ich)
	{
		// handle spaces explicitly
		if (prop_arr [ich] == LineBreakProp_SP)
		{
			action_arr [ich-1] = LineBreakAction_PROHIBITED;	// apply rule LB 7: SP
			continue;	// do not update cls
		}

		if (   prop_arr [ich] == LineBreakProp_BK
		    || prop_arr [ich] == LineBreakProp_NL
		    || prop_arr [ich] == LineBreakProp_LF)
		{
			action_arr [ich-1] = LineBreakAction_PROHIBITED;
			cls = LineBreakProp_BK;
			continue;
		}

		if (prop_arr [ich] == LineBreakProp_CR)
		{
			action_arr [ich-1] = LineBreakAction_PROHIBITED;
			cls = LineBreakProp_CR;
			continue;
		}

		// handle complex scripts in a separate function
		if (cls == LineBreakProp_SA || prop_arr [ich] == LineBreakProp_SA)
		{
			const size_t		len_cplx = len - (ich-1);
			if (len_cplx > 0)
			{
				ich += find_complex_line_breaks (
					&action_arr [ich-1],
					&prop_arr [ich-1],
					cls,
					len_cplx
				);
				// At this point, we could have ich > len if no linebreak is found.
			}
			if (ich < len)
			{
				cls = prop_arr [ich];
			}
			continue;
		}

		assert (cls < LineBreakProp_NBR_ELT_TABLE);
		assert (prop_arr [ich] < LineBreakProp_NBR_ELT_TABLE);

		// lookup pair table information in brkPairs[before, after];
		LineBreakAction	brk = _table_break_line [cls] [prop_arr [ich]];

		action_arr [ich-1] = brk;	// save break action in output array

		if (brk == LineBreakAction_INDIRECT)	// resolve indirect break
		{
			if (prop_arr [ich - 1] == LineBreakProp_SP)	// if context is A SP * B
			{
				action_arr [ich-1] = LineBreakAction_INDIRECT;	// break opportunity
			}
			else
			{
				action_arr [ich-1] = LineBreakAction_PROHIBITED;	// no break opportunity
			}
		}
		else if (brk == LineBreakAction_COMBINING_PROHIBITED)	// this is the case OP SP* CM
		{
			action_arr [ich-1] = LineBreakAction_COMBINING_PROHIBITED;	// no break allowed
			if (prop_arr [ich-1] != LineBreakProp_SP)
			{
				continue;	// apply rule 9: X CM* -> X
			}
		}
		else if (brk == LineBreakAction_COMBINING_INDIRECT)	// resolve combining mark break
		{
			action_arr [ich-1] = LineBreakAction_PROHIBITED;	// don't break before CM
			if (prop_arr [ich-1] == LineBreakProp_SP)
			{
				action_arr [ich-1] = LineBreakAction_COMBINING_INDIRECT;	// apply rule SP ?
			}
			else	// apply rule 9: X CM * -> X
			{
				continue;	// don't update cls
			}
		}

		cls = prop_arr [ich];	// save cls of current character
	}

	// Makes sure we won't write anything past the array. This could happen
	// when using the find_complex_line_breaks() function previously.
	if (ich > len)
	{
		ich = len;
	}

	// always break at the end
	action_arr [ich-1] = LineBreakAction_EXPLICIT;

	return (ich);
}



/*
==============================================================================
Name: find_word_breaks
Description:

	Computes word breaks possibilities following each character of a string.
	
	The string should have been previously processed in order to submit to the
	function an array of word break properties (each element correspounding to
	a character).

	The function returns an array of break actions, indicating if a word break
	occurs after the correspounding character.

	The end of the string is always treated as a word break.

	Algorithm from:

	Unicode Standard Annex #29: Unicode Text Segmentation
	Section 4: Word Boundaries
	http://www.unicode.org/reports/tr29/

Input parameters:
	- prop_arr: Word break properties which has been previously extracted for
		each character of the string. The function does not requires the string
		to be null-terminated.
	- len: Number of character in the string, > 0
Output parameters:
	- action_arr: Array indicating for each processed character of the string
		wether there is a word break afert it (true) or not (false). The array
		should have the same length as the input string; memory should have been
		reserved consequently. The type is UByte and not bool to allow easier
		memory management with std::vector (std::vector <bool> is generally
		specialised and is not implemented as a bool array)
Throws: Nothing
==============================================================================
*/

void	CharData::find_word_breaks (uint8_t action_arr [], const WordBreakProp prop_arr [], size_t len)
{
	assert (action_arr != nullptr);
	assert (prop_arr != nullptr);
	assert (len > 0);

	WordBreakProp	prev_2 = WordBreakProp_OTHER;
	size_t			pos = 0;
	while (pos < len - 1)
	{
		WordBreakProp	prev = prop_arr [pos    ];
		WordBreakProp	next = prop_arr [pos + 1];

		// WB3
		if (prev == WordBreakProp_CR && next == WordBreakProp_LF)
		{
			action_arr [pos] = uint8_t (false);
			++ pos;
			prev_2 = prev;
			continue;
		}

		// WB3a, WB3b
		else if (   prev == WordBreakProp_NEWLINE
		         || prev == WordBreakProp_CR
		         || prev == WordBreakProp_LF
		         || next == WordBreakProp_NEWLINE
		         || next == WordBreakProp_CR
		         || next == WordBreakProp_LF)
		{
			action_arr [pos] = uint8_t (true);
			++ pos;
			prev_2 = prev;
			continue;
		}

		// WB4
		do
		{
			next = prop_arr [pos + 1];
		   if (   next != WordBreakProp_EXTEND
			    && next != WordBreakProp_FORMAT)
			{
				break;
			}
			action_arr [pos] = uint8_t (false);
			++ pos;
		}
		while (pos < len - 1);

		// WB5 - WB14
		if (pos < len - 1)
		{
			int				prev_table = prev;
			if (prev_2 == WordBreakProp_ALETTER)
			{
				if (prev == WordBreakProp_MIDLETTER)
				{
					prev_table = WordBreakPropTablePrev_ALETTER_MIDLETTER;
				}
				else if (prev == WordBreakProp_MIDNUMLET)
				{
					prev_table = WordBreakPropTablePrev_ALETTER_MIDNUMLET;
				}
			}
			else if (prev_2 == WordBreakProp_NUMERIC)
			{
				if (prev == WordBreakProp_MIDNUM)
				{
					prev_table = WordBreakPropTablePrev_NUMERIC_MIDNUM;
				}
				else if (prev == WordBreakProp_MIDNUMLET)
				{
					prev_table = WordBreakPropTablePrev_NUMERIC_MIDNUMLET;
				}
			}

			// Looks up 2nd next character
			WordBreakProp	next_2 = WordBreakProp_OTHER;
			size_t			next_2_pos = pos + 2;
			while (next_2_pos < len)
			{
				next_2 = prop_arr [next_2_pos];
				if (   next_2 != WordBreakProp_EXTEND
				    && next_2 != WordBreakProp_FORMAT)
				{
					break;
				}
				++ next_2_pos;
			}

			int				next_table = next;
			if (next_2 == WordBreakProp_ALETTER)
			{
				if (next == WordBreakProp_MIDLETTER)
				{
					next_table = WordBreakPropTableNext_MIDLETTER_ALETTER;
				}
				else if (next == WordBreakProp_MIDNUMLET)
				{
					next_table = WordBreakPropTableNext_MIDNUMLET_ALETTER;
				}
			}
			else if (next_2 == WordBreakProp_NUMERIC)
			{
				if (next == WordBreakProp_MIDNUM)
				{
					next_table = WordBreakPropTableNext_MIDNUM_NUMERIC;
				}
				else if (next == WordBreakProp_MIDNUMLET)
				{
					next_table = WordBreakPropTableNext_MIDNUMLET_NUMERIC;
				}
			}

			action_arr [pos] =
				uint8_t (_table_break_word [prev_table] [next_table]);
		}

		++ pos;
		prev_2 = prev;
	}

	// WB2
	action_arr [len - 1] = uint8_t (true);
}



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



const CharDataInfo &	CharData::use_info (char32_t c)
{
	int				index = 0;

	if (c < MAX_NBR_CODE_POINTS)
	{
		const int		page_code = _index_page_list [c >> PAGE_SIZE_L2];
		const int		direct_mask = 1 << DIRECT_L2;

		if ((page_code & direct_mask) != 0)
		{
			const int		index_mask = (1 << DIRECT_L2) - 1;
			index = page_code & index_mask;
		}

		else
		{
			const int		page_pos = page_code << PAGE_SIZE_L2;
			const int		page_mask = (1 << PAGE_SIZE_L2) - 1;
			const int		pos_in_page = c & page_mask;
			index = _index_info_list [page_pos + pos_in_page];
		}
	}

	return (_info_unique_list [index]);
}



size_t	CharData::find_complex_line_breaks_default (LineBreakAction action_arr [], const LineBreakProp prop_arr [], LineBreakProp cls, size_t len)
{
	fstb::unused (cls);
	assert (action_arr != nullptr);
	assert (prop_arr != nullptr);
	assert (len > 0);

	size_t			ich;
	for (ich = 1; ich < len; ++ich)
	{
		action_arr [ich-1] = LineBreakAction_PROHIBITED; // by default: no break
		
		//if (prop_arr [ich] != LineBreakProp_SA)
		
		//#* RDI changed here : works better that way
		if (prop_arr [ich-1] != LineBreakProp_SA)
		{
			break;
		}
	}

	return (ich);
}



#if defined (oo) || defined (SS) || defined (cc) || defined (CC) || defined (XX)
	#error We should find other symbols for the table!
#endif

#define	oo	LineBreakAction_DIRECT
#define	SS	LineBreakAction_INDIRECT
#define	cc	LineBreakAction_COMBINING_INDIRECT
#define	CC	LineBreakAction_COMBINING_PROHIBITED
#define	XX	LineBreakAction_PROHIBITED

const LineBreakAction	CharData::_table_break_line [LineBreakProp_NBR_ELT_TABLE] [LineBreakProp_NBR_ELT_TABLE] =
{
	{ XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, XX, CC, XX, XX, XX, XX, XX, XX },
	{ oo, XX, SS, SS, XX, XX, XX, XX, SS, SS, oo, oo, oo, oo, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ XX, XX, SS, SS, SS, XX, XX, XX, SS, SS, SS, SS, SS, SS, SS, SS, SS, SS, XX, cc, XX, SS, SS, SS, SS, SS },
	{ SS, XX, SS, SS, SS, XX, XX, XX, SS, SS, SS, SS, SS, SS, SS, SS, SS, SS, XX, cc, XX, SS, SS, SS, SS, SS },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, oo, oo, oo, oo, oo, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, oo, oo, oo, oo, oo, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, oo, SS, oo, oo, oo, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, oo, SS, SS, oo, oo, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ SS, XX, SS, SS, SS, XX, XX, XX, oo, oo, SS, SS, SS, oo, SS, SS, oo, oo, XX, cc, XX, SS, SS, SS, SS, SS },
	{ SS, XX, SS, SS, SS, XX, XX, XX, oo, oo, SS, SS, oo, oo, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, XX, SS, SS, SS, XX, XX, XX, SS, SS, SS, SS, oo, SS, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, oo, SS, SS, oo, SS, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, SS, oo, oo, oo, SS, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, oo, oo, oo, oo, SS, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, XX, SS, oo, SS, XX, XX, XX, oo, oo, SS, oo, oo, oo, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, XX, SS, oo, SS, XX, XX, XX, oo, oo, oo, oo, oo, oo, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ SS, XX, SS, SS, SS, XX, XX, XX, SS, SS, SS, SS, SS, SS, SS, SS, SS, SS, XX, cc, XX, SS, SS, SS, SS, SS },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, oo, oo, oo, oo, oo, SS, SS, oo, XX, XX, cc, XX, oo, oo, oo, oo, oo },
	{ oo, oo, oo, oo, oo, oo, oo, oo, oo, oo, oo, oo, oo, oo, oo, oo, oo, oo, XX, oo, oo, oo, oo, oo, oo, oo },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, oo, SS, SS, oo, SS, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, oo },
	{ SS, XX, SS, SS, SS, XX, XX, XX, SS, SS, SS, SS, SS, SS, SS, SS, SS, SS, XX, cc, XX, SS, SS, SS, SS, SS },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, SS, oo, oo, oo, SS, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, SS, SS },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, SS, oo, oo, oo, SS, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, SS },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, SS, oo, oo, oo, SS, SS, SS, oo, oo, XX, cc, XX, SS, SS, SS, SS, oo },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, SS, oo, oo, oo, SS, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, SS, SS },
	{ oo, XX, SS, SS, SS, XX, XX, XX, oo, SS, oo, oo, oo, SS, SS, SS, oo, oo, XX, cc, XX, oo, oo, oo, oo, SS }
};

#undef	oo
#undef	SS
#undef	cc
#undef	CC
#undef	XX



// Begin: automatically generated data



const CharDataInfo	CharData::_info_unique_list [] = 
{
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_NOT_ASSIGNED       , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_XX, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_CONTROL            , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_OTHER       , BidiClass_BN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_CONTROL            , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_BA, WordBreakProp_OTHER       , BidiClass_S   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_CONTROL            , 0, 0, GraphemeBreakProp_LF     , LineBreakProp_LF, WordBreakProp_LF          , BidiClass_B   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_CONTROL            , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_BK, WordBreakProp_NEWLINE     , BidiClass_S   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_CONTROL            , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_BK, WordBreakProp_NEWLINE     , BidiClass_WS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_CONTROL            , 0, 0, GraphemeBreakProp_CR     , LineBreakProp_CR, WordBreakProp_CR          , BidiClass_B   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_CONTROL            , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_OTHER       , BidiClass_B   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_CONTROL            , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_OTHER       , BidiClass_S   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SEPARATOR_SPACE          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_SP, WordBreakProp_OTHER       , BidiClass_WS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_EX, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_QU, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_ET  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_CURRENCY          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PR, WordBreakProp_OTHER       , BidiClass_ET  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PO, WordBreakProp_OTHER       , BidiClass_ET  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_QU, WordBreakProp_MIDNUMLET   , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OPEN         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_OP, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_CLOSE        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_CL, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PR, WordBreakProp_OTHER       , BidiClass_ES  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IS, WordBreakProp_MIDNUM      , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_HY, WordBreakProp_OTHER       , BidiClass_ES  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IS, WordBreakProp_MIDNUMLET   , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_SY, WordBreakProp_OTHER       , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 1, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 2, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 3, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 4, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 5, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 6, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 7, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 8, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 9, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IS, WordBreakProp_MIDLETTER   , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IS, WordBreakProp_MIDNUM      , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000061, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000062, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000063, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000064, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000065, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000066, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000067, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000068, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000069, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00006A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00006B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00006C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00006D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00006E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00006F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000070, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000071, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000072, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000073, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000074, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000075, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000076, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000077, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000078, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000079, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00007A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PR, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_CONNECTOR    , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_EXTENDNUMLET, BidiClass_ON  },
	{ 0x000041, 0x000000, 0x000041, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000042, 0x000000, 0x000042, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000043, 0x000000, 0x000043, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000044, 0x000000, 0x000044, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000045, 0x000000, 0x000045, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000046, 0x000000, 0x000046, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000047, 0x000000, 0x000047, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000048, 0x000000, 0x000048, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000049, 0x000000, 0x000049, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00004A, 0x000000, 0x00004A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00004B, 0x000000, 0x00004B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00004C, 0x000000, 0x00004C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00004D, 0x000000, 0x00004D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00004E, 0x000000, 0x00004E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00004F, 0x000000, 0x00004F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000050, 0x000000, 0x000050, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000051, 0x000000, 0x000051, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000052, 0x000000, 0x000052, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000053, 0x000000, 0x000053, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000054, 0x000000, 0x000054, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000055, 0x000000, 0x000055, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000056, 0x000000, 0x000056, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000057, 0x000000, 0x000057, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000058, 0x000000, 0x000058, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000059, 0x000000, 0x000059, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00005A, 0x000000, 0x00005A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BA, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_CONTROL            , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_NL, WordBreakProp_NEWLINE     , BidiClass_B   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SEPARATOR_SPACE          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_GL, WordBreakProp_OTHER       , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_OP, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_CURRENCY          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PO, WordBreakProp_OTHER       , BidiClass_ET  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_INITIAL_QUOTE, 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_QU, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_BA, WordBreakProp_FORMAT      , BidiClass_BN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PO, WordBreakProp_OTHER       , BidiClass_ET  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PR, WordBreakProp_OTHER       , BidiClass_ET  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BB, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x00039C, 0x000000, 0x00039C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_MIDLETTER   , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_FINAL_QUOTE  , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_QU, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x0000E0, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000E1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000E2, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000E3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000E4, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000E5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000E6, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000E7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000E8, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000E9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000EA, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000EB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000EC, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000ED, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000EE, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000EF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000F0, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000F1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000F2, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000F3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000F4, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000F5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000F6, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x0000F8, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000F9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000FA, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000FB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000FC, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000FD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000FE, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C0, 0x000000, 0x0000C0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C1, 0x000000, 0x0000C1, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C2, 0x000000, 0x0000C2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C3, 0x000000, 0x0000C3, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C4, 0x000000, 0x0000C4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C5, 0x000000, 0x0000C5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C6, 0x000000, 0x0000C6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C7, 0x000000, 0x0000C7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C8, 0x000000, 0x0000C8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000C9, 0x000000, 0x0000C9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000CA, 0x000000, 0x0000CA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000CB, 0x000000, 0x0000CB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000CC, 0x000000, 0x0000CC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000CD, 0x000000, 0x0000CD, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000CE, 0x000000, 0x0000CE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000CF, 0x000000, 0x0000CF, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000D0, 0x000000, 0x0000D0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000D1, 0x000000, 0x0000D1, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000D2, 0x000000, 0x0000D2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000D3, 0x000000, 0x0000D3, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000D4, 0x000000, 0x0000D4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000D5, 0x000000, 0x0000D5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000D6, 0x000000, 0x0000D6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000D8, 0x000000, 0x0000D8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000D9, 0x000000, 0x0000D9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000DA, 0x000000, 0x0000DA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000DB, 0x000000, 0x0000DB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000DC, 0x000000, 0x0000DC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000DD, 0x000000, 0x0000DD, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0000DE, 0x000000, 0x0000DE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000178, 0x000000, 0x000178, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000101, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000100, 0x000000, 0x000100, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000103, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000102, 0x000000, 0x000102, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000105, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000104, 0x000000, 0x000104, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000107, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000106, 0x000000, 0x000106, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000109, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000108, 0x000000, 0x000108, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00010B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00010A, 0x000000, 0x00010A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00010D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00010C, 0x000000, 0x00010C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00010F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00010E, 0x000000, 0x00010E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000111, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000110, 0x000000, 0x000110, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000113, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000112, 0x000000, 0x000112, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000115, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000114, 0x000000, 0x000114, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000117, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000116, 0x000000, 0x000116, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000119, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000118, 0x000000, 0x000118, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00011B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00011A, 0x000000, 0x00011A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00011D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00011C, 0x000000, 0x00011C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00011F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00011E, 0x000000, 0x00011E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000121, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000120, 0x000000, 0x000120, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000123, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000122, 0x000000, 0x000122, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000125, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000124, 0x000000, 0x000124, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000127, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000126, 0x000000, 0x000126, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000129, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000128, 0x000000, 0x000128, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00012B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00012A, 0x000000, 0x00012A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00012D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00012C, 0x000000, 0x00012C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00012F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00012E, 0x000000, 0x00012E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000133, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000132, 0x000000, 0x000132, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000135, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000134, 0x000000, 0x000134, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000137, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000136, 0x000000, 0x000136, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00013A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000139, 0x000000, 0x000139, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00013C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00013B, 0x000000, 0x00013B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00013E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00013D, 0x000000, 0x00013D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000140, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00013F, 0x000000, 0x00013F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000142, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000141, 0x000000, 0x000141, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000144, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000143, 0x000000, 0x000143, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000146, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000145, 0x000000, 0x000145, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000148, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000147, 0x000000, 0x000147, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00014B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00014A, 0x000000, 0x00014A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00014D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00014C, 0x000000, 0x00014C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00014F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00014E, 0x000000, 0x00014E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000151, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000150, 0x000000, 0x000150, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000153, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000152, 0x000000, 0x000152, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000155, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000154, 0x000000, 0x000154, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000157, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000156, 0x000000, 0x000156, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000159, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000158, 0x000000, 0x000158, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00015B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00015A, 0x000000, 0x00015A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00015D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00015C, 0x000000, 0x00015C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00015F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00015E, 0x000000, 0x00015E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000161, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000160, 0x000000, 0x000160, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000163, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000162, 0x000000, 0x000162, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000165, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000164, 0x000000, 0x000164, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000167, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000166, 0x000000, 0x000166, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000169, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000168, 0x000000, 0x000168, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00016B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00016A, 0x000000, 0x00016A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00016D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00016C, 0x000000, 0x00016C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00016F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00016E, 0x000000, 0x00016E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000171, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000170, 0x000000, 0x000170, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000173, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000172, 0x000000, 0x000172, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000175, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000174, 0x000000, 0x000174, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000177, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000176, 0x000000, 0x000176, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000FF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00017A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000179, 0x000000, 0x000179, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00017C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00017B, 0x000000, 0x00017B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00017E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00017D, 0x000000, 0x00017D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000243, 0x000000, 0x000243, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000253, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000183, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000182, 0x000000, 0x000182, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000185, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000184, 0x000000, 0x000184, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000254, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000188, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000187, 0x000000, 0x000187, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000256, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000257, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00018C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00018B, 0x000000, 0x00018B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001DD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000259, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00025B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000192, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000191, 0x000000, 0x000191, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000260, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000263, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001F6, 0x000000, 0x0001F6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000269, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000268, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000199, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000198, 0x000000, 0x000198, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00023D, 0x000000, 0x00023D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00026F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000272, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000220, 0x000000, 0x000220, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000275, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001A1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001A0, 0x000000, 0x0001A0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001A3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001A2, 0x000000, 0x0001A2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001A5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001A4, 0x000000, 0x0001A4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000280, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001A8, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001A7, 0x000000, 0x0001A7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000283, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001AD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001AC, 0x000000, 0x0001AC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000288, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001B0, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001AF, 0x000000, 0x0001AF, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00028A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00028B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001B4, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001B3, 0x000000, 0x0001B3, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001B6, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001B5, 0x000000, 0x0001B5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000292, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001B9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001B8, 0x000000, 0x0001B8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001BD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001BC, 0x000000, 0x0001BC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001F7, 0x000000, 0x0001F7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001C6, 0x0001C5, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001C4, 0x0001C6, 0x0001C5, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001C4, 0x000000, 0x0001C5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001C9, 0x0001C8, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001C7, 0x0001C9, 0x0001C8, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001C7, 0x000000, 0x0001C8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001CC, 0x0001CB, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001CA, 0x0001CC, 0x0001CB, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001CA, 0x000000, 0x0001CB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001CE, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001CD, 0x000000, 0x0001CD, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001D0, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001CF, 0x000000, 0x0001CF, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001D2, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001D1, 0x000000, 0x0001D1, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001D4, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001D3, 0x000000, 0x0001D3, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001D6, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001D5, 0x000000, 0x0001D5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001D8, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001D7, 0x000000, 0x0001D7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001DA, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001D9, 0x000000, 0x0001D9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001DC, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001DB, 0x000000, 0x0001DB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00018E, 0x000000, 0x00018E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001DF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001DE, 0x000000, 0x0001DE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001E1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001E0, 0x000000, 0x0001E0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001E3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001E2, 0x000000, 0x0001E2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001E5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001E4, 0x000000, 0x0001E4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001E7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001E6, 0x000000, 0x0001E6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001E9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001E8, 0x000000, 0x0001E8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001EB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001EA, 0x000000, 0x0001EA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001ED, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001EC, 0x000000, 0x0001EC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001EF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001EE, 0x000000, 0x0001EE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001F3, 0x0001F2, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001F1, 0x0001F3, 0x0001F2, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001F1, 0x000000, 0x0001F2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001F5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001F4, 0x000000, 0x0001F4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000195, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001BF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001F9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001F8, 0x000000, 0x0001F8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001FB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001FA, 0x000000, 0x0001FA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001FD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001FC, 0x000000, 0x0001FC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0001FF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001FE, 0x000000, 0x0001FE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000201, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000200, 0x000000, 0x000200, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000203, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000202, 0x000000, 0x000202, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000205, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000204, 0x000000, 0x000204, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000207, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000206, 0x000000, 0x000206, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000209, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000208, 0x000000, 0x000208, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00020B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00020A, 0x000000, 0x00020A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00020D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00020C, 0x000000, 0x00020C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00020F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00020E, 0x000000, 0x00020E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000211, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000210, 0x000000, 0x000210, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000213, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000212, 0x000000, 0x000212, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000215, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000214, 0x000000, 0x000214, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000217, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000216, 0x000000, 0x000216, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000219, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000218, 0x000000, 0x000218, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00021B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00021A, 0x000000, 0x00021A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00021D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00021C, 0x000000, 0x00021C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00021F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00021E, 0x000000, 0x00021E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00019E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000223, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000222, 0x000000, 0x000222, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000225, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000224, 0x000000, 0x000224, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000227, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000226, 0x000000, 0x000226, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000229, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000228, 0x000000, 0x000228, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00022B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00022A, 0x000000, 0x00022A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00022D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00022C, 0x000000, 0x00022C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00022F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00022E, 0x000000, 0x00022E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000231, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000230, 0x000000, 0x000230, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000233, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000232, 0x000000, 0x000232, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C65, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00023C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00023B, 0x000000, 0x00023B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00019A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C66, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000242, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000241, 0x000000, 0x000241, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000180, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000289, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00028C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000247, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000246, 0x000000, 0x000246, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000249, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000248, 0x000000, 0x000248, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00024B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00024A, 0x000000, 0x00024A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00024D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00024C, 0x000000, 0x00024C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00024F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00024E, 0x000000, 0x00024E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C6F, 0x000000, 0x002C6F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C6D, 0x000000, 0x002C6D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000181, 0x000000, 0x000181, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000186, 0x000000, 0x000186, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000189, 0x000000, 0x000189, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00018A, 0x000000, 0x00018A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00018F, 0x000000, 0x00018F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000190, 0x000000, 0x000190, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000193, 0x000000, 0x000193, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000194, 0x000000, 0x000194, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000197, 0x000000, 0x000197, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000196, 0x000000, 0x000196, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C62, 0x000000, 0x002C62, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00019C, 0x000000, 0x00019C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C6E, 0x000000, 0x002C6E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00019D, 0x000000, 0x00019D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00019F, 0x000000, 0x00019F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C64, 0x000000, 0x002C64, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001A6, 0x000000, 0x0001A6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001A9, 0x000000, 0x0001A9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001AE, 0x000000, 0x0001AE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000244, 0x000000, 0x000244, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001B1, 0x000000, 0x0001B1, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001B2, 0x000000, 0x0001B2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000245, 0x000000, 0x000245, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0001B7, 0x000000, 0x0001B7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BB, WordBreakProp_ALETTER     , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_MARK_NONSPACING          , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_CM, WordBreakProp_EXTEND      , BidiClass_NSM },
	{ 0x000399, 0x000000, 0x000399, 0, Categ_MARK_NONSPACING          , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_CM, WordBreakProp_EXTEND      , BidiClass_NSM },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_MARK_NONSPACING          , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_GL, WordBreakProp_EXTEND      , BidiClass_NSM },
	{ 0x000000, 0x000371, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000370, 0x000000, 0x000370, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000373, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000372, 0x000000, 0x000372, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000377, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000376, 0x000000, 0x000376, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003FD, 0x000000, 0x0003FD, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003FE, 0x000000, 0x0003FE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003FF, 0x000000, 0x0003FF, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003AC, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_MIDLETTER   , BidiClass_ON  },
	{ 0x000000, 0x0003AD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003AE, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003AF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003CC, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003CD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003CE, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003B1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003B2, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003B3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003B4, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003B5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003B6, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003B7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003B8, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003B9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003BA, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003BB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003BC, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003BD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003BE, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003BF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003C0, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003C1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003C3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003C4, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003C5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003C6, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003C7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003C8, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003C9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003CA, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003CB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000386, 0x000000, 0x000386, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000388, 0x000000, 0x000388, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000389, 0x000000, 0x000389, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00038A, 0x000000, 0x00038A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000391, 0x000000, 0x000391, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000392, 0x000000, 0x000392, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000393, 0x000000, 0x000393, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000394, 0x000000, 0x000394, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000395, 0x000000, 0x000395, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000396, 0x000000, 0x000396, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000397, 0x000000, 0x000397, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000398, 0x000000, 0x000398, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000399, 0x000000, 0x000399, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00039A, 0x000000, 0x00039A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00039B, 0x000000, 0x00039B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00039D, 0x000000, 0x00039D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00039E, 0x000000, 0x00039E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00039F, 0x000000, 0x00039F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003A0, 0x000000, 0x0003A0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003A1, 0x000000, 0x0003A1, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003A3, 0x000000, 0x0003A3, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003A4, 0x000000, 0x0003A4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003A5, 0x000000, 0x0003A5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003A6, 0x000000, 0x0003A6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003A7, 0x000000, 0x0003A7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003A8, 0x000000, 0x0003A8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003A9, 0x000000, 0x0003A9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003AA, 0x000000, 0x0003AA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003AB, 0x000000, 0x0003AB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00038C, 0x000000, 0x00038C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00038E, 0x000000, 0x00038E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00038F, 0x000000, 0x00038F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003D7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003CF, 0x000000, 0x0003CF, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003D9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003D8, 0x000000, 0x0003D8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003DB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003DA, 0x000000, 0x0003DA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003DD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003DC, 0x000000, 0x0003DC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003DF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003DE, 0x000000, 0x0003DE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003E1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003E0, 0x000000, 0x0003E0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003E3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003E2, 0x000000, 0x0003E2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003E5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003E4, 0x000000, 0x0003E4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003E7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003E6, 0x000000, 0x0003E6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003E9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003E8, 0x000000, 0x0003E8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003EB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003EA, 0x000000, 0x0003EA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003ED, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003EC, 0x000000, 0x0003EC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003EF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003EE, 0x000000, 0x0003EE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003F9, 0x000000, 0x0003F9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003F8, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003F7, 0x000000, 0x0003F7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003F2, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0003FB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0003FA, 0x000000, 0x0003FA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00037B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00037C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00037D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000450, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000451, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000452, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000453, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000454, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000455, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000456, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000457, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000458, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000459, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00045A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00045B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00045C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00045D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00045E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00045F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000430, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000431, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000432, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000433, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000434, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000435, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000436, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000437, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000438, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000439, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00043A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00043B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00043C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00043D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00043E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00043F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000440, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000441, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000442, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000443, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000444, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000445, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000446, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000447, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000448, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000449, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00044A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00044B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00044C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00044D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00044E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00044F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000410, 0x000000, 0x000410, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000411, 0x000000, 0x000411, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000412, 0x000000, 0x000412, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000413, 0x000000, 0x000413, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000414, 0x000000, 0x000414, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000415, 0x000000, 0x000415, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000416, 0x000000, 0x000416, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000417, 0x000000, 0x000417, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000418, 0x000000, 0x000418, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000419, 0x000000, 0x000419, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00041A, 0x000000, 0x00041A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00041B, 0x000000, 0x00041B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00041C, 0x000000, 0x00041C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00041D, 0x000000, 0x00041D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00041E, 0x000000, 0x00041E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00041F, 0x000000, 0x00041F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000420, 0x000000, 0x000420, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000421, 0x000000, 0x000421, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000422, 0x000000, 0x000422, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000423, 0x000000, 0x000423, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000424, 0x000000, 0x000424, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000425, 0x000000, 0x000425, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000426, 0x000000, 0x000426, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000427, 0x000000, 0x000427, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000428, 0x000000, 0x000428, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000429, 0x000000, 0x000429, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00042A, 0x000000, 0x00042A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00042B, 0x000000, 0x00042B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00042C, 0x000000, 0x00042C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00042D, 0x000000, 0x00042D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00042E, 0x000000, 0x00042E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00042F, 0x000000, 0x00042F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000400, 0x000000, 0x000400, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000401, 0x000000, 0x000401, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000402, 0x000000, 0x000402, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000403, 0x000000, 0x000403, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000404, 0x000000, 0x000404, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000405, 0x000000, 0x000405, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000406, 0x000000, 0x000406, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000407, 0x000000, 0x000407, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000408, 0x000000, 0x000408, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000409, 0x000000, 0x000409, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00040A, 0x000000, 0x00040A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00040B, 0x000000, 0x00040B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00040C, 0x000000, 0x00040C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00040D, 0x000000, 0x00040D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00040E, 0x000000, 0x00040E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00040F, 0x000000, 0x00040F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000461, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000460, 0x000000, 0x000460, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000463, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000462, 0x000000, 0x000462, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000465, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000464, 0x000000, 0x000464, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000467, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000466, 0x000000, 0x000466, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000469, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000468, 0x000000, 0x000468, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00046B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00046A, 0x000000, 0x00046A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00046D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00046C, 0x000000, 0x00046C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00046F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00046E, 0x000000, 0x00046E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000471, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000470, 0x000000, 0x000470, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000473, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000472, 0x000000, 0x000472, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000475, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000474, 0x000000, 0x000474, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000477, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000476, 0x000000, 0x000476, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000479, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000478, 0x000000, 0x000478, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00047B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00047A, 0x000000, 0x00047A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00047D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00047C, 0x000000, 0x00047C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00047F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00047E, 0x000000, 0x00047E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000481, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000480, 0x000000, 0x000480, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_MARK_ENCLOSING           , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_CM, WordBreakProp_EXTEND      , BidiClass_NSM },
	{ 0x000000, 0x00048B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00048A, 0x000000, 0x00048A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00048D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00048C, 0x000000, 0x00048C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00048F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00048E, 0x000000, 0x00048E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000491, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000490, 0x000000, 0x000490, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000493, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000492, 0x000000, 0x000492, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000495, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000494, 0x000000, 0x000494, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000497, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000496, 0x000000, 0x000496, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000499, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000498, 0x000000, 0x000498, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00049B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00049A, 0x000000, 0x00049A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00049D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00049C, 0x000000, 0x00049C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00049F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00049E, 0x000000, 0x00049E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004A1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004A0, 0x000000, 0x0004A0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004A3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004A2, 0x000000, 0x0004A2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004A5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004A4, 0x000000, 0x0004A4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004A7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004A6, 0x000000, 0x0004A6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004A9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004A8, 0x000000, 0x0004A8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004AB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004AA, 0x000000, 0x0004AA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004AD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004AC, 0x000000, 0x0004AC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004AF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004AE, 0x000000, 0x0004AE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004B1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004B0, 0x000000, 0x0004B0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004B3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004B2, 0x000000, 0x0004B2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004B5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004B4, 0x000000, 0x0004B4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004B7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004B6, 0x000000, 0x0004B6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004B9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004B8, 0x000000, 0x0004B8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004BB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004BA, 0x000000, 0x0004BA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004BD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004BC, 0x000000, 0x0004BC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004BF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004BE, 0x000000, 0x0004BE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004CF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004C2, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004C1, 0x000000, 0x0004C1, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004C4, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004C3, 0x000000, 0x0004C3, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004C6, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004C5, 0x000000, 0x0004C5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004C8, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004C7, 0x000000, 0x0004C7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004CA, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004C9, 0x000000, 0x0004C9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004CC, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004CB, 0x000000, 0x0004CB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004CE, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004CD, 0x000000, 0x0004CD, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004C0, 0x000000, 0x0004C0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004D1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004D0, 0x000000, 0x0004D0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004D3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004D2, 0x000000, 0x0004D2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004D5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004D4, 0x000000, 0x0004D4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004D7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004D6, 0x000000, 0x0004D6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004D9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004D8, 0x000000, 0x0004D8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004DB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004DA, 0x000000, 0x0004DA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004DD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004DC, 0x000000, 0x0004DC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004DF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004DE, 0x000000, 0x0004DE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004E1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004E0, 0x000000, 0x0004E0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004E3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004E2, 0x000000, 0x0004E2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004E5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004E4, 0x000000, 0x0004E4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004E7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004E6, 0x000000, 0x0004E6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004E9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004E8, 0x000000, 0x0004E8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004EB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004EA, 0x000000, 0x0004EA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004ED, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004EC, 0x000000, 0x0004EC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004EF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004EE, 0x000000, 0x0004EE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004F1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004F0, 0x000000, 0x0004F0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004F3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004F2, 0x000000, 0x0004F2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004F5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004F4, 0x000000, 0x0004F4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004F7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004F6, 0x000000, 0x0004F6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004F9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004F8, 0x000000, 0x0004F8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004FB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004FA, 0x000000, 0x0004FA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004FD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004FC, 0x000000, 0x0004FC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0004FF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0004FE, 0x000000, 0x0004FE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000501, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000500, 0x000000, 0x000500, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000503, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000502, 0x000000, 0x000502, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000505, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000504, 0x000000, 0x000504, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000507, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000506, 0x000000, 0x000506, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000509, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000508, 0x000000, 0x000508, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00050B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00050A, 0x000000, 0x00050A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00050D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00050C, 0x000000, 0x00050C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00050F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00050E, 0x000000, 0x00050E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000511, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000510, 0x000000, 0x000510, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000513, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000512, 0x000000, 0x000512, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000515, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000514, 0x000000, 0x000514, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000517, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000516, 0x000000, 0x000516, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000519, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000518, 0x000000, 0x000518, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00051B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00051A, 0x000000, 0x00051A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00051D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00051C, 0x000000, 0x00051C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00051F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00051E, 0x000000, 0x00051E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000521, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000520, 0x000000, 0x000520, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000523, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000522, 0x000000, 0x000522, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000561, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000562, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000563, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000564, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000565, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000566, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000567, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000568, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000569, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00056A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00056B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00056C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00056D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00056E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00056F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000570, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000571, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000572, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000573, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000574, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000575, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000576, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000577, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000578, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000579, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00057A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00057B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00057C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00057D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00057E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00057F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000580, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000581, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000582, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000583, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000584, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000585, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000586, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000531, 0x000000, 0x000531, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000532, 0x000000, 0x000532, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000533, 0x000000, 0x000533, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000534, 0x000000, 0x000534, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000535, 0x000000, 0x000535, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000536, 0x000000, 0x000536, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000537, 0x000000, 0x000537, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000538, 0x000000, 0x000538, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000539, 0x000000, 0x000539, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00053A, 0x000000, 0x00053A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00053B, 0x000000, 0x00053B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00053C, 0x000000, 0x00053C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00053D, 0x000000, 0x00053D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00053E, 0x000000, 0x00053E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00053F, 0x000000, 0x00053F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000540, 0x000000, 0x000540, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000541, 0x000000, 0x000541, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000542, 0x000000, 0x000542, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000543, 0x000000, 0x000543, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000544, 0x000000, 0x000544, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000545, 0x000000, 0x000545, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000546, 0x000000, 0x000546, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000547, 0x000000, 0x000547, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000548, 0x000000, 0x000548, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000549, 0x000000, 0x000549, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00054A, 0x000000, 0x00054A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00054B, 0x000000, 0x00054B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00054C, 0x000000, 0x00054C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00054D, 0x000000, 0x00054D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00054E, 0x000000, 0x00054E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00054F, 0x000000, 0x00054F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000550, 0x000000, 0x000550, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000551, 0x000000, 0x000551, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000552, 0x000000, 0x000552, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000553, 0x000000, 0x000553, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000554, 0x000000, 0x000554, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000555, 0x000000, 0x000555, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000556, 0x000000, 0x000556, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IS, WordBreakProp_MIDNUM      , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BA, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BA, WordBreakProp_OTHER       , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_EX, WordBreakProp_OTHER       , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_MIDLETTER   , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_AL, WordBreakProp_FORMAT      , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_AL  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_CURRENCY          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PO, WordBreakProp_OTHER       , BidiClass_AL  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IS, WordBreakProp_MIDNUM      , BidiClass_AL  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_EX, WordBreakProp_OTHER       , BidiClass_AL  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_AL  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_AL  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 1, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 2, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 3, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 4, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 5, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 6, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 7, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 8, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 9, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_MIDNUM      , BidiClass_AN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_AL  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_AL  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_AL, WordBreakProp_FORMAT      , BidiClass_BN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 1, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 2, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 3, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 4, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 5, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 6, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 7, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 8, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 9, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_MARK_SPACING_COMBINING   , 0, 0, GraphemeBreakProp_SP_MARK, LineBreakProp_CM, WordBreakProp_EXTEND      , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BA, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 1, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 2, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 3, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 4, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 5, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 6, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 7, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 8, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 9, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NU, WordBreakProp_NUMERIC     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_MARK_SPACING_COMBINING   , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_CM, WordBreakProp_EXTEND      , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_MARK_NONSPACING          , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_CM, WordBreakProp_EXTEND      , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PO, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_SA, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_SA, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_MARK_NONSPACING          , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_SA, WordBreakProp_EXTEND      , BidiClass_NSM },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_PREPEND, LineBreakProp_SA, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_SA, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BB, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BB, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_GL, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_EX, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_EX, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BA, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_MARK_SPACING_COMBINING   , 0, 0, GraphemeBreakProp_SP_MARK, LineBreakProp_BA, WordBreakProp_EXTEND      , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_MARK_SPACING_COMBINING   , 0, 0, GraphemeBreakProp_SP_MARK, LineBreakProp_SA, WordBreakProp_EXTEND      , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_SA, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x002D00, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D01, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D02, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D03, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D04, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D05, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D06, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D07, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D08, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D09, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D0A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D0B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D0C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D0D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D0E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D0F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D10, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D11, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D12, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D13, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D14, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D15, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D16, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D17, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D18, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D19, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D1A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D1B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D1C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D1D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D1E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D1F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D20, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D21, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D22, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D23, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D24, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002D25, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_L      , LineBreakProp_JL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_V      , LineBreakProp_JV, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_T      , LineBreakProp_JT, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SEPARATOR_SPACE          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BA, WordBreakProp_OTHER       , BidiClass_WS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_SA, WordBreakProp_FORMAT      , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BA, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BB, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SEPARATOR_SPACE          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_GL, WordBreakProp_OTHER       , BidiClass_WS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_SA, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x00A77D, 0x000000, 0x00A77D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C63, 0x000000, 0x002C63, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E01, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E00, 0x000000, 0x001E00, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E03, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E02, 0x000000, 0x001E02, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E05, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E04, 0x000000, 0x001E04, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E07, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E06, 0x000000, 0x001E06, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E09, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E08, 0x000000, 0x001E08, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E0B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E0A, 0x000000, 0x001E0A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E0D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E0C, 0x000000, 0x001E0C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E0F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E0E, 0x000000, 0x001E0E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E11, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E10, 0x000000, 0x001E10, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E13, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E12, 0x000000, 0x001E12, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E15, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E14, 0x000000, 0x001E14, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E17, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E16, 0x000000, 0x001E16, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E19, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E18, 0x000000, 0x001E18, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E1B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E1A, 0x000000, 0x001E1A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E1D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E1C, 0x000000, 0x001E1C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E1F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E1E, 0x000000, 0x001E1E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E21, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E20, 0x000000, 0x001E20, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E23, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E22, 0x000000, 0x001E22, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E25, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E24, 0x000000, 0x001E24, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E27, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E26, 0x000000, 0x001E26, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E29, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E28, 0x000000, 0x001E28, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E2B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E2A, 0x000000, 0x001E2A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E2D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E2C, 0x000000, 0x001E2C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E2F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E2E, 0x000000, 0x001E2E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E31, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E30, 0x000000, 0x001E30, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E33, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E32, 0x000000, 0x001E32, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E35, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E34, 0x000000, 0x001E34, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E37, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E36, 0x000000, 0x001E36, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E39, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E38, 0x000000, 0x001E38, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E3B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E3A, 0x000000, 0x001E3A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E3D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E3C, 0x000000, 0x001E3C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E3F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E3E, 0x000000, 0x001E3E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E41, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E40, 0x000000, 0x001E40, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E43, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E42, 0x000000, 0x001E42, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E45, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E44, 0x000000, 0x001E44, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E47, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E46, 0x000000, 0x001E46, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E49, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E48, 0x000000, 0x001E48, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E4B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E4A, 0x000000, 0x001E4A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E4D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E4C, 0x000000, 0x001E4C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E4F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E4E, 0x000000, 0x001E4E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E51, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E50, 0x000000, 0x001E50, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E53, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E52, 0x000000, 0x001E52, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E55, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E54, 0x000000, 0x001E54, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E57, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E56, 0x000000, 0x001E56, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E59, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E58, 0x000000, 0x001E58, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E5B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E5A, 0x000000, 0x001E5A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E5D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E5C, 0x000000, 0x001E5C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E5F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E5E, 0x000000, 0x001E5E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E61, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E60, 0x000000, 0x001E60, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E63, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E62, 0x000000, 0x001E62, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E65, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E64, 0x000000, 0x001E64, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E67, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E66, 0x000000, 0x001E66, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E69, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E68, 0x000000, 0x001E68, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E6B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E6A, 0x000000, 0x001E6A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E6D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E6C, 0x000000, 0x001E6C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E6F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E6E, 0x000000, 0x001E6E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E71, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E70, 0x000000, 0x001E70, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E73, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E72, 0x000000, 0x001E72, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E75, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E74, 0x000000, 0x001E74, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E77, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E76, 0x000000, 0x001E76, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E79, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E78, 0x000000, 0x001E78, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E7B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E7A, 0x000000, 0x001E7A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E7D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E7C, 0x000000, 0x001E7C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E7F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E7E, 0x000000, 0x001E7E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E81, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E80, 0x000000, 0x001E80, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E83, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E82, 0x000000, 0x001E82, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E85, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E84, 0x000000, 0x001E84, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E87, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E86, 0x000000, 0x001E86, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E89, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E88, 0x000000, 0x001E88, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E8B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E8A, 0x000000, 0x001E8A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E8D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E8C, 0x000000, 0x001E8C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E8F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E8E, 0x000000, 0x001E8E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E91, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E90, 0x000000, 0x001E90, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E93, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E92, 0x000000, 0x001E92, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001E95, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001E94, 0x000000, 0x001E94, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0000DF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EA1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EA0, 0x000000, 0x001EA0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EA3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EA2, 0x000000, 0x001EA2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EA5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EA4, 0x000000, 0x001EA4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EA7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EA6, 0x000000, 0x001EA6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EA9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EA8, 0x000000, 0x001EA8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EAB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EAA, 0x000000, 0x001EAA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EAD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EAC, 0x000000, 0x001EAC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EAF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EAE, 0x000000, 0x001EAE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EB1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EB0, 0x000000, 0x001EB0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EB3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EB2, 0x000000, 0x001EB2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EB5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EB4, 0x000000, 0x001EB4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EB7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EB6, 0x000000, 0x001EB6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EB9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EB8, 0x000000, 0x001EB8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EBB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EBA, 0x000000, 0x001EBA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EBD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EBC, 0x000000, 0x001EBC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EBF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EBE, 0x000000, 0x001EBE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EC1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EC0, 0x000000, 0x001EC0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EC3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EC2, 0x000000, 0x001EC2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EC5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EC4, 0x000000, 0x001EC4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EC7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EC6, 0x000000, 0x001EC6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EC9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EC8, 0x000000, 0x001EC8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001ECB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001ECA, 0x000000, 0x001ECA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001ECD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001ECC, 0x000000, 0x001ECC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001ECF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001ECE, 0x000000, 0x001ECE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001ED1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001ED0, 0x000000, 0x001ED0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001ED3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001ED2, 0x000000, 0x001ED2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001ED5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001ED4, 0x000000, 0x001ED4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001ED7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001ED6, 0x000000, 0x001ED6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001ED9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001ED8, 0x000000, 0x001ED8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EDB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EDA, 0x000000, 0x001EDA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EDD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EDC, 0x000000, 0x001EDC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EDF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EDE, 0x000000, 0x001EDE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EE1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EE0, 0x000000, 0x001EE0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EE3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EE2, 0x000000, 0x001EE2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EE5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EE4, 0x000000, 0x001EE4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EE7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EE6, 0x000000, 0x001EE6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EE9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EE8, 0x000000, 0x001EE8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EEB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EEA, 0x000000, 0x001EEA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EED, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EEC, 0x000000, 0x001EEC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EEF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EEE, 0x000000, 0x001EEE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EF1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EF0, 0x000000, 0x001EF0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EF3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EF2, 0x000000, 0x001EF2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EF5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EF4, 0x000000, 0x001EF4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EF7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EF6, 0x000000, 0x001EF6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EF9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EF8, 0x000000, 0x001EF8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EFB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EFA, 0x000000, 0x001EFA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EFD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EFC, 0x000000, 0x001EFC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001EFF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001EFE, 0x000000, 0x001EFE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F08, 0x000000, 0x001F08, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F09, 0x000000, 0x001F09, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F0A, 0x000000, 0x001F0A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F0B, 0x000000, 0x001F0B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F0C, 0x000000, 0x001F0C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F0D, 0x000000, 0x001F0D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F0E, 0x000000, 0x001F0E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F0F, 0x000000, 0x001F0F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F00, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F01, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F02, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F03, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F04, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F05, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F06, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F07, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F18, 0x000000, 0x001F18, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F19, 0x000000, 0x001F19, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F1A, 0x000000, 0x001F1A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F1B, 0x000000, 0x001F1B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F1C, 0x000000, 0x001F1C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F1D, 0x000000, 0x001F1D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F10, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F11, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F12, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F13, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F14, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F15, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F28, 0x000000, 0x001F28, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F29, 0x000000, 0x001F29, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F2A, 0x000000, 0x001F2A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F2B, 0x000000, 0x001F2B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F2C, 0x000000, 0x001F2C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F2D, 0x000000, 0x001F2D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F2E, 0x000000, 0x001F2E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F2F, 0x000000, 0x001F2F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F20, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F21, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F22, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F23, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F24, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F25, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F26, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F27, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F38, 0x000000, 0x001F38, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F39, 0x000000, 0x001F39, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F3A, 0x000000, 0x001F3A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F3B, 0x000000, 0x001F3B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F3C, 0x000000, 0x001F3C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F3D, 0x000000, 0x001F3D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F3E, 0x000000, 0x001F3E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F3F, 0x000000, 0x001F3F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F30, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F31, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F32, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F33, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F34, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F35, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F36, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F37, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F48, 0x000000, 0x001F48, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F49, 0x000000, 0x001F49, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F4A, 0x000000, 0x001F4A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F4B, 0x000000, 0x001F4B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F4C, 0x000000, 0x001F4C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F4D, 0x000000, 0x001F4D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F40, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F41, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F42, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F43, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F44, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F45, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F59, 0x000000, 0x001F59, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F5B, 0x000000, 0x001F5B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F5D, 0x000000, 0x001F5D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F5F, 0x000000, 0x001F5F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F51, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F53, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F55, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F57, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F68, 0x000000, 0x001F68, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F69, 0x000000, 0x001F69, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F6A, 0x000000, 0x001F6A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F6B, 0x000000, 0x001F6B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F6C, 0x000000, 0x001F6C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F6D, 0x000000, 0x001F6D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F6E, 0x000000, 0x001F6E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F6F, 0x000000, 0x001F6F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F60, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F61, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F62, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F63, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F64, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F65, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F66, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F67, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FBA, 0x000000, 0x001FBA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FBB, 0x000000, 0x001FBB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FC8, 0x000000, 0x001FC8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FC9, 0x000000, 0x001FC9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FCA, 0x000000, 0x001FCA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FCB, 0x000000, 0x001FCB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FDA, 0x000000, 0x001FDA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FDB, 0x000000, 0x001FDB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FF8, 0x000000, 0x001FF8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FF9, 0x000000, 0x001FF9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FEA, 0x000000, 0x001FEA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FEB, 0x000000, 0x001FEB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FFA, 0x000000, 0x001FFA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FFB, 0x000000, 0x001FFB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F88, 0x000000, 0x001F88, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F89, 0x000000, 0x001F89, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F8A, 0x000000, 0x001F8A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F8B, 0x000000, 0x001F8B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F8C, 0x000000, 0x001F8C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F8D, 0x000000, 0x001F8D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F8E, 0x000000, 0x001F8E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F8F, 0x000000, 0x001F8F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F80, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F81, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F82, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F83, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F84, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F85, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F86, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F87, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F98, 0x000000, 0x001F98, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F99, 0x000000, 0x001F99, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F9A, 0x000000, 0x001F9A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F9B, 0x000000, 0x001F9B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F9C, 0x000000, 0x001F9C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F9D, 0x000000, 0x001F9D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F9E, 0x000000, 0x001F9E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001F9F, 0x000000, 0x001F9F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F90, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F91, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F92, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F93, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F94, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F95, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F96, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F97, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FA8, 0x000000, 0x001FA8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FA9, 0x000000, 0x001FA9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FAA, 0x000000, 0x001FAA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FAB, 0x000000, 0x001FAB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FAC, 0x000000, 0x001FAC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FAD, 0x000000, 0x001FAD, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FAE, 0x000000, 0x001FAE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FAF, 0x000000, 0x001FAF, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FA0, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FA1, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FA2, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FA3, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FA4, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FA5, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FA6, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FA7, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FB8, 0x000000, 0x001FB8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FB9, 0x000000, 0x001FB9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FBC, 0x000000, 0x001FBC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FB0, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FB1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F70, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F71, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FB3, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FCC, 0x000000, 0x001FCC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F72, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F73, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F74, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F75, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FC3, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FD8, 0x000000, 0x001FD8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FD9, 0x000000, 0x001FD9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FD0, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FD1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F76, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F77, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FE8, 0x000000, 0x001FE8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FE9, 0x000000, 0x001FE9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FEC, 0x000000, 0x001FEC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FE0, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FE1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F7A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F7B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FE5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x001FFC, 0x000000, 0x001FFC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F78, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F79, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F7C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001F7D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001FF3, 0x000000, 0, Categ_LETTER_TITLECASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_ZW, WordBreakProp_FORMAT      , BidiClass_BN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_CM, WordBreakProp_EXTEND      , BidiClass_BN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_FORMAT      , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_FORMAT      , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_GL, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_B2, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_INITIAL_QUOTE, 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_QU, WordBreakProp_MIDNUMLET   , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_FINAL_QUOTE  , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_QU, WordBreakProp_MIDNUMLET   , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IN, WordBreakProp_MIDNUMLET   , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IN, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BA, WordBreakProp_MIDLETTER   , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SEPARATOR_LINE           , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_BK, WordBreakProp_NEWLINE     , BidiClass_WS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SEPARATOR_PARAGRAPH      , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_BK, WordBreakProp_NEWLINE     , BidiClass_B   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_FORMAT      , BidiClass_LRE },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_FORMAT      , BidiClass_RLE },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_FORMAT      , BidiClass_PDF },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_FORMAT      , BidiClass_LRO },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_FORMAT      , BidiClass_RLO },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PO, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IS, WordBreakProp_MIDNUM      , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_WJ, WordBreakProp_FORMAT      , BidiClass_BN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_FORMAT      , BidiClass_BN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_ES  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PO, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_PR, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x0000E5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_ET  },
	{ 0x000000, 0x00214E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002132, 0x000000, 0x002132, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002170, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002171, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002172, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002173, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002174, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002175, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002176, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002177, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002178, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002179, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00217A, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00217B, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00217C, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00217D, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00217E, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00217F, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002160, 0x000000, 0x002160, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002161, 0x000000, 0x002161, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002162, 0x000000, 0x002162, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002163, 0x000000, 0x002163, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002164, 0x000000, 0x002164, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002165, 0x000000, 0x002165, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002166, 0x000000, 0x002166, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002167, 0x000000, 0x002167, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002168, 0x000000, 0x002168, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002169, 0x000000, 0x002169, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00216A, 0x000000, 0x00216A, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00216B, 0x000000, 0x00216B, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00216C, 0x000000, 0x00216C, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00216D, 0x000000, 0x00216D, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00216E, 0x000000, 0x00216E, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00216F, 0x000000, 0x00216F, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002184, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002183, 0x000000, 0x002183, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x0024D0, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024D1, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024D2, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024D3, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024D4, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024D5, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024D6, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024D7, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024D8, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024D9, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024DA, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024DB, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024DC, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024DD, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024DE, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024DF, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E0, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E1, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E2, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E3, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E4, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E5, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E6, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E7, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E8, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x0024E9, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024B6, 0x000000, 0x0024B6, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024B7, 0x000000, 0x0024B7, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024B8, 0x000000, 0x0024B8, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024B9, 0x000000, 0x0024B9, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024BA, 0x000000, 0x0024BA, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024BB, 0x000000, 0x0024BB, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024BC, 0x000000, 0x0024BC, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024BD, 0x000000, 0x0024BD, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024BE, 0x000000, 0x0024BE, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024BF, 0x000000, 0x0024BF, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C0, 0x000000, 0x0024C0, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C1, 0x000000, 0x0024C1, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C2, 0x000000, 0x0024C2, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C3, 0x000000, 0x0024C3, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C4, 0x000000, 0x0024C4, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C5, 0x000000, 0x0024C5, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C6, 0x000000, 0x0024C6, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C7, 0x000000, 0x0024C7, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C8, 0x000000, 0x0024C8, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024C9, 0x000000, 0x0024C9, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024CA, 0x000000, 0x0024CA, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024CB, 0x000000, 0x0024CB, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024CC, 0x000000, 0x0024CC, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024CD, 0x000000, 0x0024CD, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024CE, 0x000000, 0x0024CE, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0024CF, 0x000000, 0x0024CF, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AI, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_QU, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_EX, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x002C30, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C31, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C32, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C33, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C34, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C35, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C36, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C37, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C38, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C39, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C3A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C3B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C3C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C3D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C3E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C3F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C40, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C41, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C42, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C43, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C44, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C45, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C46, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C47, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C48, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C49, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C4A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C4B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C4C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C4D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C4E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C4F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C50, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C51, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C52, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C53, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C54, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C55, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C56, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C57, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C58, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C59, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C5A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C5B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C5C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C5D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C5E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C00, 0x000000, 0x002C00, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C01, 0x000000, 0x002C01, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C02, 0x000000, 0x002C02, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C03, 0x000000, 0x002C03, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C04, 0x000000, 0x002C04, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C05, 0x000000, 0x002C05, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C06, 0x000000, 0x002C06, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C07, 0x000000, 0x002C07, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C08, 0x000000, 0x002C08, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C09, 0x000000, 0x002C09, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C0A, 0x000000, 0x002C0A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C0B, 0x000000, 0x002C0B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C0C, 0x000000, 0x002C0C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C0D, 0x000000, 0x002C0D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C0E, 0x000000, 0x002C0E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C0F, 0x000000, 0x002C0F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C10, 0x000000, 0x002C10, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C11, 0x000000, 0x002C11, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C12, 0x000000, 0x002C12, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C13, 0x000000, 0x002C13, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C14, 0x000000, 0x002C14, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C15, 0x000000, 0x002C15, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C16, 0x000000, 0x002C16, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C17, 0x000000, 0x002C17, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C18, 0x000000, 0x002C18, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C19, 0x000000, 0x002C19, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C1A, 0x000000, 0x002C1A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C1B, 0x000000, 0x002C1B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C1C, 0x000000, 0x002C1C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C1D, 0x000000, 0x002C1D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C1E, 0x000000, 0x002C1E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C1F, 0x000000, 0x002C1F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C20, 0x000000, 0x002C20, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C21, 0x000000, 0x002C21, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C22, 0x000000, 0x002C22, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C23, 0x000000, 0x002C23, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C24, 0x000000, 0x002C24, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C25, 0x000000, 0x002C25, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C26, 0x000000, 0x002C26, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C27, 0x000000, 0x002C27, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C28, 0x000000, 0x002C28, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C29, 0x000000, 0x002C29, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C2A, 0x000000, 0x002C2A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C2B, 0x000000, 0x002C2B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C2C, 0x000000, 0x002C2C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C2D, 0x000000, 0x002C2D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C2E, 0x000000, 0x002C2E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C61, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C60, 0x000000, 0x002C60, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00026B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001D7D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00027D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00023A, 0x000000, 0x00023A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00023E, 0x000000, 0x00023E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C68, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C67, 0x000000, 0x002C67, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C6A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C69, 0x000000, 0x002C69, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C6C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C6B, 0x000000, 0x002C6B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000251, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000271, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000250, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C73, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C72, 0x000000, 0x002C72, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C76, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C75, 0x000000, 0x002C75, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C81, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C80, 0x000000, 0x002C80, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C83, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C82, 0x000000, 0x002C82, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C85, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C84, 0x000000, 0x002C84, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C87, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C86, 0x000000, 0x002C86, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C89, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C88, 0x000000, 0x002C88, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C8B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C8A, 0x000000, 0x002C8A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C8D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C8C, 0x000000, 0x002C8C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C8F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C8E, 0x000000, 0x002C8E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C91, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C90, 0x000000, 0x002C90, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C93, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C92, 0x000000, 0x002C92, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C95, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C94, 0x000000, 0x002C94, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C97, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C96, 0x000000, 0x002C96, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C99, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C98, 0x000000, 0x002C98, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C9B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C9A, 0x000000, 0x002C9A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C9D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C9C, 0x000000, 0x002C9C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002C9F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002C9E, 0x000000, 0x002C9E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CA1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CA0, 0x000000, 0x002CA0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CA3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CA2, 0x000000, 0x002CA2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CA5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CA4, 0x000000, 0x002CA4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CA7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CA6, 0x000000, 0x002CA6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CA9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CA8, 0x000000, 0x002CA8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CAB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CAA, 0x000000, 0x002CAA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CAD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CAC, 0x000000, 0x002CAC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CAF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CAE, 0x000000, 0x002CAE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CB1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CB0, 0x000000, 0x002CB0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CB3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CB2, 0x000000, 0x002CB2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CB5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CB4, 0x000000, 0x002CB4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CB7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CB6, 0x000000, 0x002CB6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CB9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CB8, 0x000000, 0x002CB8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CBB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CBA, 0x000000, 0x002CBA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CBD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CBC, 0x000000, 0x002CBC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CBF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CBE, 0x000000, 0x002CBE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CC1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CC0, 0x000000, 0x002CC0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CC3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CC2, 0x000000, 0x002CC2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CC5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CC4, 0x000000, 0x002CC4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CC7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CC6, 0x000000, 0x002CC6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CC9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CC8, 0x000000, 0x002CC8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CCB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CCA, 0x000000, 0x002CCA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CCD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CCC, 0x000000, 0x002CCC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CCF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CCE, 0x000000, 0x002CCE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CD1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CD0, 0x000000, 0x002CD0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CD3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CD2, 0x000000, 0x002CD2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CD5, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CD4, 0x000000, 0x002CD4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CD7, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CD6, 0x000000, 0x002CD6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CD9, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CD8, 0x000000, 0x002CD8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CDB, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CDA, 0x000000, 0x002CDA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CDD, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CDC, 0x000000, 0x002CDC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CDF, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CDE, 0x000000, 0x002CDE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CE1, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CE0, 0x000000, 0x002CE0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x002CE3, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x002CE2, 0x000000, 0x002CE2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A0, 0x000000, 0x0010A0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A1, 0x000000, 0x0010A1, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A2, 0x000000, 0x0010A2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A3, 0x000000, 0x0010A3, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A4, 0x000000, 0x0010A4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A5, 0x000000, 0x0010A5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A6, 0x000000, 0x0010A6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A7, 0x000000, 0x0010A7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A8, 0x000000, 0x0010A8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010A9, 0x000000, 0x0010A9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010AA, 0x000000, 0x0010AA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010AB, 0x000000, 0x0010AB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010AC, 0x000000, 0x0010AC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010AD, 0x000000, 0x0010AD, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010AE, 0x000000, 0x0010AE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010AF, 0x000000, 0x0010AF, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B0, 0x000000, 0x0010B0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B1, 0x000000, 0x0010B1, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B2, 0x000000, 0x0010B2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B3, 0x000000, 0x0010B3, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B4, 0x000000, 0x0010B4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B5, 0x000000, 0x0010B5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B6, 0x000000, 0x0010B6, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B7, 0x000000, 0x0010B7, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B8, 0x000000, 0x0010B8, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010B9, 0x000000, 0x0010B9, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010BA, 0x000000, 0x0010BA, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010BB, 0x000000, 0x0010BB, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010BC, 0x000000, 0x0010BC, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010BD, 0x000000, 0x0010BD, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010BE, 0x000000, 0x0010BE, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010BF, 0x000000, 0x0010BF, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010C0, 0x000000, 0x0010C0, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010C1, 0x000000, 0x0010C1, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010C2, 0x000000, 0x0010C2, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010C3, 0x000000, 0x0010C3, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010C4, 0x000000, 0x0010C4, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x0010C5, 0x000000, 0x0010C5, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SEPARATOR_SPACE          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_WS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_CL, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_KATAKANA    , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_KATAKANA    , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_KATAKANA    , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_KATAKANA    , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_KATAKANA    , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_KATAKANA    , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_KATAKANA    , BidiClass_L   },
	{ 0x000000, 0x00A641, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A640, 0x000000, 0x00A640, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A643, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A642, 0x000000, 0x00A642, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A645, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A644, 0x000000, 0x00A644, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A647, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A646, 0x000000, 0x00A646, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A649, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A648, 0x000000, 0x00A648, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A64B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A64A, 0x000000, 0x00A64A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A64D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A64C, 0x000000, 0x00A64C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A64F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A64E, 0x000000, 0x00A64E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A651, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A650, 0x000000, 0x00A650, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A653, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A652, 0x000000, 0x00A652, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A655, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A654, 0x000000, 0x00A654, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A657, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A656, 0x000000, 0x00A656, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A659, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A658, 0x000000, 0x00A658, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A65B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A65A, 0x000000, 0x00A65A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A65D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A65C, 0x000000, 0x00A65C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A65F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A65E, 0x000000, 0x00A65E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A663, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A662, 0x000000, 0x00A662, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A665, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A664, 0x000000, 0x00A664, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A667, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A666, 0x000000, 0x00A666, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A669, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A668, 0x000000, 0x00A668, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A66B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A66A, 0x000000, 0x00A66A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A66D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A66C, 0x000000, 0x00A66C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A681, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A680, 0x000000, 0x00A680, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A683, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A682, 0x000000, 0x00A682, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A685, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A684, 0x000000, 0x00A684, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A687, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A686, 0x000000, 0x00A686, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A689, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A688, 0x000000, 0x00A688, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A68B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A68A, 0x000000, 0x00A68A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A68D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A68C, 0x000000, 0x00A68C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A68F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A68E, 0x000000, 0x00A68E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A691, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A690, 0x000000, 0x00A690, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A693, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A692, 0x000000, 0x00A692, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A695, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A694, 0x000000, 0x00A694, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A697, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A696, 0x000000, 0x00A696, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A723, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A722, 0x000000, 0x00A722, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A725, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A724, 0x000000, 0x00A724, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A727, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A726, 0x000000, 0x00A726, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A729, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A728, 0x000000, 0x00A728, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A72B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A72A, 0x000000, 0x00A72A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A72D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A72C, 0x000000, 0x00A72C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A72F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A72E, 0x000000, 0x00A72E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A733, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A732, 0x000000, 0x00A732, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A735, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A734, 0x000000, 0x00A734, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A737, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A736, 0x000000, 0x00A736, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A739, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A738, 0x000000, 0x00A738, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A73B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A73A, 0x000000, 0x00A73A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A73D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A73C, 0x000000, 0x00A73C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A73F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A73E, 0x000000, 0x00A73E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A741, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A740, 0x000000, 0x00A740, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A743, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A742, 0x000000, 0x00A742, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A745, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A744, 0x000000, 0x00A744, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A747, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A746, 0x000000, 0x00A746, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A749, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A748, 0x000000, 0x00A748, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A74B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A74A, 0x000000, 0x00A74A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A74D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A74C, 0x000000, 0x00A74C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A74F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A74E, 0x000000, 0x00A74E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A751, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A750, 0x000000, 0x00A750, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A753, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A752, 0x000000, 0x00A752, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A755, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A754, 0x000000, 0x00A754, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A757, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A756, 0x000000, 0x00A756, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A759, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A758, 0x000000, 0x00A758, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A75B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A75A, 0x000000, 0x00A75A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A75D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A75C, 0x000000, 0x00A75C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A75F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A75E, 0x000000, 0x00A75E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A761, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A760, 0x000000, 0x00A760, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A763, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A762, 0x000000, 0x00A762, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A765, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A764, 0x000000, 0x00A764, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A767, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A766, 0x000000, 0x00A766, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A769, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A768, 0x000000, 0x00A768, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A76B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A76A, 0x000000, 0x00A76A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A76D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A76C, 0x000000, 0x00A76C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A76F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A76E, 0x000000, 0x00A76E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A77A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A779, 0x000000, 0x00A779, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A77C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A77B, 0x000000, 0x00A77B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x001D79, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A77F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A77E, 0x000000, 0x00A77E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A781, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A780, 0x000000, 0x00A780, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A783, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A782, 0x000000, 0x00A782, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A785, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A784, 0x000000, 0x00A784, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00A787, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A786, 0x000000, 0x00A786, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x00A78C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00A78B, 0x000000, 0x00A78B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BB, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_LV     , LineBreakProp_H2, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_LVT    , LineBreakProp_H3, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_SURROGATE          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_SG, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_PRIVATE_USE        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_XX, WordBreakProp_OTHER       , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_IS, WordBreakProp_MIDLETTER   , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_CONNECTOR    , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_EXTENDNUMLET, BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_CL, WordBreakProp_MIDNUM      , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_CL, WordBreakProp_MIDNUMLET   , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_MIDNUM      , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_NS, WordBreakProp_MIDLETTER   , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_ET  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_ES  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_DASH         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_ES  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_MIDNUMLET   , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_CS  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 1, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 2, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 3, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 4, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 5, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 6, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 7, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 8, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x000000, 0x000000, 9, Categ_NUMBER_DECIMAL_DIGIT     , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_EN  },
	{ 0x000000, 0x00FF41, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF42, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF43, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF44, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF45, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF46, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF47, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF48, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF49, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF4A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF4B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF4C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF4D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF4E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF4F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF50, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF51, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF52, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF53, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF54, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF55, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF56, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF57, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF58, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF59, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x00FF5A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MODIFIER          , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x00FF21, 0x000000, 0x00FF21, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF22, 0x000000, 0x00FF22, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF23, 0x000000, 0x00FF23, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF24, 0x000000, 0x00FF24, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF25, 0x000000, 0x00FF25, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF26, 0x000000, 0x00FF26, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF27, 0x000000, 0x00FF27, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF28, 0x000000, 0x00FF28, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF29, 0x000000, 0x00FF29, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF2A, 0x000000, 0x00FF2A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF2B, 0x000000, 0x00FF2B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF2C, 0x000000, 0x00FF2C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF2D, 0x000000, 0x00FF2D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF2E, 0x000000, 0x00FF2E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF2F, 0x000000, 0x00FF2F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF30, 0x000000, 0x00FF30, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF31, 0x000000, 0x00FF31, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF32, 0x000000, 0x00FF32, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF33, 0x000000, 0x00FF33, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF34, 0x000000, 0x00FF34, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF35, 0x000000, 0x00FF35, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF36, 0x000000, 0x00FF36, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF37, 0x000000, 0x00FF37, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF38, 0x000000, 0x00FF38, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF39, 0x000000, 0x00FF39, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x00FF3A, 0x000000, 0x00FF3A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_ID, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_KATAKANA    , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_LETTER_MODIFIER          , 0, 0, GraphemeBreakProp_EXTEND , LineBreakProp_NS, WordBreakProp_EXTEND      , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_OTHER_FORMAT             , 0, 0, GraphemeBreakProp_CONTROL, LineBreakProp_CM, WordBreakProp_FORMAT      , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_CB, WordBreakProp_OTHER       , BidiClass_ON  },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_LETTER            , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_ON  },
	{ 0x000000, 0x010428, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010429, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01042A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01042B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01042C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01042D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01042E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01042F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010430, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010431, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010432, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010433, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010434, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010435, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010436, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010437, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010438, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010439, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01043A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01043B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01043C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01043D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01043E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01043F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010440, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010441, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010442, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010443, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010444, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010445, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010446, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010447, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010448, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x010449, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01044A, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01044B, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01044C, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01044D, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01044E, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x01044F, 0x000000, 0, Categ_LETTER_UPPERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010400, 0x000000, 0x010400, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010401, 0x000000, 0x010401, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010402, 0x000000, 0x010402, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010403, 0x000000, 0x010403, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010404, 0x000000, 0x010404, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010405, 0x000000, 0x010405, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010406, 0x000000, 0x010406, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010407, 0x000000, 0x010407, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010408, 0x000000, 0x010408, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010409, 0x000000, 0x010409, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01040A, 0x000000, 0x01040A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01040B, 0x000000, 0x01040B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01040C, 0x000000, 0x01040C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01040D, 0x000000, 0x01040D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01040E, 0x000000, 0x01040E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01040F, 0x000000, 0x01040F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010410, 0x000000, 0x010410, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010411, 0x000000, 0x010411, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010412, 0x000000, 0x010412, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010413, 0x000000, 0x010413, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010414, 0x000000, 0x010414, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010415, 0x000000, 0x010415, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010416, 0x000000, 0x010416, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010417, 0x000000, 0x010417, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010418, 0x000000, 0x010418, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010419, 0x000000, 0x010419, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01041A, 0x000000, 0x01041A, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01041B, 0x000000, 0x01041B, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01041C, 0x000000, 0x01041C, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01041D, 0x000000, 0x01041D, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01041E, 0x000000, 0x01041E, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x01041F, 0x000000, 0x01041F, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010420, 0x000000, 0x010420, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010421, 0x000000, 0x010421, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010422, 0x000000, 0x010422, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010423, 0x000000, 0x010423, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010424, 0x000000, 0x010424, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010425, 0x000000, 0x010425, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010426, 0x000000, 0x010426, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x010427, 0x000000, 0x010427, 0, Categ_LETTER_LOWERCASE         , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_ALETTER     , BidiClass_L   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_NUMBER_OTHER             , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_PUNCTUATION_OTHER        , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_BA, WordBreakProp_OTHER       , BidiClass_R   },
	{ 0x000000, 0x000000, 0x000000, 0, Categ_SYMBOL_MATH              , 0, 0, GraphemeBreakProp_OTHER  , LineBreakProp_AL, WordBreakProp_OTHER       , BidiClass_L   }
};

const int16_t	CharData::_index_info_list [] = 
{
// 0
	    1,     1,     1,     1,     1,     1,     1,     1,
	    1,     2,     3,     4,     5,     6,     1,     1,
	    1,     1,     1,     1,     1,     1,     1,     1,
	    1,     1,     1,     1,     7,     7,     7,     8,
	    9,    10,    11,    12,    13,    14,    15,    16,
	   17,    18,    15,    19,    20,    21,    22,    23,
	   24,    25,    26,    27,    28,    29,    30,    31,
	   32,    33,    34,    35,    36,    36,    36,    10,
	   15,    37,    38,    39,    40,    41,    42,    43,
	   44,    45,    46,    47,    48,    49,    50,    51,
	   52,    53,    54,    55,    56,    57,    58,    59,
	   60,    61,    62,    17,    63,    18,    64,    65,
	   64,    66,    67,    68,    69,    70,    71,    72,
	   73,    74,    75,    76,    77,    78,    79,    80,
	   81,    82,    83,    84,    85,    86,    87,    88,
	   89,    90,    91,    17,    92,    18,    36,     1,
	    1,     1,     1,     1,     1,    93,     1,     1,
	    1,     1,     1,     1,     1,     1,     1,     1,
	    1,     1,     1,     1,     1,     1,     1,     1,
	    1,     1,     1,     1,     1,     1,     1,     1,
	   94,    95,    96,    13,    13,    13,    97,    98,
	   99,    97,   100,   101,    36,   102,    97,    64,
	  103,   104,   105,   105,   106,   107,    98,   108,
	   99,   105,   100,   109,   110,   110,   110,    95,
	  111,   112,   113,   114,   115,   116,   117,   118,
	  119,   120,   121,   122,   123,   124,   125,   126,
	  127,   128,   129,   130,   131,   132,   133,   134,
	  135,   136,   137,   138,   139,   140,   141,   142,
	  143,   144,   145,   146,   147,   148,   149,   150,
	  151,   152,   153,   154,   155,   156,   157,   158,
	  159,   160,   161,   162,   163,   164,   165,   134,
	  166,   167,   168,   169,   170,   171,   172,   173,
// 1
	  174,   175,   176,   177,   178,   179,   180,   181,
	  182,   183,   184,   185,   186,   187,   188,   189,
	  190,   191,   192,   193,   194,   195,   196,   197,
	  198,   199,   200,   201,   202,   203,   204,   205,
	  206,   207,   208,   209,   210,   211,   212,   213,
	  214,   215,   216,   217,   218,   219,   220,   221,
	   45,    74,   222,   223,   224,   225,   226,   227,
	  142,   228,   229,   230,   231,   232,   233,   234,
	  235,   236,   237,   238,   239,   240,   241,   242,
	  243,   142,   244,   245,   246,   247,   248,   249,
	  250,   251,   252,   253,   254,   255,   256,   257,
	  258,   259,   260,   261,   262,   263,   264,   265,
	  266,   267,   268,   269,   270,   271,   272,   273,
	  274,   275,   276,   277,   278,   279,   280,   281,
	  282,   283,   284,   285,   286,   287,   288,   289,
	  290,   291,   292,   293,   294,   295,   296,    84,
	  297,   298,   299,   300,   301,   302,   303,   304,
	  305,   306,   307,   308,   309,   142,   310,   311,
	  312,   313,   314,   315,   316,   317,   318,   319,
	  320,   321,   322,   142,   323,   324,   325,   326,
	  327,   328,   329,   330,   331,   332,   333,   334,
	  335,   336,   142,   142,   337,   338,   339,   340,
	  341,   342,   343,   344,   345,   346,   347,   348,
	  349,   350,   142,   351,   352,   353,   142,   354,
	  351,   351,   351,   351,   355,   356,   357,   358,
	  359,   360,   361,   362,   363,   364,   365,   366,
	  367,   368,   369,   370,   371,   372,   373,   374,
	  375,   376,   377,   378,   379,   380,   381,   382,
	  383,   384,   385,   386,   387,   388,   389,   390,
	  391,   392,   393,   394,   395,   396,   397,   398,
	  142,   399,   400,   401,   402,   403,   404,   405,
	  406,   407,   408,   409,   410,   411,   412,   413,
// 2
	  414,   415,   416,   417,   418,   419,   420,   421,
	  422,   423,   424,   425,   426,   427,   428,   429,
	  430,   431,   432,   433,   434,   435,   436,   437,
	  438,   439,   440,   441,   442,   443,   444,   445,
	  446,   142,   447,   448,   449,   450,   451,   452,
	  453,   454,   455,   456,   457,   458,   459,   460,
	  461,   462,   463,   464,   142,   142,   142,   142,
	  142,   142,   465,   466,   467,   468,   469,   142,
	  142,   470,   471,   472,   473,   474,   475,   476,
	  477,   478,   479,   480,   481,   482,   483,   484,
	  485,   486,   142,   487,   488,   142,   489,   490,
	  142,   491,   142,   492,   142,   142,   142,   142,
	  493,   142,   142,   494,   142,   142,   142,   142,
	  495,   496,   142,   497,   142,   142,   142,   498,
	  142,   499,   500,   142,   142,   501,   142,   142,
	  142,   142,   142,   142,   142,   502,   142,   142,
	  503,   142,   142,   504,   142,   142,   142,   142,
	  505,   506,   507,   508,   509,   142,   142,   142,
	  142,   142,   510,   142,   351,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   512,   512,   511,   511,   511,   511,   511,
	  511,   511,    64,    64,    64,    64,   512,   513,
	  514,   513,   513,   513,   514,   513,   512,   512,
	  515,   511,    64,    64,    64,    64,    64,    64,
	   99,    99,    99,    99,    64,    99,    64,   106,
	  511,   511,   511,   511,   511,    64,    64,    64,
	   64,    64,    64,    64,   512,    64,   511,    64,
	   64,    64,    64,    64,    64,    64,    64,    64,
	   64,    64,    64,    64,    64,    64,    64,    64,
// 3
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   517,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   518,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   518,   518,   518,   518,
	  518,   518,   518,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  519,   520,   521,   522,   512,    64,   523,   524,
	    0,     0,   511,   525,   526,   527,    35,     0,
	    0,     0,     0,     0,    64,    64,   528,   529,
	  530,   531,   532,     0,   533,     0,   534,   535,
	  142,   536,   537,   538,   539,   540,   541,   542,
	  543,   544,   545,   546,   547,   548,   549,   550,
	  551,   552,     0,   553,   554,   555,   556,   557,
	  558,   559,   560,   561,   562,   563,   564,   565,
	  142,   566,   567,   568,   569,   570,   571,   572,
	  573,   574,   575,   576,   107,   577,   578,   579,
	  580,   581,   582,   582,   583,   584,   585,   586,
	  587,   588,   589,   590,   591,   592,   593,   594,
	  567,   573,   595,   595,   595,   585,   580,   596,
	  597,   598,   599,   600,   601,   602,   603,   604,
	  605,   606,   607,   608,   609,   610,   611,   612,
	  613,   614,   615,   616,   617,   618,   619,   620,
	  575,   581,   621,   142,   543,   570,    36,   622,
	  623,   624,   625,   626,   142,   627,   628,   629,
// 4
	  630,   631,   632,   633,   634,   635,   636,   637,
	  638,   639,   640,   641,   642,   643,   644,   645,
	  646,   647,   648,   649,   650,   651,   652,   653,
	  654,   655,   656,   657,   658,   659,   660,   661,
	  662,   663,   664,   665,   666,   667,   668,   669,
	  670,   671,   672,   673,   674,   675,   676,   677,
	  678,   679,   680,   681,   682,   683,   684,   685,
	  686,   687,   688,   689,   690,   691,   692,   693,
	  694,   695,   696,   697,   698,   699,   700,   701,
	  702,   703,   704,   705,   706,   707,   708,   709,
	  710,   711,   712,   713,   714,   715,   716,   717,
	  718,   719,   720,   721,   722,   723,   724,   725,
	  726,   727,   728,   729,   730,   731,   732,   733,
	  734,   735,   736,   737,   738,   739,   740,   741,
	  742,   743,   744,   745,   746,   747,   748,   749,
	  750,   751,   752,   753,   754,   755,   756,   757,
	  758,   759,   760,   516,   516,   516,   516,   516,
	  761,   761,   762,   763,   764,   765,   766,   767,
	  768,   769,   770,   771,   772,   773,   774,   775,
	  776,   777,   778,   779,   780,   781,   782,   783,
	  784,   785,   786,   787,   788,   789,   790,   791,
	  792,   793,   794,   795,   796,   797,   798,   799,
	  800,   801,   802,   803,   804,   805,   806,   807,
	  808,   809,   810,   811,   812,   813,   814,   815,
	  816,   817,   818,   819,   820,   821,   822,   823,
	  824,   825,   826,   827,   828,   829,   830,   831,
	  832,   833,   834,   835,   836,   837,   838,   839,
	  840,   841,   842,   843,   844,   845,   846,   847,
	  848,   849,   850,   851,   852,   853,   854,   855,
	  856,   857,   858,   859,   860,   861,   862,   863,
	  864,   865,   866,   867,   868,   869,   870,   871,
	  872,   873,   874,   875,   876,   877,   878,   879,
// 5
	  880,   881,   882,   883,   884,   885,   886,   887,
	  888,   889,   890,   891,   892,   893,   894,   895,
	  896,   897,   898,   899,   900,   901,   902,   903,
	  904,   905,   906,   907,   908,   909,   910,   911,
	  912,   913,   914,   915,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,   916,   917,   918,   919,   920,   921,   922,
	  923,   924,   925,   926,   927,   928,   929,   930,
	  931,   932,   933,   934,   935,   936,   937,   938,
	  939,   940,   941,   942,   943,   944,   945,   946,
	  947,   948,   949,   950,   951,   952,   953,     0,
	    0,   511,   954,   954,   954,   954,   954,   954,
	    0,   955,   956,   957,   958,   959,   960,   961,
	  962,   963,   964,   965,   966,   967,   968,   969,
	  970,   971,   972,   973,   974,   975,   976,   977,
	  978,   979,   980,   981,   982,   983,   984,   985,
	  986,   987,   988,   989,   990,   991,   992,   142,
	    0,   993,   994,     0,     0,     0,     0,     0,
	    0,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   995,   516,
	  996,   516,   516,   996,   516,   516,   997,   516,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,     0,     0,     0,     0,     0,
	  998,   998,   998,   999,  1000,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 6
	 1001,  1001,  1001,  1001,     0,     0,    36,    36,
	 1002,    14,    14,  1003,    20,  1004,    97,    97,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,  1005,     0,     0,  1005,  1005,
	    0,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1007,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,     0,
	 1008,  1009,  1010,  1011,  1012,  1013,  1014,  1015,
	 1016,  1017,    14,  1018,  1019,  1020,  1006,  1006,
	  516,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1005,  1006,   516,   516,
	  516,   516,   516,   516,   516,  1001,   761,   516,
	  516,   516,   516,   516,   516,  1007,  1007,   516,
	  516,    97,   516,   516,   516,   516,  1006,  1006,
	   24,    25,    26,    27,    28,    29,    30,    31,
	   32,    33,  1006,  1006,  1006,  1021,  1021,  1006,
// 7
	 1020,  1020,  1020,  1020,  1020,  1020,  1020,  1020,
	 1020,  1020,  1020,  1020,  1020,  1020,     0,  1022,
	 1006,   516,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,     0,     0,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,  1006,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1023,  1024,  1025,  1026,  1027,  1028,  1029,  1030,
	 1031,  1032,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,  1033,  1033,    97,    15,
	   35,    10,  1033,     0,     0,     0,     0,     0,
// 8
	    0,   516,   516,  1034,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,     0,     0,   516,   351,  1034,  1034,
	 1034,   516,   516,   516,   516,   516,   516,   516,
	  516,  1034,  1034,  1034,  1034,   516,     0,     0,
	  351,   516,   516,   516,   516,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   516,   516,  1035,  1035,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	  954,   511,   351,     0,     0,     0,     0,     0,
	    0,     0,     0,   351,   351,   351,   351,   351,
	    0,   516,  1034,  1034,     0,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,     0,   351,
	  351,     0,     0,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,     0,     0,     0,   351,   351,
	  351,   351,     0,     0,   516,   351,  1046,  1034,
	 1034,   516,   516,   516,   516,     0,     0,  1034,
	 1034,     0,     0,  1034,  1034,   516,   351,     0,
	    0,     0,     0,     0,     0,     0,     0,  1046,
	    0,     0,     0,     0,   351,   351,     0,   351,
	  351,   351,   516,   516,     0,     0,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	  351,   351,    13,    13,  1047,  1047,  1047,  1047,
	 1047,  1047,   760,     0,     0,     0,     0,     0,
// 9
	    0,   516,   516,  1034,     0,   351,   351,   351,
	  351,   351,   351,     0,     0,     0,     0,   351,
	  351,     0,     0,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,     0,   351,   351,     0,
	  351,   351,     0,     0,   516,     0,  1034,  1034,
	 1034,   516,   516,     0,     0,     0,     0,   516,
	  516,     0,     0,   516,   516,   516,     0,     0,
	    0,   516,     0,     0,     0,     0,     0,     0,
	    0,   351,   351,   351,   351,     0,   351,     0,
	    0,     0,     0,     0,     0,     0,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	  516,   516,   351,   351,   351,   516,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,   516,   516,  1034,     0,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,     0,   351,
	  351,   351,     0,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,     0,   351,   351,   351,
	  351,   351,     0,     0,   516,   351,  1034,  1034,
	 1034,   516,   516,   516,   516,   516,     0,   516,
	  516,  1034,     0,  1034,  1034,   516,     0,     0,
	  351,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   516,   516,     0,     0,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	    0,    13,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 10
	    0,   516,  1034,  1034,     0,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,     0,   351,
	  351,     0,     0,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,     0,   351,   351,   351,
	  351,   351,     0,     0,   516,   351,  1046,   516,
	 1034,   516,   516,   516,   516,     0,     0,  1034,
	 1034,     0,     0,  1034,  1034,   516,     0,     0,
	    0,     0,     0,     0,     0,     0,   516,  1046,
	    0,     0,     0,     0,   351,   351,     0,   351,
	  351,   351,   516,   516,     0,     0,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	  760,   351,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,   516,   351,     0,   351,   351,   351,
	  351,   351,   351,     0,     0,     0,   351,   351,
	  351,     0,   351,   351,   351,   351,     0,     0,
	    0,   351,   351,     0,   351,     0,   351,   351,
	    0,     0,     0,   351,   351,     0,     0,     0,
	  351,   351,   351,     0,     0,     0,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,     0,     0,     0,     0,  1046,  1034,
	  516,  1034,  1034,     0,     0,     0,  1034,  1034,
	 1034,     0,  1034,  1034,  1034,   516,     0,     0,
	  351,     0,     0,     0,     0,     0,     0,  1046,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	 1047,  1047,  1047,    97,    97,    97,    97,    97,
	   97,    13,    97,     0,     0,     0,     0,     0,
// 11
	    0,  1034,  1034,  1034,     0,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,     0,   351,   351,   351,
	  351,   351,     0,     0,     0,   351,   516,   516,
	  516,  1034,  1034,  1034,  1034,     0,   516,   516,
	  516,     0,   516,   516,   516,   516,     0,     0,
	    0,     0,     0,     0,     0,   516,   516,     0,
	  351,   351,     0,     0,     0,     0,     0,     0,
	  351,   351,   516,   516,     0,     0,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1048,  1048,  1048,  1048,  1048,  1048,  1048,   760,
	    0,     0,  1034,  1034,     0,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,     0,   351,   351,   351,
	  351,   351,     0,     0,   516,   351,  1034,  1049,
	 1034,  1034,  1046,  1034,  1034,     0,  1049,  1034,
	 1034,     0,  1034,  1034,   516,   516,     0,     0,
	    0,     0,     0,     0,     0,  1046,  1046,     0,
	    0,     0,     0,     0,     0,     0,   351,     0,
	  351,   351,   516,   516,     0,     0,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	    0,    97,    97,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 12
	    0,     0,  1034,  1034,     0,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,     0,     0,     0,   351,  1046,  1034,
	 1034,   516,   516,   516,   516,     0,  1034,  1034,
	 1034,     0,  1034,  1034,  1034,   516,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,  1046,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   516,   516,     0,     0,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	 1047,  1047,  1047,  1047,  1047,  1047,     0,     0,
	    0,  1050,   351,   351,   351,   351,   351,   351,
	    0,     0,  1034,  1034,     0,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,     0,
	    0,     0,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,     0,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,     0,   351,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	    0,     0,   516,     0,     0,     0,     0,  1046,
	 1034,  1034,   516,   516,   516,     0,   516,     0,
	 1034,  1034,  1034,  1034,  1034,  1034,  1034,  1046,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,  1034,  1034,   954,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 13
	    0,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1052,  1053,  1052,  1052,  1053,  1053,  1053,  1053,
	 1053,  1053,  1053,     0,     0,     0,     0,    13,
	 1054,  1054,  1054,  1054,  1054,  1052,  1055,  1053,
	 1053,  1053,  1053,  1053,  1053,  1053,  1053,   954,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,  1035,  1035,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,  1051,  1051,     0,  1051,     0,     0,  1051,
	 1051,     0,  1051,     0,     0,  1051,     0,     0,
	    0,     0,     0,     0,  1051,  1051,  1051,  1051,
	    0,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	    0,  1051,  1051,  1051,     0,  1051,     0,  1051,
	    0,     0,  1051,  1051,     0,  1051,  1051,  1051,
	 1052,  1053,  1052,  1052,  1053,  1053,  1053,  1053,
	 1053,  1053,     0,  1053,  1053,  1051,     0,     0,
	 1054,  1054,  1054,  1054,  1054,     0,  1055,     0,
	 1053,  1053,  1053,  1053,  1053,  1053,     0,     0,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,  1051,  1051,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 14
	  351,  1056,  1056,  1056,  1057,   954,  1057,  1057,
	 1058,  1057,  1057,  1035,  1058,  1059,  1059,  1059,
	 1059,  1059,  1058,   760,  1060,   760,   760,   760,
	  516,   516,   760,   760,   760,   760,   760,   760,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,  1047,  1047,  1061,   516,   760,   516,
	  760,   516,    17,    18,    17,    18,  1034,  1034,
	  351,   351,   351,   351,   351,   351,   351,   351,
	    0,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,     0,     0,
	    0,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,  1062,
	  516,   516,   516,   516,   516,  1035,   516,   516,
	  351,   351,   351,   351,     0,     0,     0,     0,
	  516,   516,   516,   516,   516,   516,   516,   516,
	    0,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,     0,  1061,  1061,
	  760,   760,   760,   760,   760,   760,   516,   760,
	  760,   760,   760,   760,   760,     0,   760,   760,
	 1057,  1057,  1035,  1057,   954,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 15
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1063,  1063,  1053,  1053,  1053,
	 1053,  1063,  1053,  1053,  1053,  1053,  1053,  1053,
	 1063,  1053,  1053,  1063,  1063,  1053,  1053,  1051,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,  1035,  1035,   954,   954,   954,   954,
	 1051,  1051,  1051,  1051,  1051,  1051,  1063,  1063,
	 1053,  1053,  1051,  1051,  1051,  1051,  1053,  1053,
	 1053,  1051,  1063,  1063,  1063,  1051,  1051,  1063,
	 1063,  1063,  1063,  1063,  1063,  1063,  1051,  1051,
	 1051,  1053,  1053,  1053,  1053,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1053,  1063,  1063,  1053,  1053,  1063,
	 1063,  1063,  1063,  1063,  1063,  1053,  1051,  1063,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,     0,     0,  1064,  1064,
	 1065,  1066,  1067,  1068,  1069,  1070,  1071,  1072,
	 1073,  1074,  1075,  1076,  1077,  1078,  1079,  1080,
	 1081,  1082,  1083,  1084,  1085,  1086,  1087,  1088,
	 1089,  1090,  1091,  1092,  1093,  1094,  1095,  1096,
	 1097,  1098,  1099,  1100,  1101,  1102,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   954,   511,     0,     0,     0,
// 16
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,  1103,  1103,  1103,  1103,  1103,  1103,
	 1103,  1103,     0,     0,     0,     0,     0,  1103,
	 1104,  1104,  1104,  1104,  1104,  1104,  1104,  1104,
	 1104,  1104,  1104,  1104,  1104,  1104,  1104,  1104,
	 1104,  1104,  1104,  1104,  1104,  1104,  1104,  1104,
	 1104,  1104,  1104,  1104,  1104,  1104,  1104,  1104,
	 1104,  1104,  1104,  1104,  1104,  1104,  1104,  1104,
	 1104,  1104,  1104,  1104,  1104,  1104,  1104,  1104,
	 1104,  1104,  1104,  1104,  1104,  1104,  1104,  1104,
	 1104,  1104,  1104,  1104,  1104,  1104,  1104,  1104,
	 1104,  1104,  1104,     0,     0,     0,     0,     0,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,  1105,  1105,  1105,  1105,  1105,  1105,
	 1105,  1105,     0,     0,     0,     0,     0,     0,
// 17
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,     0,   351,   351,   351,   351,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,     0,   351,   351,   351,   351,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
// 18
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,   351,   351,   351,   351,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,     0,     0,     0,     0,   516,
	  760,  1035,   954,   954,   954,   954,   954,   954,
	  954,  1047,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,  1047,  1047,  1047,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 19
	    0,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
// 20
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   954,   954,   351,
	  351,   351,   351,   351,   351,   351,   351,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1106,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,    17,    18,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,  1035,  1035,  1035,  1107,  1107,
	 1107,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 21
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,   351,   351,
	  351,   351,   516,   516,   516,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   516,   516,   516,  1035,  1035,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   516,   516,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,   351,   351,
	  351,     0,   516,   516,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1108,  1108,  1063,  1053,
	 1053,  1053,  1053,  1053,  1053,  1053,  1063,  1063,
	 1063,  1063,  1063,  1063,  1063,  1063,  1053,  1063,
	 1063,  1053,  1053,  1053,  1053,  1053,  1053,  1053,
	 1053,  1053,  1053,  1053,  1035,  1035,  1109,  1055,
	 1035,   954,  1035,    13,  1051,  1053,     0,     0,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,     0,     0,     0,     0,
	 1048,  1048,  1048,  1048,  1048,  1048,  1048,  1048,
	 1048,  1048,     0,     0,     0,     0,     0,     0,
// 22
	   15,    15,    10,    10,  1110,  1110,  1111,    15,
	   10,    10,    15,   516,   516,   516,  1112,     0,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   511,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   516,   351,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 23
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,     0,     0,
	  516,   516,   516,  1034,  1034,  1034,  1034,   516,
	  516,  1034,  1034,  1034,     0,     0,     0,     0,
	 1034,  1034,   516,  1034,  1034,  1034,  1034,  1034,
	 1034,   516,   516,   516,     0,     0,     0,     0,
	   97,     0,     0,     0,    10,    10,  1036,  1037,
	 1038,  1039,  1040,  1041,  1042,  1043,  1044,  1045,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,     0,     0,
	 1051,  1051,  1051,  1051,  1051,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1051,  1051,     0,     0,     0,     0,     0,     0,
	 1063,  1063,  1063,  1063,  1063,  1063,  1063,  1063,
	 1063,  1063,  1063,  1063,  1063,  1063,  1063,  1063,
	 1063,  1051,  1051,  1051,  1051,  1051,  1051,  1051,
	 1063,  1063,     0,     0,     0,     0,     0,     0,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,     0,     0,  1113,  1113,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
// 24
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   516,
	  516,  1034,  1034,  1034,     0,     0,   954,   954,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 25
	  516,   516,   516,   516,  1034,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   516,  1034,   516,   516,
	  516,   516,   516,  1034,   516,  1034,  1034,  1034,
	 1034,  1034,   516,  1034,  1034,   351,   351,   351,
	  351,   351,   351,   351,     0,     0,     0,     0,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,  1035,  1035,  1035,  1035,  1035,  1035,
	 1035,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,     0,     0,     0,
	  516,   516,  1034,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,  1034,   516,   516,   516,   516,  1034,  1034,
	  516,   516,  1034,     0,     0,     0,   351,   351,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 26
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,  1034,  1034,  1034,  1034,
	 1034,  1034,  1034,  1034,   516,   516,   516,   516,
	  516,   516,   516,   516,  1034,  1034,   516,   516,
	    0,     0,     0,  1035,  1035,  1035,  1035,  1035,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,     0,   351,   351,   351,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  511,   511,   511,   511,   511,   511,  1035,  1035,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 27
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   511,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  511,  1114,   142,   142,   142,  1115,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   511,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  511,   511,   511,   511,   511,   511,   511,   511,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,   516,   516,
// 28
	 1116,  1117,  1118,  1119,  1120,  1121,  1122,  1123,
	 1124,  1125,  1126,  1127,  1128,  1129,  1130,  1131,
	 1132,  1133,  1134,  1135,  1136,  1137,  1138,  1139,
	 1140,  1141,  1142,  1143,  1144,  1145,  1146,  1147,
	 1148,  1149,  1150,  1151,  1152,  1153,  1154,  1155,
	 1156,  1157,  1158,  1159,  1160,  1161,  1162,  1163,
	 1164,  1165,  1166,  1167,  1168,  1169,  1170,  1171,
	 1172,  1173,  1174,  1175,  1176,  1177,  1178,  1179,
	 1180,  1181,  1182,  1183,  1184,  1185,  1186,  1187,
	 1188,  1189,  1190,  1191,  1192,  1193,  1194,  1195,
	 1196,  1197,  1198,  1199,  1200,  1201,  1202,  1203,
	 1204,  1205,  1206,  1207,  1208,  1209,  1210,  1211,
	 1212,  1213,  1214,  1215,  1216,  1217,  1218,  1219,
	 1220,  1221,  1222,  1223,  1224,  1225,  1226,  1227,
	 1228,  1229,  1230,  1231,  1232,  1233,  1234,  1235,
	 1236,  1237,  1238,  1239,  1240,  1241,  1242,  1243,
	 1244,  1245,  1246,  1247,  1248,  1249,  1250,  1251,
	 1252,  1253,  1254,  1255,  1256,  1257,  1258,  1259,
	 1260,  1261,  1262,  1263,  1264,  1265,   142,   142,
	  142,   142,   142,  1213,   142,   142,  1266,   142,
	 1267,  1268,  1269,  1270,  1271,  1272,  1273,  1274,
	 1275,  1276,  1277,  1278,  1279,  1280,  1281,  1282,
	 1283,  1284,  1285,  1286,  1287,  1288,  1289,  1290,
	 1291,  1292,  1293,  1294,  1295,  1296,  1297,  1298,
	 1299,  1300,  1301,  1302,  1303,  1304,  1305,  1306,
	 1307,  1308,  1309,  1310,  1311,  1312,  1313,  1314,
	 1315,  1316,  1317,  1318,  1319,  1320,  1321,  1322,
	 1323,  1324,  1325,  1326,  1327,  1328,  1329,  1330,
	 1331,  1332,  1333,  1334,  1335,  1336,  1337,  1338,
	 1339,  1340,  1341,  1342,  1343,  1344,  1345,  1346,
	 1347,  1348,  1349,  1350,  1351,  1352,  1353,  1354,
	 1355,  1356,  1357,  1358,  1359,  1360,  1361,  1362,
// 29
	 1363,  1364,  1365,  1366,  1367,  1368,  1369,  1370,
	 1371,  1372,  1373,  1374,  1375,  1376,  1377,  1378,
	 1379,  1380,  1381,  1382,  1383,  1384,     0,     0,
	 1385,  1386,  1387,  1388,  1389,  1390,     0,     0,
	 1391,  1392,  1393,  1394,  1395,  1396,  1397,  1398,
	 1399,  1400,  1401,  1402,  1403,  1404,  1405,  1406,
	 1407,  1408,  1409,  1410,  1411,  1412,  1413,  1414,
	 1415,  1416,  1417,  1418,  1419,  1420,  1421,  1422,
	 1423,  1424,  1425,  1426,  1427,  1428,     0,     0,
	 1429,  1430,  1431,  1432,  1433,  1434,     0,     0,
	  142,  1435,   142,  1436,   142,  1437,   142,  1438,
	    0,  1439,     0,  1440,     0,  1441,     0,  1442,
	 1443,  1444,  1445,  1446,  1447,  1448,  1449,  1450,
	 1451,  1452,  1453,  1454,  1455,  1456,  1457,  1458,
	 1459,  1460,  1461,  1462,  1463,  1464,  1465,  1466,
	 1467,  1468,  1469,  1470,  1471,  1472,     0,     0,
	 1473,  1474,  1475,  1476,  1477,  1478,  1479,  1480,
	 1481,  1482,  1483,  1484,  1485,  1486,  1487,  1488,
	 1489,  1490,  1491,  1492,  1493,  1494,  1495,  1496,
	 1497,  1498,  1499,  1500,  1501,  1502,  1503,  1504,
	 1505,  1506,  1507,  1508,  1509,  1510,  1511,  1512,
	 1513,  1514,  1515,  1516,  1517,  1518,  1519,  1520,
	 1521,  1522,   142,  1523,   142,     0,   142,   142,
	 1524,  1525,  1526,  1527,  1528,    64,   574,    64,
	   64,    64,   142,  1529,   142,     0,   142,   142,
	 1530,  1531,  1532,  1533,  1534,    64,    64,    64,
	 1535,  1536,   142,   142,     0,     0,   142,   142,
	 1537,  1538,  1539,  1540,     0,    64,    64,    64,
	 1541,  1542,   142,   142,   142,  1543,   142,   142,
	 1544,  1545,  1546,  1547,  1548,    64,    64,    64,
	    0,     0,   142,  1549,   142,     0,   142,   142,
	 1550,  1551,  1552,  1553,  1554,   106,    64,     0,
// 30
	 1106,  1106,  1106,  1106,  1106,  1106,  1106,  1112,
	 1106,  1106,  1106,  1555,  1556,  1556,  1557,  1558,
	  994,  1559,   994,   994,  1560,  1561,  1562,    15,
	 1563,  1564,    17,   101,   101,   109,    17,   101,
	 1562,  1562,    15,    15,  1565,  1566,  1566,  1567,
	 1568,  1569,  1570,  1571,  1572,  1573,  1574,    94,
	   14,    14,    14,    14,    14,  1575,  1575,  1575,
	   15,   101,   109,  1562,  1576,  1576,    15,    65,
	   65,    15,    15,    15,  1577,    17,    18,  1576,
	 1576,  1576,    15,    15,    15,    15,    15,    15,
	   15,    15,    36,    15,    65,    15,  1110,    15,
	 1110,  1110,  1110,  1110,    15,  1110,  1110,  1106,
	 1578,  1022,  1022,  1022,  1022,     0,     0,     0,
	    0,     0,  1579,  1579,  1579,  1579,  1579,  1579,
	 1580,   142,     0,     0,   105,  1580,  1580,  1580,
	 1580,  1580,  1581,  1581,    36,    17,    18,   100,
	 1580,   105,   105,   105,   105,  1580,  1580,  1580,
	 1580,  1580,  1581,  1581,    36,    17,    18,     0,
	  511,   511,   511,   511,   511,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	   13,    13,    13,    13,    13,    13,    13,    96,
	   13,    13,    13,    13,    13,    13,    13,    13,
	   13,    13,    13,    13,    13,    13,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   761,   761,   761,
	  761,   516,   761,   761,   761,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 31
	   97,    97,   595,  1582,    97,    98,    97,   595,
	   97,  1582,   142,   595,   595,   595,   142,   142,
	  595,   595,   595,   100,    97,   595,  1583,    97,
	   97,   595,   595,   595,   595,   595,    97,    97,
	   97,    98,    98,    97,   595,    97,   559,    97,
	  595,    97,    47,  1584,   595,   595,  1585,   142,
	  595,   595,  1586,   595,   142,   351,   351,   351,
	  351,   142,    97,    97,   142,   142,   595,   595,
	   36,    36,    36,    36,    36,   595,   142,   142,
	  142,   142,    97,    36,    97,    97,  1587,   760,
	    0,     0,     0,  1048,   110,   110,  1048,  1048,
	 1048,  1048,  1048,   110,  1048,  1048,   110,  1048,
	 1588,  1589,  1590,  1591,  1592,  1593,  1594,  1595,
	 1596,  1597,  1598,  1599,  1600,  1601,  1602,  1603,
	 1604,  1605,  1606,  1607,  1608,  1609,  1610,  1611,
	 1612,  1613,  1614,  1615,  1616,  1617,  1618,  1619,
	 1107,  1107,  1107,  1620,  1621,  1107,  1107,  1107,
	 1107,     0,     0,     0,     0,     0,     0,     0,
	  134,   134,   134,   134,   134,    98,    98,    98,
	   98,    98,    36,    36,    97,    97,    97,    97,
	   36,    97,    97,    36,    97,    97,    36,    97,
	   97,    97,    97,    97,    97,    97,    36,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    36,    36,
	   97,    97,   134,    97,   134,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
// 32
	  134,    36,   134,   134,    36,    36,    36,   134,
	  134,    36,    36,   134,    36,    36,    36,   134,
	   36,   134,    19,   104,    36,   134,    36,    36,
	   36,    36,   134,    36,    36,   134,   134,   134,
	  134,    36,    36,   134,    36,   134,    36,   134,
	  134,   134,   134,   134,   134,    36,   134,    36,
	   36,    36,    36,    36,   134,   134,   134,   134,
	   36,    36,    36,    36,   134,   134,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	  134,    36,    36,    36,   134,    36,    36,    36,
	   36,    36,   134,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	  134,   134,    36,    36,   134,   134,   134,   134,
	   36,    36,   134,   134,    36,    36,   134,   134,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,   134,   134,    36,    36,   134,   134,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,   134,    36,    36,
	   36,   134,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,   134,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,   134,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
// 33
	   97,    97,    97,    97,    97,    97,    97,    97,
	   36,    36,    36,    36,    97,    97,    97,    97,
	   97,    97,    98,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   36,    36,    97,    97,    97,    97,    97,    97,
	   97,    17,    18,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,    97,    36,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,   760,    97,    97,
	   97,    97,    97,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    36,    36,    36,    36,
	   36,    36,    97,    97,    97,    97,    97,    97,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 34
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  110,   110,   110,   110,   110,   110,   110,   110,
	  110,   110,   110,   110,   110,   110,   110,   110,
	  110,   110,   110,   110,   110,   110,   110,   110,
	  110,   110,   110,   110,   110,   110,   110,   110,
	  110,   110,   110,   110,   110,   110,   110,   110,
	  105,   105,   105,   105,   105,   105,   105,   105,
	  105,   105,   105,   105,   105,   105,   105,   105,
	  105,   105,   105,   105,  1622,  1622,  1622,  1622,
	 1622,  1622,  1622,  1622,  1622,  1622,  1622,  1622,
	 1622,  1622,  1622,  1622,  1622,  1622,  1622,  1622,
	 1622,  1622,  1622,  1622,  1622,  1622,  1623,  1624,
	 1625,  1626,  1627,  1628,  1629,  1630,  1631,  1632,
	 1633,  1634,  1635,  1636,  1637,  1638,  1639,  1640,
	 1641,  1642,  1643,  1644,  1645,  1646,  1647,  1648,
	 1649,  1650,  1651,  1652,  1653,  1654,  1655,  1656,
	 1657,  1658,  1659,  1660,  1661,  1662,  1663,  1664,
	 1665,  1666,  1667,  1668,  1669,  1670,  1671,  1672,
	 1673,  1674,   110,   110,   110,   110,   110,   110,
	  110,   110,   110,   110,   110,   110,   110,   110,
	  110,   110,   110,   110,   110,   110,   110,  1048,
// 35
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    97,    97,    97,    97,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   98,    98,    98,    98,    98,    98,    98,    98,
	   97,    97,    98,    98,    98,    98,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   98,    98,    97,    98,    98,    98,    98,    98,
	   98,    98,    97,    97,    97,    97,    97,    97,
	   97,    97,    98,    98,    97,    97,    98,   134,
	   97,    97,    97,    97,    98,    98,    97,    97,
	   98,   134,    97,    97,    97,    97,    98,    98,
	   98,    97,    97,    98,    97,    97,    98,    98,
	   98,    98,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    98,    98,    98,    98,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    98,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   36,    36,    36,    36,    36,    36,    36,    36,
// 36
	   97,    97,    97,    97,    97,    98,    98,    97,
	   97,    98,    97,    97,    97,    97,    98,    98,
	   97,    97,    97,    97,    98,    98,    98,    98,
	   97,    97,    97,    97,    98,    97,    98,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   98,    97,    98,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   98,    98,    97,    98,    98,    98,    97,    98,
	   98,    98,    98,    97,    98,    98,    97,   134,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,     0,     0,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,   760,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,     0,     0,     0,
	   97,    97,    97,    97,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 37
	    0,    97,    97,    97,    97,     0,    97,    97,
	   97,    97,     0,     0,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	    0,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,     0,    97,     0,    97,
	   97,    97,    97,     0,     0,     0,    97,     0,
	   97,    97,    97,  1675,  1675,  1675,  1675,     0,
	    0,    97,  1676,  1676,    97,    97,    97,    97,
	   17,    18,    17,    18,    17,    18,    17,    18,
	   17,    18,    17,    18,    17,    18,   110,   110,
	  110,   110,   110,   110,   110,   110,   110,   110,
	  110,   110,   110,   110,   110,   110,   110,   110,
	  110,   110,   110,   110,   110,   110,   110,   110,
	  110,   110,   110,   110,    97,     0,     0,     0,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	    0,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,     0,
	   36,    36,    36,    36,    36,    17,    18,    36,
	   36,    36,    36,     0,    36,     0,     0,     0,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    17,    18,
	   17,    18,    17,    18,    17,    18,    17,    18,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
// 38
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    17,    18,    17,    18,    17,
	   18,    17,    18,    17,    18,    17,    18,    17,
	   18,    17,    18,    17,    18,    17,    18,    17,
	   18,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   17,    18,    17,    18,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    17,    18,    36,    36,
// 39
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    36,    36,    36,
	   36,    36,    36,    36,    36,    97,    97,    36,
	   36,    36,    36,    36,    36,     0,     0,     0,
	   97,    97,    97,    97,    97,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 40
	 1677,  1678,  1679,  1680,  1681,  1682,  1683,  1684,
	 1685,  1686,  1687,  1688,  1689,  1690,  1691,  1692,
	 1693,  1694,  1695,  1696,  1697,  1698,  1699,  1700,
	 1701,  1702,  1703,  1704,  1705,  1706,  1707,  1708,
	 1709,  1710,  1711,  1712,  1713,  1714,  1715,  1716,
	 1717,  1718,  1719,  1720,  1721,  1722,  1723,     0,
	 1724,  1725,  1726,  1727,  1728,  1729,  1730,  1731,
	 1732,  1733,  1734,  1735,  1736,  1737,  1738,  1739,
	 1740,  1741,  1742,  1743,  1744,  1745,  1746,  1747,
	 1748,  1749,  1750,  1751,  1752,  1753,  1754,  1755,
	 1756,  1757,  1758,  1759,  1760,  1761,  1762,  1763,
	 1764,  1765,  1766,  1767,  1768,  1769,  1770,     0,
	 1771,  1772,  1773,  1774,  1775,  1776,  1777,  1778,
	 1779,  1780,  1781,  1782,  1783,  1784,  1785,  1786,
	    0,   142,  1787,  1788,   142,  1789,  1790,   142,
	  142,   142,   142,   142,   142,   511,     0,     0,
	 1791,  1792,  1793,  1794,  1795,  1796,  1797,  1798,
	 1799,  1800,  1801,  1802,  1803,  1804,  1805,  1806,
	 1807,  1808,  1809,  1810,  1811,  1812,  1813,  1814,
	 1815,  1816,  1817,  1818,  1819,  1820,  1821,  1822,
	 1823,  1824,  1825,  1826,  1827,  1828,  1829,  1830,
	 1831,  1832,  1833,  1834,  1835,  1836,  1837,  1838,
	 1839,  1840,  1841,  1842,  1843,  1844,  1845,  1846,
	 1847,  1848,  1849,  1850,  1851,  1852,  1853,  1854,
	 1855,  1856,  1857,  1858,  1859,  1860,  1861,  1862,
	 1863,  1864,  1865,  1866,  1867,  1868,  1869,  1870,
	 1871,  1872,  1873,  1874,  1875,  1876,  1877,  1878,
	 1879,  1880,  1881,  1882,  1883,  1884,  1885,  1886,
	 1887,  1888,  1889,  1890,   142,    97,    97,    97,
	   97,    97,    97,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,    10,  1110,  1110,  1110,  1048,    10,  1110,
// 41
	 1891,  1892,  1893,  1894,  1895,  1896,  1897,  1898,
	 1899,  1900,  1901,  1902,  1903,  1904,  1905,  1906,
	 1907,  1908,  1909,  1910,  1911,  1912,  1913,  1914,
	 1915,  1916,  1917,  1918,  1919,  1920,  1921,  1922,
	 1923,  1924,  1925,  1926,  1927,  1928,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,   511,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
// 42
	   11,    11,   101,   109,   101,   109,    11,    11,
	   11,   101,   109,    11,   101,   109,  1110,  1110,
	 1110,  1110,  1110,  1110,  1110,  1110,    15,   994,
	   95,  1110,  1929,    15,   101,   109,    15,    15,
	  101,   109,    17,    18,    17,    18,    17,    18,
	   17,    18,  1110,  1110,  1110,  1110,    10,   512,
	 1110,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,     0,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 43
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,     0,     0,     0,     0,
// 44
	 1931,  1932,  1932,  1933,  1930,  1934,  1935,  1936,
	   17,    18,    17,    18,    17,    18,    17,    18,
	   17,    18,  1930,  1930,    17,    18,    17,    18,
	   17,    18,    17,    18,  1937,    17,    18,    18,
	 1930,  1936,  1936,  1936,  1936,  1936,  1936,  1936,
	 1936,  1936,   516,   516,   516,   516,   516,   516,
	 1938,  1939,  1939,  1939,  1939,  1939,  1930,  1930,
	 1936,  1936,  1936,  1934,  1940,  1933,  1930,  1930,
	    0,  1941,  1935,  1941,  1935,  1941,  1935,  1941,
	 1935,  1941,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1941,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1941,  1935,  1941,  1935,  1941,
	 1935,  1935,  1935,  1935,  1935,  1935,  1941,  1935,
	 1935,  1935,  1935,  1935,  1935,  1941,  1941,     0,
	    0,   516,   516,  1942,  1942,  1943,  1943,  1935,
	 1944,  1945,  1946,  1945,  1946,  1945,  1946,  1945,
	 1946,  1945,  1946,  1946,  1946,  1946,  1946,  1946,
	 1946,  1946,  1946,  1946,  1946,  1946,  1946,  1946,
	 1946,  1946,  1946,  1946,  1946,  1946,  1946,  1946,
	 1946,  1946,  1946,  1945,  1946,  1946,  1946,  1946,
	 1946,  1946,  1946,  1946,  1946,  1946,  1946,  1946,
	 1946,  1946,  1946,  1946,  1946,  1946,  1946,  1946,
	 1946,  1946,  1946,  1946,  1946,  1946,  1946,  1946,
	 1946,  1946,  1946,  1945,  1946,  1945,  1946,  1945,
	 1946,  1946,  1946,  1946,  1946,  1946,  1945,  1946,
	 1946,  1946,  1946,  1946,  1946,  1945,  1945,  1946,
	 1946,  1946,  1946,  1576,  1947,  1947,  1947,  1946,
// 45
	    0,     0,     0,     0,     0,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,     0,     0,
	    0,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,     0,
	 1949,  1949,  1950,  1950,  1950,  1950,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1945,  1945,  1945,  1945,  1945,  1945,  1945,  1945,
	 1945,  1945,  1945,  1945,  1945,  1945,  1945,  1945,
// 46
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1930,  1930,     0,
	 1950,  1950,  1950,  1950,  1950,  1950,  1950,  1950,
	 1950,  1950,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1930,  1951,  1951,  1951,  1951,  1951,  1951,  1951,
	 1951,  1951,  1951,  1951,  1951,  1951,  1951,  1951,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1930,  1930,  1930,  1949,
	 1950,  1950,  1950,  1950,  1950,  1950,  1950,  1950,
	 1950,  1950,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1951,  1951,  1951,  1951,  1951,  1951,  1951,
	 1951,  1951,  1951,  1951,  1951,  1951,  1951,  1951,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1930,  1930,  1930,  1930,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,     0,
// 47
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1952,  1952,  1952,  1952,  1952,  1952,  1952,  1952,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1930,
	 1930,  1930,  1930,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1930,  1930,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1949,
	 1949,  1949,  1949,  1949,  1949,  1949,  1949,  1930,
// 48
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
// 49
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 50
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1934,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
// 51
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,  1948,  1948,  1948,
	 1948,  1948,  1948,  1948,  1948,     0,     0,     0,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,  1930,
	 1930,  1930,  1930,  1930,  1930,  1930,  1930,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 52
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   511,  1110,    10,  1110,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,   351,   351,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1953,  1954,  1955,  1956,  1957,  1958,  1959,  1960,
	 1961,  1962,  1963,  1964,  1965,  1966,  1967,  1968,
	 1969,  1970,  1971,  1972,  1973,  1974,  1975,  1976,
	 1977,  1978,  1979,  1980,  1981,  1982,  1983,  1984,
	    0,     0,  1985,  1986,  1987,  1988,  1989,  1990,
	 1991,  1992,  1993,  1994,  1995,  1996,   351,   516,
	  761,   761,   761,    15,     0,     0,     0,     0,
	    0,     0,     0,     0,   516,   516,    15,   512,
	 1997,  1998,  1999,  2000,  2001,  2002,  2003,  2004,
	 2005,  2006,  2007,  2008,  2009,  2010,  2011,  2012,
	 2013,  2014,  2015,  2016,  2017,  2018,  2019,  2020,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 53
	   64,    64,    64,    64,    64,    64,    64,    64,
	   64,    64,    64,    64,    64,    64,    64,    64,
	   64,    64,    64,    64,    64,    64,    64,   512,
	  512,   512,   512,   512,   512,   512,   512,   512,
	   64,    64,  2021,  2022,  2023,  2024,  2025,  2026,
	 2027,  2028,  2029,  2030,  2031,  2032,  2033,  2034,
	  142,   142,  2035,  2036,  2037,  2038,  2039,  2040,
	 2041,  2042,  2043,  2044,  2045,  2046,  2047,  2048,
	 2049,  2050,  2051,  2052,  2053,  2054,  2055,  2056,
	 2057,  2058,  2059,  2060,  2061,  2062,  2063,  2064,
	 2065,  2066,  2067,  2068,  2069,  2070,  2071,  2072,
	 2073,  2074,  2075,  2076,  2077,  2078,  2079,  2080,
	 2081,  2082,  2083,  2084,  2085,  2086,  2087,  2088,
	 2089,  2090,  2091,  2092,  2093,  2094,  2095,  2096,
	  511,   142,   142,   142,   142,   142,   142,   142,
	  142,  2097,  2098,  2099,  2100,  2101,  2102,  2103,
	 2104,  2105,  2106,  2107,  2108,  2109,  2110,  2111,
	  512,  2112,  2112,  2113,  2114,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,   351,   351,   351,   351,   351,
// 54
	  351,   351,   516,   351,   351,   351,   516,   351,
	  351,   351,   351,   516,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,  1034,  1034,   516,   516,  1034,
	   97,    97,    97,    97,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,  2115,  2115,    10,    10,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1034,  1034,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,  1034,  1034,  1034,  1034,
	 1034,  1034,  1034,  1034,  1034,  1034,  1034,  1034,
	 1034,  1034,  1034,  1034,   516,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,  1035,  1035,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 55
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   516,   516,
	  516,   516,   516,   516,   516,   516,  1035,  1035,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,  1034,  1034,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,   954,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 56
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   516,   516,   516,   516,   516,   516,  1034,
	 1034,   516,   516,  1034,  1034,   516,   516,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   516,   351,   351,   351,   351,
	  351,   351,   351,   351,   516,  1034,     0,     0,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,   954,  1035,  1035,  1035,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 57
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
// 58
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
// 59
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
// 60
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
// 61
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
// 62
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
// 63
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
// 64
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2116,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2116,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,  2117,  2117,  2117,  2117,
	 2117,  2117,  2117,  2117,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 65
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,     0,     0,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,     0,     0,     0,     0,     0,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 66
	  142,   142,   142,   142,   142,   142,   142,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,   142,   142,   142,   142,   142,
	    0,     0,     0,     0,     0,   998,   516,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,  1581,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,     0,
	  998,   998,   998,   998,   998,     0,   998,     0,
	  998,   998,     0,   998,   998,     0,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
// 67
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,    17,    18,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	    0,     0,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1003,    97,     0,     0,
// 68
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	   35,  1932,  1932,  2120,    35,    10,    10,    17,
	   18,  1566,     0,     0,     0,     0,     0,     0,
	  516,   516,   516,   516,   516,   516,   516,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1933,  1938,  1938,  2121,  2121,    17,    18,    17,
	   18,    17,    18,    17,    18,    17,    18,    17,
	   18,    17,    18,    17,    18,  1933,  1933,    17,
	   18,  1933,  1933,  1933,  1933,  2121,  2121,  2121,
	 2122,  1933,  2123,     0,  2124,  2125,    10,    10,
	 1938,    17,    18,    17,    18,    17,    18,  2126,
	 1933,  1933,  2127,  2128,  2129,  2129,  2129,     0,
	 1933,    13,    14,  1933,     0,     0,     0,     0,
	 1006,  1006,  1006,  1006,  1006,     0,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,  1006,  1006,  1006,
	 1006,  1006,  1006,  1006,  1006,     0,     0,  1578,
// 69
	    0,    10,  1933,  2126,    13,    14,  1933,  2130,
	   17,    18,  1933,  2127,  2122,  2128,  2123,  2131,
	 2132,  2133,  2134,  2135,  2136,  2137,  2138,  2139,
	 2140,  2141,  2125,  2124,  2129,  2129,  2129,    10,
	 1933,  2142,  2143,  2144,  2145,  2146,  2147,  2148,
	 2149,  2150,  2151,  2152,  2153,  2154,  2155,  2156,
	 2157,  2158,  2159,  2160,  2161,  2162,  2163,  2164,
	 2165,  2166,  2167,    17,  1933,    18,  2168,  2121,
	 2168,  2169,  2170,  2171,  2172,  2173,  2174,  2175,
	 2176,  2177,  2178,  2179,  2180,  2181,  2182,  2183,
	 2184,  2185,  2186,  2187,  2188,  2189,  2190,  2191,
	 2192,  2193,  2194,    17,  2129,    18,  2129,    17,
	   18,  1932,    17,    18,  1932,  1576,  2195,  1945,
	 1945,  1945,  1945,  1945,  1945,  1945,  1945,  1945,
	 1947,  2195,  2195,  2195,  2195,  2195,  2195,  2195,
	 2195,  2195,  2195,  2195,  2195,  2195,  2195,  2195,
	 2195,  2195,  2195,  2195,  2195,  2195,  2195,  2195,
	 2195,  2195,  2195,  2195,  2195,  2195,  2195,  2195,
	 2195,  2195,  2195,  2195,  2195,  2195,  2195,  2195,
	 2195,  2195,  2195,  2195,  2195,  2195,  2196,  2196,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,     0,
	    0,     0,   351,   351,   351,   351,   351,   351,
	    0,     0,   351,   351,   351,   351,   351,   351,
	    0,     0,   351,   351,   351,   351,   351,   351,
	    0,     0,   351,   351,   351,     0,     0,     0,
	   96,    13,  2129,  2168,  1930,    13,    13,     0,
	   97,    36,    36,    36,    36,    97,    97,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,  2197,  2197,  2197,  2198,    98,     0,     0,
// 70
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,     0,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,     0,   351,   351,     0,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,     0,     0,     0,     0,     0,
// 71
	 1035,  1110,  1061,     0,     0,     0,     0,  1047,
	 1047,  1047,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,  1047,  1047,     0,     0,     0,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	 2199,  2199,  2199,  2199,  2199,  2199,  2199,  2199,
	 2199,  2199,  2199,  2199,  2199,  2199,  2199,  2199,
	 2199,  2199,  2199,  2199,  2199,  2199,  2199,  2199,
	 2199,  2199,  2199,  2199,  2199,  2199,  2199,  2199,
	 2199,  2199,  2199,  2199,  2199,  2199,  2199,  2199,
	 2199,  2199,  2199,  2199,  2199,  2199,  2199,  2199,
	 2199,  2199,  2199,  2199,  2199,  1048,  1048,  1048,
	 1048,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,  1048,     0,     0,     0,     0,     0,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   516,     0,     0,
// 72
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 73
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,     0,
	 1047,  1047,  1047,  1047,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,  1107,   351,   351,   351,   351,   351,   351,
	  351,   351,  1107,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,     0,  1035,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,     0,     0,     0,     0,
	  351,   351,   351,   351,   351,   351,   351,   351,
	 1035,  1107,  1107,  1107,  1107,  1107,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 74
	 2200,  2201,  2202,  2203,  2204,  2205,  2206,  2207,
	 2208,  2209,  2210,  2211,  2212,  2213,  2214,  2215,
	 2216,  2217,  2218,  2219,  2220,  2221,  2222,  2223,
	 2224,  2225,  2226,  2227,  2228,  2229,  2230,  2231,
	 2232,  2233,  2234,  2235,  2236,  2237,  2238,  2239,
	 2240,  2241,  2242,  2243,  2244,  2245,  2246,  2247,
	 2248,  2249,  2250,  2251,  2252,  2253,  2254,  2255,
	 2256,  2257,  2258,  2259,  2260,  2261,  2262,  2263,
	 2264,  2265,  2266,  2267,  2268,  2269,  2270,  2271,
	 2272,  2273,  2274,  2275,  2276,  2277,  2278,  2279,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,     0,     0,
	 1036,  1037,  1038,  1039,  1040,  1041,  1042,  1043,
	 1044,  1045,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 75
	  998,   998,   998,   998,   998,   998,     0,     0,
	  998,     0,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,     0,   998,
	  998,     0,     0,     0,   998,     0,     0,   998,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 76
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,  2280,  2280,
	 2280,  2280,     0,     0,     0,     0,     0,  1110,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,     0,     0,     0,     0,     0,   996,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 77
	  998,   516,   516,   516,     0,   516,   516,     0,
	    0,     0,     0,     0,   516,   516,   516,   516,
	  998,   998,   998,   998,     0,   998,   998,   998,
	    0,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,   998,   998,   998,   998,
	  998,   998,   998,   998,     0,     0,     0,     0,
	  516,   516,   516,     0,     0,     0,     0,   516,
	 2280,  2280,  2280,  2280,  2280,  2280,  2280,  2280,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 2281,  2281,  2281,  2281,  2281,  2281,  2281,  2281,
	  996,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 78
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,   351,
	  351,   351,   351,   351,   351,   351,   351,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 79
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,  1107,  1107,  1107,  1107,  1107,
	 1107,  1107,  1107,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1035,  1035,  1035,  1035,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 80
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 81
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,     0,
	    0,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,  1046,  1034,   516,
	  516,   516,   760,   760,   760,  1034,  1046,  1046,
	 1046,  1046,  1046,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,   516,   516,   516,   516,   516,
	  516,   516,   516,   760,   760,   516,   516,   516,
	  516,   516,   516,   516,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   516,   516,   516,   516,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,   760,   760,
	  760,   760,   760,   760,   760,   760,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 82
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,   516,   516,   516,    97,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 83
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1047,  1047,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,  1047,  1047,  1047,  1047,  1047,  1047,
	 1047,  1047,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 84
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   142,   142,
	  142,   142,   142,   142,   142,     0,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   595,     0,   595,   595,
	    0,     0,   595,     0,     0,   595,   595,     0,
	    0,   595,   595,   595,   595,     0,   595,   595,
	  595,   595,   595,   595,   595,   595,   142,   142,
	  142,   142,     0,   142,     0,   142,   142,   142,
	  142,   142,   142,   142,     0,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
// 85
	  142,   142,   142,   142,   595,   595,     0,   595,
	  595,   595,   595,     0,     0,   595,   595,   595,
	  595,   595,   595,   595,   595,     0,   595,   595,
	  595,   595,   595,   595,   595,     0,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  595,   595,     0,   595,   595,   595,   595,     0,
	  595,   595,   595,   595,   595,     0,   595,     0,
	    0,     0,   595,   595,   595,   595,   595,   595,
	  595,     0,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
// 86
	  142,   142,   142,   142,   142,   142,   142,   142,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,     0,     0,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,  2282,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,  2282,   142,   142,   142,   142,
	  142,   142,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,  2282,   142,   142,   142,   142,
// 87
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,  2282,   142,   142,
	  142,   142,   142,   142,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,  2282,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,  2282,
	  142,   142,   142,   142,   142,   142,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,  2282,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,  2282,   142,   142,   142,   142,   142,   142,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,   595,   595,   595,   595,   595,   595,   595,
	  595,  2282,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,   142,   142,   142,   142,   142,
	  142,   142,   142,  2282,   142,   142,   142,   142,
	  142,   142,   595,   142,     0,     0,    24,    25,
	   26,    27,    28,    29,    30,    31,    32,    33,
	   24,    25,    26,    27,    28,    29,    30,    31,
	   32,    33,    24,    25,    26,    27,    28,    29,
	   30,    31,    32,    33,    24,    25,    26,    27,
	   28,    29,    30,    31,    32,    33,    24,    25,
	   26,    27,    28,    29,    30,    31,    32,    33,
// 88
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,     0,     0,     0,     0,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,    97,    97,    97,    97,
	   97,    97,    97,    97,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 89
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 90
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,  1935,  1935,
	 1935,  1935,  1935,  1935,  1935,  1935,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 91
	    0,  1579,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	 1579,  1579,  1579,  1579,  1579,  1579,  1579,  1579,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 92
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	  516,   516,   516,   516,   516,   516,   516,   516,
	    0,     0,     0,     0,     0,     0,     0,     0,
	    0,     0,     0,     0,     0,     0,     0,     0,
// 93
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,  2119,  2119,
	 2119,  2119,  2119,  2119,  2119,  2119,     0,     0
};

const uint16_t	CharData::_index_page_list [MAX_NBR_CODE_POINTS >> PAGE_SIZE_L2] = 
{
	// U+000000
	     0,      1,      2,      3,      4,      5,      6,      7,
	0x8000,      8,      9,     10,     11,     12,     13,     14,
	    15,     16,     17,     18,     19, 0x815F,     20,     21,
	    22,     23,     24,     25,     26,     27,     28,     29,
	    30,     31,     32,     33,     34,     35,     36,     37,
	0x82F8,     38, 0x8024,     39,     40,     41,     42,     43,
	    44,     45,     46,     47, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F,     48, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,     49,
	    50, 0x879C, 0x879C, 0x879C,     51, 0x815F,     52,     53,
	    54,     55,     56, 0x8000,     57,     58,     59,     60,
	    61,     62,     63,     57,     58,     59,     60,     61,
	    62,     63,     57,     58,     59,     60,     61,     62,
	    63,     57,     58,     59,     60,     61,     62,     63,
	    57,     58,     59,     60,     61,     62,     63,     57,
	    58,     59,     60,     61,     62,     63,     57,     64,
	0x8846, 0x8846, 0x8846, 0x8846, 0x8846, 0x8846, 0x8846, 0x8846,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x878F,     65,     66, 0x83EE,     67,     68,     69,
	// U+010000
	    70,     71,     72,     73,     74, 0x8000, 0x8000, 0x8000,
	    75,     76,     77, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x815F, 0x815F, 0x815F,     78,     79, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	    80,     81,     82,     83,     84,     85,     86,     87,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	    88, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+020000
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,
	0x878F, 0x878F, 0x878F, 0x878F, 0x878F, 0x878F,     89, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x878F, 0x878F,     90, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+030000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+040000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+050000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+060000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+070000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+080000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+090000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+0A0000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+0B0000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+0C0000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+0D0000
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+0E0000
	    91,     92, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000, 0x8000,
	// U+0F0000
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,     93,
	// U+100000
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,
	0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847, 0x8847,     93
};

const bool	CharData::_table_break_grapheme [GraphemeBreakProp_NBR_ELT] [GraphemeBreakProp_NBR_ELT] = 
{
	{ 1, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1 },
	{ 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1 },
	{ 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0 },
	{ 1, 1, 1, 1, 0, 1, 0, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 0, 1, 0, 0, 0, 1, 0, 0 },
	{ 1, 1, 1, 1, 0, 1, 0, 1, 0, 0, 1, 1 },
	{ 1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 1, 1 },
	{ 1, 1, 1, 1, 0, 1, 0, 1, 0, 0, 1, 1 },
	{ 1, 1, 1, 1, 0, 1, 0, 1, 1, 0, 1, 1 }
};

const bool	CharData::_table_break_word [WordBreakPropTablePrev_NBR_ELT] [WordBreakPropTableNext_NBR_ELT] = 
{
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 0, 0, 0, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 0, 1, 1, 0, 0 },
	{ 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1 },
	{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1 }
};



// End: automatically generated data



}	// namespace unicode
}	// namespace txt
}	// namespace fstb



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
