/*****************************************************************************

        MidiVeloMapper.hpp
        Author: Laurent de Soras, 2022

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law.You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://www.wtfpl.net/ for more details.

*Tab=3***********************************************************************/



#pragma once
#if ! defined (MidiVeloMapper_CODEHEADER_INCLUDED)
#define MidiVeloMapper_CODEHEADER_INCLUDED



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include "midi.h"

#include <algorithm>

#include <cassert>
#include <cmath>



/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



template <int DR>
constexpr double	MidiVeloMapper <DR>::_dynamic_range;



template <int DR>
template <typename V>
double	MidiVeloMapper <DR>::conv_velo_to_gain (V velo) noexcept
{
	assert (velo >= V (0));
	assert (velo <= V (midi::_max_1b));

	const auto     gs = _v2g_mul * double (velo) + _v2g_add;

	return gs * gs;
}



// Returned velocity is always > 0, so it is legal for a Note On.
template <int DR>
uint8_t	MidiVeloMapper <DR>::conv_gain_to_velo (double gain) noexcept
{
	assert (gain >= 0);
	assert (gain <= 1);

	const auto     gs   = sqrt (gain);
	auto           velo = (gs - _v2g_add) / _v2g_mul;
	velo = std::max (velo, 1.0);

	return uint8_t (round (velo));
}



template <int DR>
double	MidiVeloMapper <DR>::conv_gain_to_velo_f (double gain) noexcept
{
	assert (gain >= 0);
	assert (gain <= 1);

	const auto     gs   = sqrt (gain);
	auto           velo = (gs - _v2g_add) / _v2g_mul;
	velo = std::max (velo, 0);

	return velo;
}



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



template <int DR>
double	MidiVeloMapper <DR>::compute_v2g_r () noexcept
{
	return pow (10.0, _dynamic_range / 20.0);
}

template <int DR>
double	MidiVeloMapper <DR>::compute_v2g_add () noexcept
{
	const auto     r = compute_v2g_r ();
	return (midi::_max_1b / sqrt (r) - 1.0) / double (midi::_max_1b - 1);
}

template <int DR>
double	MidiVeloMapper <DR>::compute_v2g_mul () noexcept
{
	const auto     b = compute_v2g_add ();
	return (1.0 - b) / double (midi::_max_1b);
}

template <int DR>
const double	MidiVeloMapper <DR>::_v2g_add = compute_v2g_add ();

template <int DR>
const double	MidiVeloMapper <DR>::_v2g_mul = compute_v2g_mul ();



#endif // MidiVeloMapper_CODEHEADER_INCLUDED



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
