/*****************************************************************************

        MidiVeloMapper.h
        Author: Laurent de Soras, 2022

Maps a velocity to a linear gain in [0 ; 1]

Template parameters:

- DR: velocity dynamic range in dB. This is the ratio between gain at minimum
	velo (1) and unit gain at maximum velo (127). For example with DR = 40 dB,
	Velo   Gain
	  1 -> 0.01
	 64 -> 0.30
	100 -> 0.65
	127 -> 1.00

Reference:
Roger B. Dannenberg,
The Interpretation of MIDI Velocity,
Proceedings of the 2006 International Computer Music Conference, pp. 193-196

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law.You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://www.wtfpl.net/ for more details.

*Tab=3***********************************************************************/



#pragma once
#if ! defined (MidiVeloMapper_HEADER_INCLUDED)
#define MidiVeloMapper_HEADER_INCLUDED



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include <cstdint>



template <int DR = 40>
class MidiVeloMapper
{
	static_assert (DR > 0, "DR must be > 0");

/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

public:

	static constexpr double _dynamic_range = double (DR);

	template <typename V>
	static double  conv_velo_to_gain (V velo) noexcept;

	static uint8_t conv_gain_to_velo (double gain) noexcept;
	static double  conv_gain_to_velo_f (double gain) noexcept;



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

protected:



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

private:

	static double  compute_v2g_r () noexcept;
	static double  compute_v2g_add () noexcept;
	static double  compute_v2g_mul () noexcept;

	static const double  _v2g_add;
	static const double  _v2g_mul;


/*\\\ FORBIDDEN MEMBER FUNCTIONS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

private:

	               MidiVeloMapper ()                               = delete;
	               MidiVeloMapper (const MidiVeloMapper &other)    = delete;
	               MidiVeloMapper (MidiVeloMapper &&other)         = delete;
	MidiVeloMapper &
	               operator = (const MidiVeloMapper &other)        = delete;
	MidiVeloMapper &
	               operator = (MidiVeloMapper &&other)             = delete;
	bool           operator == (const MidiVeloMapper &other) const = delete;
	bool           operator != (const MidiVeloMapper &other) const = delete;

}; // class MidiVeloMapper



#include "MidiVeloMapper.hpp"



#endif // MidiVeloMapper_HEADER_INCLUDED



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
