/*****************************************************************************

        MidiState.h
        Author: Laurent de Soras, 2022

Use:

reset ();
while (x as new byte from a midi stream)
{
	push_byte (x);
	if (is_sysex_ready ())
	{
		get_sysex (m);
		// do something with m
	}
	if (is_message_ready ())
	{
		get_message (m);
		// do something with m
	}
}

There can be both SysEx and standard messages after pushing a byte, for
example when a status command without parameter terminates a SysEx.

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*Tab=3***********************************************************************/



#pragma once
#if ! defined (MidiState_HEADER_INCLUDED)
#define	MidiState_HEADER_INCLUDED



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include <vector>

#include <cstdint>



class MidiState
{

/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

public:

	void           reset ();
	void           push_byte (uint8_t val);

	bool           is_message_ready () const noexcept;
	void           get_message (std::vector <uint8_t> &msg) const;
	bool           is_sysex_ready () const noexcept;
	void           get_sysex (std::vector <uint8_t> &msg) const;
	bool           is_stream_illformed () const noexcept;
	uint8_t        get_running_status () const noexcept;

	static bool    is_status (uint8_t x) noexcept;
	static bool    is_channel (uint8_t categ) noexcept;
	static bool    is_realtime (uint8_t s) noexcept;
	static int     get_param_len (uint8_t s) noexcept;
	static uint8_t extract_channel (uint8_t s) noexcept;
	static uint8_t extract_cmd_categ (uint8_t s) noexcept;



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

protected:



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

private:

	static uint8_t update_running_status (uint8_t status) noexcept;

	uint8_t        _status = 0; // >= 0x80, 0 = currently unknown or not set
	uint8_t        _msg_rt = 0; // Real-time message. 0 = no message
	int            _length = 0; // Number of expected parameters, when known

	// Parameters (without the status). Can be a whole SysEx message
	std::vector <uint8_t>
	               _data;

	// Stores the sysex content, when the sysex is terminated.
	std::vector <uint8_t>
	               _sysex;

	// Indicates a System Exlcusive message is available. It can occur the
	// same time as a standard message (most likely a system realtime one).
	bool           _sysex_ready_flag = false;

	// Indicates that data bytes were received without any running status,
	// or a SysEx end was received without SysEx context
	bool           _illformed_stream_flag = false;



/*\\\ FORBIDDEN MEMBER FUNCTIONS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

private:

#if 0
	               MidiState ()                               = delete;
	               MidiState (const MidiState &other)         = delete;
	               MidiState (MidiState &&other)              = delete;
	MidiState &    operator = (const MidiState &other)        = delete;
	MidiState &    operator = (MidiState &&other)             = delete;
#endif
	bool           operator == (const MidiState &other) const = delete;
	bool           operator != (const MidiState &other) const = delete;

};	// class MidiState



//#include "MidiState.hpp"



#endif	// MidiState_HEADER_INCLUDED



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
