/*****************************************************************************

        MidiFileTrack.cpp
        Author: Laurent de Soras, 2022

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*Tab=3***********************************************************************/



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include "MidiFile.h"

#include <stdexcept>

#include <cassert>
#include <cstddef>



/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



// The whole file should be loaded in memory.
// end_ptr sets the limit of the valid memory area provided by the caller.
// It is either the end or past the end of the file, or the end of the valid
// input buffer if the file was truncated for some reason.
// Throws: std::runtime_error if the file cannot be fully decoded.
MidiFile::MidiFile (const uint8_t *mthd_ptr, const uint8_t *end_ptr, MidiFileTrack::NoteRepeat rep_strategy)
{
	const auto     len_avail = ptrdiff_t (end_ptr - mthd_ptr);

	if (len_avail < 8)
	{
		throw std::overflow_error ("Truncated MThd chunk.");
	}
	if (   mthd_ptr [0] != uint8_t ('M')
	    || mthd_ptr [1] != uint8_t ('T')
	    || mthd_ptr [2] != uint8_t ('h')
	    || mthd_ptr [3] != uint8_t ('d'))
	{
		throw std::runtime_error ("Not an MThd chunk");
	}
	const auto     len = ptrdiff_t (
		  (int32_t (mthd_ptr [4]) << 24)
		| (int32_t (mthd_ptr [5]) << 16)
		| (int32_t (mthd_ptr [6]) <<  8)
		|  int32_t (mthd_ptr [7])
	);
	if (len < 6)
	{
		throw std::runtime_error ("MThd content is too short");
	}

	const auto     payload_ptr = mthd_ptr + 8;
	if (end_ptr - payload_ptr < len)
	{
		throw std::overflow_error ("Truncated MThd payload");
	}
	auto           track_ptr = payload_ptr + len;

	const auto     format = (uint16_t (payload_ptr [0]) << 8) | payload_ptr [1];
	if (format > 2)
	{
		throw std::runtime_error ("Unknown MIDI file format");
	}
	_format = Format (format);

	const auto     nbr_trk = (int (payload_ptr [2]) << 8) | payload_ptr [3];
	if (_format == Format::_single && nbr_trk != 1)
	{
		throw std::runtime_error ("Multiple tracks found in single track mode");
	}

	const auto     div_msb = int8_t (payload_ptr [4]);
	const auto     div_lsb = payload_ptr [5];
	if (div_msb >= 0)
	{
		_time_info._nbr_ticks = (int (div_msb) << 8) | div_lsb;
	}
	else
	{
		TimeSmpte      smpte;
		switch (div_msb)
		{
		case -24: // Film
		case -25: // PAL
		case -30: // ATSC
			smpte._fps_num = -div_msb;
			break;
		case -29: // NTSC
			smpte._fps_num = 30000;
			smpte._fps_den = 1001;
			break;
		default:
			throw std::runtime_error ("Illegal SMPTE framerate");
		}
		_time_info._smpte = smpte;

		if (div_lsb == 0)
		{
			throw std::runtime_error ("SMPTE frame resolution cannot be null");
		}
		_time_info._nbr_ticks = div_lsb;
	}

	// Tracks
	while (int (_track_arr.size ()) < nbr_trk && track_ptr < end_ptr)
	{
		_track_arr.emplace_back (std::make_unique <MidiFileTrack> (
			track_ptr, end_ptr, rep_strategy
		));
	}
	if (int (_track_arr.size ()) < nbr_trk)
	{
		throw std::runtime_error ("Missing track(s)");
	}
}



MidiFile::Format	MidiFile::get_format () const noexcept
{
	return _format;
}



MidiFile::TimeInfo	MidiFile::get_time_info () const noexcept
{
	return _time_info;
}



int	MidiFile::get_nbr_tracks () const noexcept
{
	return int (_track_arr.size ());
}



const MidiFileTrack &	MidiFile::use_track (int trk_idx) const noexcept
{
	assert (trk_idx >= 0);
	assert (trk_idx < get_nbr_tracks ());

	return *(_track_arr [trk_idx]);
}



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



uint8_t	MidiFile::read_byte (const uint8_t * &cur_ptr, const uint8_t *end_ptr)
{
	if (cur_ptr >= end_ptr)
	{
		throw std::overflow_error ("Too few bytes in the stream");
	}
	const auto     val = *cur_ptr;
	++ cur_ptr;

	return val;
}



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
