/*****************************************************************************

        GRAOUMF TRACKER 2
        Author: Laurent de Soras, 1996-2016

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*****************************************************************************/



/*\\\ FICHIERS INCLUDE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include	"archi.h"
#include	"base.h"
#include	"base_ct.h"
#include	"Directory.h"
#include	"file.h"
#include	"file_ct.h"
#include	"fnames.h"
#include	"log.h"
#include	"memory.h"

#include <algorithm>
#include <tuple>

#include	<cassert>
#include	<cstdio>
#include	<cstdlib>
#include	<cstring>



/*\\\ CONSTANTES PUPBLIQUES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

const int	Directory::SORT_MASK         = 0x0F;
const int	Directory::SORT_NONE         = 0x00;
const int	Directory::SORT_BY_NAME      = 0x01;
const int	Directory::SORT_BY_EXTENSION = 0x02;
const int	Directory::SORT_BY_DATE      = 0x03;
const int	Directory::SORT_BY_SIZE      = 0x04;
const int	Directory::SORT_INVERTED     = 0x10;	/* Flag d'inversion du tri */



/*\\\ CONSTANTES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ TYPES & STRUCTURES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ DEFINITION DES VARIABLES DE CLASSE PUBLIQUES \\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ DEFINITION DES VARIABLES DE CLASSE PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ METHODES PUBLIQUES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*==========================================================================*/
/*      Nom: check_ok                                                       */
/*      Description: Verifie l'intergrite de l'objet.                       */
/*      Retour: 0 si l'objet parait correct.                                */
/*==========================================================================*/

int	Directory::check_ok () const
{
	return (0);
}



/*==========================================================================*/
/*      Nom: self_display                                                   */
/*      Description: Affiche "textuellement" le contenu courant de l'objet. */
/*==========================================================================*/

void	Directory::self_display () const
{

	/*** A faire ***/

}



/*==========================================================================*/
/*      Nom: build                                                          */
/*      Description: Lit le repertoire specifie et construit l'objet avec   */
/*                   les informations recuperees.                           */
/*      Parametres en entree:                                               */
/*        - dir_path_0: pointeur sur le chemin du repertoire a charger.     */
/*        - mask_0: poitneur sur le masque de filtrage.                     */
/*      Parametres en entree/sortie:                                        */
/*      Retour: 0 si tout s'est bien passe.                                 */
/*==========================================================================*/

int	Directory::build (const char *dir_path_0, const char *mask_0)
{
	int            ret_val = 0;

	if (clear ())
	{
		LOG_printf ("Directory::build: Error: couldn't clear directory.\n");
		ret_val = -1;
	}

	while (ret_val == 0)
	{
		ret_val = add_file (dir_path_0, mask_0);
		if (ret_val == 1)
		{
			ret_val = 0;
			break;
		}
	}

	return (ret_val);
}



/*==========================================================================*/
/*      Nom: clear                                                          */
/*      Description: Efface le contenu courant du repertoire.               */
/*      Retour: 0 si tout s'est bien passe.                                 */
/*==========================================================================*/

int	Directory::clear ()
{
	_file_info_arr.clear ();
	_first_file_flag = true;

	return 0;
}



/*==========================================================================*/
/*      Nom: sort                                                           */
/*      Description: Trie les fichiers du repertoire selon un critere       */
/*                   donne.                                                 */
/*      Parametres en entree:                                               */
/*        - sort_type: type du tri.                                         */
/*==========================================================================*/

void		Directory::sort (int sort_type)
{
	const int      inv = (sort_type & Directory::SORT_INVERTED) ? -1 : 1;

	switch (sort_type & Directory::SORT_MASK)
	{
	case	Directory::SORT_BY_NAME:
		std::sort (_file_info_arr.begin (), _file_info_arr.end (),
			[inv] (const FILE_DIR_ENTRY &lhs, const FILE_DIR_ENTRY &rhs)
			{
				return (std::make_tuple (
					! lhs.is_dir (),
					inv * BASE_compare_string (lhs.name.c_str (), rhs.name.c_str ()),
					inv * lhs.timestamp (),
					inv * lhs.length
				) < std::make_tuple (
					! rhs.is_dir (),
					0,
					inv * rhs.timestamp (),
					inv * rhs.length
				));
			}
		);
		break;

	case	Directory::SORT_BY_EXTENSION:
		std::sort (_file_info_arr.begin (), _file_info_arr.end (),
			[inv] (const FILE_DIR_ENTRY &lhs, const FILE_DIR_ENTRY &rhs)
			{
				const auto     l_pos = lhs.name.rfind ('.');
				const auto     r_pos = rhs.name.rfind ('.');
				const auto     l_str = lhs.name.substr (l_pos) + lhs.name.substr (0, l_pos);
				const auto     r_str = rhs.name.substr (r_pos) + rhs.name.substr (0, r_pos);
				return (std::make_tuple (
					! lhs.is_dir (),
					inv * BASE_compare_string (l_str.c_str (), r_str.c_str ()),
					inv * lhs.timestamp (),
					inv * lhs.length
				) < std::make_tuple (
					! rhs.is_dir (),
					0,
					inv * rhs.timestamp (),
					inv * rhs.length
				));
			}
		);
		break;

	case	Directory::SORT_BY_DATE:
		std::sort (_file_info_arr.begin (), _file_info_arr.end (),
			[inv] (const FILE_DIR_ENTRY &lhs, const FILE_DIR_ENTRY &rhs)
			{
				return (std::make_tuple (
					! lhs.is_dir (),
					inv * lhs.timestamp (),
					inv * BASE_compare_string (lhs.name.c_str (), rhs.name.c_str ()),
					inv * lhs.length
				) < std::make_tuple (
					! rhs.is_dir (),
					inv * rhs.timestamp (),
					0,
					inv * rhs.length
				));
			}
		);
		break;

	case	Directory::SORT_BY_SIZE:
		std::sort (_file_info_arr.begin (), _file_info_arr.end (),
			[inv] (const FILE_DIR_ENTRY &lhs, const FILE_DIR_ENTRY &rhs)
			{
				return (std::make_tuple (
					! lhs.is_dir (),
					inv * lhs.length,
					inv * BASE_compare_string (lhs.name.c_str (), rhs.name.c_str ()),
					inv * lhs.timestamp ()
				) < std::make_tuple (
					! rhs.is_dir (),
					inv * rhs.length,
					0,
					inv * rhs.timestamp ()
				));
			}
		);
		break;
	}
}



/*==========================================================================*/
/*      Nom: get_length                                                     */
/*      Description: renvoie le nombre de fichiers du repertoire.           */
/*      Retour: Nombre de fichiers.                                         */
/*==========================================================================*/

int		Directory::get_length () const
{
	return int (_file_info_arr.size ());
}



/*==========================================================================*/
/*      Nom: get_file_info                                                  */
/*      Description: Renvoie les informations d'un fichier donne.           */
/*      Parametres en entree:                                               */
/*        - position: numero du fichier dans le repertoire (0 a ...)        */
/*      Retour: reference sur le descripteur du fichier.                    */
/*==========================================================================*/

const FILE_DIR_ENTRY	&Directory::get_file_info (int position) const
{
	assert (position >= 0);
	assert (position < get_length ());

	return (_file_info_arr [position]);
}



/*==========================================================================*/
/*      Nom: get_selected_flag                                              */
/*      Description: Renvoie l'etat de selection d'un fichier dans le       */
/*                   repertoire.                                            */
/*      Parametres en entree:                                               */
/*        - position: numero du fichier dans le repertoire (0 a ...)        */
/*      Retour: true si le fichier est selectionne, false sinon.            */
/*==========================================================================*/

bool		Directory::get_selected_flag (int position) const
{
	assert (position >= 0);
	assert (position < get_length ());

	return (_file_info_arr [position].selected);
}



/*==========================================================================*/
/*      Nom: set_selected_flag                                              */
/*      Description: Fixe le flag de selection d'un fichier dans le         */
/*                   repertoire.                                            */
/*      Parametres en entree:                                               */
/*        - position: numero du fichier dans le repertoire (0 a ...)        */
/*        - state_flag: true si on doit selectionner le fichier, false si   */
/*                      on doit le deselectionner.                          */
/*==========================================================================*/

void		Directory::set_selected_flag (int position, bool state_flag)
{
	assert (position >= 0);
	assert (position < get_length ());

	_file_info_arr [position].selected = state_flag;
}



/*\\\ METHODES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*==========================================================================*/
/*      Nom: add_file                                                       */
/*      Description: Ajoute un fichier au repertoire.                       */
/*      Parametres en entree:                                               */
/*        - dir_path_0: pointeur sur le chemin du repertoire a charger.     */
/*        - mask_0: poitneur sur le masque de filtrage.                     */
/*      Retour: 0 si tout s'est bien passe,                                 */
/*              1 si on a fini de lister le repertoire,                     */
/*              -1 si erreur.                                               */
/*==========================================================================*/

int	Directory::add_file (const char *dir_path_0, const char *mask_0)
{
	FILE_DIR_ENTRY	temp_file_info;

	/* Premier fichier */
	if (_first_file_flag)
	{
		std::string    full_path = dir_path_0;
		full_path += "*.*";
		_first_file_flag = false;
		if (! FILE_get_first_file (full_path.c_str (), &temp_file_info))
		{
			return 1;
		}
	}

	/* Fichiers suivants */
	else
	{
		if (! FILE_get_next_file (&temp_file_info))
		{
			return 1;
		}
	}

	/* Ces deux repertoires virtuels ne doivent pas figurer dans la liste */
	if (   temp_file_info.name == "."
	    || temp_file_info.name == "..")
	{
		return 0;
	}

	/* Teste le fichier pour voir s'il est conforme au masque */
	if ((temp_file_info.attrib & FILE_attrib_dir) == 0)
	{
		if (! FNAM_test_filename (temp_file_info.name.c_str (), mask_0))
		{
			return 0;
		}
	}

	/* Ajoute l'entree dans le repertoire */
	_file_info_arr.push_back (temp_file_info);

	return 0;
}



/****************************************************************************/
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

