/*****************************************************************************

        GRAOUMF TRACKER 2
        Author: Laurent de Soras, 1996-2016

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*****************************************************************************/



/*\\\ FICHIERS INCLUDE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<math.h>

#include	"archi.h"
#include	"base.h"
#include	"base_ct.h"
#include	"EditString.h"
#include	"edstring.h"
#include	"log.h"
#include	"intrface.h"



/*\\\ CONSTANTES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ TYPES & STRUCTURES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PROTOTYPES DES FONCTIONS PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ VARIABLES EXTERNES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ VARIABLES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ FONCTIONS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*==========================================================================*/
/*      Nom: EDIT_edit_string                                               */
/*      Description: Edite une chaine de caracteres.                        */
/*      Parametres en entree:                                               */
/*        - object: numero de l'objet (toujours EDBOXTEXT) qui doit servir  */
/*                  de support a l'edition.                                 */
/*        - max_len: longueur maximum du texte editable (sans le 0). Peut   */
/*                   etre superieure a la longueur de la chaine de          */
/*                   l'EDBOXTEXT. Si le type de la chaine est MSC, maxlen   */
/*                   n'est pas pris en compte. En mode FIX, maxlen doit     */
/*                   etre > 1 si le point est dans la chaine (au moins un   */
/*                   caractere).                                            */
/*        - type: Type de chaine a editer.                                  */
/*      Parametres en entree/sortie:                                        */
/*        - text_0: pointeur sur la chaine a editer, se termine par un 0.   */
/*                  Une quantite de memoire suffisante (au moins maxlen +   */
/*                  1) aura deja du etre reservee.                          */
/*      Retour: 0 si tout s'est bien passe, -1 s'il y a eu un probleme, ou  */
/*              -2 si l'utilisateur a annule (text_0 est alors inchange).   */
/*==========================================================================*/

signed int	EDIT_edit_string (char *text_0, int object, int max_len, int type)
{
	EditString	ed_str (text_0, object, max_len, type);

	if (ed_str.check_ok ())
	{
		LOG_printf ("EDIT_edit_string: Error: EditString oject is not valid.\n");
		return (-1);
	}

	return (ed_str.manage ());
}



/*==========================================================================*/
/*      Nom: EDIT_edit_string_base                                          */
/*      Description: Edite une chaine de caracteres, et plus                */
/*                   particulierement une valeur numerique entiere.         */
/*      Parametres en entree:                                               */
/*        - object: numero de l'objet (toujours EDBOXTEXT) qui doit servir  */
/*                  de support a l'edition.                                 */
/*        - max_len: longueur maximum du texte editable (sans le 0). Peut   */
/*                   etre superieure a la longueur de la chaine de          */
/*                   l'EDBOXTEXT.                                           */
/*        - base: base de la valeur a editer, 10 ou 16. Si base vaut une    */
/*                autre valeur, l'edition de la chaine est alphanumerique.  */
/*      Parametres en entree/sortie:                                        */
/*        - text_0: pointeur sur la chaine a editer, se termine par un 0.   */
/*                  Une quantite de memoire suffisante (au moins maxlen +   */
/*                  1) aura deja du etre reservee.                          */
/*      Retour: 0 si tout s'est bien passe, -1 s'il y a eu un probleme, ou  */
/*              -2 si l'utilisateur a annule (text_0 est alors inchange).   */
/*==========================================================================*/

signed int	EDIT_edit_string_base (char *text_0, int object, int max_len, int base)
{
	switch (base)
	{
	case	10:
		return (EDIT_edit_string (text_0, object, max_len, EditString_TYPE_DEC));
		break;

	case	16:
		return (EDIT_edit_string (text_0, object, max_len, EditString_TYPE_HEXA));
		break;

	default:
		return (EDIT_edit_string (text_0, object, max_len, EditString_TYPE_ALPHA));
		break;

	}
}



/*==========================================================================*/
/*      Nom: EDIT_edit_splpos                                               */
/*      Description: Edite une position/longueur de sample suivant l'unite  */
/*                   choisie dans les preferences.                          */
/*      Parametres en entree:                                               */
/*        - object: numero de l'objet (toujours EDBOXTEXT) qui doit servir  */
/*                  de support a l'edition.                                 */
/*        - spl_pos_ptr: pointeur sur la position en samples.               */
/*        - freq_ech: frequence d'echantillonnage (en samples/seconde)      */
/*        - nbr_chan: nombre de cannaux du sample (1, 2 ou plus)            */
/*        - nbr_bytes: nombre d'octets par samples (1 ou 2)                 */
/*      Retour: 0 si tout s'est bien passe, -1 si probleme, ou -2 si        */
/*              l'utilisateur a annule (*spl_pos_ptr alors inchange)        */
/*==========================================================================*/

signed int	EDIT_edit_splpos (int object, unsigned long *spl_pos_ptr, ULWORD freq_ech, int nbr_chan, int nbr_bytes)
{
	ULWORD	minutes;
	ULWORD	secondes;
	ULWORD	centiemes;
	ULWORD	temp;
	signed int	result;
	int		edit_type [] =
	{
		EditString_TYPE_HEXA,
		EditString_TYPE_DEC,
		EditString_TYPE_MSC,
		EditString_TYPE_FIX
	};
	char		string_0 [8+1];

	INTR_splpos_2_string (string_0, *spl_pos_ptr, freq_ech, nbr_chan, nbr_bytes);
	result = EDIT_edit_string (string_0, object, 8, edit_type [INTR_splpos_type]);
	if (result < 0)
	{
		return (result);	/* Erreur ou annulation */
	}

	switch (INTR_splpos_type)
	{
	case	INTR_SPLPOS_TYPE_HEXA:
		temp = strtol (string_0, NULL, 16);
		if (INTR_splpos_byte_flag)
		{
			temp /= (nbr_chan * nbr_bytes);
		}
		*spl_pos_ptr = temp;
		break;

	case	INTR_SPLPOS_TYPE_DECI:
		temp = atol (string_0);
		if (INTR_splpos_byte_flag)
		{
			temp /= (nbr_chan * nbr_bytes);
		}
		*spl_pos_ptr = temp;
		break;

	case	INTR_SPLPOS_TYPE_MSC:
		string_0 [2] = 0;
		string_0 [5] = 0;
		minutes = atol (string_0);
		secondes = atol (string_0 + 3);
		centiemes = atol (string_0 + 6);
		*spl_pos_ptr = (ULWORD) (  (minutes * 60 + secondes) * freq_ech
										 + ceil ((double)(centiemes * freq_ech) / 100));
		break;

	case	INTR_SPLPOS_TYPE_SM:
		*spl_pos_ptr = (ULWORD) (ceil (atof (string_0) * freq_ech));
		break;

	}

	return (0);
}



/*==========================================================================*/
/*      Nom: EDIT_edit_time                                                 */
/*      Description: Edite un temps selon le mode de representation choisi. */
/*      Parametres en entree:                                               */
/*        - object: numero de l'objet (toujours EDBOXTEXT) qui doit servir  */
/*                  de support a l'edition.                                 */
/*        - unit: unite de representation du temps.                         */
/*        - tempo: tempo pour la representation sous forme de fraction de   */
/*                 beat. -1 = tempo courant.                                */
/*      Parametres en entree/sortie:                                        */
/*        - sec: temps, en secondes. Ce parametre est inchange en cas       */
/*               d'annulation.                                              */
/*      Retour: 0 si tout s'est bien passe, -1 si probleme, ou -2 si        */
/*              l'utilisateur a annule (sec reste alors inchange).          */
/*==========================================================================*/

signed int	EDIT_edit_time (int object, int unit, double &sec, double tempo)
{
	signed int	result;
	char		string_0 [15+1];

	INTR_time_2_string (string_0, unit, sec, 15, tempo);
	BASE_trim_string (string_0);

	switch (unit)
	{

	case	INTR_TIME_UNIT_SEC:
	case	INTR_TIME_UNIT_MSEC:
	case	INTR_TIME_UNIT_SPL:
	case	INTR_TIME_UNIT_HZ:
		result = EDIT_edit_string (string_0, object, 15, EditString_TYPE_FLO);
		break;

	case	INTR_TIME_UNIT_BEATS:
	case	INTR_TIME_UNIT_NOTE:
		result = EDIT_edit_string (string_0, object, 15, EditString_TYPE_ALPHA);
		break;
	
	}

	if (result < 0)
	{
		return (result);
	}

	INTR_string_2_time (string_0, unit, sec, tempo);

	return (0);
}



/****************************************************************************/
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

