// foobar2000 component for playing hes file

#include "foo_input_hes.h"
#include "loadpe/loadpe.h"
#include "cfg_int_t_atomic.h"
//#include "stdafx.h"

#define DLL_NAME "HES_PLAYER.DLL"
#define HES_HEADER_MAGIC 0x4D534548

const unsigned int			input_hes::m_update_sample = 588;
const unsigned int			input_hes::m_channels = 2;
const unsigned int			input_hes::m_bps = 16;
const int					input_hes::m_silence_level = 8;	//approximately equal to 72dB on 16bps

typedef signed short hes_sample_t;


// Preferences variable
extern cfg_int cfg_play_infinitely;
extern cfg_int cfg_default_song_len;
extern cfg_int cfg_default_fade_len;
extern cfg_int cfg_default_loop_count;
extern cfg_int cfg_detect_silence;
extern cfg_int cfg_silence_len;
extern cfg_int cfg_ignore_playlist;
extern cfg_int cfg_default_subsong_max;
extern cfg_int cfg_volume;
extern cfg_int cfg_display_track_num;
extern cfg_uint_atomic cfg_channel_muting;
extern cfg_int cfg_sample_rate;
extern cfg_int cfg_support_hes;


bool input_hes::g_is_our_path(const char *p_path, const char *p_extension)
{
	return (::stricmp_utf8(p_extension, "hes") == 0 && cfg_support_hes > 0);
}

// Load hes_player.dll
bool input_hes::loaddll()
{
	if (m_dll_hnd) {
		return true;
	}

	pfc::string8 path = core_api::get_my_full_path();
	path.truncate(path.scan_filename());
	path += DLL_NAME;
	p_HESPlayerSetUp HESPlayerSetUp = 0;

#ifdef _DEBUG
	m_dll_hnd = uLoadLibrary(path);	//fobO͕XbhōĐĂ͂ȂB
#else
	m_dll_hnd = XLoadLibraryW(pfc::stringcvt::string_wide_from_utf8(path));
#endif

	
	if (m_dll_hnd)
	{
#ifdef _DEBUG
		HESPlayerSetUp = (p_HESPlayerSetUp)GetProcAddress(m_dll_hnd, "HESPlayerSetUp");
#else
		HESPlayerSetUp = (p_HESPlayerSetUp)XGetProcAddress(m_dll_hnd, "HESPlayerSetUp");
#endif
		
		if (HESPlayerSetUp)
		{
			m_api = (*HESPlayerSetUp)();
		}
	}

	if (!m_dll_hnd) {
		console::info("Could not locate HES_PLAYER.DLL.");
		return false;
	}

	if (!m_api) 
	{
		console::info("HES_PLAYER.DLL function lookups failed.");
		return false;
	}

	return true;
}

// Free hes_player.dll
void input_hes::freedll()
{
	if (!m_dll_hnd) {
		return;
	}

	if (m_dll_hnd) {
#ifdef _DEBUG
		FreeLibrary(m_dll_hnd);
#else
		XFreeLibrary(m_dll_hnd);
#endif	
		m_dll_hnd = 0;
	}
}

// Destructor
input_hes::~input_hes()
{
	if (m_dll_hnd && m_api)
		m_api->p_HES_Deinit();
	freedll();
}

static t_uint16 read_le16(const t_uint8* p_data)
{
	return (p_data[0x01] << 8) | (p_data[0x00] << 0);
}

static t_uint32 read_le32(const t_uint8* p_data)
{
	return	(p_data[0x03] << 24) | (p_data[0x02] << 16) | (p_data[0x01] << 8) | (p_data[0x00] << 0);
}

// Open file
void input_hes::open(service_ptr_t<file> p_filehint, const char *p_path, t_input_open_reason p_reason, abort_callback &p_abort)
{
	if (p_reason == input_open_info_write)
	{
		throw exception_io_unsupported_format(); //our input does not support retagging.
	}

	m_file = p_filehint;	
	input_open_file_helper(m_file, p_path, p_reason, p_abort);

	// read input file
	m_file_len = m_file->get_size(p_abort);
	if (m_file_len <= 0x20)
		throw exception_io_data("Invalid input file");	//hes's header size is 0x20 byte.
	m_file_buf_len = t_size(m_file_len + 16);
	m_file_buf.set_size(m_file_buf_len);
	m_file->read(m_file_buf.get_ptr(), m_file_len, p_abort);

	// check hes's header
	if (!(read_le32(m_file_buf.get_ptr()) == HES_HEADER_MAGIC || m_file_len > 0x220 && read_le32(m_file_buf.get_ptr() + 0x200) == HES_HEADER_MAGIC))
		throw exception_io_data("Invalid input file");


	// load settings
	m_first_subsong			= static_cast<unsigned int>(m_file_buf[5]);
	m_play_infinitely		= cfg_play_infinitely ? true : false;;
	m_total_subsong			= cfg_default_subsong_max + 1;
	m_sample_rate			= cfg_sample_rate > 0 ? cfg_sample_rate : 44100;
	m_default_song_len		= cfg_default_song_len;
	m_default_fade_len		= cfg_default_fade_len;
	m_default_loop_count	= cfg_default_loop_count;
	m_detect_silence		= cfg_detect_silence ? true : false;
	m_silence_len			= cfg_silence_len;
	m_display_track_num		= cfg_display_track_num ? true : false;
	m_volume				= static_cast<double>(cfg_volume) * .01;
	m_exm3u_ignore			= cfg_ignore_playlist ? true : false;


	//load extended m3u if it exists
	if (!m_exm3u_ignore)
	{
		pfc::string_replace_extension exm3u8_path(p_path, "m3u8");
		int exm3u_ret = 1;
		pfc::array_t<t_uint8> exm3u_buf;
		int exm3u_len;

		if (filesystem::g_exists(exm3u8_path, p_abort))
		{
			filesystem::g_open(p_filehint, exm3u8_path, filesystem::open_mode_read, p_abort);
			exm3u_len = static_cast<int>(p_filehint->get_size(p_abort));
			exm3u_buf.set_size(exm3u_len + 16);
			p_filehint->read(exm3u_buf.get_ptr(), exm3u_len, p_abort);
			exm3u_ret = m_exm3u.load(exm3u_buf.get_ptr(), exm3u_len, true);
		}

		if (exm3u_ret != 0)
		{
			pfc::string_replace_extension exm3u_path(p_path, "m3u");
			if (filesystem::g_exists(exm3u_path, p_abort))
			{
				filesystem::g_open(p_filehint, exm3u_path, filesystem::open_mode_read, p_abort);
				exm3u_len = static_cast<int>(p_filehint->get_size(p_abort));
				exm3u_buf.set_size(exm3u_len + 16);
				p_filehint->read(exm3u_buf.get_ptr(), exm3u_len, p_abort);
				exm3u_ret = m_exm3u.load(exm3u_buf.get_ptr(), exm3u_len, false);
			}
		}

		if (exm3u_ret == 0)
		{
			//console::formatter() << "load m3u";
			m_exm3u_load = 1;
			m_exm3u_utf8_enc = m_exm3u.get_utf8_flag();
			m_exm3u_album_info = m_exm3u.info();
			m_exm3u_line = m_exm3u.size();
			if (m_exm3u_line < 0) m_exm3u_line = 0;
		}
		else if (exm3u_ret == -1)
		{
			m_exm3u_load = -1;
			console::info("m3u load error !");
		}
	}
}

// Check metadata is empty or not and metadata's encoding
inline void input_hes::check_encode_meta_set(file_info &p_info, const char* name, const char* value)
{
	if (!value || '\0' == value[0] || !name)
		return;

	if (m_exm3u_utf8_enc)
		p_info.meta_add(name, value);
	else
		p_info.meta_add(name, pfc::stringcvt::string_utf8_from_ansi(value));
}

void input_hes::check_encode_meta_set_multi_field(file_info &p_info, const char* name, const char* value)
{
	if (!value || '\0' == value[0] || !name)
		return;

	const char* walk = 0;
	pfc::stringcvt::string_utf8_from_ansi converter;
	if (m_exm3u_utf8_enc)
		walk = value;
	else
	{
		converter.convert(value);
		walk = converter.get_ptr();
	}

	if (!walk)
		return;

	const char* begin = walk;
	t_size len = 0;
	while (1)
	{
		if (walk[0] == '\0' || (walk[0] == ',' && walk[1] == ' '))
		{
			if (len > 0)
				p_info.meta_add_ex(name, ~0, begin, len);
				
			if (walk[0] == '\0') break;
			len = 0;
			walk += 2;
			begin = walk;
			continue;
		}
		len++;
		walk++;
	}
}

// Get file information 
void input_hes::get_info(t_uint32 p_subsong, file_info &p_info, abort_callback &p_abort)
{
	if (!m_set_len)
	{
		//get play time information from m_exm3u if it exist
		if (m_exm3u_load == 1 && p_subsong < static_cast<unsigned int>(m_exm3u_line))
		{
			int exm3u_intro = m_exm3u[p_subsong].intro;
			int exm3u_loop = m_exm3u[p_subsong].loop;
			int exm3u_length = m_exm3u[p_subsong].length;
			int exm3u_repeat = m_exm3u[p_subsong].repeat;
			int exm3u_fade = m_exm3u[p_subsong].fade;

			if (exm3u_intro != -1 && exm3u_loop != -1)
			{
				if (exm3u_repeat > 0)
				{
					m_song_len = exm3u_intro + exm3u_loop * exm3u_repeat;
				}
				else if (exm3u_repeat == 0)
				{
					m_song_len = exm3u_intro + exm3u_loop;
					m_play_infinitely = true;
				}
				else if (exm3u_repeat == -1)
				{
					if (m_default_loop_count > 0)  m_song_len = exm3u_intro + exm3u_loop * m_default_loop_count;
					else if (m_default_loop_count == 0) { m_song_len = exm3u_intro + exm3u_loop; m_play_infinitely = true; }
				}
			}
			else if (exm3u_length != -1)
			{
				m_song_len = exm3u_length;
			}
			else
			{
				m_song_len = m_default_song_len;
			}

			if (exm3u_fade != -1) m_fade_len = exm3u_fade;
			else m_fade_len = m_default_fade_len;

			m_index_num = m_exm3u[p_subsong].track;
		}
		else
		{
			m_song_len = m_default_song_len;
			m_fade_len = m_default_fade_len;
			m_index_num = p_subsong;
		}

		if (m_decode_flags & input_flag_no_looping)
			m_play_infinitely = false;
	}

	p_info.set_length(static_cast<double>(m_song_len + m_fade_len) * .001);
	p_info.info_set_int("channels", m_channels);
	p_info.info_set_int("bitspersample", m_bps);
	p_info.info_set("codec", "HES");
	p_info.info_set("encoding", "synthesized");
	p_info.meta_add("system", "PC Engine");

	//display 0-base song number
	p_info.info_set_int("song_number", m_index_num);

	if (m_exm3u_load == 1)
	{
		check_encode_meta_set(p_info,				"album",			m_exm3u_album_info.title);
		check_encode_meta_set_multi_field(p_info,	"artist",			m_exm3u_album_info.artist);
		check_encode_meta_set_multi_field(p_info,	"album artist",		m_exm3u_album_info.albumartist);
		check_encode_meta_set(p_info,				"date",				m_exm3u_album_info.date);
		check_encode_meta_set_multi_field(p_info,	"genre",			m_exm3u_album_info.genre);
		check_encode_meta_set_multi_field(p_info,	"composer",			m_exm3u_album_info.composer);
		check_encode_meta_set(p_info,				"copyright",		m_exm3u_album_info.copyright);
		check_encode_meta_set(p_info,				"comment",			m_exm3u_album_info.comment);
		check_encode_meta_set(p_info,				"enginner",			m_exm3u_album_info.engineer);
		check_encode_meta_set(p_info,				"sequencer",		m_exm3u_album_info.sequencer);
		check_encode_meta_set(p_info,				"dumper",			m_exm3u_album_info.ripping);
		check_encode_meta_set(p_info,				"tagger",			m_exm3u_album_info.tagging);
		check_encode_meta_set(p_info,				"system",			m_exm3u_album_info.system);
		if (p_subsong < static_cast<unsigned int>(m_exm3u_line))
			check_encode_meta_set(p_info, "title", m_exm3u[p_subsong].name);

		if (m_display_track_num)
		{
			p_info.meta_add("tracknumber", pfc::string_formatter() << (p_subsong + 1));
			p_info.meta_add("totaltracks", pfc::string_formatter() << m_exm3u_line);
		}
	}
	else
	{
		if (m_display_track_num && p_subsong >= m_first_subsong)
		{
			p_info.meta_add("tracknumber", pfc::string_formatter() << (p_subsong + 1 - m_first_subsong));
			unsigned int total_tracks;
			if (m_total_subsong <= m_first_subsong) total_tracks = 1;
			else total_tracks = m_total_subsong - m_first_subsong;
			p_info.meta_add("totaltracks", pfc::string_formatter() << total_tracks);
			
		}
	}
	
}

// Get Dynamic Information
bool input_hes::decode_get_dynamic_info(file_info &p_out, double &p_timestamp_delta)
{
	if (m_dynamic_info)
	{
		m_dynamic_info = false;
		p_out.info_set_int("samplerate", m_sample_rate);		
		p_timestamp_delta = -(audio_math::samples_to_time(m_update_done, m_sample_rate));
		//p_timestamp_delta = 0;
		return true;
	}
	return false;
}

// Get SubSong Information
t_uint32 input_hes::get_subsong(unsigned p_index)
{
	if (m_exm3u_load == 1) return p_index;
	else return m_first_subsong + p_index;
}

unsigned input_hes::get_subsong_count()
{
	if (m_exm3u_load == 1) return m_exm3u_line;
	else 
	{ 
		if (m_total_subsong <= m_first_subsong) return 1;
		else return m_total_subsong - m_first_subsong;
	}
}



// Decode Initialize
void input_hes::decode_initialize(t_uint32 p_subsong, unsigned int p_flags, abort_callback &p_abort)
{
	m_set_len = false;
	m_decode_flags = p_flags;

	//load hes_player.dll
	if (loaddll() == false)
	{
		throw exception_io_data("Unable to load HES_PLAYER.DLL");
	}


	//get play time information from m_exm3u if it exist
	if (m_exm3u_load == 1 && p_subsong < static_cast<unsigned int>(m_exm3u_line))
	{
		int exm3u_intro = m_exm3u[p_subsong].intro;
		int exm3u_loop = m_exm3u[p_subsong].loop;
		int exm3u_length = m_exm3u[p_subsong].length;
		int exm3u_repeat = m_exm3u[p_subsong].repeat;
		int exm3u_fade = m_exm3u[p_subsong].fade;

		if (exm3u_intro != -1 && exm3u_loop != -1)
		{
			if (exm3u_repeat > 0)
			{
				m_song_len = exm3u_intro + exm3u_loop * exm3u_repeat;
			}
			else if (exm3u_repeat == 0)
			{
				m_song_len = exm3u_intro + exm3u_loop;
				m_play_infinitely = true;
			}
			else if (exm3u_repeat == -1)
			{
				if (m_default_loop_count > 0)  m_song_len = exm3u_intro + exm3u_loop * m_default_loop_count;
				else if (m_default_loop_count == 0) { m_song_len = exm3u_intro + exm3u_loop; m_play_infinitely = true; }
			}
		}
		else if (exm3u_length != -1)
		{
			m_song_len = exm3u_length;
		}
		else
		{
			m_song_len = m_default_song_len;
		}

		if (exm3u_fade != -1) m_fade_len = exm3u_fade;
		else m_fade_len = m_default_fade_len;

		m_index_num = m_exm3u[p_subsong].track;
	}
	else
	{
		m_song_len = m_default_song_len;
		m_fade_len = m_default_fade_len;
		m_index_num = p_subsong;
	}

	if (m_decode_flags & input_flag_no_looping)
		m_play_infinitely = false;

	m_song_sample = audio_math::time_to_samples(double(m_song_len) * .001, m_sample_rate);
	m_fade_sample = audio_math::time_to_samples(double(m_fade_len) * .001, m_sample_rate);
	m_silence_sample = audio_math::time_to_samples(m_silence_len, m_sample_rate);
	m_set_len = true;

	// allocate buffer for decode.
	m_decode_buf_len = m_update_sample * m_channels * (m_bps / 8) * 2;		// [byte]
	m_decode_buf.set_size(m_decode_buf_len + 16);
	
	//load hes file
	if (!(m_api->p_HES_Init(m_file_buf.get_ptr(), m_file_len, m_index_num, m_sample_rate)))
		throw exception_io_data("Invalid input file");
	m_api->p_HES_SetChannelMuting(cfg_channel_muting.load());

	m_dynamic_info = true;
	m_update_done = 0;

}

// Decode
bool input_hes::decode_run(audio_chunk &p_chunk, abort_callback &p_abort)
{
	// decide how many samples we want from hes_player
	t_uint64 update_sample = m_update_sample;

	if ((!m_play_infinitely && m_played_sample >= m_song_sample + m_fade_sample) || (m_detect_silence && m_detected_silence >= m_silence_sample))
	{
		return false;
	}

	if (!m_play_infinitely && (m_played_sample + update_sample) > (m_song_sample + m_fade_sample))
	{
		update_sample = (m_song_sample + m_fade_sample) - m_played_sample;
	}

	//Channel Muting
	if (m_api->p_HES_GetChannelMuting() != cfg_channel_muting.load())
	{
		m_api->p_HES_SetChannelMuting(cfg_channel_muting.load());
	}

	// Update samples. 
	m_api->p_HES_UpdateSamples(m_decode_buf.get_ptr(), m_decode_buf_len, update_sample);

	// Check Silence
	if (m_detect_silence)
	{
		const hes_sample_t* p_decode_buf = (const hes_sample_t*)m_decode_buf.get_ptr();
		t_uint64 i;
		bool silence(true);
		for (i = 0; i < update_sample; i++)
		{
			if (m_detected_silence >= m_silence_sample)
				break;

			for (int j = 0; j < m_channels; j++)
			{
				if (p_decode_buf[j] > m_silence_level || p_decode_buf[j] < m_silence_level * (-1))
					silence = false;
			}
	
			if (silence)
				m_detected_silence++;
			else
			{
				m_detected_silence = 0;
				silence = true;
			}
			p_decode_buf += m_channels;
		}
	}


	p_chunk.set_data_fixedpoint(
		m_decode_buf.get_ptr(),
		update_sample * m_channels * (m_bps / 8),
		m_sample_rate, m_channels, m_bps,
		audio_chunk::g_guess_channel_config(m_channels));


	// Volume or Fadeout
	if (!m_play_infinitely && m_fade_sample && m_played_sample + update_sample >= m_song_sample || m_volume != 1.00)
	{
		audio_sample* p_sample = p_chunk.get_data();
		t_uint64 i;

		for (i = m_played_sample; i < (m_played_sample + update_sample); i++)
		{
			if (m_play_infinitely || i < m_song_sample)
			{
				p_sample[0] = static_cast<audio_sample>(p_sample[0] * m_volume);
				p_sample[1] = static_cast<audio_sample>(p_sample[1] * m_volume);
			}
			else if ( i < (m_song_sample + m_fade_sample))
			{
				double scale = static_cast<double>(m_song_sample + m_fade_sample - i) / static_cast<double>(m_fade_sample);
				//scale = scale * scale * (3 - 2 * scale);	// s curve fade out
				p_sample[0] = static_cast<audio_sample>(p_sample[0] * m_volume * scale);
				p_sample[1] = static_cast<audio_sample>(p_sample[1] * m_volume * scale);
			}
			else
			{
				p_sample[0] = 0;
				p_sample[1] = 0;
			}
			p_sample += 2;
		}
	}

	if (!m_play_infinitely)
		m_played_sample += update_sample;
	m_update_done = update_sample;
	return true;
}


// Whether can seek or not.
bool input_hes::decode_can_seek()
{
	if (m_play_infinitely)
		return false;
	return true;
}


// Seek
void input_hes::decode_seek(double p_seconds, abort_callback &p_abort)
{
	// compare desired seek pos with current pos
	t_uint64 seek_sample = ::audio_math::time_to_samples(p_seconds, m_sample_rate);
	if (seek_sample < 0 || seek_sample > m_song_sample + m_fade_sample)
		throw exception_io_data("Seek is out of range");
	
	m_dynamic_info = true;
	m_update_done = 0;

	if (m_played_sample > seek_sample)
	{
		m_api->p_HES_Reset(m_index_num);
		m_played_sample = 0;
	}
	
	while (seek_sample - m_played_sample > m_update_sample)
	{
		p_abort.check();
		if (!m_api->p_HES_UpdateSamples(m_decode_buf.get_ptr(), m_decode_buf_len, m_update_sample))
			throw exception_io_data("Seek Error !");
			
		m_played_sample += m_update_sample;
	}
	t_uint64 remain_sample = seek_sample - m_played_sample;
	if (remain_sample > 0)
	{
		m_api->p_HES_UpdateSamples(m_decode_buf.get_ptr(), m_decode_buf_len, remain_sample);
	}
	m_played_sample = seek_sample;
	m_detected_silence = 0;
}


static input_factory_t<input_hes> g_input_hes_factory;

// version info
DECLARE_COMPONENT_VERSION(input_hes::g_get_name(),
"0.16_beta",
"Ootake v2.87\n"
"http://www.ouma.jp/ootake/\n"
"Copyright (C) 2006-2018 Kitao Nakamura\n"
"\n"
"M3u_Playlist in Game_Music_Emu\n"
"Copyright (C) 2006 Shay Green\n"
"modified by kode54\n"
"\n"
"libloadpe\n"
"Copyright (C) 2007 Ku-Zu\n"
)

DECLARE_FILE_TYPE("HES file", "*.hes");

// This will prevent users from renaming your component around (important for proper troubleshooter behaviors) or loading multiple instances of it.
VALIDATE_COMPONENT_FILENAME("foo_input_hes.dll");

