/***************************************************************************************\

Module:         OS driver interface

Description:    This is a special interface used by the Parhelia driver to use 
                functionnalities provided by the OS-dependent driver. 

                All operating system supporting Parhelia driver MUST implements those 
                calls on its own. 

    Copyright (C) 2004, Matrox Graphics Inc.
    All Rights Reserved.


\***************************************************************************************/

#ifndef INC_GENDRVOS_H
#define INC_GENDRVOS_H

/*-------------------------------------------------------------------------------------*\
                                 H E A D E R   F I L E S
\*-------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------*\
                          C O N S T A N T S   A N D   T Y P E S
\*-------------------------------------------------------------------------------------*/

typedef struct
{
    // Filled by Parhelia driver
    ULONG       ulCommonDataSize;       // Size of common data buffer
    ULONG       ulCommonLockedDataSize; // Size of common locked data buffer
    ULONG       ulSharedDataSize;       // Size of shared data buffer
    ULONG       ulSharedLockedDataSize; // Size of shared locked data buffer

    // Returned by the OS-driver
    VOID*       pvCommonData;           // Address of common data buffer
    VOID*       pvCommonLockedData;     // Address of common locked data buffer
    VOID*       pvSharedData;           // Address of shared data buffer
    VOID*       pvSharedLockedData;     // Address of shared locked data buffer
    
} OsCommonBufferData;

/*-------------------------------------------------------------------------------------*\
                   G L O B A L   V A R I A B L E   R E F E R E N C E S
\*-------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------*\
                 I N T E R F A C E   F U N C T I O N   P R O T O T Y P E S
\*-------------------------------------------------------------------------------------*/

/***************************************************************************************\

Function:       OsGetCommonBuffers

Description:    Return pointers to common buffers (buffers used by all devices instances).
                The OS-driver is responsible of allocating buffers and keeping track of 
                reference count to them.

                Four pointers should be returned:
                
                    1- CommonData buffer allocated in system memory
                    
                    2- CommonLockedData buffer allocated in contiguous PCI memory
                    
                    3- SharedData buffer allocated in system memory 
                       and initialized by the OS-dependent driver
                       
                    4- SharedLockedData buffer allocated in contiguous PCI memory
                       and initialized by the OS-dependent driver
                
Parameters:     hDriver             Handle to OS driver
                poCommonBufferData  Pointer holding sizes of buffers, 
                                    and pointers returned by the OS-driver
                
Return Value:   LBOOL               TRUE if succeed
                                    FALSE otherwise

Comments:       If buffer sizes doesn't match those of existing buffers, failure
                must be returned.

\***************************************************************************************/
LBOOL   OsGetCommonBuffers(IN     HANDLE hContext, 
                           IN OUT OsCommonBufferData* poCommonBufferData);
    
/***************************************************************************************\

Function:       OsReleaseCommonBuffers

Description:    Release reference to all common buffers. This means that device device
                as been closed, and do not need anymore the common buffers. When the
                last device referencing the buffers is closed, ressources should be
                freed.
                
Parameters:     hDriver             Handle to OS driver
                
Return Value:   LBOOL               TRUE if succeed
                                    FALSE otherwise

Comments:       None.

\***************************************************************************************/
LBOOL   OsReleaseCommonBuffers(IN HANDLE hContext);    

/***************************************************************************************\

Function:       OsInterlocked*

Description:    Method that handle race-conditions with data shared between 
                user-level driver, kernel driver and interrupt handler.
                
Parameters:     Non applicable.

Return Value:   Non applicable.

Comments:       None.

\***************************************************************************************/
EXTERN_C STACK_LINKAGE LONG    OsInterlockedIncrement(IN OUT PLONG plValue);
EXTERN_C STACK_LINKAGE LONG    OsInterlockedDecrement(IN OUT PLONG plValue);
EXTERN_C STACK_LINKAGE LONG    OsInterlockedExchange(IN OUT PLONG plOld, IN LONG lNew);

/***************************************************************************************\

Function:       OsAllocLockedPages

Description:    Allocate a number of locked pages in memory. Those pages may
                resides in normal system memory, or in AGP memory. Also, caller may
                has specified if a contiguous set of pages is requested (allowing 
                direct access from the hardware)
                
Parameters:     hDriver             Handle to OS driver
                ulNbPages           Number of pages to allocate
                bAgp                Allocate pages in AGP memory
                bContiguous         Multiple pages must be contiguous
                ppvVirtAddress      Returned pointer to the first byte of the first 
                                    page allocated (may be NULL)
                
Return Value:   DWORD               Physical address of the first byte of the first page 
                                    0 if failed.

Comments:       None.

\***************************************************************************************/
EXTERN_C STACK_LINKAGE DWORD   OsAllocLockedPages(IN  HANDLE hContext, 
                                                  IN  ULONG  ulNbPages, 
                                                  IN  BOOL   bAgp, 
                                                  IN  BOOL   bContiguous,
                                                  OUT VOID** ppvVirtAddress);

/***************************************************************************************\

Function:       OsFreeLockedPages

Description:    Free a number of locked memory pages. Pages might resides in
                system or AGP memory. OS should keep track itself of the allocation 
                type.
                
Parameters:     hDriver             Handle to OS driver
                dwPhysAddress       Physical address to the first byte of the first page
                pvVirtAddress       Virtual address to the first byte of the first page
                ulNbPages           Number of pages to free
                
Return Value:   BOOL                TRUE is succeed, 
                                    FALSE otherwise.
                                    
Comments:       The dwPhysAddress and ulNbPages fields correspond to the same used when
                allocating the pages

\***************************************************************************************/
EXTERN_C STACK_LINKAGE BOOL   OsFreeLockedPages(IN HANDLE hContext, 
                                                IN DWORD dwPhysAddress,
                                                IN VOID* pvVirtAddress,
                                                IN ULONG ulNbPages);

/***************************************************************************************\

Function:       OsMapPhysicalMemory

Description:    Map to kernel space a block of memory starting from a physical address.
                The block may resides anywhere in the system (IO, AGP, PCI, ...)
                
Parameters:     hDriver             Handle to OS driver
                dwPhysicalAddress   Start physical address of the memory block.
                ulSize              Size in bytes of mapping
                
Return Value:   MEMHANDLE           The kernel virtual memhandle pointing to this 
                                    physical block. NULL if failed.
Comments:       None.

\***************************************************************************************/
EXTERN_C STACK_LINKAGE MEMHANDLE   OsMapPhysicalMemory(IN HANDLE hContext, 
                                                       IN DWORD dwPhysicalAddress,
                                                       IN ULONG ulSize);

/***************************************************************************************\

Function:       OsUnmapMemory

Description:    Unmap a block of kernel virtual address.
                
Parameters:     hDriver             Handle to OS driver
                pvVirtualAddress    Kernel virtual mapping
                ulSize              Size in bytes of mappingOsWriteRegistryValue
                
Return Value:   BOOL                TRUE if succeed
                                    FALSE otherwise
Comments:       None.

\***************************************************************************************/
EXTERN_C STACK_LINKAGE BOOL   OsUnmapMemory(IN HANDLE hContext, 
                                            IN MEMHANDLE pvVirtualAddress,
                                            IN ULONG ulSize);

/***************************************************************************************\

Function:       OsReadRegistryValue

Description:    Read value in OS registry.
                
Parameters:     hDriver             Handle to OS driver
                pszRegistryPath     Registry section (or path)
                pszRegistryName     Name of registry
                pvRegValue          Pointer to data block receiving the registry value
                ulSizeValue         Size of output data block
                
Return Value:   BOOL                TRUE is succeed, 
                                    FALSE otherwise.
Comments:      

\***************************************************************************************/
EXTERN_C STACK_LINKAGE BOOL    OsReadRegistryValue(IN  HANDLE hContext,
                                                  IN  PSTR pszRegistryPath,
                                                  IN  PSTR pszRegistryName,
                                                  OUT VOID* pvRegValue,
                                                  IN  ULONG ulSizeValue);

/***************************************************************************************\

Function:       OsWriteRegistryValue

Description:    Write value in OS registry.
                
Parameters:     hDriver             Handle to OS driver
                pszRegistryPath     Registry section (or path)
                pszRegistryName     Name of registry
                pvRegValue          Pointer to data block holding the registry value
                ulSizeValue         Size of input data block
                
Return Value:   BOOL                TRUE is succeed, 
                                    FALSE otherwise.
Comments:      

\***************************************************************************************/
EXTERN_C STACK_LINKAGE BOOL    OsWriteRegistryValue(IN  HANDLE hContext,
                                                   IN  PSTR pszRegistryPath,
                                                   IN  PSTR pszRegistryName,
                                                   OUT VOID* pvRegValue,
                                                   IN  ULONG ulSizeValue);

/***************************************************************************************\

Function:       OsReadDrvSwitchValue

Description:    Read value of a driver switch.
                
Parameters:     hDriver             Handle to OS driver
                szSwitchID          String switch ID 
                pvSwitchValue       Pointer to data block holding the switch value
                ulSizeValue         Size of output data block
                
Return Value:   BOOL                TRUE is succeed, 
                                    FALSE otherwise.
Comments:      

\***************************************************************************************/
EXTERN_C STACK_LINKAGE BOOL    OsReadDrvSwitchValue(IN  HANDLE  hContext,
                                                   IN  PSTR szSwitchID,
                                                   OUT VOID* pvSwitchValue,
                                                   IN  ULONG ulSizeValue);

/***************************************************************************************\

Function:       OsGetCurrentProcessId

Description:    Return the PID of the current process.

Parameters:     hDriver             Handle to OS driver

Return Value:   DWORD               PID of current process, 0 if none or failed.

Comments:

\***************************************************************************************/
EXTERN_C STACK_LINKAGE DWORD   OsGetCurrentProcessId(IN  HANDLE hContext);

/***************************************************************************************\

Function:       OsSchedule

Description:    Call the OS schedule to yield process

Parameters:     NONE

Return Value:   NONE

Comments:

\***************************************************************************************/
EXTERN_C STACK_LINKAGE void OsSchedule( void );

/***************************************************************************************\
 *
 * Function:       OsGetRegisterHandle
 *
 * Description:    return OS register handle used by client fcn to access register
 *
 * Parameters:     hCtx             Handle to OS context
 *
 * Return Value:   handle to pass to client fcn
 *
 * Comments:       none
 *
 * \***************************************************************************************/
EXTERN_C STACK_LINKAGE void* OsGetRegisterHandle( HANDLE hContext );


/*-------------------------------------------------------------------------------------*\
                             I N L I N E S   A N D   M A C R O S
\*-------------------------------------------------------------------------------------*/

#endif  /* #ifndef INC_GENDRVOS_H */
