/***************************************************************************************\

File Name:      mtx_ioctl.h

Description:    MTX ioctl definitions.  It can be directly included by any apps that want
                to communicate to the MTX driver, but most of them will prefer
                the core specific version depending on the board they want to run.

References:     None.

    Copyright (c) 2004, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#ifndef INC_MTX_IOCTL_H
#define INC_MTX_IOCTL_H

/*-------------------------------------------------------------------------------------*\
                                 H E A D E R   F I L E S
\*-------------------------------------------------------------------------------------*/

#include <linux/ioctl.h>

/*-------------------------------------------------------------------------------------*\
                          C O N S T A N T S   A N D   T Y P E S
\*-------------------------------------------------------------------------------------*/

#include "mtx_mem_type.h"

/* Type of value requests */
typedef enum {

    MTX_REQUEST_DRIVER_PARM_WRITE,      /* Request a module parameter (W) */
    MTX_REQUEST_DRIVER_PARM_READ       /* Request a module parameter (R) */
    
} mtx_value_req_type;

/* GET MEMORY INFO */
typedef struct mtx_ioctl_get_memory_info {

    unsigned long   fb_base;            /* framebuffer base */
    unsigned long   fb_size;            /* framebuffer size */
    
    unsigned long   reg_base;           /* control registers base */
    unsigned long   reg_size;           /* control registers size */

    unsigned long   agp_base;           /* AGP space base */
    unsigned long   agp_size;           /* AGP space size */
    
} mtx_ioctl_get_memory_info_t;


/* ALLOC/GET BUFFER */
typedef struct mtx_ioctl_buffer {

    mtx_memory_type     type;           /* Buffer memory type */
    mtx_memory_scope    scope;          /* Buffer sharing scope */
    unsigned long       createNewRegion;/* Create a new region */

    unsigned long       size;           /* Desired size of buffer */
    unsigned long       base;           /* Buffer returned physical address */

} mtx_ioctl_buffer_t;

#define MTX_MODULE_PARM_MAX_LENGTH          50

/* DRIVER VALUE REQUESTS */
typedef struct mtx_ioctl_value {

    mtx_value_req_type  type;           /* Type of request */
    unsigned long       value;          /* 32 bits value (returned by kernel if type is R) */

    char  parm_name[MTX_MODULE_PARM_MAX_LENGTH];    /* Name of parameter (if needed) */

} mtx_ioctl_value_t;

typedef struct 
{
    /* Input value */
    UINT32  iBusNumber;
    UINT32  iDeviceNumber;
    UINT32  iFunction;

    /* Output value */
    UINT32  iMinor;
} MtxIoctlPacket_GetMinor;

/*-------------------------------------------------------------------------------------*\
                   G L O B A L   V A R I A B L E   R E F E R E N C E S
\*-------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------*\
                 I N T E R F A C E   F U N C T I O N   P R O T O T Y P E S
\*-------------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------------*\
                             I N L I N E S   A N D   M A C R O S
\*-------------------------------------------------------------------------------------*/

#define MTX_MAJOR_NUMBER                    0 /* Use dynamic major number allocation */
#define MTX_DRIVER_NAME                     "mtx"

#define MTX_MINOR_INVALID                   ~0

#define MTX_DEVICE_FILE_PREFIX              "mtx"
#define MTX_DEVICE_FILE_PATH                "/dev/matrox"

#define MTX_IOCTL_MAGIC                 'F'
#define MTX_IO(nr, offset)              _IO(MTX_IOCTL_MAGIC, (nr + offset))
#define MTX_IOR(nr, offset, arg)        _IOR(MTX_IOCTL_MAGIC, (nr + offset), arg)
#define MTX_IOW(nr, offset, arg)        _IOW(MTX_IOCTL_MAGIC, (nr + offset), arg)
#define MTX_IOWR(nr, offset, arg)       _IOWR(MTX_IOCTL_MAGIC, (nr + offset), arg)

/* core ioctl number offsets and range */
#define MTX_IOCTL_NROFF                 0
#define MTX_PARHL_IOCTL_NROFF           20
#define MTX_IOCTL_RANGE                 20 /* always a max range of 20 ioctls */

/* XXX: Always update MTX_IOCTL_MAXNR when adding a new ioctl. */
#define MTX_IOCTL_MAXNR                 (MTX_PARHL_IOCTL_NROFF + MTX_IOCTL_RANGE)

#define MTX_IOCTL_GET_MEMORY_INFO       MTX_IOR(0x00, MTX_IOCTL_NROFF, mtx_ioctl_get_memory_info_t)
#define MTX_IOCTL_GET_BUFFER            MTX_IOWR(0x01, MTX_IOCTL_NROFF, mtx_ioctl_buffer_t)
#define MTX_IOCTL_RELEASE_BUFFER        MTX_IOW(0x02, MTX_IOCTL_NROFF, mtx_ioctl_buffer_t)
#define MTX_IOCTL_DRIVER_VALUE_REQUEST  MTX_IOWR(0x03, MTX_IOCTL_NROFF, mtx_ioctl_value_t)
#define MTX_IOCTL_ADD_USER              MTX_IO(0x04, MTX_IOCTL_NROFF)
#define MTX_IOCTL_REMOVE_USER           MTX_IOW(0x05, MTX_IOCTL_NROFF, unsigned long)
#define MTX_IOCTL_GET_MINOR             MTX_IOWR(0x06, MTX_IOCTL_NROFF, MtxIoctlPacket_GetMinor)
#define MTX_IOCTL_HARD_RESET            MTX_IO(0x07, MTX_IOCTL_NROFF)  /* for debugging only */

#endif  /* #ifndef INC_MTX_IOCTL_H */
