/******************************************************************************\

    clientlx.c

    Copyright 1997, 2005, Matrox Graphics Inc.

    All Rights Reserved.

\******************************************************************************/
#include "xf86_OSproc.h"
#include "xf86Pci.h"
#include "xf86_ansic.h"
#include "client.h"
#include "mga.h"

CLIENTTABLE GClientFunctions = {
    (ULONG (DECL *)(LPVOID,ULONG,LPVOID))    MGAClientReadConfigSpaceByte,
    (ULONG (DECL *)(LPVOID,ULONG,LPVOID))    MGAClientReadConfigSpaceDword,
    (ULONG (DECL *)(LPVOID,ULONG,UCHAR))     MGAClientWriteConfigSpaceByte,
    (ULONG (DECL *)(LPVOID,ULONG,ULONG))     MGAClientWriteConfigSpaceDword,
    (ULONG (DECL *)(LPVOID,ULONG,ULONG))     MGAClientOpenRegisterBase,
    (ULONG (DECL *)(LPVOID))                 MGAClientCloseRegisterBase,
    (ULONG (DECL *)(LPVOID,ULONG,LPVOID))    MGAClientReadRegisterByte,
    (ULONG (DECL *)(LPVOID,ULONG,LPVOID))    MGAClientReadRegisterDword,
    (ULONG (DECL *)(LPVOID,ULONG,UCHAR))     MGAClientWriteRegisterByte,
    (ULONG (DECL *)(LPVOID,ULONG,ULONG))     MGAClientWriteRegisterDword,
    (ULONG (DECL *)(LPVOID,ULONG,ULONG))     MGAClientOpenMemoryBase,
    (ULONG (DECL *)(LPVOID))                 MGAClientCloseMemoryBase,
    (ULONG (DECL *)(LPVOID,ULONG,LPVOID))    MGAClientReadMemoryByte,
    (ULONG (DECL *)(LPVOID,ULONG,LPVOID))    MGAClientReadMemoryDword,
    (ULONG (DECL *)(LPVOID,ULONG,UCHAR))     MGAClientWriteMemoryByte,
    (ULONG (DECL *)(LPVOID,ULONG,ULONG))     MGAClientWriteMemoryDword,
    (ULONG (DECL *)(LPVOID,ULONG,ULONG))     MGAClientOpenSystemDevice,
    (ULONG (DECL *)(LPVOID))                 MGAClientCloseSystemDevice,
    (ULONG (DECL *)(LPVOID,ULONG,LPVOID))    MGAClientReadSystemDeviceByte,
    (ULONG (DECL *)(LPVOID,ULONG,LPVOID))    MGAClientReadSystemDeviceDword,
    (ULONG (DECL *)(LPVOID,ULONG,UCHAR))     MGAClientWriteSystemDeviceByte,
    (ULONG (DECL *)(LPVOID,ULONG,ULONG))     MGAClientWriteSystemDeviceDword,
    (ULONG (DECL *)(LPVOID,ULONG))           MGAClientWait,
    (ULONG (DECL *)(LPVOID,LPVOID,LPVOID))   MGAClientGetBiosInfo,
    (ULONG (DECL *)(LPVOID,LPVOID,ULONG))    MGAClientReadDDCEdid
};

/******************************************************************************\

    Function      : MGAClientReadConfigSpaceByte

    Description   : Read a Byte from the configuration space.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specifib board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    UCHAR *pucByte | pointer to a byte that will receive
                    the data

    Return Val : ULONG

\******************************************************************************/
ULONG MGAClientReadConfigSpaceByte(LPBOARDHANDLE pBoard, ULONG ulOffset,
                                UCHAR *pucByte)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);
    MGAPtr pMga = (MGAPtr)pClientStruct->pMga;

    ASSERT_HANDLER(pBoard);

    *pucByte = pciReadByte(pMga->PciTag,ulOffset);

    return 0;
}


/******************************************************************************\

    Function      : MGAClientReadConfigSpaceDword

    Description   : Read a Dword from the configuration space.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset  | Offset of the Byte to be read.
                    ULONG *pulDword | Dword to receive the data

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientReadConfigSpaceDword(LPBOARDHANDLE pBoard, ULONG ulOffset,
                                 ULONG *pulDword)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);
    MGAPtr pMga = (MGAPtr)pClientStruct->pMga;

    ASSERT_HANDLER(pBoard);

    *pulDword = pciReadLong(pMga->PciTag,ulOffset);

    return 0;
}


/******************************************************************************\

    Function      : MGAClientWriteConfigSpaceByte

    Description   : Write a Byte from the configuration space.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    UCHAR ucByte   | Byte to receive the data

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientWriteConfigSpaceByte(LPBOARDHANDLE pBoard, ULONG ulOffset,
                                 UCHAR ucByte)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);
    MGAPtr pMga = (MGAPtr)pClientStruct->pMga;

    ASSERT_HANDLER(pBoard);

    pciWriteByte(pMga->PciTag,ulOffset, ucByte);

    return 0;
}


/******************************************************************************\

    Function      : MGAClientWriteConfigSpaceDword

    Description   : Write a Dword from the configuration space.

    I/O Desc.     : LPBOARDHANDLE pBoard     | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    ULONG ulDword  | Dword containing the data to  be written

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientWriteConfigSpaceDword(LPBOARDHANDLE pBoard, ULONG ulOffset,
                                  ULONG ulDword)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);
    MGAPtr pMga = (MGAPtr)pClientStruct->pMga;

    ASSERT_HANDLER(pBoard);

    pciWriteLong(pMga->PciTag,ulOffset, ulDword);

    return 0;
}


/******************************************************************************\

    Function      : MGAClientOpenRegisterBase

    Description   : Map the register base for future call to ClientReadRegisterX
                    and ClientWriteRegisterX.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulAddress | Physical address of the Register aperture
                    ULONG ulSize    | Size in Byte of the Register Aperture

    Return Val : ULONG

\******************************************************************************/
ULONG MGAClientOpenRegisterBase(LPBOARDHANDLE pBoard, ULONG ulAddress, ULONG ulSize)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);
    MGAPtr pMga = (MGAPtr)pClientStruct->pMga;

    ASSERT_HANDLER(pBoard);

    pClientStruct->ulRegisterBase = (ULONG) pMga->IOBase;

    return 0;
}


/******************************************************************************\

    Function      : MGAClientCloseRegisterBase

    Description   : Unmap the register base address and free resources needed
                    to address it.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.

    Return Val : ULONG

\******************************************************************************/
ULONG MGAClientCloseRegisterBase(LPBOARDHANDLE pBoard)
{
    ASSERT_HANDLER(pBoard);
    return 0;
}


/******************************************************************************\

    Function      : MGAClientReadRegisterByte

    Description   : Read a byte from the Register space.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure.
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    UCHAR *pucByte | pointer to the byte that will receive
                    the data.

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientReadRegisterByte(LPBOARDHANDLE pBoard, ULONG ulOffset,
                             UCHAR *pucByte)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);

    ASSERT_HANDLER(pBoard);

    *pucByte = *((UCHAR *)(pClientStruct->ulRegisterBase + ulOffset));

    return 0;
}


/******************************************************************************\

    Function      : MGAClientReadRegisterDword

    Description   : Read a Dword from the Register space.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset  | Offset of the Byte to be read.
                    ULONG *pulDword | pointer to the dword that will receive
                    the data.

    Return Val : ULONG

\******************************************************************************/
ULONG MGAClientReadRegisterDword(LPBOARDHANDLE pBoard, ULONG ulOffset,
                              ULONG *pulDword)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);

    ASSERT_HANDLER(pBoard);

    *pulDword = *((ULONG *)(pClientStruct->ulRegisterBase + ulOffset));

    return 0;
}


/******************************************************************************\

    Function      : MGAClientWriteRegisterByte

    Description   : Write a Byte from the Register space.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    UCHAR ucByte   | CHAR to receive the data.

    Return Val : ULONG

\******************************************************************************/
ULONG MGAClientWriteRegisterByte(LPBOARDHANDLE pBoard, ULONG ulOffset,
                              UCHAR ucByte)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);

    ASSERT_HANDLER(pBoard);

    *((UCHAR *)(pClientStruct->ulRegisterBase + ulOffset)) = ucByte;

    return 0;
}


/******************************************************************************\

    Function      : MGAClientWriteRegisterSpaceDword

    Description   : Write a Dword from the Register space.

    I/O Desc.     : LPBOARDHANDLE *| pBoard  | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    ULONG ulDword  | Dword to receive the data

    Return Val : ULONG

\******************************************************************************/
ULONG MGAClientWriteRegisterDword(LPBOARDHANDLE pBoard, ULONG ulOffset,
                               ULONG ulDword)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);

    ASSERT_HANDLER(pBoard);

    *((ULONG *)(pClientStruct->ulRegisterBase + ulOffset)) = ulDword;

    return 0;
}


/******************************************************************************\

    Function      : MGAClientOpenMemoryBase

    Description   : Map the Memory base for future call to ClientReadMemoryX
                    and ClientWriteMemoryX.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulAddress | Physical address of the Register aperture
                    ULONG ulSize    | Size in Byte of the Register Aperture
    Return Val : ULONG

\******************************************************************************/
ULONG MGAClientOpenMemoryBase(LPBOARDHANDLE pBoard, ULONG ulAddress, ULONG ulSize)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);
    MGAPtr pMga = (MGAPtr) pClientStruct->pMga;

    ASSERT_HANDLER(pBoard);

    pClientStruct->ulFrameBufferBase = (ULONG) pMga->FbBase;

    return 0;
}


/******************************************************************************\

    Function      : MGAClientCloseMemoryBase

    Description   : Unmap the Frame Buffer aperture and free resources
                    needed to address it.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientCloseMemoryBase(LPBOARDHANDLE pBoard)
{
    ASSERT_HANDLER(pBoard);
    return 0;
}


/******************************************************************************\

    Function      : MGAClientReadMemoryByte

    Description   : Read a Byte from the Frame Buffer space.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    UCHAR *ucByte  | CHAR to receive the data

    Return Val : ULONG
\******************************************************************************/
ULONG MGAClientReadMemoryByte(LPBOARDHANDLE pBoard, ULONG ulOffset, UCHAR *pucByte)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);

    ASSERT_HANDLER(pBoard);

    *pucByte = *((UCHAR *)(pClientStruct->ulFrameBufferBase + ulOffset));

    return 0;
}


/******************************************************************************\

    Function      : MGAClientReadMemoryDword

    Description   : Read a Dword from the Frame Buffer Space.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    ULONG *uDword  | Dword to receive the data

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientReadMemoryDword(LPBOARDHANDLE pBoard, ULONG ulOffset,
                            ULONG *pulDword)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);

    ASSERT_HANDLER(pBoard);

    *pulDword = *((ULONG *)(pClientStruct->ulFrameBufferBase + ulOffset));

    return 0;
}


/******************************************************************************\

   Function      : MGAClientWriteMemoryByte

   Description   : Write a Byte from the Frame Buffer space.

   I/O Desc.     : LPBOARDHANDLE pBoard     | Handle to the board structure
                   containing all the information about a specific board.
                   ULONG ulOffset | Offset of the Byte to be read.
                   UCHAR ucByte   | CHAR to receive the data

   Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientWriteMemoryByte(LPBOARDHANDLE pBoard, ULONG ulOffset, UCHAR ucByte)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);

    ASSERT_HANDLER(pBoard);

    *((UCHAR *)(pClientStruct->ulFrameBufferBase + ulOffset)) = ucByte;

    return 0;
}


/******************************************************************************\

    Function      : MGAClientWriteMemoryDword

    Description   : Write a Dword from the Frame Buffer space.

    I/O desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    ULONG ulDword  | Dword to receive the data

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientWriteMemoryDword(LPBOARDHANDLE pBoard, ULONG ulOffset,
                            ULONG ulDword)
{
    LPCLIENTDATA pClientStruct = MATROXGetClientPointer(pBoard);

    ASSERT_HANDLER(pBoard);

    *((ULONG *)(pClientStruct->ulFrameBufferBase + ulOffset)) = ulDword;

    return 0;
}


/******************************************************************************\

    Function      : MGAClientOpenSystemDevice

    Description   : Map a System device aperture for future call to
                    ClientReadSystemDeviceX and ClientWriteSystemDeviceX.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulAddress | Physical address of the Register aperture
                    ULONG ulSize    | Size in Byte of the Register Aperture

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientOpenSystemDevice(LPBOARDHANDLE pBoard, ULONG ulAddress,
                                    ULONG ulSize)
{
    return 1;
}


/******************************************************************************\

    Function      : MGAClientCloseSystemDevice

    Description   : Unmap the System Device aperture address and free
                    resources needed to address it.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientCloseSystemDevice (LPBOARDHANDLE pBoard)
{
    return 1;
}


/******************************************************************************\

    Function      : MGAClientReadSystemDeviceByte

    Description   : Read a Byte from the device Space.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    ULONG pucByte  | Byte to read the data

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientReadSystemDeviceByte(LPBOARDHANDLE pBoard, ULONG ulOffset,
                                 UCHAR *pucByte)
{
    return 1;
}


/******************************************************************************\

    Function      : MGAClientReadSystemDeviceDword

    Description   : Read a Dword from the Frame Buffer Space

    I/O Desc.     : LPBOARDHANDLE pBoard     | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    ULONG ulDword  | Dword to Read the data

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientReadSystemDeviceDword(LPBOARDHANDLE pBoard, ULONG ulOffset,
                                  ULONG *pulDword)
{
    return 1;
}


/******************************************************************************\

    Function      : MGAClientWriteSystemByte

    Description   : Write a Byte from the System Device Aperture

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    UCHAR ucByte   | Byte to receive the data
    Return Val : ULONG

\******************************************************************************/
ULONG MGAClientWriteSystemDeviceByte(LPBOARDHANDLE pBoard, ULONG ulOffset,
                                  UCHAR ucByte)
{
    return 1;
}


/******************************************************************************\

    Function      : MGAClientWriteSystemDword

    Description   : Write a Dword from the System Device Aperture.

    I/O Desc.     : LPBOARDHANDLE pBoard     | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulOffset | Offset of the Byte to be read.
                    ULONG uDword   | Dword to receive the data

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientWriteSystemDeviceDword(LPBOARDHANDLE pBoard, ULONG ulOffset,
                                   ULONG ulDword)
{
    return 1;
}


/******************************************************************************\

    Function      : MGAClientWait

    Description   : Wait for ulDelayus micro-seconds.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    ULONG ulDelayus | Delay in uSec

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientWait (LPBOARDHANDLE pBoard, ULONG ulDelayus)
{
    int i;
    ULONG ulTmp;

    ASSERT_HANDLER(pBoard);

    for(i = 0; i < ulDelayus * 3; i++)
    {
        MGAClientReadRegisterDword(pBoard,0x1e14,&ulTmp);
    }

    return 0;
}


/******************************************************************************\

    Function      : MGAClientGetBiosInfo

    Description   : This function will be call if no PINS can be found
                    in physical EEPROM.

    I/O Desc.     :  LPBOARDHANDLE pBoard | Handle to the board structure
                     containing all the information about a specific board.
                     LPUCHAR    | Buffer where we copy bios pins.
                     ULONG      | Bios version

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientGetBiosInfo(LPBOARDHANDLE pBoard, LPUCHAR pucPins,LPULONG pulBIOSVersion)
{
    Bool bNotFound = TRUE;
    UCHAR ucBIOS[32768];
    UCHAR ucTmpByte;
    UCHAR ucCheckSum;
    UCHAR ucPinsIndex;
    UCHAR ucPinsSize;
    ULONG ulTmpDword;
    ULONG ulPinsOffset = 0;
    ULONG ulPCIINFOffset;

    ASSERT_HANDLER(pBoard);

    xf86ReadBIOS(0xc0000,0,ucBIOS,32768);

    if(ucBIOS[0] == 0x55)
    {
        if(ucBIOS[1] == 0xaa)
        {
            while((ulPinsOffset < 0x10000) && bNotFound)
            {
                ulTmpDword = *(ULONG *)(ucBIOS + ulPinsOffset);
                ucPinsSize = (UCHAR) (ulTmpDword >> 16);
                if(((ulTmpDword & 0x0000ffff) == 0x0000412e)
                    && ucPinsSize <= 128)
                {
                    ucCheckSum = 0;
                    for(ucPinsIndex = 0;ucPinsIndex < ucPinsSize; ucPinsIndex++)
                    {
                        pucPins[ucPinsIndex] = ucBIOS[ulPinsOffset +
                                               ucPinsIndex];
                        ucCheckSum += pucPins[ucPinsIndex];
                    }
                    if(ucCheckSum == 0)
                    {
                        bNotFound = FALSE;
                    }
                }
                ulPinsOffset++;
            }

            if(bNotFound)
            {
                return 1;
            }

            ulPCIINFOffset = *(ULONG *)(ucBIOS + 0x18);
            ulPCIINFOffset &= 0x0000ffff;
            ulTmpDword = *(ULONG *)(ucBIOS + ulPCIINFOffset);

            if(ulTmpDword == 0x52494350)  /* "PCIR" */
            {
                ulPCIINFOffset += 0x12;
                ulTmpDword = *(ULONG *)(ucBIOS + ulPCIINFOffset);
                *pulBIOSVersion = ((ULONG) ((ulTmpDword & 0xf0) >> 4) << 16) |
                                  ((ulTmpDword &0xf) << 12) | ((ulTmpDword >> 8)
                                  & 0xff);
            }
            else
            {
                return 1;
            }
        }
        else
        {
           return 1;
        }
    }
    else
    {
        return 1;
    }

    if(!*pulBIOSVersion)
    {
        ucTmpByte = ucBIOS[5];
        *pulBIOSVersion = ((ULONG) (ucTmpByte >> 4) << 16) | ((ULONG)
                          (ucTmpByte & 0x0f) << 12);
    }

    return 0;
}

/******************************************************************************\

    Function      : MGAClientCallBiosInt10

    Description   : Call the BIOS Int10h with specified parameters.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    LPBIOSREGS pBiosRegs | Pointor to the Bios register
                    structure.

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientCallBiosInt10(LPBOARDHANDLE pBoard, LPBIOSREGS pBiosRegs)
{
   ASSERT_HANDLER(pBoard);

   return 1;
}


/******************************************************************************\

    Function      : MGAClientReadDDCEdid

    Description   : Not implemented.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    LPBIOSREGS pBiosRegs | Pointor to the Bios register
                    structure.

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientReadDDCEdid(LPBOARDHANDLE pBoard, LPUCHAR pEdid,
                               ULONG ulMonitorIndex)
{
   ASSERT_HANDLER(pBoard);

   return 1;
}


/******************************************************************************\

    Function      : MGAClientCustomCall

    Description   : Not implemented.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    LPBIOSREGS pBiosRegs | Pointor to the Bios register
                    structure.

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientCustomCall(LPBOARDHANDLE pBoard, ULONG ulServiceNumber,
                       LPVOID pInData, LPVOID pOutData)
{
   ASSERT_HANDLER(pBoard);

   return 1;
}


/******************************************************************************\

    Function      : MGAClientApplyEpromPatch

    Description   : Not implemented.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    LPBIOSREGS pBiosRegs | Pointor to the Bios register
                    structure.

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientApplyEpromPatch(LPBOARDHANDLE pBoard)
{
   ASSERT_HANDLER(pBoard);

   return 1;
}


/******************************************************************************\

    Function      : MGAClientDetectHostInterface

    Description   : Not implemented.

    I/O Desc.     : LPBOARDHANDLE pBoard | Handle to the board structure
                    containing all the information about a specific board.
                    LPBIOSREGS pBiosRegs | Pointor to the Bios register
                    structure.

    Return Val    : ULONG

\******************************************************************************/
ULONG MGAClientDetectHostInterface(LPBOARDHANDLE pBoard, LPULONG pulData)
{
   ASSERT_HANDLER(pBoard);

   return 1;
}

