/***************************************************************************************\

Module Name:    MtlVector.h

Description:    None.

References:     None.

    Copyright (c) 2002, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#ifndef INC_MTLVECTOR_H
#define INC_MTLVECTOR_H

//#define TESTWITHSTD     // This is used to test the Vector with the standard algorithm.
// --------------------------------------------------------------------------------------
//                        I N C L U D E S   A N D   U S I N G S
// --------------------------------------------------------------------------------------
#include "MtlXMemory.h"
// --------------------------------------------------------------------------------------
//                        C O N S T A N T S   A N D   T Y P E S
// --------------------------------------------------------------------------------------

// Create the namespace for the Matrox Template Library
// Defines are required to avoid tabulation reformating by VC editor.
#define MTLBEGIN        namespace Mtl{
#define MTLEND          }

#ifdef TESTWITHSTD
#ifdef MTLBEGIN
#undef MTLBEGIN
#define BOOL            bool
#define TRUE            true
#define FALSE           false
#endif

#define Vector          vector

#define ValueType       value_type     
#define Pointer         pointer        
#define ConstPointer    const_pointer  
#define Reference       reference      
#define ConstReference  const_reference
#define SizeType        size_type      
#define DifferenceType  difference_type

#define Iterator        iterator
#define ConstIterator   const_iterator
//#define ReverseIterator reverse_iterator
#define ConstReverseIterator   const_reverse_iterator

#define Assign          assign
#define Back            back
#define Begin           begin
#define Capacity        capacity
#define Clear           clear
#define Empty           empty
#define End             end
#define Erase           erase
#define Front           front
#define Insert          insert
#define MaxSize         max_size
#define PushBack        push_back
#define PopBack         pop_back
#define RBegin          rbegin
#define REnd            rend
#define Reserve         reserve
#define Resize          resize
#define Size            size

#endif

// --------------------------------------------------------------------------------------
//                 G L O B A L   V A R I A B L E   R E F E R E N C E S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//               G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//     I N L I N E S,  M A C R O S,  A N D   T E M P L A T E   D E F I N I T I O N S
// --------------------------------------------------------------------------------------
MTLBEGIN   
/***************************************************************************************\

Class:          ReverseIterator

Description:    The template class is an Iterator adaptor that describes a reverse Iterator 
                object that behaves like a random-access or bidirectional Iterator, only in 
                reverse. It enables the backward traversal of a range.

Comments:       None.

\***************************************************************************************/
template<class IteratorType,
	     class ValueType,
	     class ReferenceType    = ValueType&,
	     class PointerType      = ValueType *,
	     class DifferenceType   = unsigned long>
class ReverseIterator 
{
public:
	typedef ReverseIterator<IteratorType, 
                            ValueType, 
                            ReferenceType, 
                            PointerType, 
                            DifferenceType>     ReverseIteratorType;
    // Constructs a default ReverseIterator or a ReverseIterator from an underlying Iterator.
	ReverseIterator() {}
	explicit ReverseIterator(IteratorType itX): itCurrent(itX) {}
    // Recovers the underlying Iterator from its ReverseIterator.
	IteratorType Base() const 
        {return (itCurrent); }
    // Returns the element that a ReverseIterator addresses.
	ReferenceType operator*() const 
        {return (*(itCurrent - 1)); }
    // Returns a pointer to the element addressed by the ReverseIterator.
	PointerType operator->() const {return (&**this); }
    // Increments the ReverseIterator to the next element.
	ReverseIteratorType& operator++()
		{--itCurrent;
		return (*this); }
	ReverseIteratorType operator++(int)
		{ReverseIteratorType itTemp = *this;
		--itCurrent;
		return (itTemp); }
    // Decrements the ReverseIterator to the previous element.
	ReverseIteratorType& operator--()
		{++itCurrent;
		return (*this); }
	ReverseIteratorType operator--(int)
		{ReverseIteratorType itTemp = *this;
		++itCurrent;
		return (itTemp); }
    // Adds a specified offset from a ReverseIterator.
	ReverseIteratorType& operator+=(DifferenceType ulOffset)
		{itCurrent -= ulOffset;
		return (*this); }
    // Adds an offset to an Iterator and returns the new ReverseIterator 
    // addressing the inserted element at the new offset position.
	ReverseIteratorType operator+(DifferenceType ulOffset) const
		{return (ReverseIteratorType(itCurrent - ulOffset)); }
    // Subtracts a specified offset from a ReverseIterator.
	ReverseIteratorType& operator-=(DifferenceType ulOffset)
		{itCurrent += ulOffset;
		return (*this); }
    // Subtracts an offset from a ReverseIterator and returns a ReverseIterator 
    // addressing the element at the offset position.
	ReverseIteratorType operator-(DifferenceType ulOffset) const
		{return (ReverseIteratorType(itCurrent + ulOffset)); }
    // Returns a reference to an element offset from the element addressed by a 
    // ReverseIterator by a specified number of positions.
	ReferenceType operator[](DifferenceType ulOffset) const
		{return (*(*this + ulOffset)); }
protected:
	IteratorType itCurrent;
	
};  // class ReverseIterator

// Tests if the Iterator object on the left side of the operator is equal to the 
// Iterator object on the right side.
template<class IteratorType, class ValueType, class ReferenceType, class PointerType,
	class DifferenceType> inline
	bool __cdecl operator==(
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itLeft,
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itRight)
	{return (itLeft.Base() == itRight.Base()); }
// Tests if the Iterator object on the left side of the operator is not equal to 
// the Iterator object on the right side.
template<class IteratorType, class ValueType, class ReferenceType, class PointerType,
	class DifferenceType> inline
	bool __cdecl operator!=(
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itLeft,
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itRight)
	{return (!(itLeft == itRight)); }
template<class IteratorType, class ValueType, class ReferenceType, class PointerType,
	class DifferenceType> inline
	bool __cdecl operator<(
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itLeft,
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itRight)
	{return (itRight.Base() < itLeft.Base()); }
template<class IteratorType, class ValueType, class ReferenceType, class PointerType,
	class DifferenceType> inline
	bool __cdecl operator>(
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itLeft,
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itRight)
	{return (itRight < itLeft); }
template<class IteratorType, class ValueType, class ReferenceType, class PointerType,
	class DifferenceType> inline
	bool __cdecl operator<=(
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itLeft,
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itRight)
	{return (!(itRight < itLeft)); }
template<class IteratorType, class ValueType, class ReferenceType, class PointerType,
	class DifferenceType> inline
	bool __cdecl operator>=(
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itLeft,
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itRight)
	{return (!(itLeft < itRight)); }
template<class IteratorType, class ValueType, class ReferenceType, class PointerType,
	class DifferenceType> inline
	DifferenceType __cdecl operator-(
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itLeft,
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itRight)
	{return (itRight.Base() - itLeft.Base()); }
template<class IteratorType, class ValueType, class ReferenceType, class PointerType,
	class DifferenceType> inline
	ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType> __cdecl operator+(DifferenceType ulOffset,
		const ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>& itRight)
	{return (ReverseIterator<IteratorType, ValueType, ReferenceType, PointerType, DifferenceType>
            (itRight.Base() - ulOffset)); }

/***************************************************************************************\

Class:          Vector

Description:    The Vector class is a template class of sequence containers that arrange 
                elements of a given type in a linear arrangement and allow fast random 
                access to any element. They should be the preferred container for a 
                sequence when random-access performance is at a premium.

Comments:       Vectors allow constant time insertions and deletions at the end of the 
                sequence. Inserting or deleting elements in the middle of a vector 
                requires linear time.

                Vector reallocation occurs when a member function must increase the 
                sequence contained in the vector object beyond its current storage 
                capacity. Other insertions and erasures may alter various storage 
                addresses within the sequence. In all such cases, Iterators or references 
                that point at altered portions of the sequence become invalid. If no 
                reallocation happens, only Iterators and references before the 
                insertion/deletion point remain valid.

\***************************************************************************************/
template<class ValueType, class AllocatorType = allocator<ValueType> >
class Vector 
{
public:
    // ----- Class Types -----
    typedef typename AllocatorType::pointer          Pointer;
    typedef typename AllocatorType::const_pointer    ConstPointer;
    typedef typename AllocatorType::reference        Reference;
    typedef typename AllocatorType::const_reference  ConstReference;
    typedef typename AllocatorType::size_type        SizeType;
    typedef typename AllocatorType::difference_type  DifferenceType;
	typedef Pointer                         Iterator;
	typedef ConstPointer                    ConstIterator;

	typedef ReverseIterator<ConstIterator, 
                            ValueType,
		                    ConstReference, 
                            ConstPointer, 
                            DifferenceType>     ConstReverseIterator;
	typedef ReverseIterator<Iterator, 
                            ValueType,
		                    Reference, 
                            Pointer, 
                            DifferenceType>		ReverseIterator;

    // The default constructor. Creates a vector of length zero.
	explicit Vector(const AllocatorType& oAllocator = AllocatorType())
        : m_oAllocator(oAllocator), m_itFirst(0), m_itLast(0), m_itEnd(0) {}
    // The destructor. Releases any allocated memory for this vector.    
	~Vector()
		{Destroy(m_itFirst, m_itLast);
		m_oAllocator.deallocate(m_itFirst, m_itEnd - m_itFirst);  //operator delete(m_itFirst);
		m_itFirst = 0, m_itLast = 0, m_itEnd = 0; }
    // Reserves a minimum length of storage for a vector object, allocating space if necessary.    
	inline BOOL Reserve(SizeType ulReserveSize);
    // Returns the number of elements that the vector could contain without allocating more storage.
	SizeType Capacity() const {return (m_itFirst == 0 ? 0 : m_itEnd - m_itFirst); }
    // Returns a random-access Iterator (ConstIterator for constant vector) that points 
    // to the first element.
	Iterator Begin() {return (m_itFirst); }
	ConstIterator Begin() const {return ((ConstIterator)m_itFirst); }
    // Returns a random-access Iterator (ConstIterator for constant vector) that points 
    // to the past-the-end value.
	Iterator End() {return (m_itLast); }
	ConstIterator End() const {return ((ConstIterator)m_itLast); }
    // Return a random-access ReverseIterator (ConstReverseIterator for constant vector) 
    // that points to the past-the-end value. 
	ReverseIterator RBegin() {return (ReverseIterator(End())); }
	ConstReverseIterator RBegin() const {return (ConstReverseIterator(End())); }
    // Return a random-access ReverseIterator (ConstReverseIterator for constant vector) 
    // that points to the first element. 
	ReverseIterator REnd() {return (ReverseIterator(Begin())); }
	ConstReverseIterator REnd() const {return (ConstReverseIterator(Begin())); }
    // Specifies a new size for a vector.
	inline BOOL Resize(SizeType ulNewSize, const ValueType& oValue = ValueType());
    // Return the number of elements currently stored in the vector.
	SizeType Size() const {return (m_itFirst == 0 ? 0 : m_itLast - m_itFirst); }
    // Returns the maximum length of the vector.
	SizeType MaxSize() const { return (m_oAllocator.max_size()); }
    // Returns TRUE if the vector contains no elements (i.e., if Begin() == End()), FALSE otherwise.
	bool Empty() const {return (Size() == 0); }
/*
	ConstReference at(SizeType itPosition) const
		{if (Size() <= itPosition)
			_Xran();
		return (*(Begin() + itPosition)); }
	Reference at(SizeType itPosition)
		{if (Size() <= itPosition)
			_Xran();
		return (*(Begin() + itPosition)); }
*/
    // Returns a reference to the vector element at a specified position.
    // The index ulPosition must be between 0 and the size less one.
    ConstReference operator[](SizeType ulPosition) const {return (*(Begin() + ulPosition)); }
    // Returns a reference to the vector element at a specified position.
    // The index ulPosition must be between 0 and the size less one.
    // The result can be used as an lvalue. 
	Reference operator[](SizeType ulPosition) {return (*(Begin() + ulPosition)); }
    // Returns a reference to the first element in a vector.
	Reference Front() {return (*Begin()); }
	ConstReference Front() const {return (*Begin()); }
    // Returns a reference to the last element of the vector.
	Reference Back() {return (*(End() - 1)); }
	ConstReference Back() const {return (*(End() - 1)); }
    // Insert the element oValue at the end of the vector.
	BOOL PushBack(const ValueType& oValue) {return(Insert(End(), oValue)); }
    // Erases the last element of the vector. Undefined if the vector is empty.
	void PopBack() {Erase(End() - 1); }
    // Erases all elements in a vector, then inserts new elements from the range [first, last).
	BOOL Assign(ConstPointer cpFirst, ConstPointer cpLast)
		{Erase(Begin(), End());
		return(Insert(Begin(), cpFirst, cpLast)); }
    // Erases all elements in a vector, then inserts ulNumberOfCopies instances of 
    // the (default if not specified) value of type ValueType.
	BOOL Assign(SizeType ulNumberOfCopies, const ValueType& oValue = ValueType())
		{Erase(Begin(), End());
		return(Insert(Begin(), ulNumberOfCopies, oValue)); }
    // Inserts an element into a vector before the specified position.
	BOOL Insert(Iterator itPosition, const ValueType& oValue = ValueType())
		{return (Insert(itPosition, 1, oValue));}
    // Inserts ulNumberOfCopies copies of the element into the list before the 
    // specified position.
	BOOL Insert(Iterator itPosition, SizeType ulNumberOfCopies, const ValueType& oValue);
    // Inserts copies of the elements in the range [first, last] before itPosition.
	BOOL Insert(Iterator itPosition, ConstPointer cpFirst, ConstPointer cpLast);
    // Deletes the vector element pointed to by the iterator position. Returns 
    // an iterator pointing to the element following the deleted element, 
    // or end() if the deleted element was the last one in this vector.
	Iterator Erase(Iterator itPosition);
    // Deletes the vector elements in the range (first, last). Returns an 
    // iterator pointing to the element following the last deleted element, 
    // or end() if there were no elements in the deleted range.
	Iterator Erase(Iterator itFirst, Iterator itLast);
    // Erases all elements from the vector.
    void Clear() {Erase(Begin(), End()); }
protected:
	void Destroy(Iterator itFirst, Iterator itLast)
	{
        for (; itFirst != itLast; ++itFirst)
        // Call the ValueType destructor here.
        m_oAllocator.destroy(itFirst);  //itFirst->~ValueType();
    }
	Iterator Copy(ConstIterator itFirst, ConstIterator itLast, Iterator itPosition)
	{
        for (; itFirst != itLast; ++itPosition, ++itFirst)
        m_oAllocator.construct(itPosition, *itFirst); //new ((void *)itPosition) ValueType(*itFirst); 
		return (itPosition); 
    }
	void FillSize(Iterator itFirst, SizeType ulSize, const ValueType &oValue)
	{
        for (; 0 < ulSize; --ulSize, ++itFirst)
		m_oAllocator.construct(itFirst, oValue);  //new ((void *)itFirst) ValueType(oValue);
    }
    void FillRange(Iterator itFirst, Iterator itLast, const ValueType& oValue) 
    {
        for ( ; itFirst != itLast; ++itFirst)
            *itFirst = oValue;
    }
    Iterator CopyForward(Iterator itFirst, Iterator itLast, Iterator itPosition)
    {
        for (; itFirst != itLast; ++itPosition, ++itFirst)
        *itPosition = *itFirst;
        return (itPosition); 
    }
    Iterator CopyBackward(Iterator itFirst, Iterator itLast, Iterator itPosition)
    {
        while (itFirst != itLast)
        *--itPosition = *--itLast;
        return (itPosition); 
    }
    
    /*
	void _Xran() const
		{_THROW(out_of_range, "invalid vector<T> subscript"); }
        */
    AllocatorType m_oAllocator;
	Iterator m_itFirst, m_itLast, m_itEnd;
};  // class Vector
/*
		// vector TEMPLATE OPERATORS
template<class T> inline
	bool operator==(const Vector<T>& _X,
		const Vector<T>& _Y)
	{return (_X._Eq(_Y)); }
template<class T> inline
	bool operator!=(const Vector<T>& _X,
		const Vector<T>& _Y)
	{return (!(_X == _Y)); }
template<class T> inline
	bool operator<(const Vector<T>& _X,
		const Vector<T>& _Y)
	{return (_X._Lt(_Y)); }
template<class T> inline
	bool operator>(const Vector<T>& _X,
		const Vector<T>& _Y)
	{return (_Y < _X); }
template<class T> inline
	bool operator<=(const Vector<T>& _X,
		const Vector<T>& _Y)
	{return (!(_Y < _X)); }
template<class T> inline
	bool operator>=(const Vector<T>& _X,
		const Vector<T>& _Y)
	{
    return (!(_X < _Y)); 
    }
*/

/***************************************************************************************\

Function:       Vector<ValueType>::Reserve

Description:    Reserves a minimum length of storage for a vector object, allocating 
                space if necessary.

Parameters:     ulReserveSize   The minimum length of storage to be allocated for the vector.

Return Value:   BOOL    Return TRUE if allocation/insertion succed, FALSE otherwise.

Comments:       Increases the capacity of self in anticipation of adding new elements. 
                Reserve itself does not add any new elements. After a call to Reserve, 
                Capacity() is greater than or equal to ulReserveSize and subsequent 
                insertions will not cause a reallocation until the size of the vector 
                exceeds ulReserveSize. Reallocation does not occur if ulReserveSize is 
                less than Capacity(). If reallocation does occur, then all iterators and 
                references pointing to elements in the vector are invalidated. reserve 
                takes at most linear time in the size of self.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
BOOL Vector<ValueType, AllocatorType>::Reserve(SizeType ulReserveSize)
{
    if (Capacity() < ulReserveSize)
    {
        // Allocate a new Vector.
        Iterator itStartOfNewVector = m_oAllocator.allocate(ulReserveSize, (void *)0);  //(Iterator)operator new (ulReserveSize * sizeof(ValueType));
        if(itStartOfNewVector == NULL)
            return FALSE;
        // Copy the current Vector into the new one.
        Copy(m_itFirst, m_itLast, itStartOfNewVector);
        // Destroy all current Vector elements(call the destructor).
        Destroy(m_itFirst, m_itLast);
        // Deallocate the memory of current Vector.
        m_oAllocator.deallocate(m_itFirst, m_itEnd - m_itFirst);  //operator delete(m_itFirst);
        // Set New vector as current.
        m_itEnd = itStartOfNewVector + ulReserveSize;
        m_itLast = itStartOfNewVector + Size();
        m_itFirst = itStartOfNewVector; 
    }
    return TRUE;
}

/***************************************************************************************\

Function:       Vector<ValueType>::Resize

Description:    Specifies a new size for a vector.

Parameters:     ulNewSize   The new size of the vector. 
                oValue      The value of new elements added to the vector if the new size 
                            is larger that the original size. If the value is omitted, 
                            the new objects are assigned the default value. 

Return Value:   BOOL    Return TRUE if allocation/insertion succed, FALSE otherwise.

Comments:       If the container's Size is less than the requested size, ulNewSize, 
                elements are added to the vector until it reaches the requested size. 
                If the container's size is larger than the requested size, the elements 
                closest to the end of the container are deleted until the container 
                reaches the size ulNewSize. If the present size of the container is the 
                same as the requested size, no action is taken.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
BOOL Vector<ValueType, AllocatorType>::Resize(SizeType ulNewSize, const ValueType& oValue)
{
    if (Size() < ulNewSize)
    {
        if(!Insert(End(), ulNewSize - Size(), oValue))
            return FALSE;
    }
	else if (ulNewSize < Size())
		Erase(Begin() + ulNewSize, End());

    return TRUE;
}

/***************************************************************************************\

Function:       Vector<ValueType>::Insert

Description:    Inserts ulNumberOfCopies copies of the element into the list before the 
                specified position.

Parameters:     itPosition          The position in the vector where the first element is inserted. 
                ulNumberOfCopies    The number of elements(copies) being inserted into the vector. 
                oValue              The value of the element being inserted into the vector.

Return Value:   BOOL                Return TRUE if allocation/insertion succed, FALSE otherwise.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
BOOL Vector<ValueType, AllocatorType>::Insert(Iterator itPosition, SizeType ulNumberOfCopies, const ValueType& oValue)
{
    if (SizeType(m_itEnd - m_itLast) < ulNumberOfCopies)
	{
        // Not enough free place in current vector: need to allocate a new one.
        // Evaluate the new size.
        SizeType ulNewSize = Size() + (ulNumberOfCopies < Size() ? Size() : ulNumberOfCopies);
        // Allocate the new vector.
		Iterator itStartOfNewVector = m_oAllocator.allocate(ulNewSize, (void *)0);  //(Iterator)operator new (ulNewSize * sizeof(ValueType));
        // Validate the allocation of new memory.
        if(itStartOfNewVector == NULL)
            return FALSE;
        // copy the first part of current vector into new vector.
		Iterator itInsertHere = Copy(m_itFirst, itPosition, itStartOfNewVector);
        // Insert new elements.
        FillSize(itInsertHere, ulNumberOfCopies, oValue);
        // Copy the last part of the current vector into the new one.
        Copy(itPosition, m_itLast, itInsertHere + ulNumberOfCopies);
        // destroy the current vector.
        Destroy(m_itFirst, m_itLast);
        m_oAllocator.deallocate(m_itFirst, m_itEnd - m_itFirst);   //operator delete(m_itFirst);
        // Set new vector as current.
        m_itEnd = itStartOfNewVector + ulNewSize;
        m_itLast = itStartOfNewVector + Size() + ulNumberOfCopies;
        m_itFirst = itStartOfNewVector; 
    }
    else if (SizeType(m_itLast - itPosition) < ulNumberOfCopies)
    {
        Copy(itPosition, m_itLast, itPosition + ulNumberOfCopies);
        FillSize(m_itLast, ulNumberOfCopies - (m_itLast - itPosition), oValue);
        FillRange(itPosition, m_itLast, oValue);
        m_itLast += ulNumberOfCopies; }
    else if (0 < ulNumberOfCopies)
    {
        Copy(m_itLast - ulNumberOfCopies, m_itLast, m_itLast);
        CopyBackward(itPosition, m_itLast - ulNumberOfCopies, m_itLast);
        FillRange(itPosition, itPosition + ulNumberOfCopies, oValue);
        m_itLast += ulNumberOfCopies; 
    }
    return TRUE;
}

/***************************************************************************************\

Function:       Vector<ValueType>::Insert

Description:    Inserts copies of the elements in the range [first, last] before itPosition.

Parameters:     itPosition  The position in the vector where the first element is inserted.
                cpFirst     The position of the first element in the range of elements to be copied.
                cpLast      The position of the first element beyond the range of elements to be copied.

Return Value:   BOOL        Return TRUE if allocation/insertion succed, FALSE otherwise.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
BOOL Vector<ValueType, AllocatorType>::Insert(Iterator itPosition, ConstPointer cpFirst, ConstPointer cpLast)
{
    SizeType ulInsertSize = 0;
    ulInsertSize += cpLast - cpFirst; // _Distance(cpFirst, cpLast, ulInsertSize);
    if (SizeType(m_itEnd - m_itLast) < ulInsertSize)
    {
        // Not enough free place in current vector: need to allocate a new one.
        // Evaluate the new size.
        SizeType ulNewSize = Size() + (ulInsertSize < Size() ? Size() : ulInsertSize);
        // Allocate the new vector.
        Iterator itStartOfNewVector = m_oAllocator.allocate(ulNewSize, (void *)0);  //(Iterator)operator new (ulNewSize * sizeof(ValueType));
        // Validate the allocation of new memory.
        if(itStartOfNewVector == NULL)
            return FALSE;
        // copy the first part of current vector into new vector.
        Iterator itInsertHere = Copy(m_itFirst, itPosition, itStartOfNewVector);
        // Copy new elements.
        itInsertHere = Copy(cpFirst, cpLast, itInsertHere);
        // Copy the last part of the current vector into the new one.
        Copy(itPosition, m_itLast, itInsertHere);
        // destroy the current vector.
        Destroy(m_itFirst, m_itLast);
        m_oAllocator.deallocate(m_itFirst, m_itEnd - m_itFirst); //operator delete(m_itFirst);
        // Set new vector as current.
        m_itEnd = itStartOfNewVector + ulNewSize;
        m_itLast = itStartOfNewVector + Size() + ulInsertSize;
        m_itFirst = itStartOfNewVector; 
    }
    else if (SizeType(m_itLast - itPosition) < ulInsertSize)
    {
        Copy(itPosition, m_itLast, itPosition + ulInsertSize);
        Copy(cpFirst + (m_itLast - itPosition), cpLast, m_itLast);
        CopyForward(cpFirst, cpFirst + (m_itLast - itPosition), itPosition);
        m_itLast += ulInsertSize; 
    }
    else if (0 < ulInsertSize)
    {
        Copy(m_itLast - ulInsertSize, m_itLast, m_itLast);
        CopyBackward(itPosition, m_itLast - ulInsertSize, m_itLast);
        CopyForward(cpFirst, cpLast, itPosition);
        m_itLast += ulInsertSize; 
    }
    return TRUE;
}

/***************************************************************************************\

Function:       Vector<ValueType>::Erase

Description:    Deletes the vector element pointed to by the iterator position. Returns 
                an iterator pointing to the element following the deleted element, 
                or end() if the deleted element was the last one in this vector.

Parameters:     itPosition  Position of the element to be removed from the vector. 

Return Value:   Iterator.

Comments:       None.
          
\***************************************************************************************/
template <class ValueType, class AllocatorType>
Vector<ValueType, AllocatorType>::Iterator Vector<ValueType, AllocatorType>::Erase(Iterator itPosition)
{
    CopyForward(itPosition + 1, End(), itPosition);
    Destroy(m_itLast - 1, m_itLast);
    --m_itLast;
    return (itPosition); 
}

/***************************************************************************************\

Function:       Vector<ValueType>::Erase

Description:    Deletes the vector elements in the range (first, last). Returns an 
                iterator pointing to the element following the last deleted element, 
                or end() if there were no elements in the deleted range.

Parameters:     itFirst     Position of the first element removed from the vector. 
                itLast      Position just beyond the last element removed from the vector.

Return Value:   Vector<ValueType>::Iterator.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
Vector<ValueType, AllocatorType>::Iterator Vector<ValueType, AllocatorType>::Erase(Iterator itFirst, Iterator itLast)
{
    Iterator itStart = CopyForward(itLast, End(), itFirst);
    Destroy(itStart, End());
    m_itLast = itStart;
    return (itFirst); 
}

MTLEND

#ifdef TESTWITHSTD

#undef Vector

#undef ValueType       
#undef Pointer         
#undef ConstPointer    
#undef Reference       
#undef ConstReference  
#undef SizeType        
#undef DifferenceType  

#undef Iterator     
#undef ConstIterator
#undef ReverseIterator
#undef ConstReverseIterator

#undef Assign
#undef Back     
#undef Begin
#undef Capacity  
#undef Clear  
#undef Empty    
#undef End      
#undef Erase    
#undef Front    
#undef Insert   
#undef MaxSize
#undef PushBack 
#undef PopBack  
#undef RBegin
#undef REnd
#undef Reserve  
#undef Resize
#undef Size 
    
#endif

#undef MTLBEGIN
#undef MTLEND


#endif  // #ifndef INC_MTLVECTOR_H
